package gov.va.nvap.web.util.file;

import java.io.BufferedReader;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.DataFormatter;
import org.apache.poi.ss.usermodel.FormulaEvaluator;
import org.apache.poi.ss.usermodel.RichTextString;
import org.apache.poi.ss.usermodel.Row;
import org.apache.poi.xssf.usermodel.XSSFFormulaEvaluator;
import org.apache.poi.xssf.usermodel.XSSFSheet;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;

/**
 * @author Nusrath Mohammed
 *
 * Helper class to import Excel and CSV files containing patients.
 *
 */
public class FileUploader {

    static private final Log LOG = LogFactory.getLog(FileUploader.class);

    /**
     * This method parses CSV file with extension .csv
     *
     * @param csvFile
     * @return List of Map Objects
     * @throws RuntimeException
     */
    public static List<Map<String, String>> readCSV(FileItem fileItem) {

        BufferedReader br = null;
        String line = "";
        String cvsSplitBy = ",";

        List<Map<String, String>> mapList = new ArrayList<Map<String, String>>();
        List<String> headers = new ArrayList<String>();

        try {
            int rowNum = 0;
            br = new BufferedReader(new InputStreamReader(fileItem.getInputStream()));
            while ((line = br.readLine()) != null) {
                Map<String, String> patientInfo = new HashMap<String, String>();
                // use comma as separator
                String[] cells = line.split(cvsSplitBy);

                if (rowNum == 0) {
                    for (String pd : cells) {
                        headers.add(pd);
                    }

                    rowNum++;
                    continue;
                }

				// Get the map containing positions of the three columns (SSN, first name, and last name)
                Map<String, Integer> columnPositions = getColumnPositions(headers);

                Iterator<String> columnsIterator = columnPositions.keySet()
                        .iterator();
                while (columnsIterator.hasNext()) {
                    String name = columnsIterator.next();
                    String value = cells[columnPositions.get(name)];

                    patientInfo.put(name, value);
                }

                mapList.add(new HashMap<String, String>(patientInfo));
            }
        } catch (FileNotFoundException e1) {
            throw new RuntimeException("File you are trying to upload could not be found. Please try again.");
        } catch (IOException e2) {
            throw new RuntimeException("File you are trying to upload could not be read. Please try again.");
        } finally {
            if (br != null) {
                try {
                    br.close();
                } catch (IOException e3) {
                    throw new RuntimeException("There was an error processing the file you are trying to upload. Please try again.");
                }
            }
        }

        return mapList;
    }

    /**
     * This method parse Excel file with extensions xls and xlsx
     *
     * @param String fileName
     * @throws RuntimeException
     */
    public static List<Map<String, String>> readExcel(FileItem fileItem) {

        XSSFWorkbook workbook = null;
		// all key value pair mapping ( holding column name and its respective value per row.
        List<Map<String, String>> mapList = new ArrayList<Map<String, String>>();
        InputStream in = null;
        try {
            // Get the workbook instance for XLS file
        	in = fileItem.getInputStream();
            workbook = new XSSFWorkbook(in);

            DataFormatter objDefaultFormat = new DataFormatter();
            FormulaEvaluator objFormulaEvaluator = new XSSFFormulaEvaluator((XSSFWorkbook) workbook);

            // Get first sheet from the workbook
            XSSFSheet sheet = workbook.getSheetAt(0);

            // GET the header row
            Row headerRow = sheet.getRow(0);

            // LIst of headers from excel
            List<String> headers = new ArrayList<String>();
            Iterator<Cell> cells = headerRow.cellIterator();
            while (cells.hasNext()) {
                Cell cell = (Cell) cells.next();
                RichTextString value = cell.getRichStringCellValue();
                headers.add(value.getString());
            }

            // There must be at least 3 headers.
            if (headers.size() < 3) {
                mapList = null;
                return mapList;
            }

            // Get the map containing positions of the three columns (SSN, first name, and last name)
            Map<String, Integer> columnPositions = getColumnPositions(headers);

            // Get iterator to all the rows in current sheet
            int rowNumber = 0;
            Iterator<Row> rowIterator = sheet.iterator();
            while (rowIterator.hasNext()) {
                Row row = (Row) rowIterator.next();
                if (rowNumber == 0) {
                    rowNumber++;
                    continue;
                }

                Map<String, String> patientInfo = new HashMap<String, String>();

                Iterator<String> columnsIterator = columnPositions.keySet().iterator();
                while (columnsIterator.hasNext()) {
                    String name = columnsIterator.next();
                    Cell cellValue = row.getCell(columnPositions.get(name));
                    // This will evaluate the cell, And any type of cell will return string value
                    objFormulaEvaluator.evaluate(cellValue);

                    String cellValueStr = objDefaultFormat.formatCellValue(cellValue, objFormulaEvaluator);

                    patientInfo.put(name, cellValueStr);
                }
                mapList.add(patientInfo);
            }

        } catch (FileNotFoundException e1) {
            throw new RuntimeException("File you are trying to upload could not be found. Please try again.");
        } catch (IOException e2) {
            throw new RuntimeException("File you are trying to upload could not be read. Please try again.");
        } finally {
            if (workbook != null) {
                try {
                    workbook.close();
                } catch (Exception e3) {
                    throw new RuntimeException("There was an error processing the file you are trying to upload. Please try again.");
                }
            }
            if (in != null) {
                try {
                    in.close();
                } catch (Exception e3) {
                    throw new RuntimeException("There was an error processing the file you are trying to upload. Please try again.");
                }
            }
        }

        return mapList;
    }

    /**
     * Gets the map containing positions of the three columns (SSN, first name, and last name)
     *
     * @param headers
     * @return Map
     */
    private static Map<String, Integer> getColumnPositions(List<String> headers) {
        Map<String, Integer> positions = new HashMap<String, Integer>();
        int i = 0;

        for (String columnName : headers) {
        	
            if (columnName.toLowerCase().contains("ssn")) {
                positions.put("ssn", i);
            } else if (columnName.toLowerCase().contains("first")) {
                positions.put("firstname", i);
            } else if (columnName.toLowerCase().contains("last")) {
                positions.put("lastname", i);
            }
            i++;
        }

        return positions;
    }
}
