package gov.va.nvap.web.util;

import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.ConsentDirectiveReferenceType;
import gov.va.nvap.web.util.date.DateUtil;

import java.text.MessageFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.Map;
import java.util.PropertyResourceBundle;
import java.util.ResourceBundle;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;

import org.apache.commons.validator.Arg;
import org.apache.commons.validator.Field;
import org.apache.commons.validator.GenericValidator;
import org.apache.commons.validator.ValidatorAction;
import org.apache.commons.validator.util.ValidatorUtils;

/**
 * Validator for checking fields.
 * 
 * @author David Vazquez Modified By: Asha Amritraj
 */
public class FieldChecks {
	/**
	 * Add the list of error messages to the list.
	 */
	static private void addMessage(final Field field,
			final ValidatorAction action,
			final ArrayList<ValidatorMessage> messages) {
		final PropertyResourceBundle prb = (PropertyResourceBundle) ResourceBundle
				.getBundle("gov.va.nvap.web.application");
		final Arg[] args = field.getArgs("");
		final String[] messageArgs = new String[args.length];
		int i = 0;
		for (final Arg arg : args) {
			messageArgs[i++] = prb.getString(arg.getKey());
		}
		final ValidatorMessage message = new ValidatorMessage();
		message.setProperty(field.getProperty());
		message.setMessage(MessageFormat.format(prb.getString(action.getMsg()),
				(Object[]) messageArgs));
		messages.add(message);
	}

	/**
	 * Validate date pattern.
	 */
	static public boolean validateDate(final Object bean, final Field field,
			final ValidatorAction action, final HttpServletRequest request,
			final ArrayList<ValidatorMessage> messages) {
		boolean ret;
		final String value = ValidatorUtils.getValueAsString(bean,
				field.getProperty());
		final String datePattern = field.getVarValue("datePattern");

		ret = GenericValidator.isBlankOrNull(value)
				|| (!GenericValidator.isBlankOrNull(datePattern) && GenericValidator
						.isDate(value, datePattern, false));

		if (!ret) {
			FieldChecks.addMessage(field, action, messages);
		}

		return ret;
	}

	/**
	 * Validate if the start and end dates are correct.
	 */
	static public boolean validateDateAfter(final Object bean,
			final Field field, final ValidatorAction action,
			final HttpServletRequest request, final ArrayList messages) {
		boolean ret = false;
		final String value = ValidatorUtils.getValueAsString(bean,
				field.getProperty());
		final String datePattern = field.getVarValue("datePattern");
		final String dateField = field.getVarValue("dateField");

		if (!GenericValidator.isBlankOrNull(value)) {
			if (!GenericValidator.isBlankOrNull(datePattern)
					&& !GenericValidator.isBlankOrNull(dateField)) {
				final String dateFieldValue = ValidatorUtils.getValueAsString(
						bean, dateField);
				if (!GenericValidator.isBlankOrNull(dateFieldValue)) {
					if (GenericValidator.isDate(value, datePattern, false)
							&& GenericValidator.isDate(dateFieldValue,
									datePattern, false)) {
						final SimpleDateFormat formatter = new SimpleDateFormat(
								datePattern);
						try {
							final Date d1 = formatter.parse(value);
							final Date d2 = formatter.parse(dateFieldValue);
							// Added check not to return error if the start/end
							// dates are the same
							if (d1.equals(d2)) {
								ret = true;
							} else {
								ret = d1.after(d2);
							}
						} catch (final Throwable t) {
						}
					}
				} else {
					ret = true;
				}
			}
		} else {
			ret = true;
		}

		if (!ret) {
			FieldChecks.addMessage(field, action, messages);
		}

		return ret;
	}

	static public boolean validateDateAfterAuthorizationAndBeforeExpiration(
			final Object bean, final Field field, final ValidatorAction action,
			final HttpServletRequest request, final ArrayList messages) {
		boolean ret = false;
		final String value = ValidatorUtils.getValueAsString(bean,
				field.getProperty());
		final String consentType = ValidatorUtils.getValueAsString(bean,
				"consentType");
		final String datePattern = field.getVarValue("datePattern");

		if ("NwHIN Organization Restriction Revocation".equals(consentType)) {
			return true;
		}

		if (!GenericValidator.isBlankOrNull(value)) {
			if (!GenericValidator.isBlankOrNull(datePattern)) {
				final SimpleDateFormat formatter = new SimpleDateFormat(
						datePattern);
				try {
					final Date d1 = formatter.parse(value);
					final Date fieldDate = DateUtil.voidTime(d1);
					final HttpSession session = request.getSession(false);
					final ConsentDirectiveReferenceType reference = (ConsentDirectiveReferenceType) getConsentDirectiveType(
							session, consentType);
					if (NullChecker.isNotEmpty(reference)) {
						Date authorizationDate = reference.getOptinDate();
						authorizationDate = DateUtil
								.voidTime(authorizationDate);
						Date expirationDate = reference.getExpirationDate();
						expirationDate = DateUtil.voidTime(expirationDate);
						if (fieldDate.after(authorizationDate)
								&& fieldDate.before(expirationDate)) {
							ret = true;
						} else if (fieldDate.equals(authorizationDate)) {
							ret = true;
						} else {
							ret = false;
						}
					}

				} catch (final ParseException ex) {
				}
			}
		} else {
			ret = true;
		}

		if (!ret) {
			FieldChecks.addMessage(field, action, messages);
		}
		return ret;
	}

	static public ConsentDirectiveReferenceType getConsentDirectiveType(
			HttpSession session, String consentType) {
		Object consentDirectiveReferenceObj = null;
		if ("SSA Revocation".equals(consentType)) {
			consentDirectiveReferenceObj = session
					.getAttribute("ssaConsentDirective");
		} else if ("NwHIN Revocation".equals(consentType)) {
			consentDirectiveReferenceObj = session
					.getAttribute("consentDirective");
		} else if ("NwHIN Organization Restriction Revocation"
				.equals(consentType)) {
			consentDirectiveReferenceObj = session
					.getAttribute("nwhinOrganizationRestrictionsDirective");
		}
		if (ConsentDirectiveReferenceType.class
				.isInstance(consentDirectiveReferenceObj)) {
			final ConsentDirectiveReferenceType reference = (ConsentDirectiveReferenceType) consentDirectiveReferenceObj;
			return reference;
		}
		return null;
	}

	static public boolean validateDateAfterAuthorization(final Object bean,
			final Field field, final ValidatorAction action,
			final HttpServletRequest request, final ArrayList messages) {
		boolean ret = false;
		final String value = ValidatorUtils.getValueAsString(bean,
				field.getProperty());
		final String consentType = ValidatorUtils.getValueAsString(bean,
				"consentType");
		final String datePattern = field.getVarValue("datePattern");

		if (!GenericValidator.isBlankOrNull(value)) {
			if (!GenericValidator.isBlankOrNull(datePattern)) {
				final SimpleDateFormat formatter = new SimpleDateFormat(
						datePattern);
				try {
					final Date d1 = formatter.parse(value);
					final Date fieldDate = DateUtil.voidTime(d1);
					final HttpSession session = request.getSession(false);
					final ConsentDirectiveReferenceType reference = (ConsentDirectiveReferenceType) getConsentDirectiveType(
							session, consentType);
					if (NullChecker.isNotEmpty(reference)) {
						Date authorizationDate = reference.getOptinDate();
						authorizationDate = DateUtil
								.voidTime(authorizationDate);
						if (fieldDate.after(authorizationDate)) {
							ret = true;
						} else if (fieldDate.equals(authorizationDate)) {
							ret = true;
						} else {
							ret = false;
						}
					}

				} catch (final ParseException ex) {
				}
			}
		} else {
			ret = true;
		}

		if (!ret) {
			FieldChecks.addMessage(field, action, messages);
		}
		return ret;
	}

	/**
	 * Validate if the date is before current date.
	 */
	static public boolean validateDateBeforeNow(final Object bean,
			final Field field, final ValidatorAction action,
			final HttpServletRequest request, final ArrayList messages) {
		boolean ret = false;
		final String value = ValidatorUtils.getValueAsString(bean,
				field.getProperty());
		final String datePattern = field.getVarValue("datePattern");

		if (!GenericValidator.isBlankOrNull(value)) {
			if (!GenericValidator.isBlankOrNull(datePattern)) {
				final SimpleDateFormat formatter = new SimpleDateFormat(
						datePattern);
				try {
					final Date d1 = formatter.parse(value);
					final Date fieldDate = DateUtil.voidTime(d1);
					final Date now = DateUtil.voidTime(new Date());
					if (fieldDate.before(now) || fieldDate.equals(now)) {
						ret = true;
					}
				} catch (final ParseException ex) {
				}
			}
		} else {
			ret = true;
		}

		if (!ret) {
			FieldChecks.addMessage(field, action, messages);
		}
		return ret;
	}

	static public boolean validateDateWithinFiveYears(final Object bean,
			final Field field, final ValidatorAction action,
			final HttpServletRequest request, final ArrayList messages) {
		final String consentType = ValidatorUtils.getValueAsString(bean,
				"consentType");
		if (!"NwHIN Authorization".equals(consentType) && 
				!"NwHIN Organization Restriction Modification".equals(consentType) && 
				!"NwHIN Organization Restriction Authorization".equals(consentType)) {
			return true;
		}
		
		return FieldChecks.validateDateWithinNYears(bean, field, action,
				request, messages, -5);
		
	}

	static public boolean validateOrganizationRequired(final Object bean,
			final Field field, final ValidatorAction action,
			final HttpServletRequest request, final ArrayList messages) {
		final String consentType = ValidatorUtils.getValueAsString(bean,
				"consentType");
		
		if (!"NwHIN Organization Restriction Modification".equals(consentType) && 
				!"NwHIN Organization Restriction Authorization".equals(consentType)) {
			return true;
		}
		
		return FieldChecks.validateRequired(bean, field, action,
			request, messages);
		
	}
	
	static public boolean validateDateWithinNYears(final Object bean,
			final Field field, final ValidatorAction action,
			final HttpServletRequest request, final ArrayList messages,
			int lookbackYears) {
		boolean ret = false;
		final String value = ValidatorUtils.getValueAsString(bean,
				field.getProperty());
		final String datePattern = field.getVarValue("datePattern");

		if (!GenericValidator.isBlankOrNull(value)) {
			if (!GenericValidator.isBlankOrNull(datePattern)) {
				final SimpleDateFormat formatter = new SimpleDateFormat(
						datePattern);
				try {
					final Date d1 = formatter.parse(value);
					final Date fieldDate = DateUtil.voidTime(d1);
					final Date now = DateUtil.voidTime(new Date());
					final Calendar dateCal = Calendar.getInstance();
					dateCal.setTime(now);
					// End date should be 5 years after begin date
					dateCal.add(Calendar.YEAR, lookbackYears);
					dateCal.add(Calendar.HOUR, 24);
					final Date fiveYearBefore = dateCal.getTime();

					// the other validateDateBeforeNow will catch the date being
					// before today or today
					if (fieldDate.before(fiveYearBefore)) {
						ret = false;
					} else {
						ret = true;
					}

				} catch (final ParseException ex) {
				}
			}
		} else {
			ret = true;
		}

		if (!ret) {
			FieldChecks.addMessage(field, action, messages);
		}
		return ret;

	}

	static public boolean validateDateWithinTwoYears(final Object bean,
			final Field field, final ValidatorAction action,
			final HttpServletRequest request, final ArrayList messages) {
		final String consentType = ValidatorUtils.getValueAsString(bean,
				"consentType");
		if (!"SSA Authorization".equals(consentType)) {
			return true;
		}
		return FieldChecks.validateDateWithinNYears(bean, field, action,
				request, messages, -2);
	}

	/**
	 * Validate SSN field.
	 */
	static public boolean validateMask(final Object bean, final Field field,
			final ValidatorAction action, final HttpServletRequest request,
			final ArrayList messages) {
		boolean ret;
		final String value = ValidatorUtils.getValueAsString(bean,
				field.getProperty());
		final String mask = field.getVarValue("mask");

		ret = GenericValidator.isBlankOrNull(value)
				|| (!GenericValidator.isBlankOrNull(mask) && GenericValidator
						.matchRegexp(value, mask));

		if (!ret) {
			FieldChecks.addMessage(field, action, messages);
		}

		return ret;
	}

	/**
	 * Validate request id associated with the session.
	 */
	static public boolean validateRequestId(final Object bean,
			final Field field, final ValidatorAction action,
			final HttpServletRequest request, final ArrayList messages) {
		boolean ret = true;
		final String value = ValidatorUtils.getValueAsString(bean,
				field.getProperty());
		final HttpSession session = request.getSession(false);
		final String sessionAttribute = field.getVarValue("sessionAttribute");

		if (session != null) {
			final String requestId = (String) session
					.getAttribute(sessionAttribute);
			ret = !GenericValidator.isBlankOrNull(value)
					&& value.equals(requestId);
		}

		if (!ret) {
			FieldChecks.addMessage(field, action, messages);
		}

		return ret;
	}

	/**
	 * Validate required fields.
	 */
	static public boolean validateRequired(final Object bean,
			final Field field, final ValidatorAction action,
			final HttpServletRequest request, final ArrayList messages) {
		boolean ret;
		final String value = ValidatorUtils.getValueAsString(bean,
				field.getProperty());

		ret = !GenericValidator.isBlankOrNull(value);

		if (!ret) {
			FieldChecks.addMessage(field, action, messages);
		}

		return ret;
	}

	/**
	 * Checks fields if they are not null or empty for jsp, if so, replace with
	 * non-deliminated space for aesthetics
	 */
	static public Map<String, Object> replaceEmptyOrNullWithSpace(
			final Map<String, Object> inList) {
		// iterate over all keys, check value of each key, looking for null or
		// empty
		// if find one, replace with non-deliminated space
		Map<String, Object> outList = new HashMap<String, Object>();
		Iterator<String> i = inList.keySet().iterator();
		String tempKey;
		while (i.hasNext()) {
			tempKey = i.next();
			if (NullChecker.isNullOrEmpty(inList.get(tempKey))) {
				outList.put(tempKey, " ");
			} else {
				outList.put(tempKey, inList.get(tempKey));
			}
		}
		return outList;
	}

}
