package gov.va.nvap.web.report;

import gov.va.med.nhin.adapter.audit.ActionType;
import gov.va.med.nhin.adapter.audit.ActionValuesType;
import gov.va.med.nhin.adapter.audit.AuditSummariesType;
import gov.va.med.nhin.adapter.audit.AuditSummaryType;
import gov.va.med.nhin.adapter.audit.FieldType;
import gov.va.med.nhin.adapter.audit.GetAuditsSummary;
import gov.va.med.nhin.adapter.audit.GetAuditsSummaryResponse;
import gov.va.med.nhin.adapter.audit.GroupByFieldsType;
import gov.va.med.nhin.adapter.audit.StringValuesType;
import gov.va.med.nhin.adapter.audit.SummaryFieldsType;
import gov.va.nvap.common.date.GregorianDateUtil;
import gov.va.nvap.common.sort.BubbleSortListMap;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.OrganizationType;
import gov.va.nvap.service.adapter.audit.AdapterAuditManager;
import gov.va.nvap.service.audit.AuditException;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.helper.document.DocumentHelper;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.util.xls.ExcelExporter;
import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.ejb.EJB;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.poi.ss.usermodel.Workbook;
import org.json.JSONException;
import org.json.JSONObject;

public class ReceivedNwHINDocumentSummaryReport extends ResponseDispatcherHttpServlet {
	
    private static final long serialVersionUID = 3721503355079491829L;
    private AdapterAuditManager adapterAuditManager;

	@Override
	protected void unspecified(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		final HttpSession session = request.getSession(false);
		session.removeAttribute("results");
        ReportHelper.setDefaultSearchDates(session);
		request.setAttribute("facilities", this.getFacilityHelper().getAllVistAFacilities());
        this.getReportHelper().getReceivedDocsOrgLists(request);
		
		this.forward(request, response, "show");
	}
    
    @EJB(beanInterface = AdapterAuditManager.class, mappedName = "AdapterAuditManager")
	public void setAdapterAuditManager(
			final AdapterAuditManager adapterAuditManager) {
		this.adapterAuditManager = adapterAuditManager;
	}
    
    /**
	 * Get the consent management helper from Spring.
	 */
	public ConsentManagementHelper getCmsHelper() {
		final ConsentManagementHelper cmsHelper = this.getBean("cmsHelper",
				ConsentManagementHelper.class);
		return cmsHelper;
	}

	/**
	 * Get the document helper from Spring.
	 */
	public DocumentHelper getDocumentHelper() {
		return this.getBean("adapterDocumentHelper", DocumentHelper.class);
	}

	/**
	 * Get the excel exporter class from Spring.
	 * 
	 * @return the excel exporter object
	 */
	public ExcelExporter getExcelExporter() {
		final ExcelExporter excelExporter = this.getBean("excelExporter",
				ExcelExporter.class);
		return excelExporter;
	}

	/**
	 * Get the facility helper from Spring.
	 */
	public FacilityHelper getFacilityHelper() {
		return this.getBean("facilityHelper", FacilityHelper.class);
	}

	public ReportHelper getReportHelper() {
		final ReportHelper reportHelper = this.getBean("reportHelper",
				ReportHelper.class);
		return reportHelper;
	}
    
    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }
    
    
    public void doSearch(final HttpServletRequest request,
			final HttpServletResponse response) throws IOException, ServletException{
        final HttpSession session = request.getSession(false);

        session.setAttribute("results", encodeIntoJSON(getResults(request, false)));

        this.forward(request, response, "searchResultsJSON");
        
    }
    
    /**
     * Returns search results of request parameter or default values if blank. For excel calls adds oid value to result map to be output on
     * excel download
     * @param request
     * @param calledForExcel
     * @return
     * @throws AuditException 
     */
    private SearchResultsPackage getResults(final HttpServletRequest request, Boolean calledForExcel) throws AuditException {
		
		// prepare audit summary and retrieve web services response
		GetAuditsSummaryResponse getAuditsSummaryResponse = prepareAuditSummary(request);
		
		final List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
		long totalCount = 0l;
		final AuditSummariesType auditSummariesType = getAuditsSummaryResponse
				.getAuditSummaries();

        List<AuditSummaryType> auditSummary = new ArrayList<AuditSummaryType>();
		if (auditSummariesType != null) {
			auditSummary = auditSummariesType.getAuditSummary();
		}
		for (final AuditSummaryType auditSummaryType2 : auditSummary) {
			final AuditSummaryType auditSummaryType = auditSummaryType2;

			final Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
			final long count = auditSummaryType.getCount();
			final SummaryFieldsType summaryFields = auditSummaryType
					.getSummaryFields();
			final List<String> summaryField = summaryFields.getSummaryField();

			if (NullChecker.isEmpty(summaryField)) {
				continue;
			}

			if (summaryField.size() < 4) {
				continue;
			}

			OrganizationType organizationType = null;
			if (NullChecker.isNotEmpty(summaryField.get(0))) {
				organizationType = (this.getCmsHelper()
						.getOrganizationByHomeCommunityId(summaryField.get(0)
								.replace("urn:oid:", "")));
                
                //adds oid to result map to be added to excel export, not included in web output
                if(calledForExcel){
                    if (!NullChecker.isNullOrEmpty(organizationType)) {
                    resultMap.put("oid", ReportDataProcessor.nullEmptyReplaceWithUnknown(organizationType.getOrgOid()));
                    } else {
                        resultMap.put("oid", "Unknown");
                    }
                }
            }
            
			if (!NullChecker.isNullOrEmpty(organizationType)) {
				resultMap.put("partnerOrg", ReportDataProcessor
						.nullEmptyReplaceWithUnknown(organizationType
								.getOrgName()));
			} else {
				resultMap.put("partnerOrg", ReportDataProcessor
						.nullEmptyReplaceWithUnknown(summaryField.get(3)));
			}
			// Extract value
			final String adapterVAFacility = summaryField.get(1);
            
			String vaFacility = "";
			if (ReportDataProcessor.isValidStation(adapterVAFacility)) {
				vaFacility = this.getFacilityHelper()
						.getFacilityNameByStationId(adapterVAFacility);
				if (NullChecker.isEmpty(vaFacility)) {
					vaFacility = summaryField.get(2);
				}
			} else {
				vaFacility = summaryField.get(2);
			}
			resultMap.put("vaFacility",
					ReportDataProcessor.fixStation(vaFacility));
            if (!NullChecker.isNullOrEmpty(adapterVAFacility)) {
                resultMap.put("vaFacilityNumber", ReportDataProcessor
                        .nullEmptyReplaceWithUnknown(adapterVAFacility));
            }
                        
			resultMap.put("total", count);
			results.add(resultMap);
			totalCount += count;

		}
        //sort result list and combine into transport package
        SearchResultsPackage packagedResults = new SearchResultsPackage( sortResults(results,request), totalCount);
        
		return packagedResults;
	}
    
    /**
     * Sort data using table specific columns
     * @param unSortedData
     * @param request
     * @return 
     */
    private List<Map<String,Object>> sortResults(List<Map<String,Object>> unSortedData, HttpServletRequest request){
        final BubbleSortListMap bSorter = new BubbleSortListMap();
        String sortValue = request.getParameter("order[0][column]") == null ? request.getParameter("sortBy") : request.getParameter("order[0][column]");
        final boolean sortAscending = request.getParameter("order[0][dir]") == null ? !"desc".equalsIgnoreCase(request.getParameter("sortOrder")) :
                        !"desc".equalsIgnoreCase(request.getParameter("order[0][dir]"));
        
        
        if("0".equals(sortValue)){
            sortValue = "partnerOrg";
        } else if ("1".equals(sortValue)){
            sortValue = "vaFacility";
        } else if ("2".equals(sortValue)){
            sortValue = "vaFacilityNumber";
        } else if ("3".equals(sortValue)){
            sortValue = "total";
        } else {
            sortValue = "partnerOrg";
        }
        return bSorter.sortByColumn(unSortedData,sortValue, sortAscending);
    }
    
    
    /**
     * Adds all request parameters to AuditSummary and calls for response
     * @param request
     * @return
     * @throws AuditException 
     */
    private GetAuditsSummaryResponse prepareAuditSummary(final HttpServletRequest request) throws AuditException{

		final Date sd = ReportHelper.getStartDate(request.getParameter("startDate"));
		final Date ed = ReportHelper.getEndDate(request.getParameter("endDate"));
        final String userId = request.getParameter("userId");
                
        final int patientTypes = (request.getParameter("patientTypes") != null)
                ? Integer.parseInt(request.getParameter("patientTypes")) : 1;
        
        final String externalPartnerOrg = (request.getParameter("organization") != null)
            ? request.getParameter("organization") : "";
        
        final String patientFacility = (request.getParameter("facility") != null)
            ? request.getParameter("facility") : "";
        
		final GetAuditsSummary getAuditsSummary = new GetAuditsSummary();

		// start time
		if (NullChecker.isNotEmpty(sd)) {
			getAuditsSummary.setFromDate(GregorianDateUtil
					.getGregorianCalendarByDate(sd));
		}
		// to time
		if (NullChecker.isNotEmpty(ed)) {
			getAuditsSummary.setToDate(GregorianDateUtil
					.getGregorianCalendarByDate(ed));
		}
        //Set User ID
        if (NullChecker.isNotEmpty(userId)) {
            final StringValuesType userIdValue = new StringValuesType();
            userIdValue.getValue().add(userId);
            userIdValue.setNotIn(false);
            getAuditsSummary.setUserIds(userIdValue);
        }
		final ActionValuesType actionValuesType = new ActionValuesType();
		final List<ActionType> actiontype = actionValuesType.getValue();
		actiontype.add(ActionType.RETRIEVE_DOCUMENT_OUT);
		actionValuesType.setNotIn(false);
		getAuditsSummary.setActions(actionValuesType);
                getAuditsSummary.setPatientTypes(patientTypes);


		// patient preferred facility
		if (!NullChecker.isNullOrEmpty(patientFacility)) {

			final StringValuesType patientFacilyNumber = new StringValuesType();
			patientFacilyNumber.setNotIn(false);
			final List<String> patientFacilyNumberList = patientFacilyNumber
					.getValue();
			patientFacilyNumberList.add(patientFacility);
			getAuditsSummary.setPatientFacilityNumbers(patientFacilyNumber);
		}

		if (!NullChecker.isNullOrEmpty(externalPartnerOrg)
				&& !externalPartnerOrg.equals("external")) {
			final StringValuesType remoteOrganizationIds = new StringValuesType();
			remoteOrganizationIds.setNotIn(false);
			final List<String> remoteOrganizationIdsList = remoteOrganizationIds
					.getValue();
			remoteOrganizationIdsList.add(externalPartnerOrg.replace("%",
					"urn:oid:"));
			getAuditsSummary.setRemoteOrganizationIds(remoteOrganizationIds);
		}

		// group by
		final GroupByFieldsType groupByFieldsType = new GroupByFieldsType();
		final List<FieldType> fieldTypeList = groupByFieldsType
				.getGroupByField();
		fieldTypeList.add(FieldType.REMOTE_ORGANIZATION_ID);
		fieldTypeList.add(FieldType.PATIENT_FACILITY_NUMBER);
		fieldTypeList.add(FieldType.PATIENT_FACILITY_NAME);
		fieldTypeList.add(FieldType.REMOTE_ORGANIZATION_NAME);
		getAuditsSummary.setGroupByFields(groupByFieldsType);

		// web services response
		GetAuditsSummaryResponse getAuditsSummaryResponse;
		try {
			getAuditsSummaryResponse = this.adapterAuditManager
					.getAuditSummary(getAuditsSummary);

            return getAuditsSummaryResponse;

        } catch (final AuditException e) {
			throw new AuditException();
		}
    }
    
    /**
     * Converts SearchResutlsPackge to json string Changing List<Map<String,Object>> to a json array with each map being its own array.
     * Object Name is "data"
     *
     * @param searchResults
     * @return
     */
    private String encodeIntoJSON(SearchResultsPackage searchResults) {
        String jsonString = "";

        try {
            StringWriter json = new StringWriter();
            JSONObject obj = new JSONObject();
            List<List> data = new ArrayList<List>();

            if(searchResults.getSearchResults() != null){
                for (Map<String, Object> row : searchResults.getSearchResults()) {
                    List<String> dataItem = new ArrayList<String>();

                    for (String key : row.keySet()) {
                        dataItem.add(row.get(key).toString());
                    }
                    data.add(dataItem);
                }
            }
            obj.put("data", data);

            obj.write(json);

            jsonString = json.toString();
        } catch (JSONException ex) {
            Logger.getLogger(ReceivedNwHINDocumentSummaryReport.class.getName()).log(Level.SEVERE, null, ex);
        }
        return jsonString;
    }
    
    /**
     * Export to excel - This method is called when the user clicks on the excel icon in the JSP. Reflection is used in the
     * ResponseDispatcherHttpServlet to call this method based on the http parameters.
     *
     * @param request HTTP Request
     * @param response HTTP Response
     * @throws ServletException
     * @throws IOException
     */
    public void exportToExcel(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        
        // results)
        SearchResultsPackage packagedResults = getResults(request, true);
        final List<Map<String, Object>> results = packagedResults.getSearchResults();

        // Create a map of key which is based on the result data key and the
        // heading value
        final Map<String, String> receivedNwhinDocumentMap = new LinkedHashMap<String, String>();

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        final LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();

        filters.put("Start Date", this.getReportHelper().getFormattedDate(ReportHelper.getStartDate(request.getParameter("startDate"))));
        filters.put("End Date", this.getReportHelper().getFormattedDate(ReportHelper.getStartDate(request.getParameter("endDate"))));
        filters.put("User ID", request.getParameter("userId"));
        filters.put("Patient Preferred Facility", ExcelExporter.getPatientPreferredFacilityFilter(getFacilityDAO(), request.getParameter("facility")));
        filters.put("eHealth Exchange Organization", ExcelExporter.getFilterValue(request.getParameter("organizationName")));
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, request.getParameter("patientTypes"));

        final Object receivedNwhinDocumentCount = packagedResults.getTotalCount();
        final Map<String, List<Object>> optionalRows = new LinkedHashMap<String, List<Object>>();
        final List<Object> receivedNwhinDocumentRow = new ArrayList<Object>();
        receivedNwhinDocumentRow.add("");
        receivedNwhinDocumentRow.add("");
        receivedNwhinDocumentRow.add("");
        receivedNwhinDocumentRow.add(receivedNwhinDocumentCount);
        optionalRows.put("Total", receivedNwhinDocumentRow);
        // Create the options selected list

        if (NullChecker.isNullOrEmpty(results)) {
            receivedNwhinDocumentMap.put("", "No records were found.");
        } else {
            // The heading is used to create the column headers and the key is used
            // to pull the data from the results
            receivedNwhinDocumentMap.put("partnerOrg", "eHealth Exchange Organization");
            receivedNwhinDocumentMap.put("vaFacility", "Patient Preferred Facility");
            receivedNwhinDocumentMap.put("vaFacilityNumber", "Patient Preferred Facility Station ID");
            receivedNwhinDocumentMap.put("oid", "OID");
            receivedNwhinDocumentMap.put("total", "Total");
        }

        // Create workbook
        final String title = "Received eHealth Exchange Documents Summary Report";
        final Workbook wb = this.getExcelExporter().exportToExcel(title, title, receivedNwhinDocumentMap, results, filterMap, optionalRows);

        // Write Excel to Stream
        this.getExcelExporter().writeExcelToStream("Received_eHealth_Exchange_Documents_Summary_Report", wb, response);
    }
    
    /**
     * Class specific to this report format, putting the searchresult list and total count long together
     * Allowing ease of parameter passing.
     */
    private class SearchResultsPackage {

        private  List<Map<String, Object>> searchResults = new ArrayList<>();
        private final  long totalCount;

        public SearchResultsPackage(List<Map<String, Object>> searchResults, long totalCount) {
            this.searchResults = searchResults;
            this.totalCount = totalCount;
        }

        public List<Map<String, Object>> getSearchResults() {
            return searchResults;
        }

        public long getTotalCount() {
            return totalCount;
        }
    }
    
}
