package gov.va.nvap.web.report;

import gov.va.med.nhin.adapter.audit.ActionType;
import gov.va.med.nhin.adapter.audit.ActionValuesType;
import gov.va.med.nhin.adapter.audit.AuditSummariesType;
import gov.va.med.nhin.adapter.audit.AuditSummaryType;
import gov.va.med.nhin.adapter.audit.FieldType;
import gov.va.med.nhin.adapter.audit.GetAuditsSummary;
import gov.va.med.nhin.adapter.audit.GetAuditsSummaryResponse;
import gov.va.med.nhin.adapter.audit.GroupByFieldsType;
import gov.va.med.nhin.adapter.audit.StringValuesType;
import gov.va.med.nhin.adapter.audit.SummaryFieldsType;
import gov.va.nvap.common.date.GregorianDateUtil;
import gov.va.nvap.common.sort.BubbleSortListMap;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.OrganizationType;
import gov.va.nvap.service.adapter.audit.AdapterAuditManager;
import gov.va.nvap.service.audit.AuditException;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.dao.UserDocumentDAO;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.patient.ExcelGeneratorThread;
import gov.va.nvap.web.user.UserHelper;
import gov.va.nvap.web.util.date.DateUtil;
import gov.va.nvap.web.util.xls.ExcelExporter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import javax.ejb.EJB;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.commons.lang.StringUtils;

/**
 * This class handles the results page for Patient Discovery Summary report.
 *
 * @author Irakli Kakushadze
 */
public class PatientDiscoverySummaryReportResults extends ResponseDispatcherHttpServlet {

    private static final long serialVersionUID = 1L;

    private AdapterAuditManager adapterAuditManager;

    @EJB(beanInterface = AdapterAuditManager.class, mappedName = "AdapterAuditManager")
    public void setAdapterAuditManager(final AdapterAuditManager adapterAuditManager) {
        this.adapterAuditManager = adapterAuditManager;
    }

    public void exportToExcel(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
        ExcelGeneratorThread exGenThread = new ExcelGeneratorThread("exGenThread");
        exGenThread.setIsSummaryReport(true);

        //Get the session final
        HttpSession session = request.getSession(false);

        // Create a map of key which is based on the result data key and the
        // heading value The heading is used to create the column headers and the
        // key is used to pull the data from the results
        final Map<String, String> patientDiscoveryMap = new LinkedHashMap<String, String>();

        final Object startDateStr = session.getAttribute("patientDiscoverySummaryQueryStartDate");
        final Object endDateStr = session.getAttribute("patientDiscoverySummaryQueryEndDate");

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        final LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();

        filters.put("Start Date", this.getReportHelper().getFormattedDate((Date) startDateStr));
        filters.put("End Date", this.getReportHelper().getFormattedDate((Date) endDateStr));
        filters.put("User ID", session.getAttribute("patientDiscoverySummaryQueryUserId"));
        filters.put("Patient Preferred Facility", ExcelExporter.getPatientPreferredFacilityFilter(getFacilityDAO(), (String) session.getAttribute("facility")));
        filters.put("eHealth Exchange Organization", ExcelExporter.getFilterValue(session.getAttribute("patientDiscoverySummaryQueryOrgName")));
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, session.getAttribute("patientTypes").toString());
        exGenThread.setFilterMap(filterMap);

        exGenThread.setHasOptionalRows(true);
        exGenThread.setOptionalRowsMap(createOptionalRowsMap());

        patientDiscoveryMap.put("partnerOrg", "eHealth Exchange Organization");
        patientDiscoveryMap.put("vaFacility", "Patient Preferred Facility");
        patientDiscoveryMap.put("auditsCount", "Audits");
        patientDiscoveryMap.put("countUniqueRealPatients", "Unique Real Patients");
        patientDiscoveryMap.put("matchesFoundRealPatients", "Matches Found for Real Patients");
        patientDiscoveryMap.put("matchFailsRealPatients", "Match Fails for Real Patients");
        patientDiscoveryMap.put("countUniqueTestPatients", "Unique Test Patients");
        patientDiscoveryMap.put("matchesFoundTestPatients", "Matches Found for Test Patients");
        patientDiscoveryMap.put("matchFailsTestPatients", "Match Fails for Test Patients");
        exGenThread.setReportMap(patientDiscoveryMap);

        // Set names for Excel generator thread
        exGenThread.setTitle("Patient Discovery Audit Summary Report");

        exGenThread.setExcelExporter(this.getExcelExporter());
        exGenThread.setAdapterAuditManager(adapterAuditManager);
        exGenThread.setReportHelper(getReportHelper());
        exGenThread.setCmsHelper(getCmsHelper());
        exGenThread.setFacilityHelper(getFacilityHelper());
        exGenThread.setUserId(UserHelper.getUserName(request));
        exGenThread.setUserDocumentDao(getUserDocumentDAO());

        // Set search attributes for getting audit results
        exGenThread.setAttributes(mapSessionAttributes(session));

        exGenThread.run();

        this.forward(request, response, "show");
    }

    private Map<String, Object> mapSessionAttributes(HttpSession session) {
        Map<String, Object> attributes = new HashMap<String, Object>();

        attributes.put("userId", (String) session
                .getAttribute("patientDiscoverySummaryQueryUserId"));

        // Add 24 hours to end date if same as start date
        Date startDate = (Date) session.getAttribute("patientDiscoverySummaryQueryStartDate");
        Date endDate = (Date) session.getAttribute("patientDiscoverySummaryQueryEndDate");
        if (NullChecker.isNotEmpty(endDate)) {
            if (endDate.equals(startDate)) {
                endDate = DateUtil.addTime(endDate, Calendar.HOUR, 24);
            }
        }
        attributes.put("startDate", startDate);
        attributes.put("endDate", endDate);

        // Set both sender and receiver org to the organization entered
        attributes.put("organization", (String) session
                .getAttribute("patientDiscoverySummaryQueryOrganization"));
        attributes.put("remoteOrganization", (String) session
                .getAttribute("patientDiscoverySummaryQueryOrganization"));

        attributes.put("facility", (String) session.getAttribute("facility"));
        attributes.put("patientTypes", (Integer) session.getAttribute("patientTypes"));
        // sort
        attributes.put("sortValue", (String) session
                .getAttribute("patientDiscoverySummarySortValue"));
        attributes.put("sortDirection", (String) session
                .getAttribute("patientDiscoverySummarySortDirection"));

        // Set Action Types
        final ActionValuesType actionsType = new ActionValuesType();
        actionsType.getValue().add(ActionType.ANNOUNCE);
        actionsType.getValue().add(ActionType.CHECK_POLICY);
        actionsType.getValue().add(ActionType.MPI_FIND_MATCH);
        actionsType.getValue().add(ActionType.ADD_PATIENT_CORRELATION);
        actionsType.setNotIn(false);
        attributes.put("actionsType", actionsType);

        // Group by
        final GroupByFieldsType groupByFieldsType = new GroupByFieldsType();
        final List<FieldType> fieldTypeList = groupByFieldsType.getGroupByField();
        fieldTypeList.add(FieldType.REMOTE_ORGANIZATION_ID);
        fieldTypeList.add(FieldType.PATIENT_FACILITY_NUMBER);
        fieldTypeList.add(FieldType.PATIENT_FACILITY_NAME);
        fieldTypeList.add(FieldType.REMOTE_ORGANIZATION_NAME);
        attributes.put("groupByFieldsType", groupByFieldsType);

        return attributes;
    }

    Map<String, String> createOptionalRowsMap() {
        Map<String, String> optionalRowsMap = new LinkedHashMap<String, String>();
        optionalRowsMap.put("totalCount", "Total audits");
        return optionalRowsMap;
    }

    @Override
    public void unspecified(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {

        final HttpSession session = request.getSession(false);
        final Date sd = (Date) session.getAttribute("patientDiscoverySummaryQueryStartDate");
        Date ed = (Date) session.getAttribute("patientDiscoverySummaryQueryEndDate");
        final String userId = (String) session.getAttribute("patientDiscoverySummaryQueryUserId");
        final String organization = (String) session.getAttribute("patientDiscoverySummaryQueryOrganization");
        final String patientFacility = (String) session.getAttribute("facility");
        final int patientTypes = (Integer) session.getAttribute("patientTypes");
        final GetAuditsSummary getAuditsSummaryRequest = new GetAuditsSummary();
        final String patientDiscoverySummarySortValue = request.getParameter("patientDiscoverySummarySortValue");
        final String patientDiscoverySummarySortDirection = request.getParameter("patientDiscoverySummarySortDirection");

        // Set start date
        if (NullChecker.isNotEmpty(sd)) {
            getAuditsSummaryRequest.setFromDate(GregorianDateUtil.getGregorianCalendarByDate(sd));
        }

        // Set end date
        if (NullChecker.isNotEmpty(ed)) {
            if (ed.equals(sd)) {
                ed = DateUtil.addTime(ed, Calendar.HOUR, 24);
            }
            getAuditsSummaryRequest.setToDate(GregorianDateUtil.getGregorianCalendarByDate(ed));
        }

        // Set user ID
        if (NullChecker.isNotEmpty(userId)) {
            final StringValuesType userIdValue = new StringValuesType();
            userIdValue.getValue().add(userId);
            userIdValue.setNotIn(false);
            getAuditsSummaryRequest.setUserIds(userIdValue);

        }

        // Set patient preferred facility
        if (!NullChecker.isNullOrEmpty(patientFacility)) {
            final StringValuesType patientFacilyNumber = new StringValuesType();
            patientFacilyNumber.setNotIn(false);
            final List<String> patientFacilyNumberList = patientFacilyNumber.getValue();
            patientFacilyNumberList.add(patientFacility);
            getAuditsSummaryRequest.setPatientFacilityNumbers(patientFacilyNumber);
        }

        // Set partner organization
        final StringValuesType orgTypes = new StringValuesType();
        if (NullChecker.isNotEmpty(organization)) {
            orgTypes.getValue().add(organization);
        }
        orgTypes.setNotIn(false);
        getAuditsSummaryRequest.setRemoteOrganizationIds(orgTypes);
        getAuditsSummaryRequest.setOrganizationIds(orgTypes);

        // Set patient types
        getAuditsSummaryRequest.setPatientTypes(patientTypes);

        // Set actioin types
        final ActionValuesType actionsType = new ActionValuesType();
        actionsType.getValue().add(ActionType.ANNOUNCE);
        actionsType.getValue().add(ActionType.CHECK_POLICY);
        actionsType.getValue().add(ActionType.MPI_FIND_MATCH);
        actionsType.getValue().add(ActionType.ADD_PATIENT_CORRELATION);
        actionsType.setNotIn(false);
        getAuditsSummaryRequest.setActions(actionsType);

        // Group by
        final GroupByFieldsType groupByFieldsType = new GroupByFieldsType();
        final List<FieldType> fieldTypeList = groupByFieldsType.getGroupByField();
        fieldTypeList.add(FieldType.REMOTE_ORGANIZATION_ID);
        fieldTypeList.add(FieldType.PATIENT_FACILITY_NUMBER);
        fieldTypeList.add(FieldType.PATIENT_FACILITY_NAME);
        fieldTypeList.add(FieldType.REMOTE_ORGANIZATION_NAME);
        getAuditsSummaryRequest.setGroupByFields(groupByFieldsType);

        try {
            // Get summary list from Exchange
            GetAuditsSummaryResponse getAuditsSummaryResponse = this.adapterAuditManager.getAuditSummary(getAuditsSummaryRequest);

            // Construct results for the jsp page
            List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
            long totalCount = 0l;
            final AuditSummariesType auditSummariesType = getAuditsSummaryResponse.getAuditSummaries();

            List<AuditSummaryType> auditSummary = new ArrayList<AuditSummaryType>();
            if (auditSummariesType != null) {
                auditSummary = auditSummariesType.getAuditSummary();
            }

            for (final AuditSummaryType auditSummaryType : auditSummary) {
                final Map<String, Object> resultMap = new HashMap<String, Object>();
                final long count = auditSummaryType.getCount();
                final SummaryFieldsType summaryFields = auditSummaryType.getSummaryFields();
                final List<String> summaryField = summaryFields.getSummaryField();

                if (NullChecker.isEmpty(summaryField)) {
                    continue;
                }

                if (summaryField.size() < 4) {
                    continue;
                }

                // Partner organization
                OrganizationType organizationType = null;
                if (NullChecker.isNotEmpty(summaryField.get(0))) {
                    organizationType = (this.getCmsHelper().getOrganizationByHomeCommunityId(summaryField.get(0).replace("urn:oid:", "")));
                }
                if (organizationType!= null && !NullChecker.isNullOrEmpty(organizationType)) {
                    resultMap.put("partnerOrg", ReportDataProcessor.nullEmptyReplaceWithUnknown(organizationType.getOrgName()));
                } else {
                    resultMap.put("partnerOrg", ReportDataProcessor.nullEmptyReplaceWithUnknown(summaryField.get(3)));
                }

                // Patient preferred facility
                final String adapterVAFacility = summaryField.get(1);
                String vaFacility;
                if (ReportDataProcessor.isValidStation(adapterVAFacility)) {
                    vaFacility = this.getFacilityHelper().getFacilityNameByStationId(adapterVAFacility);
                    if (NullChecker.isEmpty(vaFacility)) {
                        vaFacility = summaryField.get(2);
                    }
                } else {
                    vaFacility = summaryField.get(2);
                }
                resultMap.put("vaFacility", ReportDataProcessor.fixStation(vaFacility));

                // Audits count
                resultMap.put("auditsCount", count);

                // The rest of the values are simply copied from auditSummary
                resultMap.put("countUniqueRealPatients", StringUtils.isNumeric(summaryField.get(4)) ? Long.parseLong(summaryField.get(4)) : 0);
                resultMap.put("countUniqueTestPatients", StringUtils.isNumeric(summaryField.get(5)) ? Long.parseLong(summaryField.get(5)) : 0);
                resultMap.put("matchesFoundRealPatients", StringUtils.isNumeric(summaryField.get(6)) ? Long.parseLong(summaryField.get(6)) : 0);
                resultMap.put("matchesFoundTestPatients", StringUtils.isNumeric(summaryField.get(7)) ? Long.parseLong(summaryField.get(7)) : 0);
                resultMap.put("matchFailsRealPatients", StringUtils.isNumeric(summaryField.get(8)) ? Long.parseLong(summaryField.get(8)) : 0);
                resultMap.put("matchFailsTestPatients", StringUtils.isNumeric(summaryField.get(9)) ? Long.parseLong(summaryField.get(9)) : 0);

                results.add(resultMap);
                totalCount += count;
            }

            // Sort
            if (NullChecker.isNotEmpty(patientDiscoverySummarySortValue) && NullChecker.isNotEmpty(patientDiscoverySummarySortDirection)) {
                boolean ascending = !patientDiscoverySummarySortDirection.equalsIgnoreCase("DESC");
                final BubbleSortListMap bSorter = new BubbleSortListMap();
                results = bSorter.sortByColumn(results, patientDiscoverySummarySortValue, ascending);
            }

            session.setAttribute("totalCount", totalCount);
            session.setAttribute("patientDiscoverySummaryResults", results);
        } catch (final AuditException ex) {
            throw new AuditException(ex);
        }
        this.forward(request, response, "show");
    }

    /**
     * Get the consent management helper from Spring.
     * @return ConsentManagementHelper Consent management helper
     */
    private ConsentManagementHelper getCmsHelper() {
        return getBean("cmsHelper", ConsentManagementHelper.class);
    }

    /**
     * Get the excel exporter class from Spring.
     * @return the excel exporter object
     */
    private ExcelExporter getExcelExporter() {
        return getBean("excelExporter", ExcelExporter.class);
    }

    /**
     * Get the facility helper from Spring.
     * @return FacilityHelper helper
     */
    private FacilityHelper getFacilityHelper() {
        return getBean("facilityHelper", FacilityHelper.class);
    }

    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }

    /**
     * Get the report helper from Spring.
     * @return ReportHelper Report helper
     */
    private ReportHelper getReportHelper() {
        return getBean("reportHelper", ReportHelper.class);
    }

    private UserDocumentDAO getUserDocumentDAO() {
        return this.getBean("UserDocumentDAO", UserDocumentDAO.class);
    }
}