package gov.va.nvap.web.report;

import gov.va.med.nhin.adapter.audit.ActionType;
import gov.va.med.nhin.adapter.audit.ActionValuesType;
import gov.va.med.nhin.adapter.audit.AuditSummariesType;
import gov.va.med.nhin.adapter.audit.AuditSummaryType;
import gov.va.med.nhin.adapter.audit.FieldType;
import gov.va.med.nhin.adapter.audit.GetAuditsSummary;
import gov.va.med.nhin.adapter.audit.GetAuditsSummaryResponse;
import gov.va.med.nhin.adapter.audit.GroupByFieldsType;
import gov.va.med.nhin.adapter.audit.StringValuesType;
import gov.va.med.nhin.adapter.audit.SummaryFieldsType;
import gov.va.nvap.common.date.GregorianDateUtil;
import gov.va.nvap.common.sort.BubbleSortListMap;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.OrganizationType;
import gov.va.nvap.service.adapter.audit.AdapterAuditManager;
import gov.va.nvap.service.audit.AuditException;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.util.xls.ExcelExporter;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.ejb.EJB;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.ss.usermodel.Workbook;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * This class handles the search page for Patient Discovery Summary report.
 * 
 * @author Irakli Kakushadze, updated by Raul Alfaro
 */
public class PatientDiscoverySummaryReport extends ResponseDispatcherHttpServlet{
    
	/**
	 * Serial UID.
	 */
	private static final long serialVersionUID = 1L;

    private AdapterAuditManager adapterAuditManager;

    @EJB(beanInterface = AdapterAuditManager.class, mappedName = "AdapterAuditManager")
    public void setAdapterAuditManager(final AdapterAuditManager adapterAuditManager) {
        this.adapterAuditManager = adapterAuditManager;
    }
    
    /**
     * Get the consent management helper from Spring.
     * @return ConsentManagementHelper Consent management helper
     */
	public ConsentManagementHelper getCmsHelper() {
		return this.getBean("cmsHelper", ConsentManagementHelper.class);
	}

    /**
     * Get the facility helper from Spring.
     * @return FacilityHelper helper
     */
	public FacilityHelper getFacilityHelper() {
		return this.getBean("facilityHelper", FacilityHelper.class);
	}

    private ReportHelper getReportHelper() {
        return this.getBean("reportHelper", ReportHelper.class);
    }
    
    @Override
    protected void unspecified(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {

        HttpSession session = request.getSession(false);
        ReportHelper.setDefaultSearchDates(session);
        request.setAttribute("facilities", this.getFacilityHelper().getAllVistAFacilities());
        this.getReportHelper().getOrgLists(request, this.getCmsHelper());

        this.forward(request, response, "show");
    }

    /**
     *
     * @param request
     * @param response
     * @throws ServletException
     * @throws IOException
     */
    public void doSearch(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
        final HttpSession session = request.getSession(false);

        session.setAttribute("results", encodeIntoJSON(getResults(request)));

        this.forward(request, response, "searchResultsJSON");
    }

    /**
     * Returns search results and audit total packaged together in a class
     *
     * @param request
     * @return SearchResultsPackage
     * @throws AuditException
     */
    private SearchResultsPackage getResults(final HttpServletRequest request) throws AuditException {

        final Date sd = ReportHelper.getStartDate(request.getParameter("startDate"));
        final Date ed = ReportHelper.getEndDate(request.getParameter("endDate"));
        final String userId = request.getParameter("userId");
        final String organization = (request.getParameter("organization") != null)
            ? request.getParameter("organization") : "";
        final String patientFacility = (request.getParameter("facility") != null)
            ? request.getParameter("facility") : "";
        final int patientTypes = (request.getParameter("patientTypes") != null)
            ? Integer.parseInt(request.getParameter("patientTypes")) : 1;

        final GetAuditsSummary getAuditsSummaryRequest = new GetAuditsSummary();

        // Set start date
        if (NullChecker.isNotEmpty(sd)) {
            getAuditsSummaryRequest.setFromDate(GregorianDateUtil.getGregorianCalendarByDate(sd));
        }

        // Set end date
        if (NullChecker.isNotEmpty(ed)) {
            getAuditsSummaryRequest.setToDate(GregorianDateUtil.getGregorianCalendarByDate(ed));
        }

        // Set user ID
        if (NullChecker.isNotEmpty(userId)) {
            final StringValuesType userIdValue = new StringValuesType();
            userIdValue.getValue().add(userId);
            userIdValue.setNotIn(false);
            getAuditsSummaryRequest.setUserIds(userIdValue);
        }

        // Set patient preferred facility
        if (!NullChecker.isNullOrEmpty(patientFacility)) {
            final StringValuesType patientFacilyNumber = new StringValuesType();
            patientFacilyNumber.setNotIn(false);
            final List<String> patientFacilyNumberList = patientFacilyNumber.getValue();
            patientFacilyNumberList.add(patientFacility);
            getAuditsSummaryRequest.setPatientFacilityNumbers(patientFacilyNumber);
        }

        // Set partner organization
        final StringValuesType orgTypes = new StringValuesType();
        if (NullChecker.isNotEmpty(organization)) {
            orgTypes.getValue().add(organization);
        }
        orgTypes.setNotIn(false);
        getAuditsSummaryRequest.setRemoteOrganizationIds(orgTypes);
        getAuditsSummaryRequest.setOrganizationIds(orgTypes);

        // Set patient types
        getAuditsSummaryRequest.setPatientTypes(patientTypes);

        // Set actioin types
        final ActionValuesType actionsType = new ActionValuesType();
        actionsType.getValue().add(ActionType.ANNOUNCE);
        actionsType.getValue().add(ActionType.CHECK_POLICY);
        actionsType.getValue().add(ActionType.MPI_FIND_MATCH);
        actionsType.getValue().add(ActionType.ADD_PATIENT_CORRELATION);
        actionsType.setNotIn(false);
        getAuditsSummaryRequest.setActions(actionsType);

        // Group by
        final GroupByFieldsType groupByFieldsType = new GroupByFieldsType();
        final List<FieldType> fieldTypeList = groupByFieldsType.getGroupByField();
        fieldTypeList.add(FieldType.REMOTE_ORGANIZATION_ID);
        fieldTypeList.add(FieldType.PATIENT_FACILITY_NUMBER);
        fieldTypeList.add(FieldType.PATIENT_FACILITY_NAME);
        fieldTypeList.add(FieldType.REMOTE_ORGANIZATION_NAME);
        getAuditsSummaryRequest.setGroupByFields(groupByFieldsType);

        SearchResultsPackage packagedResults;
        List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();

        long totalCount = 0l;
        try {
            // Get summary list from Exchange
            GetAuditsSummaryResponse getAuditsSummaryResponse = this.adapterAuditManager.getAuditSummary(getAuditsSummaryRequest);

            // Construct results for the jsp page
            final AuditSummariesType auditSummariesType = getAuditsSummaryResponse.getAuditSummaries();

            List<AuditSummaryType> auditSummary = new ArrayList<AuditSummaryType>();
            if (auditSummariesType != null) {
                auditSummary = auditSummariesType.getAuditSummary();
            }

            for (final AuditSummaryType auditSummaryType : auditSummary) {

                final Map<String, Object> resultMap = new LinkedHashMap<String, Object>();

                final long count = auditSummaryType.getCount();
                final SummaryFieldsType summaryFields = auditSummaryType.getSummaryFields();
                final List<String> summaryField = summaryFields.getSummaryField();

                if (NullChecker.isEmpty(summaryField)) {
                    continue;
                }

                if (summaryField.size() < 4) {
                    continue;
                }

                // Partner organization
                OrganizationType organizationType = null;
                if (NullChecker.isNotEmpty(summaryField.get(0))) {
                    organizationType = (this.getCmsHelper().getOrganizationByHomeCommunityId(summaryField.get(0).replace("urn:oid:", "")));
                }
                if (!NullChecker.isNullOrEmpty(organizationType)) {
                    resultMap.put("partnerOrg", ReportDataProcessor.nullEmptyReplaceWithUnknown(organizationType.getOrgName()));
                } else {
                    resultMap.put("partnerOrg", ReportDataProcessor.nullEmptyReplaceWithUnknown(summaryField.get(3)));
                }

                // Patient preferred facility
                final String adapterVAFacility = summaryField.get(1);
                String vaFacility;
                if (ReportDataProcessor.isValidStation(adapterVAFacility)) {
                    vaFacility = this.getFacilityHelper().getFacilityNameByStationId(adapterVAFacility);
                    if (NullChecker.isEmpty(vaFacility)) {
                        vaFacility = summaryField.get(2);
                    }
                } else {
                    vaFacility = summaryField.get(2);
                }
                resultMap.put("vaFacility", ReportDataProcessor.fixStation(vaFacility));

                // Audits count
                resultMap.put("auditsCount", count);

                // The rest of the values are simply copied from auditSummary, same order as they appear on datatables.
                resultMap.put("countUniqueRealPatients", StringUtils.isNumeric(summaryField.get(4)) ? Long.parseLong(summaryField.get(4)) : 0);
                resultMap.put("matchesFoundRealPatients", StringUtils.isNumeric(summaryField.get(6)) ? Long.parseLong(summaryField.get(6)) : 0);
                resultMap.put("matchFailsRealPatients", StringUtils.isNumeric(summaryField.get(8)) ? Long.parseLong(summaryField.get(8)) : 0);
                resultMap.put("countUniqueTestPatients", StringUtils.isNumeric(summaryField.get(5)) ? Long.parseLong(summaryField.get(5)) : 0);
                resultMap.put("matchesFoundTestPatients", StringUtils.isNumeric(summaryField.get(7)) ? Long.parseLong(summaryField.get(7)) : 0);
                resultMap.put("matchFailsTestPatients", StringUtils.isNumeric(summaryField.get(9)) ? Long.parseLong(summaryField.get(9)) : 0);

                results.add(resultMap);
                totalCount += count;
            }

            //sort and put eveverything together for transport package
            packagedResults = new SearchResultsPackage( sortResults(results,request), totalCount);

        } catch (final AuditException ex) {
            throw new AuditException();
        }
        return packagedResults;
    }
    
    /**
     * sorts List<Map<String,Object>> based on map keys and request parameters.
     * @param unsortedList
     * @param request
     * @return 
     */
    private List<Map<String,Object>> sortResults(List<Map<String,Object>> unsortedList, HttpServletRequest request){
        
        String sortValue = request.getParameter("order[0][column]") == null ? request.getParameter("sortBy") : request.getParameter("order[0][column]");
        boolean sortAscending = request.getParameter("order[0][dir]") == null ? !"desc".equalsIgnoreCase(request.getParameter("sortOrder"))
                : !"desc".equalsIgnoreCase(request.getParameter("order[0][dir]"));
        
        if(unsortedList.size() > 1){
            final BubbleSortListMap bSorter = new BubbleSortListMap();
            
            if ("0".equals(sortValue)){
                sortValue = "partnerOrg";
            } else if ("1".equals(sortValue)){
                sortValue = "vaFacility";
            } else if ("2".equals(sortValue)){
                 sortValue = "auditsCount";
            } else if ("3".equals(sortValue)){
                sortValue = "countUniqueRealPatients";
            } else if ("4".equals(sortValue)){
                sortValue = "matchesFoundRealPatients";
            } else if ("5".equals(sortValue)){
                sortValue = "matchFailsRealPatients";
            } else if ("6".equals(sortValue)){
                sortValue = "countUniqueTestPatients";
            } else if ("7".equals(sortValue)){
                sortValue = "matchesFoundTestPatients";
            } else if ("8".equals(sortValue)){
                sortValue = "matchFailsTestPatients";
            } else {
                sortValue = "partnerOrg";
            }
            return bSorter.sortByColumn(unsortedList, sortValue, sortAscending);
        } else {
            return unsortedList;
        }
    }

    /**
     * Converts SearchResutlsPackge to json string Changing List<Map<String,Object>> to a json array with each map being its own array.
     * Object Name is "data" Adds object "totalCount" for total number of audits.
     *
     * @param searchResults
     * @return
     */
    private String encodeIntoJSON(SearchResultsPackage searchResults) {
        String jsonString = "";

        try {
            StringWriter json = new StringWriter();
            JSONObject obj = new JSONObject();
            List<List> data = new ArrayList<List>();

            for (Map<String, Object> row : searchResults.getSearchResults()) {
                List<String> dataItem = new ArrayList<String>();

                for (String key : row.keySet()) {
                    dataItem.add(row.get(key).toString());
                }
                data.add(dataItem);
            }

            obj.put("data", data);

            obj.write(json);

            jsonString = json.toString();
        } catch (JSONException ex) {
            Logger.getLogger(PatientDiscoverySummaryReport.class.getName()).log(Level.SEVERE, null, ex);
        }
        return jsonString;
    }

    /**
     * Pulls its own copy of of the search results and converts it to excel output to be downloaded.
     *
     * @param request
     * @param response
     * @throws ServletException
     * @throws IOException
     */
    public void exportToExcel(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {

        SearchResultsPackage results = getResults(request);

        // Create a map of key which is based on the result data key and the
        // heading value The heading is used to create the column headers and the
        // key is used to pull the data from the results
        final Map<String, String> patientDiscoveryMap = new LinkedHashMap<String, String>();

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        final LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();

        filters.put("Start Date", request.getParameter("startDate"));
        filters.put("End Date", request.getParameter("endDate"));
        filters.put("User ID", request.getParameter("userId"));
        filters.put("Patient Preferred Facility", ExcelExporter.getPatientPreferredFacilityFilter(getFacilityDAO(), request.getParameter("facility")));
        filters.put("eHealth Exchange Organization", ExcelExporter.getFilterValue(request.getParameter("organizationName")));
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, request.getParameter("patientTypes"));

        //Handle the calculation of optional row totals here.
        final Map<String, List<Object>> optionalRows = new LinkedHashMap<String, List<Object>>();
        List<Object> auditCount = new ArrayList<Object>();
        auditCount.add(results.getTotalCount());

        optionalRows.put("Total Audits", auditCount);

        if (NullChecker.isNullOrEmpty(results.getSearchResults())) {
            patientDiscoveryMap.put("", "No records were found.");
        } else {
            patientDiscoveryMap.put("partnerOrg", "eHealth Exchange Organization");
            patientDiscoveryMap.put("vaFacility", "Patient Preferred Facility");
            patientDiscoveryMap.put("auditsCount", "Audits");
            patientDiscoveryMap.put("countUniqueRealPatients", "Unique Real Patients");
            patientDiscoveryMap.put("matchesFoundRealPatients", "Matches Found for Real Patients");
            patientDiscoveryMap.put("matchFailsRealPatients", "Match Fails for Real Patients");
            patientDiscoveryMap.put("countUniqueTestPatients", "Unique Test Patients");
            patientDiscoveryMap.put("matchesFoundTestPatients", "Matches Found for Test Patients");
            patientDiscoveryMap.put("matchFailsTestPatients", "Match Fails for Test Patients");
        }

        // Create workbook
        final String title = "Patient Discovery Audit Summary Report";
        final Workbook wb = this.getExcelExporter().exportToExcel(title, title, patientDiscoveryMap, results.getSearchResults(), filterMap, optionalRows);

        // Write Excel to Stream
        this.getExcelExporter().writeExcelToStream("Patient_Discovery_Audit_Summary_Report", wb, response);
    }

    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }

    /**
     * Get the excel exporter class from Spring.
     *
     * @return the excel exporter object
     */
    private ExcelExporter getExcelExporter() {
        return getBean("excelExporter", ExcelExporter.class);
    }

    /**
     * Class specific to this report format, putting the searchresult list and total count long together
     * Allowing ease of parameter passing.
     */
    private class SearchResultsPackage {

        private final List<Map<String, Object>> searchResults;
        private final long totalCount;

        public SearchResultsPackage(List<Map<String, Object>> searchResults, long totalCount) {
            this.searchResults = searchResults;
            this.totalCount = totalCount;
        }

        public List<Map<String, Object>> getSearchResults() {
            return searchResults;
        }

        public long getTotalCount() {
            return totalCount;
        }
    }
    
}
