/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package gov.va.nvap.web.report;

import gov.va.med.nhin.adapter.audit.ActionType;
import gov.va.med.nhin.adapter.audit.ActionValuesType;
import gov.va.med.nhin.adapter.audit.AuditType;
import gov.va.med.nhin.adapter.audit.FieldType;
import gov.va.med.nhin.adapter.audit.GetAudits;
import gov.va.med.nhin.adapter.audit.GetAuditsResponse;
import gov.va.med.nhin.adapter.audit.PageInfoType;
import gov.va.med.nhin.adapter.audit.SortDirection;
import gov.va.med.nhin.adapter.audit.SortFieldType;
import gov.va.med.nhin.adapter.audit.SortFieldsType;
import gov.va.med.nhin.adapter.audit.StringValuesType;
import gov.va.nvap.common.date.GregorianDateUtil;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.service.adapter.audit.AdapterAuditManager;
import gov.va.nvap.service.audit.AuditException;
import gov.va.nvap.service.pdq.PdqService;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.dao.UserDocumentDAO;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.patient.ExcelGeneratorThread;
import gov.va.nvap.web.user.UserHelper;
import gov.va.nvap.web.util.date.DateUtil;
import gov.va.nvap.web.util.xls.CsvExporter;
import gov.va.nvap.web.util.xls.ExcelExporter;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.ejb.EJB;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.poi.ss.usermodel.Workbook;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * @author Asha Amritraj ajaxed by Raul Alfaro
 */
public class PatientDiscoveryReport extends ResponseDispatcherHttpServlet {

	/**
	 *
	 */
	private static final long serialVersionUID = -8614673990226997234L;
    private AdapterAuditManager adapterAuditManager;
    @EJB(beanInterface = PdqService.class, mappedName = "PdqService")
    PdqService pdqService;

    private final String REPORT_TITLE = "Patient Discovery Audit Detail Report";

	public ConsentManagementHelper getCmsHelper() {
		final ConsentManagementHelper cmsHelper = this.getBean("cmsHelper",
				ConsentManagementHelper.class);
		return cmsHelper;
	}

    private ReportHelper getReportHelper() {
        return this.getBean("reportHelper", ReportHelper.class);
    }


	@Override
	protected void unspecified(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		final HttpSession session = request.getSession(false);

		session.removeAttribute("patientDiscoveryQueryICN");
		session.removeAttribute("results");

        ReportHelper.setDefaultSearchDates(session);
        this.getReportHelper().getOrgLists(request, this.getCmsHelper());

		this.forward(request, response, "show");
	}

    /**
     * Main search function returns json results use primarily for datatables.
     * @param request
     * @param response
     * @throws IOException
     * @throws ServletException
     */
    public void doSearch(final HttpServletRequest request,
			final HttpServletResponse response) throws IOException, ServletException{

        final HttpSession session = request.getSession(false);

        session.setAttribute("results", encodeIntoJSON(getResults(request, false, false), request.getParameter("draw")));

        this.forward(request, response, "searchResultsJSON");

    }

    /**
     * Pull required data using request to gather query parameters, Boolean represents if all data should
     * be returned
     * @param request
     * @param getALL
     * @return
     * @throws AuditException
     */
    public SummaryPackage getResults(final HttpServletRequest request, boolean getALL, boolean exporting) throws AuditException {

        final GetAudits getAuditsRequest = prepareAuditParameters(request, getALL);

        final int recordsPerPage = (request.getParameter("length")!=null) ?
                Integer.parseInt(request.getParameter("length")) : 25;

        Long totalCount = new Long(0);

        // TODO: Anand Sastry / Weimin Shi - FIX INTEGRATION 08/29/2011 - END
        final List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        try {
            final GetAuditsResponse queryResponse = this.adapterAuditManager
                    .getAudits(getAuditsRequest);
            if(queryResponse.getPageInfo() != null) {
                totalCount = exporting ? 0 : new Long(queryResponse.getPageInfo().getTotalSize());
            }

            long realPatientMessages = 0;
            long realPatientFails = 0;
            long realPatientMatches = 0;
            long testPatientMessages = 0;
            long testPatientFails = 0;
            long testPatientMatches = 0;
            HashSet<String> uniqueRealPatients = new HashSet<String>();
            HashSet<String> uniqueTestPatients = new HashSet<String>();

            if (NullChecker.isNotEmpty(queryResponse)
                    && NullChecker.isNotEmpty(queryResponse.getAudits())
                    && NullChecker.isNotEmpty(queryResponse.getAudits().getAudit())) {
                final List<AuditType> auditTypeList = queryResponse.getAudits().getAudit();
                //grab datatables parameter for page length
                int pageSize = recordsPerPage;
                for (final AuditType auditType : auditTypeList) {
                    final Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
                    String explanationOfFailure = "N/A";
                    String patientFirstName = "";
                    String patientLastName = "";
                    String patientMiddleName = "";
                    String patientSSN = "";
                    Boolean mviFindMatchFailed = null;

                    String details = ReportDataProcessor.nullEmptyReplaceWithUnknown(auditType.getDetails());
                    if (details.startsWith("QUERY FAILED")) {
                        explanationOfFailure = "MVI: Query Failed";
                        mviFindMatchFailed = true;
                    } else if (details.startsWith("MATCH FAILED")) {
                        explanationOfFailure = "MVI: Match Not Found";
                        mviFindMatchFailed = true;
                    } else if (details.startsWith("AMBIGUOUS FAILED")) {
                        explanationOfFailure = "MVI: Ambiguous Match";
                        mviFindMatchFailed = true;
                    } else if (details.startsWith("MATCH FOUND")) {
                        mviFindMatchFailed = false;
                    } else if (details.startsWith("FAILURE - PARTNER DID NOT RESPOND WITH PATIENT INFORMATION ")) {
                        explanationOfFailure = "Partner did not respond with patient information";
                        details = details.substring("FAILURE - PARTNER DID NOT RESPOND WITH PATIENT INFORMATION ".length());
                    } else if (details.startsWith("FAILURE - OUTBOUND PD: PARTNER OUTBOUND PD HAS BEEN DISABLED. ")) {
                        explanationOfFailure = "Outbound PD: Partner outbound PD has been disabled";
                        details = details.substring("FAILURE - OUTBOUND PD: PARTNER OUTBOUND PD HAS BEEN DISABLED. ".length());
                    } else if (details.startsWith("FAILURE - OUTBOUND PD: PARTNER IS NOT REGISTERED WITH THE SYSTEM. ")) {
                        explanationOfFailure = "Outbound PD: Partner is not registered with the system";
                        details = details.substring("FAILURE - OUTBOUND PD: PARTNER IS NOT REGISTERED WITH THE SYSTEM. ".length());
                    }
                    resultMap.put("failureExplanation", explanationOfFailure);

                    ReportHelper reportHelper = getReportHelper();
                    if (Boolean.TRUE.equals(mviFindMatchFailed)) {
                        patientFirstName = getReportHelper().extractFromDetails(details, "FIRSTNAME");
                        patientLastName = reportHelper.extractFromDetails(details, "LASTNAME");
                        patientMiddleName = reportHelper.extractFromDetails(details, "MIDDLENAME");
                        patientSSN = reportHelper.extractFromDetails(details, "SSN");
                    }
                    if (patientSSN.length() == 0) {
                        patientSSN = auditType.getPatientSSN();
                    }

                    // SSN
                    ReportDataProcessor.addSsnToResultMap(resultMap, patientSSN);

                    boolean isTestPatient = this.getReportHelper().isTestPatient(resultMap.get("ssn").toString());

                    if (NullChecker.isNotEmpty(patientSSN)) {
                        if (isTestPatient) {
                            if (!uniqueTestPatients.contains(patientSSN)) {
                                uniqueTestPatients.add(patientSSN);
                            }
                        } else {
                            if (!uniqueRealPatients.contains(patientSSN)) {
                                uniqueRealPatients.add(patientSSN);
                            }
                        }
                    }

                    if (pageSize == -1 || (testPatientMessages + realPatientMessages) < pageSize) {
                        if (isTestPatient) {
                            testPatientMessages++;
                            if (Boolean.TRUE.equals(mviFindMatchFailed)) {
                                testPatientFails++;
                            } else if (Boolean.FALSE.equals(mviFindMatchFailed)) {
                                testPatientMatches++;
                            }
                        } else {
                            realPatientMessages++;
                            if (Boolean.TRUE.equals(mviFindMatchFailed)) {
                                realPatientFails++;
                            } else if (Boolean.FALSE.equals(mviFindMatchFailed)) {
                                realPatientMatches++;
                            }
                        }
                    }

                    // First name.
                    if (patientFirstName.length() == 0) {
                        patientFirstName = ReportDataProcessor.nullEmptyReplaceWithUnknown(auditType.getPatientGivenName());
                    }
                    resultMap.put("firstName", patientFirstName);

                    // Middle name.
                    if (patientMiddleName.length() == 0) {
                        patientMiddleName =  ReportDataProcessor.nullEmptyReplaceWithUnknown(auditType.getPatientMiddleName());
                    }
                    if ("Unknown".equalsIgnoreCase(patientMiddleName)) {
                        patientMiddleName = "";
                    }
                    resultMap.put("middleName", patientMiddleName);

                    // Last name.
                    if (patientLastName.length() == 0) {
                        patientLastName = ReportDataProcessor.nullEmptyReplaceWithUnknown(auditType.getPatientLastName());
                    }
                    resultMap.put("lastName", patientLastName);

                    resultMap.put("icn", ReportDataProcessor.nullEmptyReplaceWithUnknown(auditType.getPatientId()));
                    resultMap.put("auditTime", GregorianDateUtil.getDateFromGregorianCalendar(auditType.getAuditTime()));
                    resultMap.put("auditTimeFormatted",
                            this.getReportHelper()
                            .getFormattedDateTime(
                            GregorianDateUtil
                            .getDateFromGregorianCalendar(auditType
                            .getAuditTime())));
                    resultMap.put("userId", ReportDataProcessor
                            .nullEmptyReplaceWithUnknown(auditType
                            .getUserId()));
                    resultMap.put("purposeForUse", ReportDataProcessor
                            .nullEmptyReplaceWithUnknown(auditType
                            .getPurposeForUse()));
                    resultMap.put("facilityOid", ReportDataProcessor.nullEmptyReplaceWithUnknown(auditType.getOrganizationId()));
                    resultMap.put("organizationName", this.getReportHelper().findOrganizationByOid(auditType.getOrganizationId(), auditType.getOrganizationName()));

                    // if
                    // (NullChecker.isNullOrEmpty(auditType.getOrganizationId()))
                    // {
                    // logger.warning("RemoteOrganizationId Id is null/empty for audit record["
                    // + auditType.getAuditId() + "], audit details["+
                    // auditType.getDetails() + "]");
                    // }
                    resultMap.put("remoteFacilityOid", ReportDataProcessor.nullEmptyReplaceWithUnknown(auditType.getRemoteOrganizationId()));
                    resultMap.put("remoteOrganizationName", this.getReportHelper().findOrganizationByOid(auditType.getRemoteOrganizationId(), auditType.getRemoteOrganizationName()));
                    /*
                     * if
                     * (NullChecker.isNullOrEmpty(auditType.getOrganizationId(
                     * ))) { Logger.warning(
                     * "Organization Id is null/empty for audit record[" +
                     * auditType.getAuditId() + "], audit details["+
                     * auditType.getDetails() + "]"); } resultMap.put(
                     * "facility", this.getCmsHelper()
                     * .getOrganizationByHomeCommunityId(
                     * auditType.getOrganizationId())); if
                     * (NullChecker.isNullOrEmpty
                     * (auditType.getOrganizationId())) { Logger.warning(
                     * "RemoteOrganizationId Id is null/empty for audit record["
                     * + auditType.getAuditId() + "], audit details["+
                     * auditType.getDetails() + "]"); } resultMap.put(
                     * "remoteFacility", this.getCmsHelper()
                     * .getOrganizationByHomeCommunityId(
                     * auditType.getRemoteOrganizationId()));
                     */
                    if (NullChecker.isNotEmpty(auditType.getAction())) {
                        resultMap.put("message", auditType.getAction().name());
                    }

                    details = processDatails(details);

                    resultMap.put("details", details);

                    results.add(resultMap);
                }

                /*
                 * if (NullChecker.isNullOrEmpty(auditType.getOrganizationId(
                 * ))) { Logger.warning(
                 * "Organization Id is null/empty for audit record[" +
                 * auditType.getAuditId() + "], audit details["+
                 * auditType.getDetails() + "]"); } resultMap.put( "facility",
                 * this.getCmsHelper() .getOrganizationByHomeCommunityId(
                 * auditType.getOrganizationId())); if
                 * (NullChecker.isNullOrEmpty (auditType.getOrganizationId())) {
                 * Logger.warning(
                 * "RemoteOrganizationId Id is null/empty for audit record[" +
                 * auditType.getAuditId() + "], audit details["+
                 * auditType.getDetails() + "]"); } resultMap.put(
                 * "remoteFacility", this.getCmsHelper()
                 * .getOrganizationByHomeCommunityId(
                 * auditType.getRemoteOrganizationId()));
                 */
            }
            //set these outside of the null checker just in case we get an empty response we still display counts
            //put all parameters together into neat package
            SummaryPackage finishedPackage = new SummaryPackage();

            finishedPackage.setResults(results);
            finishedPackage.setRealPatientMessages(realPatientMessages);
            finishedPackage.setRealPatientFails(realPatientFails);
            finishedPackage.setRealPatientMatches(realPatientMatches);
            finishedPackage.setTestPatientMessages(testPatientMessages);
            finishedPackage.setTestPatientFails(testPatientFails);
            finishedPackage.setTestPatientMatches(testPatientMatches);
            finishedPackage.setUniqueRealPatients(uniqueRealPatients.size());
            finishedPackage.setUniqueTestPatients(uniqueTestPatients.size());
            finishedPackage.setTotalCount(totalCount);
            return finishedPackage;

        } catch (final AuditException ex) {
            throw new AuditException();
        }

    }

    /**
     * Takes parameter request and pulls all the variables needed for query search, using default values
     * where applicable
     * @param request
     * @param getALL
     * @return
     */
    public GetAudits prepareAuditParameters(HttpServletRequest request, Boolean getALL){
        final HttpSession session = request.getSession(false);

        final String icn = (String) session
                .getAttribute("patientDiscoveryQueryICN");
        final String ssn = request.getParameter("ssn");
        final String lastName = request.getParameter("lastName");
        final String firstName = request.getParameter("firstName");
        final String patientDiscoveryUserId = request.getParameter("userId");
        final Date sd = ReportHelper.getStartDate(request.getParameter("startDate"));
        Date ed = ReportHelper.getEndDate(request.getParameter("endDate"));
        final String organization = request.getParameter("organization");
        String mpiMatch = (request.getParameter("mpiMatch")!=null)?
                request.getParameter("mpiMatch") : "ALL";
        final int patientTypes = (request.getParameter("patientTypes") != null) ?
                Integer.parseInt(request.getParameter("patientTypes")) : 1; //default to real patients as is shown on the form
        int recordsPerPage = (request.getParameter("length")!=null) ?
                Integer.parseInt(request.getParameter("length")) : 25;
        final int pageNumber = (request.getParameter("start") != null)?
            Integer.parseInt(request.getParameter("start"))/Integer.parseInt(request.getParameter("length")) : 0;

        if(getALL){
            recordsPerPage = -1;
        }

        // sort
        String patientDiscoverySortValue = getSortColumnValue(request.getParameter("order[0][column]"));
        String patientDiscoverySortDirection = request.getParameter("order[0][dir]");

        //if called by export button, sort values are passed under different param
        if(patientDiscoverySortValue == null || patientDiscoverySortDirection == null){
            patientDiscoverySortValue = getSortColumnValue(request.getParameter("sortBy"));
            patientDiscoverySortDirection = request.getParameter("sortOrder");
        }

        if (mpiMatch.equalsIgnoreCase("all")) {
            mpiMatch = null;
        }

        final GetAudits getAuditsRequest = new GetAudits();
        if (NullChecker.isNotEmpty(sd)) {
            getAuditsRequest.setFromDate(GregorianDateUtil
                    .getGregorianCalendarByDate(sd));
        }
        if (NullChecker.isNotEmpty(ed)) {
            if (ed.equals(sd)) {
                ed = DateUtil.addTime(ed, Calendar.HOUR, 24);
            }
            getAuditsRequest.setToDate(GregorianDateUtil
                    .getGregorianCalendarByDate(ed));
        }

        // Set Patient Types
        getAuditsRequest.setPatientTypes(patientTypes);

        // Set Patient Ids
        if (NullChecker.isNotEmpty(icn)) {
            final StringValuesType patientIds = new StringValuesType();
            patientIds.getValue().add(icn);
            patientIds.setNotIn(false);
            getAuditsRequest.setPatientIds(patientIds);
        }
        // Set SSN
        if (NullChecker.isNotEmpty(ssn)) {
            String[] ssns = ssn.split(",");
            final StringValuesType patientSSNs = new StringValuesType();
            for (String ssnEntry : ssns) {
                patientSSNs.getValue().add(ssnEntry);
            }
            patientSSNs.setNotIn(false);
            getAuditsRequest.setPatientSSNs(patientSSNs);
        }
        // Set Last Name
        if (NullChecker.isNotEmpty(lastName)) {
            final StringValuesType patientLastNames = new StringValuesType();
            patientLastNames.getValue().add(lastName);
            patientLastNames.setNotIn(false);
            getAuditsRequest.setPatientLastNames(patientLastNames);
        }
        // Set First Name
        if (NullChecker.isNotEmpty(firstName)) {
            final StringValuesType patientFirstNames = new StringValuesType();
            patientFirstNames.getValue().add(firstName);
            patientFirstNames.setNotIn(false);
            getAuditsRequest.setPatientGivenNames(patientFirstNames);
        }
        // Set User ID
        if (NullChecker.isNotEmpty(patientDiscoveryUserId)) {
            final StringValuesType userId = new StringValuesType();
            userId.getValue().add(patientDiscoveryUserId);
            userId.setNotIn(false);
            getAuditsRequest.setUserIds(userId);
        }
        // Set Organization Ids
        final StringValuesType orgTypes = new StringValuesType();
        if (NullChecker.isNotEmpty(organization)) {
            orgTypes.getValue().add(organization);
        }
        orgTypes.setNotIn(false);
        getAuditsRequest.setRemoteOrganizationIds(orgTypes);
        getAuditsRequest.setOrganizationIds(orgTypes);

        // sorting
        if (NullChecker.isNotEmpty(patientDiscoverySortValue)) {
            final SortFieldType sortField = new SortFieldType();
            sortField.setField(FieldType.fromValue(patientDiscoverySortValue));
            if ("DESC".equalsIgnoreCase(patientDiscoverySortDirection)) {
                sortField.setDirection(SortDirection.DESC);
            } else {
                sortField.setDirection(SortDirection.ASC);
            }
            final SortFieldsType sortFieldsType = new SortFieldsType();
            sortFieldsType.getSortField().add(sortField);
            getAuditsRequest.setSortFields(sortFieldsType);
        }

        // Set Action Types
        final ActionValuesType actionsType = new ActionValuesType();
        actionsType.getValue().add(ActionType.ANNOUNCE);
        actionsType.getValue().add(ActionType.CHECK_POLICY);
        actionsType.getValue().add(ActionType.MPI_FIND_MATCH);
        actionsType.getValue().add(ActionType.ADD_PATIENT_CORRELATION);
        actionsType.setNotIn(false);
        getAuditsRequest.setActions(actionsType);
        // Set MPI Match
        if ("all".equals(mpiMatch) || NullChecker.isEmpty(mpiMatch)) {
            getAuditsRequest.setDetails(null);
        } else {
            getAuditsRequest.setDetails(mpiMatch);
        }


        //Set Page Information
        final PageInfoType pageInfoType = new PageInfoType();

        pageInfoType.setPageNumber(pageNumber);
        pageInfoType.setPageSize(recordsPerPage);
        getAuditsRequest.setPageInfo(pageInfoType);

        return getAuditsRequest;
    }

    /**
     * Returns appropriate sort field value based on column number selected
     * @param colNumber
     * @return
     */
    private String getSortColumnValue(String colNumber){
        String sortValue = "";

        if("0".equals(colNumber)) {
            sortValue = "auditTime";
        } else if ("1".equals(colNumber)){
            sortValue = "patientSSN";
        } else if ("2".equals(colNumber)){
            sortValue = "patientLastName";
        } else if ("3".equals(colNumber)){
            sortValue = "patientGivenName";
        } else if ("4".equals(colNumber)){
            sortValue = "patientMiddleName";
        } else if ("5".equals(colNumber)){
            sortValue = "organizationName";
        } else if ("6".equals(colNumber)){
            sortValue = "purposeForUse";
        } else if ("7".equals(colNumber)){
            sortValue = "remoteOrganizationName";
        } else if ("8".equals(colNumber)){
            sortValue = "action";
        } else {
            sortValue = "patientLastName";
        }
        return sortValue;
    }

    /**
     * Processes Details property as follows:
     * - Masks SSN
     * - Replaces "MOTHERSMAIDENNAME" with "MMN"
     * @param details Details property of AuditType
     * @return Processed Details property string
     */
    private String processDatails(String details) {
        // Mask SSN
        int pos1 = details.indexOf("SSN=");
        int pos2;
        String ssn;
        if (pos1 !=  -1) {
            pos2 = details.indexOf(",", pos1);
            if (pos2 != -1) {
                ssn = details.substring(pos1 + 4, pos2).trim();
                if (ssn.length() > 0) {
                    details = details.substring(0, pos1 + 4) + ReportDataProcessor.maskSsn(ssn) + details.substring(pos2);
                }
            }
        }
        // Replace "MOTHERSMAIDENNAME" with "MMN"
        if (details.contains("MOTHERSMAIDENNAME=")) {
            details = details.replace("MOTHERSMAIDENNAME=", "MMN=");
        }
        return details;
    }

    /**
     * Change resultset into json string object to be returned for datatables. Since paging isnt used
     * for this particular page, draw is optional. Total is used to determine next page button disable
     * @param searchResults
     * @return
     */
    private String encodeIntoJSON(SummaryPackage resultPackage, String draw) {
        String jsonString = "";

        try {
            StringWriter json = new StringWriter();
            JSONObject obj = new JSONObject();
            List<List> data = new ArrayList<List>();

            for (Map<String, Object> row : resultPackage.getResults()) {
                List<String> dataItem = new ArrayList<String>();

                dataItem.add(row.get("auditTimeFormatted").toString());
                dataItem.add(row.get("ssnMasked").toString());
                dataItem.add(row.get("lastName").toString());
                dataItem.add(row.get("firstName").toString());
                dataItem.add(row.get("middleName").toString());
                dataItem.add(row.get("organizationName").toString());
                dataItem.add(row.get("purposeForUse").toString());
                dataItem.add(row.get("remoteOrganizationName").toString());
                dataItem.add(row.get("message").toString());
                dataItem.add(row.get("failureExplanation").toString());
                dataItem.add(row.get("details").toString());

                //add unmasked ssn for linkToPatients, does not appear as table column
                dataItem.add(row.get("ssn").toString());
                data.add(dataItem);
            }

            obj.put("data", data);
            obj.put("recordsTotal", resultPackage.getTotalCount());
            obj.put("recordsFiltered", resultPackage.getTotalCount());
            obj.put("draw", draw);

            //Add all the calculated values to json obj to be displayed on page.

            obj.put("realPatientMessages",resultPackage.getRealPatientMessages());
            obj.put("realPatientFails",resultPackage.getRealPatientFails());
            obj.put("realPatientMatches",resultPackage.getRealPatientMatches());
            obj.put("testPatientMessages",resultPackage.getTestPatientMessages());
            obj.put("testPatientFails",resultPackage.getTestPatientFails());
            obj.put("testPatientMatches",resultPackage.getTestPatientMatches());
            obj.put("uniqueRealPatients",resultPackage.getUniqueRealPatients());
            obj.put("uniqueTestPatients",resultPackage.getUniqueTestPatients());

            obj.write(json);

            jsonString = json.toString();
        } catch (JSONException ex) {
            Logger.getLogger(PatientDiscoverySummaryReport.class.getName()).log(Level.SEVERE, null, ex);
        }

        return jsonString;
    }

    public void exportToCsv(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException, IOException {

        Map<String, String> csvReportMap = new LinkedHashMap<>();
        csvReportMap.put("Date Received (CT)", "auditTimeFormatted");
        csvReportMap.put("SSN", "ssn");
        csvReportMap.put("Patient Last Name", "lastName");
        csvReportMap.put("Patient First Name", "firstName");
        csvReportMap.put("Patient Middle Name", "middleName");
        csvReportMap.put("Sender", "organizationName");
        csvReportMap.put("Sender OID", "facilityOid");
        csvReportMap.put("Purpose of Use", "purposeForUse");
        csvReportMap.put("Receiver", "remoteOrganizationName");
        csvReportMap.put("Receiver OID", "remoteFacilityOid");
        csvReportMap.put("Message", "message");
        csvReportMap.put("Explanation of Failure", "failureExplanation");
        csvReportMap.put("Details", "details");

        int minimumThreshold = Integer.parseInt((String) this.getServletContext().getAttribute("scheduledExportMin"));
        int total = Integer.parseInt(request.getParameter("totalRows"));

        //Start a scheduled export with its own thread if greater than the minimum threshold
        if (total >= minimumThreshold) {
            ExcelGeneratorThread csvGenThread = new ExcelGeneratorThread("csvGenThread");
            csvGenThread.setDocumentType("csv");
            csvGenThread.setReportMap(csvReportMap);
            csvGenThread.setTitle(REPORT_TITLE);
            csvGenThread.setUserId(UserHelper.getUserName(request));
            csvGenThread.setAdapterAuditManager(adapterAuditManager);
            csvGenThread.setReportHelper(getReportHelper());
            csvGenThread.setUserDocumentDao(getUserDocumentDAO());

            // Set search attributes for getting audit results
            HttpSession session = request.getSession(false);
            csvGenThread.setAttributes(mapSessionAttributes(session, request));

            csvGenThread.start();
        }
        //Otherwise generate and download the export directly
        else {
            List<Map<String, Object>> results;
            PatientDiscoveryReport.SummaryPackage packedResults;
            packedResults = this.getResults(request, true, true);
            results = packedResults.getResults();

            CsvExporter csvExporter = new CsvExporter();
            csvExporter.exportToCSV(response, REPORT_TITLE.replace(" ", "_"), results, (LinkedHashMap<String, String>) csvReportMap);
        }
    }

    /**
     * Export to excel - This method is called when the user clicks on the excel
     * icon in the JSP. Reflection is used in the ResponseDispatcherHttpServlet
     * to call this method based on the http parameters.
     *
     * @param request HTTP Request
     * @param response HTTP Response
     * @throws ServletException
     * @throws IOException
     */
    public void exportToExcel(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {

        //Get the session final
        HttpSession session = request.getSession(false);

        //      Create a map of key which is based on the result data key and the
        //      heading value The heading is used to create the column headers and the
        //      key is used to pull the data from the results
        final Map<String, String> patientDiscoveryMap = new LinkedHashMap<String, String>();

        // Set the optional rows
        final Date startDate = ReportHelper.getStartDate(request.getParameter("startDate"));
        Date endDate = ReportHelper.getEndDate(request.getParameter("endDate"));
        String patientMpiResPmpi = (request.getParameter("mpiMatch")!=null)?
                request.getParameter("mpiMatch") : "ALL";

        if (patientMpiResPmpi.equalsIgnoreCase("all")) {
            patientMpiResPmpi = null;
        }

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        final LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();

        filters.put("Start Date", this.getReportHelper().getFormattedDate(startDate));
        filters.put("End Date", this.getReportHelper().getFormattedDate(endDate));
        filters.put("SSN", request.getParameter("ssn"));
        filters.put("Last Name", request.getParameter("lastName"));
        filters.put("First Name", request.getParameter("firstName"));
        filters.put("User ID", request.getParameter("userId"));
        filters.put("MPI Results", ExcelExporter.getFilterValue(patientMpiResPmpi));
        filters.put("eHealth Exchange Org", ExcelExporter.getFilterValue(request.getParameter("organizationName")));
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, request.getParameter("patientTypes"));

        patientDiscoveryMap.put("auditTimeFormatted", "Date Received (CT)");
        patientDiscoveryMap.put("ssn", "SSN");
        patientDiscoveryMap.put("lastName", "Patient Last Name");
        patientDiscoveryMap.put("firstName", "Patient First Name");
        patientDiscoveryMap.put("middleName", "Patient Middle Name");
        patientDiscoveryMap.put("organizationName", "Sender");
        patientDiscoveryMap.put("facilityOid", "Sender OID");
        patientDiscoveryMap.put("purposeForUse", "Purpose of Use");
        patientDiscoveryMap.put("remoteOrganizationName", "Receiver");
        patientDiscoveryMap.put("remoteFacilityOid", "Receiver OID");
        patientDiscoveryMap.put("message", "Message");
        patientDiscoveryMap.put("failureExplanation", "Explanation of Failure");
        patientDiscoveryMap.put("details", "Details");

        int minimumThreshold = Integer.parseInt((String) this.getServletContext().getAttribute("scheduledExportMin"));
        int total = Integer.parseInt(request.getParameter("totalRows"));

        //Start a scheduled export with its own thread if greater than the minimum threshold
        if (total >= minimumThreshold) {
            ExcelGeneratorThread exGenThread = new ExcelGeneratorThread("exGenThread");
            exGenThread.setFilterMap(filterMap);
            exGenThread.setReportMap(patientDiscoveryMap);
            exGenThread.setHasOptionalRows(true);
            exGenThread.setOptionalRowsMap(createOptionalRowsMap(Integer.valueOf(request.getParameter("patientTypes"))));

            // Set names for Excel generator thread
            exGenThread.setTitle(REPORT_TITLE);
            exGenThread.setExcelExporter(this.getExcelExporter());
            exGenThread.setAdapterAuditManager(adapterAuditManager);
            exGenThread.setReportHelper(getReportHelper());
            exGenThread.setUserId(UserHelper.getUserName(request));
            exGenThread.setUserDocumentDao(getUserDocumentDAO());

            // Set search attributes for getting audit results
            exGenThread.setAttributes(mapSessionAttributes(session, request));

            exGenThread.start();
        }
        //Otherwise generate and download the export directly
        else {
            // Create workbook
            final String title = REPORT_TITLE;

            List<Map<String, Object>> results;
            PatientDiscoveryReport.SummaryPackage packedResults;
            packedResults = this.getResults(request, true, true);
            results = packedResults.getResults();

            final Workbook wb = this.getExcelExporter().exportToExcel(title, title, patientDiscoveryMap, results, filterMap, null);

            // Write Excel workbook to Stream
            this.getExcelExporter().writeExcelToStream(REPORT_TITLE.replace(" ", "_"), wb, response);
        }
    }

    private Map<String, Object> mapSessionAttributes(HttpSession session,HttpServletRequest request ) {
        Map<String, Object> attributes = new HashMap<String, Object>();

        String mpiMatch = (request.getParameter("mpiMatch")!=null)?
                request.getParameter("mpiMatch") : "ALL";

        if (mpiMatch.equalsIgnoreCase("all")) {
            mpiMatch = null;
        }

        attributes.put("icn", (String) session
                .getAttribute("patientDiscoveryQueryICN"));
        attributes.put("ssn", request.getParameter("ssn"));
        attributes.put("lastName", request.getParameter("lastName"));
        attributes.put("firstName", request.getParameter("firstName"));
        attributes.put("userId", request.getParameter("userId"));

        // Add 24 hours to end date if same as start date
        Date startDate = ReportHelper.getStartDate(request.getParameter("startDate"));
        Date endDate = ReportHelper.getEndDate(request.getParameter("endDate"));
        if (NullChecker.isNotEmpty(endDate)) {
            if (endDate.equals(startDate)) {
                endDate = DateUtil.addTime(endDate, Calendar.HOUR, 24);
            }
        }
        attributes.put("startDate", startDate);
        attributes.put("endDate", endDate);

        // Set both sender and receiver org to the organization entered
        attributes.put("organization", request.getParameter("organization"));
        attributes.put("remoteOrganization", (String) session
                .getAttribute("patientDiscoveryQueryOrganization"));

        attributes.put("mpiMatch", mpiMatch);
        attributes.put("patientTypes", Integer.parseInt(request.getParameter("patientTypes")));
        // sort
        final String patientDiscoverySortValue = getSortColumnValue(request.getParameter("sortBy"));
        final String patientDiscoverySortDirection = request.getParameter("sortOrder");

        attributes.put("sortValue", patientDiscoverySortValue);
        attributes.put("sortDirection", patientDiscoverySortDirection);

        // Set Action Types
        final ActionValuesType actionsType = new ActionValuesType();
        actionsType.getValue().add(ActionType.ANNOUNCE);
        actionsType.getValue().add(ActionType.CHECK_POLICY);
        actionsType.getValue().add(ActionType.MPI_FIND_MATCH);
        actionsType.getValue().add(ActionType.ADD_PATIENT_CORRELATION);
        actionsType.setNotIn(false);
        attributes.put("actionsType", actionsType);

        return attributes;
    }

    Map<String, String> createOptionalRowsMap(int patientTypes) {
        Map<String, String> optionalRowsMap = new LinkedHashMap<String, String>();

        if (patientTypes == 1) {
            optionalRowsMap.put("uniqueRealPatients", "Unique patients");
            optionalRowsMap.put("realPatientMessages", "Total messages");
            optionalRowsMap.put("realPatientFails", "Match failures");
            optionalRowsMap.put("realPatientMatches", "Matches found");
        } else if (patientTypes == 2) {
            optionalRowsMap.put("uniqueTestPatients", "Unique patients");
            optionalRowsMap.put("testPatientMessages", "Total messages");
            optionalRowsMap.put("testPatientFails", "Match failures");
            optionalRowsMap.put("testPatientMatches", "Matches found");
        } else {
            optionalRowsMap.put("uniqueRealPatients", "Unique real patients");
            optionalRowsMap.put("realPatientMessages", "Total messages for real patients");
            optionalRowsMap.put("realPatientFails", "Match failures for real patients");
            optionalRowsMap.put("realPatientMatches", "Matches found for real patients");

            optionalRowsMap.put("", "");

            optionalRowsMap.put("uniqueTestPatients", "Unique test patients");
            optionalRowsMap.put("testPatientMessages", "Total messages for test patients");
            optionalRowsMap.put("testPatientFails", "Match failures for test patients");
            optionalRowsMap.put("testPatientMatches", "Matches found for test patients");
        }

        return optionalRowsMap;
    }

    /**
     * Get the excel exporter class from Spring.
     *
     * @return the excel exporter object
     */
    public ExcelExporter getExcelExporter() {
        final ExcelExporter excelExporter = this.getBean("excelExporter",
                ExcelExporter.class);
        return excelExporter;
    }

    private UserDocumentDAO getUserDocumentDAO() {
        return this.getBean("UserDocumentDAO", UserDocumentDAO.class);
    }

    @EJB(beanInterface = AdapterAuditManager.class, mappedName = "AdapterAuditManager")
    public void setAdapterAuditManager(
            final AdapterAuditManager adapterAuditManager) {
        this.adapterAuditManager = adapterAuditManager;
    }

    /*
    Class used to bring summary information together to aide in passing parameters and avoid using session
    */
    private class SummaryPackage{

        List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        long realPatientMessages = 0;
        long realPatientFails = 0;
        long realPatientMatches = 0;
        long testPatientMessages = 0;
        long testPatientFails = 0;
        long testPatientMatches = 0;
        long uniqueRealPatients = 0;
        long uniqueTestPatients = 0;
        Long totalCount;

         public List<Map<String, Object>> getResults() {
            return results;
        }

        public void setResults(List<Map<String, Object>> results) {
            this.results = results;
        }

        public long getRealPatientMessages() {
            return realPatientMessages;
        }

        public void setRealPatientMessages(long realPatientMessages) {
            this.realPatientMessages = realPatientMessages;
        }

        public long getRealPatientFails() {
            return realPatientFails;
        }

        public void setRealPatientFails(long realPatientFails) {
            this.realPatientFails = realPatientFails;
        }

        public long getRealPatientMatches() {
            return realPatientMatches;
        }

        public void setRealPatientMatches(long realPatientMatches) {
            this.realPatientMatches = realPatientMatches;
        }

        public long getTestPatientMessages() {
            return testPatientMessages;
        }

        public void setTestPatientMessages(long testPatientMessages) {
            this.testPatientMessages = testPatientMessages;
        }

        public long getTestPatientFails() {
            return testPatientFails;
        }

        public void setTestPatientFails(long testPatientFails) {
            this.testPatientFails = testPatientFails;
        }

        public long getTestPatientMatches() {
            return testPatientMatches;
        }

        public void setTestPatientMatches(long testPatientMatches) {
            this.testPatientMatches = testPatientMatches;
        }

        public long getUniqueRealPatients() {
            return uniqueRealPatients;
        }

        public void setUniqueRealPatients(int uniqueRealPatients) {
            this.uniqueRealPatients = uniqueRealPatients;
        }

        public long getUniqueTestPatients() {
            return uniqueTestPatients;
        }

        public void setUniqueTestPatients(int uniqueTestPatients) {
            this.uniqueTestPatients = uniqueTestPatients;
        }

        public Long getTotalCount() {
            return totalCount;
        }

        public void setTotalCount(Long totalCount) {
            this.totalCount = totalCount;
        }
    }
}
