package gov.va.nvap.web.report;

import gov.va.nvap.common.sort.BubbleSortListMap;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.ConsentType;
import gov.va.nvap.web.dao.ActiveConsentSummaryDAO;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.consent.audit.ActiveConsentSummary;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.util.xls.ExcelExporter;
import java.io.IOException;
import java.io.StringWriter;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.poi.ss.usermodel.Workbook;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * This servlet handles the summary Opt-In Patients report.
 * 
 * @since 12/04/2014
 * @author Irakli Kakushadze
 */
public class OptInSummaryReport extends ResponseDispatcherHttpServlet {

    /**
     *
     */
    private static final long serialVersionUID = -3952503136988071993L;

    public void doSearch(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException, ParseException {
        final HttpSession session = request.getSession(false);

        // Perform the search.
        List<List<String>> searchResponse = getResults(request).results;

        // Generate JSON results.
        session.setAttribute("results", encodeIntoJSON(searchResponse));

        // Forward the response.
        this.forward(request, response, "searchResultsJSON");
    }

    public void exportToExcel(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException, IOException {
        //Get the results from the adapter with the paginator set to -1 (all //  results) 
        PackedResults packedResults = this.getResults(request);
        final List<List<String>> results = packedResults.results;
        // Create a map of key which is based on the resultrequest data key and the heading value 
        //The heading is used to create the column headers and the
        //key is used to pull the data from the results 
        final Map<String, String> reportMap = new LinkedHashMap<String, String>();
        if (NullChecker.isNullOrEmpty(results)) {
            reportMap.put("", "No records were found.");
        } else {
            reportMap.put("Authenticating Facility", "Authenticating Facility");
            reportMap.put("eHealth Exchange Authorizations", "eHealth Exchange Authorizations");
            reportMap.put("SSA Authorizations", "SSA Authorizations");
            reportMap.put("Facility Total", "Facility Total");
        }
        
        //Remap the results to List<Map<String,Object>> so it can be passed to export class.
        List<Map<String, Object>> resultsMap = new ArrayList<Map<String, Object>>();
        for (List<String> listEntry : results) {
            Map<String, Object> mapEntry = new LinkedHashMap<String, Object>();
            mapEntry.put("Authenticating Facility", listEntry.get(0));
            mapEntry.put("eHealth Exchange Authorizations", listEntry.get(1));
            mapEntry.put("SSA Authorizations", listEntry.get(2));
            mapEntry.put("Facility Total", listEntry.get(3));
            resultsMap.add(mapEntry);
        }
        
        final Map<String, List<Object>> optionalRows = new LinkedHashMap<String, List<Object>>();
        optionalRows.put("Total", packedResults.totals);

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();

        String facilityNames = ExcelExporter.getFacilitiesFilter(this.getFacilityDAO(), request.getParameter("stationNumbers"));
        filters.put("Authenticating Facility", facilityNames);
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, request.getParameter("patientTypes"));

        // Create workbook
        final String title = "Opt-In Patients Summary Report";
        final Workbook wb = this.getExcelExporter().exportToExcel(title, title, reportMap, resultsMap, filterMap, optionalRows);

        // Write Excel to Stream
        this.getExcelExporter().writeExcelToStream("Opt-In_Patients_Summary_Report", wb, response);
    }

    @Override
    protected void unspecified(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        this.forward(request, response, "searchForm"); 
    }

    /**
     * Gets ActiveConsentSummaryDAO class from Spring.
     *
     * @return ActiveConsentSummaryDAO object
     */
    private ActiveConsentSummaryDAO getActiveConsentSummaryDAO() {
        return this.getBean("ActiveConsentSummaryDAO", ActiveConsentSummaryDAO.class);
    }

    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }
    
    /**
     * Gets ExcelExporter class from Spring.
     *
     * @return ExcelExporter object
     */
    private ExcelExporter getExcelExporter() {
        return this.getBean("excelExporter", ExcelExporter.class);
    }

    private String encodeIntoJSON(List<List<String>> searchResponse) throws ParseException {
        String returnValue = "";

        try {
            StringWriter json = new StringWriter();
            JSONObject obj = new JSONObject();
            List<List> data = new ArrayList<List>();

            if (searchResponse != null) {
                for (List<String> row : searchResponse) {
                    List<String> dataItem = new ArrayList<String>();

                    for (String cell : row) {
                        dataItem.add(cell);
                    }

                    data.add(dataItem);
                }
            }
            obj.put("data", data);

            obj.write(json);

            returnValue = json.toString();
        } catch (JSONException ex) {
            Logger.getLogger(OptInSummaryReport.class.getName()).log(Level.SEVERE, null, ex);
        }

        return returnValue;
    }
    
    /**
     * Generates the summary report for opted-in patients, which is a matrix of String values. The first column contains facility names, and
     * the last column contains totals for each row. The last row contains totals for each authorization type, and the grand total in the
     * last column. The following diagram shows the layout of the report:
     *
     * Facility1    [###]       [###]    ...    [###]     [row total]
     * Facility2    [###]       [###]    ...    [###]     [row total]
     * ...           ...         ...     ...     ...          ... 
     * FacilityN    [###]       [###]    ...    [###]     [row total]
     * Total     [col total] [col total] ... [col total] [grand total]
     *
     * @return List<List<String>> Summary report of opted-in patients.
     */
    private PackedResults getResults(HttpServletRequest request) {

        List<List<String>> results = new ArrayList<List<String>>();
        PackedResults endResults  = new PackedResults();
        try {
            ActiveConsentSummaryDAO dao = this.getActiveConsentSummaryDAO();

            ActiveConsentSummaryDAO.SummaryRequest searchRequest = new ActiveConsentSummaryDAO().new SummaryRequest();
            searchRequest.aggregateAtFacilityLevel =  "true".equals(request.getParameter("aggregateAtFacilityLevel"));
            searchRequest.includeUnknownVisn = (request.getParameter("includeUnknownVisn")==null)?
                    false : "true".equals(request.getParameter("includeUnknownVisn"));
            searchRequest.stationNumbers = ReportHelper.getStationNumbersFromRequest(request);
            
            List<ActiveConsentSummary> activeConsentSummaryList = dao.find(searchRequest);

            if ((activeConsentSummaryList != null) && (activeConsentSummaryList.size() > 0)) {
                List<ConsentType> allowedConsentTypes = Arrays.asList(ConsentType.NW_HIN_AUTHORIZATION, ConsentType.SSA_AUTHORIZATION);
                Map<String, Long> totalsByConsentType = new HashMap<String, Long>();
                List<String> row;
                Map<String, Map<String, Long>> map = new LinkedHashMap<String, Map<String, Long>>();
                int patientTypes = NullChecker.isNullOrEmpty(request.getParameter("patientTypes")) ? 1 : Integer.parseInt(request.getParameter("patientTypes"));

                for (ConsentType allowedConsentType : allowedConsentTypes) {
                    totalsByConsentType.put(allowedConsentType.value(), new Long(0));
                }
                totalsByConsentType.put("Total", new Long(0));

                for (ActiveConsentSummary activeConsentSummary : activeConsentSummaryList) {
                    if (!map.containsKey(activeConsentSummary.getFacilityName())) {
                        map.put(activeConsentSummary.getFacilityName(), new HashMap<String, Long>());
                    }
                    switch (patientTypes) {
                        case 1: // Real patients only
                            map.get(activeConsentSummary.getFacilityName()).put(activeConsentSummary.getOptinConsentType().getName(), activeConsentSummary.getCountReal());
                            break;
                        case 2: // Test patients only
                            map.get(activeConsentSummary.getFacilityName()).put(activeConsentSummary.getOptinConsentType().getName(), (activeConsentSummary.getTotal() - activeConsentSummary.getCountReal()));
                            break;
                        case 3: // Both
                            map.get(activeConsentSummary.getFacilityName()).put(activeConsentSummary.getOptinConsentType().getName(), activeConsentSummary.getTotal());
                            break;
                    }
                }

                List<Map<String, Object>> dataList = new ArrayList<Map<String, Object>>();
                for (Map.Entry<String, Map<String, Long>> facility : map.entrySet()) {
                    final Map<String, Object> dataMap = new HashMap<String, Object>();
                    dataMap.put("facility", facility.getKey());
                    long total = 0;
                    for (ConsentType allowedConsentType : allowedConsentTypes) {
                        if (facility.getValue().containsKey(allowedConsentType.value())) {
                            long value = facility.getValue().get(allowedConsentType.value());
                            dataMap.put(allowedConsentType.getValue(), value);
                            total += value;
                            totalsByConsentType.put(allowedConsentType.value(), totalsByConsentType.get(allowedConsentType.value()) + value);
                        } else {
                            dataMap.put(allowedConsentType.getValue(), 0L);
                        }
                    }
                    dataMap.put("total", total);
                    totalsByConsentType.put("Total", totalsByConsentType.get("Total") + total);
                    dataList.add(dataMap);
                }
                
                if (dataList.size() > 1) {
                    dataList = sortResults(request, dataList);
                }
                
                for (Map<String, Object> dataMap : dataList) {
                    row = new ArrayList<String>();
                    row.add((String) dataMap.get("facility"));
                    for (ConsentType allowedConsentType : allowedConsentTypes) {
                        row.add(dataMap.get(allowedConsentType.value()).toString());
                    }
                    row.add(dataMap.get("total").toString());
                    results.add(row);
                }

                row = new ArrayList<String>();
                for (ConsentType allowedConsentType : allowedConsentTypes) {
                    row.add(totalsByConsentType.get(allowedConsentType.value()).toString());
                }
                row.add(totalsByConsentType.get("Total").toString());
                
                endResults.results = results;
                endResults.totals = row;
            }
        } catch (final Exception ex) {
            throw new RuntimeException(ex);
        }
        return endResults;
    }
    
    private List<Map<String, Object>> sortResults(HttpServletRequest request, List<Map<String, Object>> data) {

        final BubbleSortListMap bSorter = new BubbleSortListMap();
        // When exporting, we'll receive sort field and sort order explicitly in the corresponding params.
        String sortValue = request.getParameter("order[0][column]") == null ? request.getParameter("sortBy") : request.getParameter("order[0][column]");
        boolean sortAscending = request.getParameter("order[0][dir]") == null ? !"desc".equalsIgnoreCase(request.getParameter("sortOrder"))
                : !"desc".equalsIgnoreCase(request.getParameter("order[0][dir]"));
        if ("0".equals(sortValue)) {
            sortValue = "facility";
        } else if ("1".equals(sortValue)) {
            sortValue = ConsentType.NW_HIN_AUTHORIZATION.value();
        } else if ("2".equals(sortValue)) {
            sortValue = ConsentType.SSA_AUTHORIZATION.value();
        } else if ("3".equals(sortValue)) {
            sortValue = "total";
        } else {
            sortValue = "facility";
        }
        return bSorter.sortByColumn(data, sortValue, sortAscending);

    }
    
    private class PackedResults{
        List<List<String>> results;
        List totals;
    }

}
