package gov.va.nvap.web.report;

import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.ConsentType;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.consent.audit.AuditedConsentSummary;
import gov.va.nvap.web.consent.audit.dao.AuditedConsentDAO;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.util.Constants;
import gov.va.nvap.web.util.xls.ExcelExporter;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.LinkedHashMap;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.poi.ss.usermodel.Workbook;

/**
 * Consent Directive Summary Report results servlet
 *
 * @author Asha Amritraj edited by Stephen Miller edited by Irakli Kakushadze
 */
public class OptInOptOutSummaryReportResults extends ResponseDispatcherHttpServlet {

    /**
     * Serial UID.
     */
    private static final long serialVersionUID = -1131510956121479062L;

    /**
     * Export to excel - This method is called when the user clicks on the excel icon in the JSP. Reflection is used in the
     * ResponseDispatcherHttpServlet to call this method based on the http parameters.
     *
     * @param request HTTP Request
     * @param response HTTP Response
     * @throws ServletException
     * @throws IOException
     */
    public void exportToExcel(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
        final HttpSession session = request.getSession(false);
        final List<Map<String, Object>> results = this.getResults(request.getSession());
        final Map<String, String> consentDirectiveSummaryMap = new LinkedHashMap<String, String>();
        // Set the optional rows
        final Object nwhinAuthorizationTotal = session.getAttribute("nwhinAuthorizationTotal");
        final Object ssaAuthorizationTotal = session.getAttribute("ssaAuthorizationTotal");
        final Object nwhinRevocationTotal = session.getAttribute("nwhinRevocationTotal");
        final Object ssaRevocationTotal = session.getAttribute("ssaRevocationTotal");
        final Object nwhinRestrictionTotal = session.getAttribute("nwhinRestrictionTotal");
        final Object nwhinRestrictionRevocationTotal = session.getAttribute("nwhinRestrictionRevocationTotal");
        final Map<String, List<Object>> optionalRows = new LinkedHashMap<String, List<Object>>();
        final List<Object> nwhinAuthorizationRow = new ArrayList<Object>();
        nwhinAuthorizationRow.add(nwhinAuthorizationTotal);
        final List<Object> ssaAuthorizationRow = new ArrayList<Object>();
        ssaAuthorizationRow.add(ssaAuthorizationTotal);
        final List<Object> nwhinRevocationRow = new ArrayList<Object>();
        nwhinRevocationRow.add(nwhinRevocationTotal);
        final List<Object> ssaRevocationRow = new ArrayList<Object>();
        ssaRevocationRow.add(ssaRevocationTotal);
        final List<Object> nwhinRestrictionRow = new ArrayList<Object>();
        nwhinRestrictionRow.add(nwhinRestrictionTotal);
        final List<Object> nwhinRestrictionRevocationRow = new ArrayList<Object>();
        nwhinRestrictionRevocationRow.add(nwhinRestrictionRevocationTotal);
        optionalRows.put("Grand Total eHealth Exchange Authorizations", nwhinAuthorizationRow);
        optionalRows.put("Grand Total SSA Authorizations", ssaAuthorizationRow);
        optionalRows.put("Grand Total eHealth Exchange Revocations", nwhinRevocationRow);
        optionalRows.put("Grand Total SSA Revocations", ssaRevocationRow);
        optionalRows.put("Grand Total eHealth Exchange Restrictions", nwhinRestrictionRow);
        optionalRows.put("Grand Total eHealth Exchange Restriction Revocations", nwhinRestrictionRevocationRow);

        // Generate filters
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        final LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();

        filters.put("Start Date", this.getReportHelper().getFormattedDate((Date) session.getAttribute("optInOptOutSummaryQueryStartDate")));
        filters.put("End Date", this.getReportHelper().getFormattedDate((Date) session.getAttribute("optInOptOutSummaryQueryEndDate")));
        filters.put("Authenticating Facility", ExcelExporter.getFacilitiesFilter(getFacilityDAO(), (String) session.getAttribute("optInOptOutSummaryQueryStationNumbers")));
        filters.put("Consent Type", ExcelExporter.getFilterValue(session.getAttribute("consentTypeName")));
        filters.put("Entered By", session.getAttribute("enteredBy"));
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, session.getAttribute("patientTypes").toString());

        if (NullChecker.isNullOrEmpty(results)) {
            consentDirectiveSummaryMap.put("", "No records were found.");
        } else {
            consentDirectiveSummaryMap.put("facilityName", "Authenticating Facility");
            consentDirectiveSummaryMap.put("consentType", "Consent Type");
            consentDirectiveSummaryMap.put("count", "Total");
        }

        // Create workbook
        String title = "Consent Directive Summary Report";
        final Workbook wb = this.getExcelExporter().exportToExcel(title, title, consentDirectiveSummaryMap, results, filterMap, optionalRows);
        // Write Excel to Stream

        this.getExcelExporter().writeExcelToStream("Consent_Directive_Summary_Report", wb, response);
    }

    @Override
    public void unspecified(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
        final HttpSession session = request.getSession(false);
        final String optInOptOutSummaryReportSortValue = request.getParameter("optInOptOutSummaryReportSortValue");
        final String optInOptOutSummaryReportSortDirection = request.getParameter("optInOptOutSummaryReportSortDirection");

        session.setAttribute("optInOptOutSummaryReportSortValue", optInOptOutSummaryReportSortValue);
        session.setAttribute("optInOptOutSummaryReportSortDirection", optInOptOutSummaryReportSortDirection);
        session.setAttribute("optInOptOutSummaryResults", this.getResults(session));

        this.forward(request, response, "show");
    }

    private List<Map<String, Object>> getResults(final HttpSession session) throws ServletException {
        final AuditedConsentDAO auditedConsentDAO = this.getBean("auditedConsentDAO", AuditedConsentDAO.class);
        final AuditedConsentDAO.SummaryRequest req = auditedConsentDAO.new SummaryRequest();
        Long nwhinAuthorizationTotal = 0L;
        Long ssaAuthorizationTotal = 0L;
        Long nwhinRevocationTotal = 0L;
        Long ssaRevocationTotal = 0L;
        Long nwhinRestrictionTotal = 0L;
        Long nwhinRestrictionRevocationTotal = 0L;

        req.aggregateAtFacilityLevel = Boolean.TRUE.equals(session.getAttribute("aggregateAtFacilityLevel"));
        req.consentType = (String) session.getAttribute("consentType");
        req.endDate = (Date) session.getAttribute("optInOptOutSummaryQueryEndDate");
        req.includeUnknownVisn = Boolean.TRUE.equals(session.getAttribute("includeUnknownVisn"));
        req.patientTypes = (Integer) session.getAttribute("patientTypes");
        req.sortDirection = (String) session.getAttribute("optInOptOutSummaryReportSortDirection");
        req.sortField = (String) session.getAttribute("optInOptOutSummaryReportSortValue");
        req.startDate = (Date) session.getAttribute("optInOptOutSummaryQueryStartDate");
        req.stationNumbers = (String) session.getAttribute("optInOptOutSummaryQueryStationNumbers");
        req.userId = (String) session.getAttribute("enteredBy");

        final List<AuditedConsentSummary> auditedConsentSummaryList = auditedConsentDAO.getEventsSummary(req);
        final List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();

        for (final AuditedConsentSummary auditSummary : auditedConsentSummaryList) {

            final Map<String, Object> resultMap = new HashMap<String, Object>();

            String displayConsentType = auditSummary.getConsentType();
            if (displayConsentType.contains("NwHIN")) {
                displayConsentType = displayConsentType.replace("NwHIN", Constants.getOrganizationName());
            }
            resultMap.put("consentType", displayConsentType);
            resultMap.put("facilityName", ReportDataProcessor.fixStation(auditSummary.getFacility()));
            resultMap.put("count", auditSummary.getCount());

            //TODO use consentType.java
            final String consentType = auditSummary.getConsentType();
            final ConsentType ct = ConsentType.fromValue(consentType);
            switch (ct) {
                case NW_HIN_AUTHORIZATION:
                    nwhinAuthorizationTotal += auditSummary.getCount();
                    break;
                case NW_HIN_REVOCATION:
                    nwhinRevocationTotal += auditSummary.getCount();
                    break;
                case SSA_AUTHORIZATION:
                    ssaAuthorizationTotal += auditSummary.getCount();
                    break;
                case SSA_REVOCATION:
                    ssaRevocationTotal += auditSummary.getCount();
                    break;
                case NW_HIN_ORGANIZATION_RESTRICTION_AUTHORIZATION:
                    nwhinRestrictionTotal += auditSummary.getCount();
                    break;
                case NW_HIN_ORGANIZATION_RESTRICTION_REVOCATION:
                    nwhinRestrictionRevocationTotal += auditSummary.getCount();
                    break;
                default:
                    try {
                        throw new Exception("unrecognized consent type: " + consentType);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                    break;
            }
            results.add(resultMap);
        }

        session.setAttribute("nwhinAuthorizationTotal", nwhinAuthorizationTotal);
        session.setAttribute("ssaAuthorizationTotal", ssaAuthorizationTotal);
        session.setAttribute("nwhinRevocationTotal", nwhinRevocationTotal);
        session.setAttribute("ssaRevocationTotal", ssaRevocationTotal);
        session.setAttribute("nwhinRestrictionTotal", nwhinRestrictionTotal);
        session.setAttribute("nwhinRestrictionRevocationTotal", nwhinRestrictionRevocationTotal);

        if (NullChecker.isEmpty(auditedConsentSummaryList)) {
            return results;
        }

        session.setAttribute("optInOptOutSummaryReportSortValue", req.sortField);
        session.setAttribute("optInOptOutSummaryReportSortDirection", req.sortDirection);

        return results;
    }

    private ExcelExporter getExcelExporter() {
        return this.getBean("excelExporter", ExcelExporter.class);
    }

    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }

    private ReportHelper getReportHelper() {
        return this.getBean("reportHelper", ReportHelper.class);
    }

}
