package gov.va.nvap.web.report;

import gov.va.nvap.common.sort.BubbleSortListMap;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.ConsentType;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.consent.audit.AuditedConsentSummary;
import gov.va.nvap.web.consent.audit.dao.AuditedConsentDAO;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.util.Constants;

import gov.va.nvap.web.util.xls.ExcelExporter;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;


import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.poi.ss.usermodel.Workbook;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Consent Directive Summary Report search servlet
 *
 * @author Asha Amritraj edited by Stephen Miller, updated by Raul Alfaro
 */
public class OptInOptOutSummaryReport extends ResponseDispatcherHttpServlet {

    /**
     * Serial UID.
     */
    private static final long serialVersionUID = 3721503355079491829L;

    @Override
    protected void unspecified(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
        final HttpSession session = request.getSession(false);
        
        ReportHelper.setDefaultSearchDates(session);
        session.removeAttribute("results");
        session.removeAttribute("sortCol");
        session.removeAttribute("sortDir");
        session.removeAttribute("unproccessedResults");

        this.forward(request, response, "show");
    }
    
    /**
     * Returns search results in JSON form. JSON contains two arrays: "data" and "summary".
     * If no request parameters are present, default values are used for search.
     * 
     * @param request
     * @param response
     * @throws ServletException
     * @throws IOException
     */
    public void doSearch(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        final HttpSession session = request.getSession(false);

        ResultPackage results = getResults(request);
        //used if only sorting is request as to not hit db again.
        session.setAttribute("unproccessedResults", results);
        session.setAttribute("sortCol", request.getParameter("order[0][column]"));
        session.setAttribute("sortDir", request.getParameter("order[0][dir]"));

        //actual results used by DT
        session.setAttribute("results", encodeIntoJSON(results));
        
        this.forward(request, response, "searchResultsJSON");
    }

    /**
     * Gathers search results and summary totals, packaging them together for ease in transport.
     * 
     * @param httpRequest
     * @return ResultPackage
     * @throws ServletException
     */
    private ResultPackage getResults(HttpServletRequest httpRequest) throws ServletException {
        HttpSession session = httpRequest.getSession(false);
        /*
        Compare sort col/dir of previous call. if they are the same, it is a new search request. If it it different its a resorting. use bubble.
        Should something go wrong, a normal search is done.
        */
        String sortColCheck = (String)session.getAttribute("sortCol");
        String sortDirCheck = (String)session.getAttribute("sortDir");
        
        if(sortColCheck != null && sortDirCheck != null){
            
            if(httpRequest.getParameter("order[0][column]") != null && httpRequest.getParameter("order[0][dir]") != null){
                /*
                if order[0][column] and order[0][dir] are null then its an excel call, new search.
                */
                if(!sortColCheck.equals(httpRequest.getParameter("order[0][column]"))
                        || !sortDirCheck.equals(httpRequest.getParameter("order[0][dir]"))){
                    //if here then dir/col were not the same as the last call which means resorting.

                    ResultPackage previousResult = (ResultPackage)session.getAttribute("unproccessedResults");

                    if(previousResult!=null){
                        previousResult.setSearchResults( sortResults(previousResult.getSearchResults(), httpRequest) );
                        //return newly sorted old result as to not call db again.
                        return previousResult;
                    }
                }
            }
        }
        
        final AuditedConsentDAO auditedConsentDAO = this.getBean("auditedConsentDAO", AuditedConsentDAO.class);
        final AuditedConsentDAO.SummaryRequest req = prepareSummaryRequest(auditedConsentDAO.new SummaryRequest(), httpRequest);
        
        Long nwhinAuthorizationTotal = 0L;
        Long ssaAuthorizationTotal = 0L;
        Long nwhinRevocationTotal = 0L;
        Long ssaRevocationTotal = 0L;
        Long nwhinRestrictionTotal = 0L;
        Long nwhinRestrictionRevocationTotal = 0L;

        final List<AuditedConsentSummary> auditedConsentSummaryList = auditedConsentDAO.getEventsSummary(req);
        final List<Map<String, Object>> endResults = new ArrayList<Map<String, Object>>();
        final Map<String, FacilityResults> combinedByFacilityResults = new LinkedHashMap<String,FacilityResults>();
        
        for (final AuditedConsentSummary auditSummary : auditedConsentSummaryList) {


            final String facility = ReportDataProcessor.fixStation(auditSummary.getFacility());
            String displayConsentType = auditSummary.getConsentType();
            if (displayConsentType.contains("NwHIN")) {
                displayConsentType = displayConsentType.replace("NwHIN", Constants.getOrganizationName());
            }
            
            //combine all reports by facility
            if(!combinedByFacilityResults.containsKey(facility)){
                //if new entry for facility, create class object and add to map
                FacilityResults facilityResults = new FacilityResults(facility);
                facilityResults.addConsentType(displayConsentType, auditSummary.getCount());
                
                combinedByFacilityResults.put(facility,facilityResults);
            } else {
                //if facility object already exists, add consent type value to existing object
                FacilityResults facilityResults = combinedByFacilityResults.get(facility);
                facilityResults.addConsentType(displayConsentType, auditSummary.getCount());
            }

            final String consentType = auditSummary.getConsentType();
            final ConsentType ct = ConsentType.fromValue(consentType);
            switch (ct) {
                case NW_HIN_AUTHORIZATION:
                    nwhinAuthorizationTotal += auditSummary.getCount();
                    break;
                case NW_HIN_REVOCATION:
                    nwhinRevocationTotal += auditSummary.getCount();
                    break;
                case SSA_AUTHORIZATION:
                    ssaAuthorizationTotal += auditSummary.getCount();
                    break;
                case SSA_REVOCATION:
                    ssaRevocationTotal += auditSummary.getCount();
                    break;
                case NW_HIN_ORGANIZATION_RESTRICTION_AUTHORIZATION:
                    nwhinRestrictionTotal += auditSummary.getCount();
                    break;
                case NW_HIN_ORGANIZATION_RESTRICTION_REVOCATION:
                    nwhinRestrictionRevocationTotal += auditSummary.getCount();
                    break;
                default:
                    try {
                        throw new Exception("unrecognized consent type: " + consentType);
                    } catch (Exception e) {
                        e.printStackTrace();
                    }
                    break;
            }
        }
        
        FacilityResults currentResultsReport;
        for(String currentFacilityKey: combinedByFacilityResults.keySet()){
            currentResultsReport = combinedByFacilityResults.get(currentFacilityKey);
            endResults.add(currentResultsReport.transformToMap());
        }
        //Summary Map used in excel output
        final Map<String, Object> summaryMap = new LinkedHashMap<String, Object>();
        
        //add everything to map so it can passed around in package and later converted to JSON
        summaryMap.put(Constants.getOrganizationName() + " Authorizations", nwhinAuthorizationTotal);
        summaryMap.put(Constants.getOrganizationName() + " Revocations", nwhinRevocationTotal);
        summaryMap.put(Constants.getOrganizationName() + " Restrictions", nwhinRestrictionTotal);
        summaryMap.put(Constants.getOrganizationName() + " Restriction Revocations", nwhinRestrictionRevocationTotal);
        summaryMap.put("SSA Authorizations", ssaAuthorizationTotal);
        summaryMap.put("SSA Revocations", ssaRevocationTotal);
        
        //Package sorted search results and summary totals
        ResultPackage resultPackage = new ResultPackage( sortResults(endResults, httpRequest), summaryMap);

        return resultPackage;
    }

    /**
     * Sort List<Map<String,Object>> based on map keys (aka column names)
     * @param unsortedList
     * @param request
     * @return 
     */
    private List<Map<String,Object>> sortResults(List<Map<String,Object>> unsortedList, HttpServletRequest request){
        
        if(unsortedList.size() > 1){
            final BubbleSortListMap bSorter = new BubbleSortListMap();
            // When exporting, we'll receive sort field and sort order explicitly in the corresponding params.
            String sortValue = request.getParameter("order[0][column]") == null ? request.getParameter("sortBy") : request.getParameter("order[0][column]");
            boolean sortAscending = request.getParameter("order[0][dir]") == null ? !"desc".equalsIgnoreCase(request.getParameter("sortOrder"))
                    : !"desc".equalsIgnoreCase(request.getParameter("order[0][dir]"));

            sortValue = getSortValue(sortValue);
        
            return bSorter.sortByColumn(unsortedList, sortValue, sortAscending);
        } else {
            return unsortedList;
        }
    }
    
    /**
     * Get sort value from column location
     * @param paramValue
     * @return 
     */
    private String getSortValue(String paramValue){
        String sortValue;
        if ("0".equals(paramValue)) {
                sortValue = "facilityName";
            } else if ("1".equals(paramValue)) {
                sortValue = Constants.getOrganizationName() + " Authorization";
            } else if ("2".equals(paramValue)) {
                sortValue = Constants.getOrganizationName() + " Revocation";
            } else if ("3".equals(paramValue)) {
                sortValue = Constants.getOrganizationName() + " Organization Restriction Authorization";
            } else if("4".equals(paramValue)){
                sortValue = Constants.getOrganizationName() + " Organization Restriction Revocation";
            } else if("5".equals(paramValue)){
                sortValue = "SSA Authorization";
            } else if("6".equals(paramValue)){
                sortValue = "SSA Revocation";
            } else {
                sortValue = "facilityName";
            }
        return sortValue;
    }
    
    /**
     * Add all of the httpServletRequest parameters to the AuditConsentDao request, using defaults if not provided.
     * 
     * @param auditRequest
     * @param httpRequest
     * @return
     */
    private AuditedConsentDAO.SummaryRequest prepareSummaryRequest(AuditedConsentDAO.SummaryRequest auditRequest,
        HttpServletRequest httpRequest) {

        auditRequest.aggregateAtFacilityLevel = "true".equals(httpRequest.getParameter("aggregateAtFacilityLevel"));
        auditRequest.consentType = "";
        auditRequest.endDate = ReportHelper.getEndDate(httpRequest.getParameter("endDate"));
        auditRequest.includeUnknownVisn =  "true".equals(httpRequest.getParameter("includeUnknownVisn"));
        auditRequest.patientTypes = (httpRequest.getParameter("patientTypes") != null) ? Integer.parseInt(httpRequest.getParameter("patientTypes")) : 1;
        auditRequest.sortDirection = "";
        auditRequest.sortField = "";
        auditRequest.startDate = ReportHelper.getStartDate(httpRequest.getParameter("startDate"));
        auditRequest.stationNumbers = ReportHelper.getStationNumbersFromRequest(httpRequest);
        auditRequest.userId = (httpRequest.getParameter("enteredBy") != null) ? (String) httpRequest.getParameter("enteredBy").trim() : "";

        return auditRequest;
    }

    /**
     * Search Results are combined into to JSON String which includes two arrays. The first contains the search results as "data", the
     * second is the summary totals array as "summary"
     *
     * @param searchResults
     * @return
     */
    private String encodeIntoJSON(ResultPackage searchResults) {
        String jsonString = "";

        try {
            StringWriter json = new StringWriter();
            JSONObject obj = new JSONObject();
            List<List> data = new ArrayList<List>();

            for (Map<String, Object> row : searchResults.getSearchResults()) {
                List<String> dataItem = new ArrayList<String>();

                for (String key : row.keySet()) {
                    dataItem.add(row.get(key).toString());
                }
                data.add(dataItem);
            }

            obj.put("data", data);

            obj.write(json);

            jsonString = json.toString();
        } catch (JSONException ex) {
            Logger.getLogger(OptInOptOutSummaryReport.class.getName()).log(Level.SEVERE, null, ex);
        }
        return jsonString;
    }

    /**
     * Export to excel - This method is called when the user clicks on the excel icon in the JSP. Reflection is used in the
     * ResponseDispatcherHttpServlet to call this method based on the http parameters.
     *
     * @param request HTTP Request
     * @param response HTTP Response
     * @throws ServletException
     * @throws IOException
     */
    public void exportToExcel(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {

        //Gather up search results
        final ResultPackage results = getResults(request);
        final Map<String, String> consentDirectiveSummaryMap = new LinkedHashMap<String, String>();

        if (NullChecker.isNullOrEmpty(results.getSearchResults())) {
            consentDirectiveSummaryMap.put("", "No records were found.");
        } else {
            consentDirectiveSummaryMap.put("facilityName", "Authenticating Facility");
            consentDirectiveSummaryMap.put(Constants.getOrganizationName() + " Authorization", Constants.getOrganizationName() + " Authorizations");
            consentDirectiveSummaryMap.put(Constants.getOrganizationName() + " Revocation", Constants.getOrganizationName() + " Revocations");
            consentDirectiveSummaryMap.put(Constants.getOrganizationName() + " Organization Restriction Authorization", Constants.getOrganizationName() + " Restrictions");
            consentDirectiveSummaryMap.put(Constants.getOrganizationName() + " Organization Restriction Revocation", Constants.getOrganizationName() + " Restriction Revocations");
            consentDirectiveSummaryMap.put("SSA Authorization", "SSA Authorizations");
            consentDirectiveSummaryMap.put("SSA Revocation", "SSA Revocations");
        }

        //Create Optional Rows
        final Map<String, List<Object>> optionalRows = new LinkedHashMap<String, List<Object>>();
        List<Object> typePlaceHolder;
        
        typePlaceHolder = new ArrayList<Object>();
        for (String key : results.getSummaryTotals().keySet()) {
            typePlaceHolder.add(results.getSummaryTotals().get(key));
        }
        optionalRows.put("Total", typePlaceHolder);

        // Generate filters
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        final LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();

        filters.put("Start Date", this.getReportHelper().getFormattedDate(request.getParameter("startDate")));
        filters.put("End Date", this.getReportHelper().getFormattedDate(request.getParameter("endDate")));
        filters.put("Authenticating Facility", ExcelExporter.getFacilitiesFilter(getFacilityDAO(), request.getParameter("stationNumbers")));
        filters.put("Entered By", request.getParameter("enteredBy"));
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, request.getParameter("patientTypes"));

        // Create workbook
        String title = "Consent Directive Summary Report";
        final Workbook wb = this.getExcelExporter().exportToExcel(title, title, consentDirectiveSummaryMap, results.getSearchResults(), filterMap, optionalRows);
        
        // Write Excel to Stream
        this.getExcelExporter().writeExcelToStream("Consent_Directive_Summary_Report", wb, response);
    }

    private ExcelExporter getExcelExporter() {
        return this.getBean("excelExporter", ExcelExporter.class);
    }

    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }

    private ReportHelper getReportHelper() {
        return this.getBean("reportHelper", ReportHelper.class);
    }

    /**
     * Neatly brings together the summary totals and the search results for passing as parameters
     * Only used within this class
     */
    private class ResultPackage {

        private List<Map<String, Object>> searchResults;
        private Map<String, Object> summaryTotals;

        public void setSearchResults(List<Map<String, Object>> searchResults) {
            this.searchResults = searchResults;
        }

        public void setSummaryTotals(Map<String, Object> summaryTotals) {
            this.summaryTotals = summaryTotals;
        }

        public ResultPackage(List<Map<String, Object>> results, Map<String, Object> summaryTotals) {
            this.searchResults = results;
            this.summaryTotals = summaryTotals;
        }

        public List<Map<String, Object>> getSearchResults() {
            return searchResults;
        }

        public Map<String, Object> getSummaryTotals() {
            return summaryTotals;
        }

    }
    
    /**
     * Class used to group all report types of a given facility.
     */
    private class FacilityResults{
        final String facilityName;
        long authorizationCount = 0;
        long restrictionsCount = 0;
        long revocationsCount = 0;
        long restrictionRevocationsCount = 0;
        long ssaAuthorizationsCount = 0;
        long ssaRevocationsCount = 0;

        final String authorizationName = Constants.getOrganizationName() + " Authorization";
        final String restrictionsName = Constants.getOrganizationName() + " Organization Restriction Authorization";
        final String revocationsName = Constants.getOrganizationName() + " Revocation";
        final String restrictionRevocationsName = Constants.getOrganizationName() + " Organization Restriction Revocation";
        final String ssaAuthorizationsName = "SSA Authorization";
        final String ssaRevocationsName = "SSA Revocation";
        

        public FacilityResults(String facilityName) {
            this.facilityName = facilityName;
        }

        public Long getAuthorization() {
            return authorizationCount;
        }

        public Long getRestrictions() {
            return restrictionsCount;
        }

        public Long getRevocations() {
            return revocationsCount;
        }
        
        public Long getRestrictionRevocations() {
            return restrictionRevocationsCount;
        }

        public Long getSsaAuthorizations() {
            return ssaAuthorizationsCount;
        }

        public Long getSsaRevocations() {
            return ssaRevocationsCount;
        }
        
        /**
         * updates class values based on parameters passed.
         * @param consentType
         * @param count 
         */
        public void addConsentType(String consentType, long count){
            
            if(consentType.equals(authorizationName)){
                this.authorizationCount = count;
            } else if (consentType.equals(restrictionsName)){
                this.restrictionsCount = count;
            } else if (consentType.equals(revocationsName)){
                this.revocationsCount = count;
            } else if (consentType.equals(restrictionRevocationsName)){
                this.restrictionRevocationsCount = count;
            } else if (consentType.equals(ssaAuthorizationsName)){
                this.ssaAuthorizationsCount = count;
            } else if (consentType.equals(ssaRevocationsName)){
                this.ssaRevocationsCount = count;
            }
        }
        
        /**
         * Combines the class variables into a Linked Hash Map.
         * @return 
         */
        public Map<String,Object> transformToMap(){
            LinkedHashMap combinedMap = new LinkedHashMap<String,Object>();
            
            combinedMap.put("facilityName", this.facilityName);
            combinedMap.put(this.authorizationName, this.authorizationCount);
            combinedMap.put(this.revocationsName, this.revocationsCount);
            combinedMap.put(this.restrictionsName, this.restrictionsCount);
            combinedMap.put(this.restrictionRevocationsName, this.restrictionRevocationsCount);
            combinedMap.put(this.ssaAuthorizationsName, this.ssaAuthorizationsCount);
            combinedMap.put(this.ssaRevocationsName, this.ssaRevocationsCount);
            
            return combinedMap;
        }
        
    }

}
