package gov.va.nvap.web.report;

import gov.va.nvap.common.transformer.TransformerException;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.common.xpath.XPathException;
import gov.va.nvap.service.audit.ConsentAuditType;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.consent.audit.AuditedConsentEx;
import gov.va.nvap.web.consent.audit.dao.AuditedConsentDAO;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.dao.UserDocumentDAO;
import gov.va.nvap.web.helper.document.DocumentHelper;
import gov.va.nvap.web.helper.document.MediaType;
import gov.va.nvap.web.helper.document.RepresentationType;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.patient.ExcelGeneratorThread;
import gov.va.nvap.web.user.UserHelper;
import gov.va.nvap.web.util.Constants;
import gov.va.nvap.web.util.FieldChecks;
import gov.va.nvap.web.util.Paginator;
import gov.va.nvap.web.util.xls.CsvExporter;
import gov.va.nvap.web.util.xls.ExcelExporter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.LinkedHashMap;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

/**
 * Consent Directive Detail Report results servlet
 *
 * @author Asha Amritraj edited by Stephen Miller
 */
public class OptInOptOutReportResults extends ResponseDispatcherHttpServlet {

    /**
     * Serial UID.
     */
    private static final long serialVersionUID = -1131510956121479062L;

    public void exportToCsv(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException, IOException {

        final List<Map<String, Object>> results = this.getResults(
            request.getSession(), new Paginator(this.getReportHelper().getConsentDirectivesReportCapAsInt()));

        CsvExporter csvExporter = new CsvExporter();

        Map<String, String> csvReportMap = new LinkedHashMap<String, String>();

        csvReportMap.put("SSN", "ssn");
        csvReportMap.put("Patient Last Name", "lastName");
        csvReportMap.put("Patient First Name", "firstName");
        csvReportMap.put("Time of Event (CT)", "auditTimeFormatted");
        csvReportMap.put("Patient Signature/Patient Deceased Date", "patientDateFormatted");
        csvReportMap.put("Purpose of Use", "purposeOfUse");
        csvReportMap.put("Consent Type", "consentType");
        csvReportMap.put("Inactivation Reason", "inactivationReason");
        csvReportMap.put("Entered By", "userId");
        csvReportMap.put("Authenticating Facility", "facilityName");
        csvReportMap.put("Authenticating Facility Station ID", "facilityId");
        csvReportMap.put("VISN", "visnName");

        csvExporter.exportToCSV(response, "Consent_Directive_Detail_Report", results, (LinkedHashMap<String, String>) csvReportMap);
    }

    /**
     * Export to excel - This method is called when the user clicks on the excel icon in the JSP. Reflection is used in the
     * ResponseDispatcherHttpServlet to call this method based on the http parameters.
     *
     * @param request HTTP Request
     * @param response HTTP Response
     * @throws ServletException
     * @throws IOException
     */
    public void exportToExcel(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException, IOException {
        ExcelGeneratorThread exGenThread = new ExcelGeneratorThread("exGenThread");
        //Get the session
        final HttpSession session = request.getSession(false);

        // Create a map of key which is based on the result data key and the heading value
        //The heading is used to create the column headers and the
        //key is used to pull the data from the results
        final Map<String, String> reportMap = new LinkedHashMap<String, String>();

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        final LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();

        final Object startDateStr = session.getAttribute("optInOptOutQueryStartDate");
        final Object endDateStr = session.getAttribute("optInOptOutQueryEndDate");

        filters.put("SSN", session.getAttribute("optInOptOutQuerySsn"));
        filters.put("Last Name", session.getAttribute("optInOptOutQueryLastName"));
        filters.put("First Name", session.getAttribute("optInOptOutQueryFirstName"));
        filters.put("Start Date", startDateStr == null ? "" : this.getReportHelper().getFormattedDate((Date) startDateStr));
        filters.put("End Date", endDateStr == null ? "" : this.getReportHelper().getFormattedDate((Date) endDateStr));
        filters.put("Authenticating Facility", ExcelExporter.getFacilitiesFilter(getFacilityDAO(), (String)session.getAttribute("optInOptOutQueryStationNumbers")));
        filters.put("Consent Type", ExcelExporter.getFilterValue(session.getAttribute("optInOptOutQueryConsentTypeName")));
        filters.put("Inactivation Reason", ExcelExporter.getFilterValue(session.getAttribute("inactivationReason")));
        filters.put("Entered By", session.getAttribute("enteredBy"));
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, session.getAttribute("patientTypes").toString());
        exGenThread.setFilterMap(filterMap);

        reportMap.put("ssn", "SSN");
        reportMap.put("lastName", "Patient Last Name");
        reportMap.put("firstName", "Patient First Name");
        reportMap.put("auditTimeFormatted", "Time of Event (CT)");
        reportMap.put("patientDateFormatted", "Patient Signature/Patient Deceased Date");
        reportMap.put("purposeOfUse", "Purpose of Use");
        reportMap.put("consentType", "Consent Type");
        reportMap.put("inactivationReason", "Inactivation Reason");
        reportMap.put("userId", "Entered By");
        reportMap.put("facilityName", "Authenticating Facility");
        reportMap.put("facilityId", "Authenticating Facility Station ID");
        reportMap.put("visnName", "VISN");
        exGenThread.setReportMap(reportMap);

        // Set names for Excel generator thread
        exGenThread.setTitle("Consent Directive Detailed Report");

        exGenThread.setExcelExporter(this.getExcelExporter());
        exGenThread.setReportHelper(getReportHelper());
        exGenThread.setUserId(UserHelper.getUserName(request));
        exGenThread.setAuditedConsentDAO(this.getBean("auditedConsentDAO", AuditedConsentDAO.class));
        exGenThread.setDocumentHelper(getDocumentHelper());
        exGenThread.setUserDocumentDao(getUserDocumentDAO());

        // Set search attributes for getting audit results
        exGenThread.setAttributes(mapSessionAttributes(session));

        exGenThread.start();

        this.forward(request, response, "show");
    }

    private Map<String, Object> mapSessionAttributes(HttpSession session) {
        Map<String, Object> attributes = new HashMap<String, Object>();

        attributes.put("ssn", (String) session
                .getAttribute("optInOptOutQuerySsn"));
        attributes.put("lastName", (String) session
                .getAttribute("optInOptOutQueryLastName"));
        attributes.put("firstName", (String) session
                .getAttribute("optInOptOutQueryFirstName"));
        attributes.put("userId", (String) session
                .getAttribute("enteredBy"));

        Date startDate = (Date) session.getAttribute("optInOptOutQueryStartDate");
        Date endDate = (Date) session.getAttribute("optInOptOutQueryEndDate");
        attributes.put("startDate", startDate);
        attributes.put("endDate", endDate);

        attributes.put("organization", (String) session
                .getAttribute("documentDisclosureQueryOrganization"));
        attributes.put("stationNumbers", (String) session.getAttribute("optInOptOutQueryStationNumbers"));
        attributes.put("patientTypes", (Integer) session.getAttribute("patientTypes"));
        attributes.put("consentType", (String) session.getAttribute("optInOptOutQueryConsentType"));
        attributes.put("inactivationReason", (String) session.getAttribute("inactivationReason"));
        attributes.put("includeUnknownVisn", Boolean.TRUE.equals(session.getAttribute("includeUnknownVisn")));
        // sort
        attributes.put("sortValue", (String) session
                .getAttribute("optInOptOutSortValue"));
        attributes.put("sortDirection", (String) session
                .getAttribute("optInOptOutSortDirection"));

        // Set Action Types
        List<String> actionTypes = new ArrayList<String>();
        actionTypes.add(ConsentAuditType.OPTIN_AUDIT_TYPE.value());
        actionTypes.add(ConsentAuditType.OPTOUT_AUDIT_TYPE.value());
        attributes.put("actionTypes", actionTypes);

        return attributes;
    }

    public DocumentHelper getDocumentHelper() {
        return this.getBean("adapterDocumentHelper", DocumentHelper.class);
    }

    /**
     * Get the excel exporter class from Spring.
     *
     * @return the excel exporter object
     */
    public ExcelExporter getExcelExporter() {
        final ExcelExporter excelExporter = this.getBean("excelExporter",
            ExcelExporter.class);
        return excelExporter;
    }

    /**
     * Get the facility helper from Spring.
     * @return FacilityHelper
     */
    public FacilityHelper getFacilityHelper() {
        return this.getBean("facilityHelper", FacilityHelper.class);
    }

    public ReportHelper getReportHelper() {
        final ReportHelper reportHelper = this.getBean("reportHelper",
            ReportHelper.class);
        return reportHelper;
    }

    private List<Map<String, Object>> getResults(final HttpSession session,
        final Paginator paginator) throws ServletException {
        final AuditedConsentDAO auditedConsentDAO = this.getBean("auditedConsentDAO", AuditedConsentDAO.class);
        final AuditedConsentDAO.DetailRequest req = auditedConsentDAO.new DetailRequest();

        req.patientFirstName = (String) session.getAttribute("optInOptOutQueryFirstName");
        req.patientLastName = (String) session.getAttribute("optInOptOutQueryLastName");
        req.patientSsn = (String) session.getAttribute("optInOptOutQuerySsn");
        req.stationNumbers = (String) session.getAttribute("optInOptOutQueryStationNumbers");
        req.consentType = (String) session.getAttribute("optInOptOutQueryConsentType");
        req.inactivationReason = (String) session.getAttribute("inactivationReason");
        req.startDate = (Date) session.getAttribute("optInOptOutQueryStartDate");
        req.endDate = (Date) session.getAttribute("optInOptOutQueryEndDate");
        req.patientTypes = (Integer) session.getAttribute("patientTypes");
        req.userId = (String) session.getAttribute("enteredBy");
        req.includeUnknownVisn = Boolean.TRUE.equals(session.getAttribute("includeUnknownVisn"));
        req.sortField = (String) session.getAttribute("optInOptOutSortValue");
        req.sortDirection = (String) session.getAttribute("optInOptOutSortDirection");
        req.currentPage = paginator.getCurrentPage();
        req.pageSize = paginator.getRecordsPerPage();

        // The result of the query is a list of arrays.
        // Each array is a [AuditedConsent, Facility, Visn]
        List<Object> auditedConsentExList = auditedConsentDAO.getEvents(req);

        final List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        if (NullChecker.isNotEmpty(auditedConsentExList)) {
            for (final Object o : auditedConsentExList) {
                AuditedConsentEx auditType = (AuditedConsentEx) o;

                final Map<String, Object> resultMap = new HashMap<String, Object>();

                // Replace SSN with Mask
                ReportDataProcessor.addSsnToResultMap(resultMap, auditType.getPatientSsn());

                resultMap.put("icn", auditType.getPatientId());
                resultMap.put("auditTime", auditType.getTimeOfEvent());
                resultMap.put("auditTimeFormatted", this.getReportHelper()
                    .getFormattedDateTime(auditType.getTimeOfEvent()));
                resultMap.put("userId", auditType.getUserId());
                resultMap.put("actionType", auditType.getActionType());

                String consentTypeRow = auditType.getConsentType();
                if (consentTypeRow.contains("NwHIN")) {
                    consentTypeRow = consentTypeRow.replace("NwHIN", Constants.getOrganizationName());
                }
                resultMap.put("consentType", consentTypeRow);
                resultMap.put("firstName", auditType.getPatientGivenName());
                resultMap.put("lastName", auditType.getPatientLastName());
                resultMap.put("purposeOfUse", auditType.getPouValue());

                String inactivationReasonRow = auditType.getOptoutReason();
                resultMap.put("inactivationReason", inactivationReasonRow);
                resultMap.put("patientDate", auditType.getCreatedDate());
                resultMap.put("patientDateFormatted", this.getReportHelper()
                    .getFormattedDate(auditType.getCreatedDate()));

				// final String vaFacility = this.getFacilityHelper()
                // .getFacilityNameByStationId(auditType.getFacility());
                // resultMap.put("facility",
                // ReportDataProcessor.fixStation(vaFacility));
                resultMap.put("facilityName", ReportDataProcessor
                    .fixStation(auditType.getFacilityName()));
                resultMap.put("facilityId", auditType.getFacility());
                resultMap.put("visnName", NullChecker.isNullOrEmpty(auditType.getVisnName()) ? "Unknown" : auditType.getVisnName());

                if (ConsentAuditType.OPTIN_AUDIT_TYPE.value().equals(
                    auditType.getActionType())
                    || ConsentAuditType.OPTOUT_AUDIT_TYPE.value().equals(
                        auditType.getActionType())) {
                    resultMap.put("auditId", auditType.getConsentAuditId());
                    // Get the CDA R2 Content from Audit
                    final String content = auditedConsentDAO
                        .getMessageContent(auditType.getConsentAuditId());
                    if (NullChecker.isNotEmpty(content)) {
                        try {
                            // Get the attachment inside the audit
                            final String attachment = this.getDocumentHelper()
                                .getPrivacyConsentDirectiveAttachment(
                                    content);
                            // Set the flag
                            if (NullChecker.isNotEmpty(attachment)) {
                                resultMap.put("hasAttachment", true);
                            } else {
                                resultMap.put("hasAttachment", false);
                            }
                        } catch (final TransformerException ex) {
                            throw new ServletException(ex);
                        } catch (final XPathException ex) {
                            throw new ServletException(ex);
                        }
                    }
                }
                results.add(FieldChecks.replaceEmptyOrNullWithSpace(resultMap));
            }
        }

        // Check if any of the patient match criteria exists
        if (NullChecker.isNotEmpty(req.patientSsn) || NullChecker.isNotEmpty(req.patientLastName)
            || NullChecker.isNotEmpty(req.patientFirstName)) {
            // Default to unknown
            String resultIcn = "";
            if (NullChecker.isNotEmpty(auditedConsentExList)) {
                final AuditedConsentEx result = (AuditedConsentEx) auditedConsentExList.get(0);
                resultIcn = result.getPatientId();

                if (NullChecker.isNotEmpty(resultIcn)) {
                    for (final Object o : auditedConsentExList) {
                        AuditedConsentEx consent = (AuditedConsentEx) o;
                        if (!resultIcn.equals(consent.getPatientId())) {
                            resultIcn = "Multiple ICNs";
                            break;
                        }
                    }
                } else if (NullChecker.isNotEmpty(result.getPatientGivenName())
                    && NullChecker.isNotEmpty(result.getPatientLastName())
                    && NullChecker.isNotEmpty(result.getPatientSsn())) {
                    final String patientGivenName = result
                        .getPatientGivenName();
                    final String patientLastName = result.getPatientLastName();
                    final String patientSsn = result.getPatientSsn();
                    boolean multipleIcns = false;
                    for (final Object o : auditedConsentExList) {
                        AuditedConsentEx consent = (AuditedConsentEx) o;
                        if (!patientGivenName.equals(consent
                            .getPatientGivenName())
                            || !patientLastName.equals(consent
                                .getPatientLastName())
                            || !patientSsn.equals(consent.getPatientSsn())) {
                            multipleIcns = true;
                            break;
                        }
                    }
                    if (multipleIcns) {
                        resultIcn = "Multiple ICNs";
                    } else {
                        resultIcn = this.getReportHelper().resolveICN(
                            result.getPatientGivenName(),
                            result.getPatientLastName(),
                            result.getPatientSsn());
                    }
                }
            } else if (NullChecker.isNotEmpty(req.patientSsn)
                && NullChecker.isNotEmpty(req.patientLastName)
                && NullChecker.isNotEmpty(req.patientFirstName)) {
                // Try to get from what was typed
                resultIcn = this.getReportHelper().resolveICN(req.patientFirstName,
                    req.patientLastName, req.patientSsn);
            }
            // Set in Session
            if (NullChecker.isEmpty(resultIcn)) {
                resultIcn = "Unknown";
            }
            session.setAttribute("optInOptOutQueryICN", resultIcn);
        }

        return results;
    }

    private boolean isPaginatorPresent(final HttpSession session) {
        return !NullChecker.isNullOrEmpty(session.getAttribute("paginator"));
    }

    public void next(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        final HttpSession session = request.getSession(false);
        if (this.isPaginatorPresent(session)) {
            final Paginator paginator = (Paginator) session
                .getAttribute("paginator");
            paginator.next(request);
            session.setAttribute("paginator", paginator);
            session.setAttribute("optInOptOutResults",
                this.getResults(session, paginator));
            this.forward(request, response, "show");
        } else {
            this.forward(request, response, "noshow");
        }
    }

    public void prev(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        final HttpSession session = request.getSession(false);
        if (this.isPaginatorPresent(session)) {
            final Paginator paginator = (Paginator) session
                .getAttribute("paginator");
            paginator.previous(request);
            session.setAttribute("paginator", paginator);
            session.setAttribute("optInOptOutResults",
                this.getResults(session, paginator));
            this.forward(request, response, "show");
        } else {
            this.forward(request, response, "noshow");
        }
    }

    @Override
    public void unspecified(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        final HttpSession session = request.getSession(false);

        // sort
        final String optInOptOutSortValue = request
            .getParameter("optInOptOutSortValue");
        final String optInOptOutSortDirection = request
            .getParameter("optInOptOutSortDirection");
        session.setAttribute("optInOptOutSortValue", optInOptOutSortValue);
        session.setAttribute("optInOptOutSortDirection",
            optInOptOutSortDirection);

        if (this.isPaginatorPresent(session)) {
            Paginator paginator = (Paginator) session.getAttribute("paginator");
            if (NullChecker.isEmpty(paginator)) {
                paginator = new Paginator(0);
                session.setAttribute("paginator", paginator);
            }
            paginator.reset();
            session.setAttribute("optInOptOutResults",
                this.getResults(session, paginator));
            this.forward(request, response, "show");
        } else {
            this.forward(request, response, "noshow");
        }
    }

    public void view(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        final HttpSession session = request.getSession(false);
        final String documentUniqueId = request
            .getParameter("documentUniqueId");

        final AuditedConsentDAO auditedConsentDAO = this.getBean(
            "auditedConsentDAO", AuditedConsentDAO.class);
        final String document = auditedConsentDAO.getMessageContent(Long
            .parseLong(documentUniqueId));

        session.setAttribute("document", document);
        session.setAttribute("documentUniqueId", documentUniqueId);

        try {
            final String attachment = this.getDocumentHelper()
                .getPrivacyConsentDirectiveAttachment(document);
            if (NullChecker.isNotEmpty(attachment)) {
                this.getDocumentHelper()
                    .writeAttachmentToStream(
                        response,
                        attachment,
                        MediaType.fromValue(this.getDocumentHelper()
                            .getPrivacyConsentDirectiveMediaType(
                                document)),
                        RepresentationType.B64);
            } else {
                session.setAttribute("informationMessage",
                    "There are no attachments associated with this consent directive.");
                this.forward(request, response, "noattachment");
            }
        } catch (final TransformerException ex) {
            throw new ServletException(ex);
        } catch (final XPathException ex) {
            throw new ServletException(ex);
        }
    }

    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }

    private UserDocumentDAO getUserDocumentDAO() {
        return this.getBean("UserDocumentDAO", UserDocumentDAO.class);
    }
}
