package gov.va.nvap.web.report;

import gov.va.nvap.common.transformer.TransformerException;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.common.xpath.XPathException;
import gov.va.nvap.service.audit.ConsentAuditType;
import gov.va.nvap.svc.facility.data.Facility;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.consent.audit.AuditedConsentEx;
import gov.va.nvap.web.consent.audit.dao.AuditedConsentDAO;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.dao.UserDocumentDAO;
import gov.va.nvap.web.helper.document.DocumentHelper;
import gov.va.nvap.web.helper.document.MediaType;
import gov.va.nvap.web.helper.document.RepresentationType;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.patient.ExcelGeneratorThread;
import gov.va.nvap.web.user.UserHelper;
import gov.va.nvap.web.util.Constants;
import gov.va.nvap.web.util.FieldChecks;
import gov.va.nvap.web.util.xls.CsvExporter;
import gov.va.nvap.web.util.xls.ExcelExporter;
import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.poi.ss.usermodel.Workbook;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Consent Directive Detail Report search servlet
 *
 * @author Asha Amritraj edited by Stephen Miller
 */
public class OptInOptOutReport extends ResponseDispatcherHttpServlet {

    private static final Logger logger = Logger.getLogger(OptInOptOutReport.class.getName());

    /**
     * Serial UID.
     */
    private static final long serialVersionUID = 3721503355079491829L;

    /**
     * Entity Manager instance.
     */
    @PersistenceContext
    private EntityManager em;

    /**
     * Get the CMS helper from Spring.
     * @return ConsentManagementHelper
     */
    public ConsentManagementHelper getCmsHelper() {
        final ConsentManagementHelper cmsHelper = this.getBean("cmsHelper",
            ConsentManagementHelper.class);
        return cmsHelper;
    }

    /**
     * Get the facility helper from Spring.
     * @return FacilityHelper
     */
    public FacilityHelper getFacilityHelper() {
        final FacilityHelper facilityHelper = this.getBean("facilityHelper",
            FacilityHelper.class);
        return facilityHelper;
    }

    /**
     * Set the entity manager (Injected by Spring).
     * @param em Entity Manager
     */
    public void setEntityManager(final EntityManager em) {
        this.em = em;
    }

    @Override
    protected void unspecified(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        final HttpSession session = request.getSession(false);
        session.removeAttribute("results");

        Collection<Facility> facilities = this.getFacilityHelper().getAllVistAFacilities();
        request.setAttribute("facilities", facilities);
        request.setAttribute("optOutReasons", this.getCmsHelper().getOptoutReasons());

        final gov.va.nvap.svc.facility.data.Facility defaultUserFacility
            = (gov.va.nvap.svc.facility.data.Facility) session.getAttribute("defaultUserFacility");

        ReportHelper.setDefaultSearchDates(session);
        this.forward(request, response, "show");
    }

     /**
     * Returns search results in JSON form. JSON contains two arrays: "data" and "summary".
     * If no request parameters are present, default values are used for search.
     *
     * @param request
     * @param response
     * @throws ServletException
     * @throws IOException
     */
    public void doSearch(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException, TransformerException, XPathException {
        final HttpSession session = request.getSession(false);

        SearchPackage results = getResults(request, false);
        session.setAttribute("results", encodeIntoJSON(results, request.getParameter("draw")));

        this.forward(request, response, "searchResultsJSON");
    }

    /**
     * Returns packaged query results.
     * @param request
     * @param getAll
     * @return
     * @throws ServletException
     * @throws TransformerException
     * @throws XPathException
     */
    private SearchPackage getResults(final HttpServletRequest request, boolean exporting) throws ServletException, TransformerException, XPathException {
        final AuditedConsentDAO auditedConsentDAO = this.getBean("auditedConsentDAO", AuditedConsentDAO.class);
        final AuditedConsentDAO.DetailRequest req = auditedConsentDAO.new DetailRequest();

        req.patientFirstName = request.getParameter("firstName");
        req.patientLastName = request.getParameter("lastName");
        req.patientSsn = request.getParameter("ssn");
        req.stationNumbers = request.getParameter("stationNumbers");
        req.consentType = request.getParameter("consentType");
        req.inactivationReason = request.getParameter("inactivationReason");
        req.startDate = ReportHelper.getStartDate(request.getParameter("startDate"));
        req.endDate = ReportHelper.getEndDate(request.getParameter("endDate"));
        req.patientTypes = Integer.parseInt(request.getParameter("patientTypes"));
        req.userId = request.getParameter("enteredBy");
        req.includeUnknownVisn = "true".equals(request.getParameter("includeUnknownVisn"));
        req.sortField = (request.getParameter("order[0][column]") !=null)
                ? getSortValue(request.getParameter("order[0][column]")) : getSortValue(request.getParameter("sortBy"));
        req.sortDirection = (request.getParameter("order[0][dir]") !=null)
                ?request.getParameter("order[0][dir]") : request.getParameter("sortOrder");
        req.currentPage = (request.getParameter("start") != null)
                ?Integer.parseInt(request.getParameter("start"))/Integer.parseInt(request.getParameter("length")) : 0;
        req.pageSize = (request.getParameter("length")!=null)
                ?Integer.parseInt(request.getParameter("length")): -1;


        // The result of the query is a list of arrays.
        // Each array is a [AuditedConsent, Facility, Visn]
        List<Object> auditedConsentExList = auditedConsentDAO.getEvents(req);

        final List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        if (NullChecker.isNotEmpty(auditedConsentExList)) {
            for (final Object o : auditedConsentExList) {
                AuditedConsentEx auditType = (AuditedConsentEx) o;

                final Map<String, Object> resultMap = new HashMap<String, Object>();

                // Replace SSN with Mask
                ReportDataProcessor.addSsnToResultMap(resultMap, auditType.getPatientSsn());

                resultMap.put("icn", auditType.getPatientId());
                resultMap.put("auditTime", auditType.getTimeOfEvent());
                resultMap.put("auditTimeFormatted", this.getReportHelper()
                    .getFormattedDateTime(auditType.getTimeOfEvent()));
                resultMap.put("userId", auditType.getUserId());
                resultMap.put("actionType", auditType.getActionType());

                String consentTypeRow = auditType.getConsentType();
                if (consentTypeRow.contains("NwHIN")) {
                    consentTypeRow = consentTypeRow.replace("NwHIN", Constants.getOrganizationName());
                }
                resultMap.put("consentType", consentTypeRow);
                resultMap.put("firstName", auditType.getPatientGivenName());
                resultMap.put("lastName", auditType.getPatientLastName());
                resultMap.put("purposeOfUse", auditType.getPouValue());

                String inactivationReasonRow = auditType.getOptoutReason();
                resultMap.put("inactivationReason", inactivationReasonRow);
                resultMap.put("patientDate", auditType.getCreatedDate());
                resultMap.put("patientDateFormatted", this.getReportHelper()
                    .getFormattedDate(auditType.getCreatedDate()));

				// final String vaFacility = this.getFacilityHelper()
                // .getFacilityNameByStationId(auditType.getFacility());
                // resultMap.put("facility",
                // ReportDataProcessor.fixStation(vaFacility));
                resultMap.put("facilityName", ReportDataProcessor
                    .fixStation(auditType.getFacilityName()));
                resultMap.put("facilityId", auditType.getFacility());
                resultMap.put("visnName", NullChecker.isNullOrEmpty(auditType.getVisnName()) ? "Unknown" : auditType.getVisnName());

                if (ConsentAuditType.OPTIN_AUDIT_TYPE.value().equals(
                    auditType.getActionType())
                    || ConsentAuditType.OPTOUT_AUDIT_TYPE.value().equals(
                        auditType.getActionType())) {
                    resultMap.put("auditId", auditType.getConsentAuditId());
                    if (!exporting) {
                        // Get the CDA R2 Content from Audit
                        final String content = auditedConsentDAO
                            .getMessageContent(auditType.getConsentAuditId());
                        if (NullChecker.isNotEmpty(content)) {
                            final String attachment = this.getDocumentHelper()
                                    .getPrivacyConsentDirectiveAttachment(
                                            content);
                            if (NullChecker.isNotEmpty(attachment)) {
                                resultMap.put("hasAttachment", true);
                            } else {
                                resultMap.put("hasAttachment", false);
                            }
                        }
                    }
                }
                results.add(FieldChecks.replaceEmptyOrNullWithSpace(resultMap));
            }
        }

        // Check if any of the patient match criteria exists
        String resultIcn = "";
        if (NullChecker.isNotEmpty(req.patientSsn) || NullChecker.isNotEmpty(req.patientLastName)
            || NullChecker.isNotEmpty(req.patientFirstName)) {
            // Default to unknown
            if (NullChecker.isNotEmpty(auditedConsentExList)) {
                final AuditedConsentEx result = (AuditedConsentEx) auditedConsentExList.get(0);
                resultIcn = result.getPatientId();

                if (NullChecker.isNotEmpty(resultIcn)) {
                    for (final Object o : auditedConsentExList) {
                        AuditedConsentEx consent = (AuditedConsentEx) o;
                        if (!resultIcn.equals(consent.getPatientId())) {
                            resultIcn = "Multiple ICNs";
                            break;
                        }
                    }
                } else if (NullChecker.isNotEmpty(result.getPatientGivenName())
                    && NullChecker.isNotEmpty(result.getPatientLastName())
                    && NullChecker.isNotEmpty(result.getPatientSsn())) {
                    final String patientGivenName = result
                        .getPatientGivenName();
                    final String patientLastName = result.getPatientLastName();
                    final String patientSsn = result.getPatientSsn();
                    boolean multipleIcns = false;
                    for (final Object o : auditedConsentExList) {
                        AuditedConsentEx consent = (AuditedConsentEx) o;
                        if (!patientGivenName.equals(consent
                            .getPatientGivenName())
                            || !patientLastName.equals(consent
                                .getPatientLastName())
                            || !patientSsn.equals(consent.getPatientSsn())) {
                            multipleIcns = true;
                            break;
                        }
                    }
                    if (multipleIcns) {
                        resultIcn = "Multiple ICNs";
                    } else {
                        resultIcn = this.getReportHelper().resolveICN(
                            result.getPatientGivenName(),
                            result.getPatientLastName(),
                            result.getPatientSsn());
                    }
                }
            } else if (NullChecker.isNotEmpty(req.patientSsn)
                && NullChecker.isNotEmpty(req.patientLastName)
                && NullChecker.isNotEmpty(req.patientFirstName)) {
                // Try to get from what was typed
                resultIcn = this.getReportHelper().resolveICN(req.patientFirstName,
                    req.patientLastName, req.patientSsn);
            }
            // Set in Session
            if (NullChecker.isEmpty(resultIcn)) {
                resultIcn = "Unknown";
            }
        }

        //Grab to total count of the search results, not just the count of the current batch but overall for pagination.
        final Long totalCount = exporting ? 0 : auditedConsentDAO.getEventsCount(req);

        return new SearchPackage(results, resultIcn, totalCount);
    }

    /**
     * changes search package into json string for passing to datatables.
     * @param packedResults
     * @return
     */
    private String encodeIntoJSON(SearchPackage packedResults, String draw){
        String returnValue = "";

        try {
            StringWriter json = new StringWriter();
            JSONObject obj = new JSONObject();
            List<List> data = new ArrayList<List>();

            for (Map<String,Object> currentRow : packedResults.getResults()) {
                List<String> dataItem = new ArrayList<String>();

                dataItem.add(currentRow.get("ssnMasked").toString());
                dataItem.add(currentRow.get("lastName").toString());
                dataItem.add(currentRow.get("firstName").toString());
                dataItem.add(currentRow.get("auditTimeFormatted").toString());
                dataItem.add(currentRow.get("patientDateFormatted").toString());
                dataItem.add(currentRow.get("purposeOfUse").toString());
                dataItem.add(currentRow.get("consentType").toString());
                dataItem.add(currentRow.get("inactivationReason").toString());
                dataItem.add(currentRow.get("userId").toString());
                dataItem.add(currentRow.get("facilityName").toString());
                dataItem.add(currentRow.get("facilityId").toString());
                dataItem.add(currentRow.get("visnName").toString());
                dataItem.add(currentRow.get("auditId").toString());
                dataItem.add(currentRow.get("ssn").toString());

                if(currentRow.containsKey("hasAttachment")){
                    dataItem.add(currentRow.get("hasAttachment").toString());
                }

                data.add(dataItem);
            }

            obj.put("data", data);
            obj.put("draw", Integer.parseInt(draw));
            obj.put("recordsTotal", packedResults.getResultCount());
            obj.put("recordsFiltered", packedResults.getResultCount());
            obj.put("icn", packedResults.getIcn());

            obj.write(json);

            returnValue = json.toString();
        } catch (JSONException ex) {
            Logger.getLogger(OptInOptOutReport.class.getName()).log(Level.SEVERE, null, ex);
        }

        return returnValue;

    }

    /**
     * convert column number into correct sort value
     * @param colNumber
     * @return
     */
    private String getSortValue(String colNumber){
        String sortValue = "";

        if(null != colNumber) switch (Integer.parseInt(colNumber)) {
            case 0:
                sortValue = "patientSsn";
                break;
            case 1:
                sortValue = "patientLastName";
                break;
            case 2:
                sortValue = "patientGivenName";
                break;
            case 3:
                sortValue = "timeOfEvent";
                break;
            case 4:
                sortValue = "createdDate";
                break;
            case 5:
                sortValue = "pouValue";
                break;
            case 6:
                sortValue = "consentType";
                break;
            case 7:
                sortValue = "optoutReason";
                break;
            case 8:
                sortValue = "userId";
                break;
            case 9:
                sortValue = "facilityName";
                break;
            case 10:
                sortValue = "facility";
                break;
            case 11:
                sortValue = "visnName";
                break;
        }
        return sortValue;
    }

    /**
     * Export to excel - This method is called when the user clicks on the excel icon in the JSP. Reflection is used in the
     * ResponseDispatcherHttpServlet to call this method based on the http parameters.
     *
     * @param request HTTP Request
     * @param response HTTP Response
     * @throws ServletException
     * @throws IOException
     */
    public void exportToExcel(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException, IOException, TransformerException, XPathException {

        // Create a map of key which is based on the result data key and the heading value
        //The heading is used to create the column headers and the
        //key is used to pull the data from the results
        final Map<String, String> reportMap = new LinkedHashMap<String, String>();

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        final LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();

        final Date startDateStr = ReportHelper.getStartDate(request.getParameter("startDate"));
        final Date endDateStr = ReportHelper.getEndDate(request.getParameter("endDate"));
        final String consentType = getConsentTypeFilterText(request.getParameter("consentType"));

        filters.put("SSN", request.getParameter("ssn"));
        filters.put("Last Name", request.getParameter("lastName"));
        filters.put("First Name", request.getParameter("firstName"));
        filters.put("Start Date", startDateStr == null ? "" : this.getReportHelper().getFormattedDate(startDateStr));
        filters.put("End Date", endDateStr == null ? "" : this.getReportHelper().getFormattedDate(endDateStr));
        filters.put("Authenticating Facility", ExcelExporter.getFacilitiesFilter(getFacilityDAO(), request.getParameter("stationNumbers")));
        filters.put("Consent Type", ExcelExporter.getFilterValue(consentType));
        if (consentType.contains("Revocation")) {
            filters.put("Inactivation Reason", ExcelExporter.getFilterValue(request.getParameter("inactivationReason")));
        }
        filters.put("Entered By", request.getParameter("enteredBy"));
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, request.getParameter("patientTypes"));

        reportMap.put("ssn", "SSN");
        reportMap.put("lastName", "Patient Last Name");
        reportMap.put("firstName", "Patient First Name");
        reportMap.put("auditTimeFormatted", "Time of Event (CT)");
        reportMap.put("patientDateFormatted", "Patient Signature/Patient Deceased Date");
        reportMap.put("purposeOfUse", "Purpose of Use");
        reportMap.put("consentType", "Consent Type");
        reportMap.put("inactivationReason", "Inactivation Reason");
        reportMap.put("userId", "Entered By");
        reportMap.put("facilityName", "Authenticating Facility");
        reportMap.put("facilityId", "Authenticating Facility Station ID");
        reportMap.put("visnName", "VISN");

        int minimumThreshold = Integer.parseInt((String) this.getServletContext().getAttribute("scheduledExportMin"));
        int total = Integer.parseInt(request.getParameter("totalRows"));

        if (total >= minimumThreshold) {
            // Start a scheduled export with its own thread if number of rows is greater than the minimum threshold
            ExcelGeneratorThread exGenThread = new ExcelGeneratorThread("exGenThread");
            exGenThread.setFilterMap(filterMap);
            exGenThread.setReportMap(reportMap);
            // Set names for Excel generator thread
            exGenThread.setTitle("Consent Directive Detail Report");

            exGenThread.setExcelExporter(this.getExcelExporter());
            exGenThread.setReportHelper(getReportHelper());
            exGenThread.setUserId(UserHelper.getUserName(request));
            exGenThread.setAuditedConsentDAO(this.getBean("auditedConsentDAO", AuditedConsentDAO.class));
            exGenThread.setDocumentHelper(getDocumentHelper());
            exGenThread.setUserDocumentDao(getUserDocumentDAO());

            // Set search attributes for getting audit results
            exGenThread.setAttributes(mapSessionAttributes(request));

            exGenThread.start();
        } else {
            // Otherwise generate and download the export directly
            logger.log(Level.SEVERE, "EXPORT_EXCEL_CDDR_BEGIN_" + Thread.currentThread().getId() + " " + System.currentTimeMillis());
            // Create workbook
            final String title = "Consent Directive Detail Report";

            List<Map<String, Object>> results;
            OptInOptOutReport.SearchPackage packedResults;
            packedResults = this.getResults(request, true);
            results = packedResults.getResults();

            final Workbook wb = this.getExcelExporter().exportToExcel(title, title, reportMap, results, filterMap, null);

            // Write Excel workbook to Stream
            this.getExcelExporter().writeExcelToStream("Consent_Directive_Detail_Report", wb, response);
            logger.log(Level.SEVERE, "EXPORT_EXCEL_CDDR_END_" + Thread.currentThread().getId() + " " + System.currentTimeMillis());
        }
    }

    private Map<String, Object> mapSessionAttributes(HttpServletRequest request) {
        Map<String, Object> attributes = new HashMap<String, Object>();

        attributes.put("ssn", request.getParameter("ssn"));
        attributes.put("lastName", request.getParameter("lastName"));
        attributes.put("firstName", request.getParameter("firstName"));
        attributes.put("userId", request.getParameter("enteredBy"));
        Date startDate = ReportHelper.getStartDate(request.getParameter("startDate"));
        Date endDate = ReportHelper.getEndDate(request.getParameter("endDate"));
        attributes.put("startDate", startDate);
        attributes.put("endDate", endDate);

        //attributes.put("organization", (String) session
                //.getAttribute("documentDisclosureQueryOrganization"));
        attributes.put("stationNumbers", (String) request.getParameter("stationNumbers"));
        attributes.put("patientTypes", Integer.parseInt(request.getParameter("patientTypes")));
        attributes.put("consentType", request.getParameter("consentType"));
        attributes.put("inactivationReason", request.getParameter("inactivationReason"));
        attributes.put("includeUnknownVisn", "true".equals(request.getParameter("includeUnknownVisn")));
        // sort
        attributes.put("sortValue", getSortValue(request.getParameter("sortBy")));
        attributes.put("sortDirection", request.getParameter("sortOrder"));

        // Set Action Types
        List<String> actionTypes = new ArrayList<String>();
        actionTypes.add(ConsentAuditType.OPTIN_AUDIT_TYPE.value());
        actionTypes.add(ConsentAuditType.OPTOUT_AUDIT_TYPE.value());
        attributes.put("actionTypes", actionTypes);

        return attributes;
    }

    public void exportToCsv(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException, IOException, TransformerException, XPathException {

        Map<String, String> csvReportMap = new LinkedHashMap<String, String>();

        csvReportMap.put("SSN", "ssn");
        csvReportMap.put("Patient Last Name", "lastName");
        csvReportMap.put("Patient First Name", "firstName");
        csvReportMap.put("Time of Event (CT)", "auditTimeFormatted");
        csvReportMap.put("Patient Signature/Patient Deceased Date", "patientDateFormatted");
        csvReportMap.put("Purpose of Use", "purposeOfUse");
        csvReportMap.put("Consent Type", "consentType");
        csvReportMap.put("Inactivation Reason", "inactivationReason");
        csvReportMap.put("Entered By", "userId");
        csvReportMap.put("Authenticating Facility", "facilityName");
        csvReportMap.put("Authenticating Facility Station ID", "facilityId");
        csvReportMap.put("VISN", "visnName");

        int minimumThreshold = Integer.parseInt((String) this.getServletContext().getAttribute("scheduledExportMin"));
        int total = Integer.parseInt(request.getParameter("totalRows"));

        if (total >= minimumThreshold) {
            // Start a scheduled export with its own thread if number of rows is greater than the minimum threshold
            ExcelGeneratorThread csvGenThread = new ExcelGeneratorThread("csvGenThread");
            csvGenThread.setDocumentType("csv");
            csvGenThread.setReportMap(csvReportMap);
            csvGenThread.setTitle("Consent Directive Detail Report");
            csvGenThread.setExcelExporter(this.getExcelExporter());
            csvGenThread.setReportHelper(getReportHelper());
            csvGenThread.setUserId(UserHelper.getUserName(request));
            csvGenThread.setDocumentHelper(getDocumentHelper());
            csvGenThread.setAuditedConsentDAO(this.getAuditedConsentDAO());
            csvGenThread.setUserDocumentDao(getUserDocumentDAO());

            // Set search attributes for getting audit results
            csvGenThread.setAttributes(mapSessionAttributes(request));

            csvGenThread.start();
        } else {
            // Otherwise generate and download the export directly
            logger.log(Level.SEVERE, "EXPORT_CSV_CDDR_BEGIN_" + Thread.currentThread().getId() + " " + System.currentTimeMillis());
            List<Map<String, Object>> results;
            SearchPackage packedResults;

            packedResults = this.getResults(request, true);
            results = packedResults.getResults();

            CsvExporter csvExporter = new CsvExporter();
            csvExporter.exportToCSV(response, "Consent_Directive_Detail_Report", results, (LinkedHashMap<String, String>) csvReportMap);
            logger.log(Level.SEVERE, "EXPORT_CSV_CDDR_BEGIN_" + Thread.currentThread().getId() + " " + System.currentTimeMillis());
        }
    }

    public void view(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        final HttpSession session = request.getSession(false);
        final String documentUniqueId = request
            .getParameter("documentUniqueId");

        final AuditedConsentDAO auditedConsentDAO = this.getBean(
            "auditedConsentDAO", AuditedConsentDAO.class);
        final String document = auditedConsentDAO.getMessageContent(Long
            .parseLong(documentUniqueId));

        session.setAttribute("document", document);
        session.setAttribute("documentUniqueId", documentUniqueId);

        try {
            final String attachment = this.getDocumentHelper()
                .getPrivacyConsentDirectiveAttachment(document);
            if (NullChecker.isNotEmpty(attachment)) {
                this.getDocumentHelper()
                    .writeAttachmentToStream(
                        response,
                        attachment,
                        MediaType.fromValue(this.getDocumentHelper()
                            .getPrivacyConsentDirectiveMediaType(
                                document)),
                        RepresentationType.B64);
            } else {
                session.setAttribute("informationMessage",
                    "There are no attachments associated with this consent directive.");
                this.forward(request, response, "noattachment");
            }
        } catch (final TransformerException ex) {
            throw new ServletException(ex);
        } catch (final XPathException ex) {
            throw new ServletException(ex);
        }
    }

    private AuditedConsentDAO getAuditedConsentDAO() {
        return this.getBean("auditedConsentDAO", AuditedConsentDAO.class);
    }

    private String getConsentTypeFilterText(String consentType) {
        if (NullChecker.isNullOrEmpty(consentType)) {
            return "ALL";
        } else if (consentType.equalsIgnoreCase("allAuthorizations")) {
            return "ALL Authorizations";
        } else if (consentType.equalsIgnoreCase("allRevocations")) {
             return "ALL Revocations";
        } else if (consentType.equalsIgnoreCase("NwHINAuthorizations")) {
             return "eHealth Exchange Authorizations";
        } else if (consentType.equalsIgnoreCase("SSAAuthorizations")) {
            return "SSA Authorizations";
        } else if (consentType.equalsIgnoreCase("NwHINRevocations")) {
            return "eHealth Exchange Revocations";
        } else if (consentType.equalsIgnoreCase("SSARevocations")) {
            return "SSA Revocations";
        } else if (consentType.equalsIgnoreCase("NwHINRestrictions")) {
            return "eHealth Exchange Restrictions";
        } else if (consentType.equalsIgnoreCase("NwHINRestrictionRevocations")) {
            return "eHealth Exchange Restriction Revocations";
        } else {
            return "";
        }
    }

    /**
     * Get the excel exporter class from Spring.
     *
     * @return the excel exporter object
     */
    public ExcelExporter getExcelExporter() {
        final ExcelExporter excelExporter = this.getBean("excelExporter",
            ExcelExporter.class);
        return excelExporter;
    }

    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }

    private UserDocumentDAO getUserDocumentDAO() {
        return this.getBean("UserDocumentDAO", UserDocumentDAO.class);
    }

    public ReportHelper getReportHelper() {
        final ReportHelper reportHelper = this.getBean("reportHelper",
            ReportHelper.class);
        return reportHelper;
    }

    public DocumentHelper getDocumentHelper() {
        return this.getBean("adapterDocumentHelper", DocumentHelper.class);
    }

    /**
     * inner class
     */
    private class SearchPackage{
        List<Map<String,Object>> results;
        String icn;
        Long resultCount;

        public Long getResultCount() {
            return resultCount;
        }

        public void setResultCount(Long resultCount) {
            this.resultCount = resultCount;
        }

        public SearchPackage(List<Map<String, Object>> results, String icn, Long count) {
            this.results = results;
            this.icn = icn;
            this.resultCount = count;
        }

        public List<Map<String, Object>> getResults() {
            return results;
        }

        public void setResults(List<Map<String, Object>> results) {
            this.results = results;
        }

        public String getIcn() {
            return icn;
        }

        public void setIcn(String icn) {
            this.icn = icn;
        }

    }

}
