package gov.va.nvap.web.report;

import gov.va.med.nhin.adapter.audit.ActionType;
import gov.va.med.nhin.adapter.audit.ActionValuesType;
import gov.va.med.nhin.adapter.audit.AuditSummariesType;
import gov.va.med.nhin.adapter.audit.AuditSummaryType;
import gov.va.med.nhin.adapter.audit.FieldType;
import gov.va.med.nhin.adapter.audit.GetAuditsSummary;
import gov.va.med.nhin.adapter.audit.GetAuditsSummaryResponse;
import gov.va.med.nhin.adapter.audit.GroupByFieldsType;
import gov.va.nvap.common.date.GregorianDateUtil;
import gov.va.nvap.common.sort.BubbleSortListMap;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.OrganizationType;
import gov.va.nvap.service.adapter.audit.AdapterAuditManager;
import gov.va.nvap.service.audit.AuditException;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.util.date.DateUtil;
import java.io.IOException;
import java.io.StringWriter;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.ejb.EJB;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.json.JSONException;
import org.json.JSONObject;

/**
 *
 * @author Johann Sonnenberg
 */
public class MonthlyReceivedDocumentsReport extends gov.va.nvap.web.app.ResponseDispatcherHttpServlet {

    private AdapterAuditManager adapterAuditManager;

    @EJB(beanInterface = AdapterAuditManager.class, mappedName = "AdapterAuditManager")
    public void setAdapterAuditManager(final AdapterAuditManager adapterAuditManager) {
        this.adapterAuditManager = adapterAuditManager;
    }

    @Override
    protected void unspecified(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        final HttpSession session = request.getSession(false);

        //this is the search form
        this.forward(request, response, "searchForm");
    }

    public void doSearch(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException, ParseException {
        final HttpSession session = request.getSession(false);
        final String draw = request.getParameter("draw");
        final String monthAndYear = request.getParameter("startYearMonth");
        final String[] mAndY = monthAndYear.split("[|]");
        final String month = mAndY[0];
        final String year = mAndY[1];
        
        try{
            SearchResultsPackage srp = this.getResults(request, false, month, year); 
            session.setAttribute("results", this.encodeServiceAuditIntoJSON(srp, Integer.parseInt(draw)));
        } catch(Exception e){
            throw new AuditException();
        }
        this.forward(request, response, "searchResultsJSON");
    }

    private ConsentManagementHelper getCmsHelper() {
        return this.getBean("cmsHelper", ConsentManagementHelper.class);
    }

    // Private methods
    private MonthlyReceivedDocumentsReport.SearchResultsPackage getResults(final HttpServletRequest request, Boolean calledForExcel, String startMonth, String startYear) throws AuditException, ParseException {
        final List<Map<String, Object>> results = new ArrayList<>();
        long totalCount = 0L;
        long totalRecordCount = 0L;
        
        // Prepare audit summary
        GetAuditsSummaryResponse getAuditsSummaryResponse = prepareAuditSummary(startMonth, startYear);

        // Retrieve the data from Exchange
        final AuditSummariesType summaryData = getAuditsSummaryResponse.getAuditSummaries();

        List<AuditSummaryType> auditSummary = summaryData == null ? new ArrayList<AuditSummaryType>() : summaryData.getAuditSummary();

        for (final AuditSummaryType auditSummaryType : auditSummary) {
            final Map<String, Object> row = new LinkedHashMap<>();
            final long count = auditSummaryType.getCount();
            final List<String> summaryField = auditSummaryType.getSummaryFields().getSummaryField();

            if (NullChecker.isEmpty(summaryField)) {
                continue;
            }

            if (summaryField.size() < 4) {
                continue;
            }

            // Add organization
            OrganizationType organizationType = null;
            if (NullChecker.isNotEmpty(summaryField.get(0))) {
                organizationType = (this.getCmsHelper().getOrganizationByHomeCommunityId(summaryField.get(0).replace("urn:oid:", "")));
            }
            if (organizationType == null) {
                row.put("partnerOrg", ReportDataProcessor.nullEmptyReplaceWithUnknown(summaryField.get(3)));
            } else {
                row.put("partnerOrg", ReportDataProcessor.nullEmptyReplaceWithUnknown(organizationType.getOrgName()));
            }

            // Add user
            if(NullChecker.isNotEmpty(summaryField.get(3))) {
                row.put("user", summaryField.get(3));
            } else {
                row.put("user", "");
            }
            

            // Add count
            row.put("total", count);

            // Add the row to the results map
            results.add(row);

            totalCount += count;
            totalRecordCount++;
        }

        // Sort the results
        List<Map<String, Object>> sortedResults = sortResults(results, request);

        // Return the packaged, sorted results
        return new MonthlyReceivedDocumentsReport.SearchResultsPackage(sortedResults, totalCount, totalRecordCount);
    }

    private GetAuditsSummaryResponse prepareAuditSummary(String startMonth, String startYear) throws AuditException, ParseException {

        Calendar calendar = Calendar.getInstance();
        //prep start date
        //year,month,date,hour,min,sec
        calendar.set(Integer.parseInt(startYear), Integer.parseInt(startMonth) - 1, 1, 0, 0, 0);
        Date startDate = calendar.getTime();
        
        //prep end date
        calendar.set(Calendar.DAY_OF_MONTH, calendar.getActualMaximum(Calendar.DAY_OF_MONTH));
        calendar.set(Calendar.HOUR_OF_DAY, 23);
        calendar.set(Calendar.MINUTE, 59);
        calendar.set(Calendar.SECOND, 59);
        Date endDate = calendar.getTime();
        
        final GetAuditsSummary getAuditsSummary = new GetAuditsSummary();

        // Start time
        getAuditsSummary.setFromDate(GregorianDateUtil.getGregorianCalendarByDate(startDate));

        // End time
        getAuditsSummary.setToDate(GregorianDateUtil.getGregorianCalendarByDate(endDate));

        // Action types
        final ActionValuesType actionValuesType = new ActionValuesType();
        final List<ActionType> actiontype = actionValuesType.getValue();
        actiontype.add(ActionType.RETRIEVE_DOCUMENT_OUT);
        actionValuesType.setNotIn(false);
        getAuditsSummary.setActions(actionValuesType);

        // Patient types
        getAuditsSummary.setPatientTypes(1);

        // Group by
        final GroupByFieldsType groupByFieldsType = new GroupByFieldsType();
        final List<FieldType> fieldTypeList = groupByFieldsType.getGroupByField();
        fieldTypeList.add(FieldType.REMOTE_ORGANIZATION_ID);
        fieldTypeList.add(FieldType.REMOTE_ORGANIZATION_NAME);
        fieldTypeList.add(FieldType.USER_ID);
        fieldTypeList.add(FieldType.USER_NAME);
        getAuditsSummary.setGroupByFields(groupByFieldsType);

        // Call Exchange
        GetAuditsSummaryResponse getAuditsSummaryResponse;
        try {
            getAuditsSummaryResponse = this.adapterAuditManager.getAuditSummary(getAuditsSummary);
            return getAuditsSummaryResponse;
        } catch (final AuditException e) {
            throw new AuditException();
        }
    }

    private String encodeServiceAuditIntoJSON(SearchResultsPackage srp, Integer draw) throws ParseException {
         try {
            StringWriter json = new StringWriter();
            JSONObject obj = new JSONObject();
            List<Map<String, Object>> results = srp.getSearchResults();
            
            obj.put("recordsTotal", srp.totalRecordCount);
            obj.put("recordsFiltered", srp.totalRecordCount);
            obj.put("draw", draw);
            
            List<List> data = new ArrayList<List>(); //overarching "data"
            
            if(results != null) { //make it null safe
                for(Map<String, Object> r : results) {
                    List<String> dataItem = new ArrayList<String>(); //each "row"
                    
                    if(r != null) {
                        dataItem.add(r.get("partnerOrg").toString()); //remote organization
                        dataItem.add(r.get("user").toString()); //user
                        dataItem.add(r.get("total").toString()); //count

                        data.add(dataItem); 
                    }
                }
            }
            
            obj.put("data", data);
            
            obj.write(json);
            
            return json.toString();
         } catch(JSONException ex) {
             Logger.getLogger(MonthlyReceivedDocumentsReport.class.getName()).log(Level.SEVERE, null, ex);
         }
        
        return null;
    }

    private List<Map<String, Object>> sortResults(List<Map<String, Object>> unSortedData, HttpServletRequest request) {
        final BubbleSortListMap bSorter = new BubbleSortListMap();
        final String sortByIndex = request.getParameter("order[0][column]");
        final String sortOrder = request.getParameter("order[0][dir]");
        String sortValue = "partnerOrg";
        final boolean sortAscending = sortOrder == null ? !"desc".equalsIgnoreCase(request.getParameter("sortOrder")) : !"desc".equalsIgnoreCase(sortOrder);

        if (sortByIndex == null) {
            if (NullChecker.isNotEmpty(request.getParameter("sortBy"))) {
                sortValue = request.getParameter("sortBy");
            }
        } else {
            switch (sortByIndex) {
                case "0":
                    sortValue = "partnerOrg";
                    break;
                case "1":
                    sortValue = "user";
                    break;
                case "2":
                    sortValue = "total";
                    break;            
            }
        }
        
        return NullChecker.isNullOrEmpty(unSortedData) || unSortedData.size() == 0 ? unSortedData : bSorter.sortByColumn(unSortedData, sortValue, sortAscending);
    }

    // Private classes
    /**
     * Private class specific to this report allowing to package report results and the total count into one object
     */
    private class SearchResultsPackage {

        private final List<Map<String, Object>> searchResults;
        private final long totalCount; //sum of all of the "total" columns
        private final long totalRecordCount; //records returned

        public SearchResultsPackage(List<Map<String, Object>> searchResults, long totalCount, long totalRecordCount) {
            this.searchResults = searchResults;
            this.totalCount = totalCount;
            this.totalRecordCount = totalRecordCount;
        }

        public List<Map<String, Object>> getSearchResults() {
            return searchResults;
        }

        public long getTotalCount() {
            return totalCount;
        }
        
        public long getTotalRecordCount() {
            return totalRecordCount;
        }

    }

}
