package gov.va.nvap.web.report;

import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.ConsentDirectiveDetailedExpirationRequest;
import gov.va.nvap.service.adapter.doc.AdapterException;
import gov.va.nvap.svc.consentmgmt.stub.dao.MailTemplateDAO;
import gov.va.nvap.svc.consentmgmt.stub.dao.PatientConsentDirDAO;
import gov.va.nvap.svc.consentmgmt.stub.data.DetailedConsentDirective;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.dao.UserDocumentDAO;
import gov.va.nvap.web.helper.document.DocumentHelper;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.patient.ExcelGeneratorThread;
import gov.va.nvap.web.user.UserHelper;
import gov.va.nvap.web.util.Constants;
import gov.va.nvap.web.util.Paginator;
import gov.va.nvap.web.util.xls.CsvExporter;
import gov.va.nvap.web.util.xls.ExcelExporter;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

/**
 * @author Zack Peterson
 */
public class ExpiringConsentReportResults extends ResponseDispatcherHttpServlet {

    /**
     * Serial UID.
     */
    private static final long serialVersionUID = -4614454894945125740L;

    /**
     * Get the Consent Management Service Helper class from Spring. The Helper class can be used to access consent management functions on
     * the server side.
     *
     * @return helper class for consent management purposes
     */
    public ConsentManagementHelper getCmsHelper() {
        final ConsentManagementHelper cmsHelper = this.getBean("cmsHelper",
            ConsentManagementHelper.class);
        return cmsHelper;
    }

    public void exportToCsv(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException, IOException {

        final HttpSession session = request.getSession(false);
        final List<Map<String, Object>> results = this.getResults(
            request.getSession(), new Paginator(this.getReportHelper().getConsentDirectivesReportCapAsInt()));

        CsvExporter csvExporter = new CsvExporter();

        Map<String, String> csvReportMap = new LinkedHashMap<String, String>();

        csvReportMap.put("SSN", "ssn");
        csvReportMap.put("Patient Last Name", "lastName");
        csvReportMap.put("Patient First Name", "firstName");
        csvReportMap.put("Patient Middle Name", "middleName");
        csvReportMap.put("Opt In Date", "optInDate");
        csvReportMap.put("Expiration Date", "expirationDate");
        csvReportMap.put("Consent Type", "consentTypeName");
        csvReportMap.put("Entered By", "userId");
        csvReportMap.put("Authenticating Facility", "facilityName");

        String reportType = "Expiring_Consent_Detail_Report";
        try {
            if (session.getAttribute("optInReport").equals("true")) {
                reportType = "Opt-In_Patients_Detail_Report";
            }
        } catch (NullPointerException e) {

        }

        csvExporter.exportToCSV(response, reportType, results, (LinkedHashMap<String, String>) csvReportMap);
    }

    public void exportToExcel(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException, IOException {
        ExcelGeneratorThread exGenThread = new ExcelGeneratorThread("exGenThread");

        final HttpSession session = request.getSession(false);

        // Create a map of key which is based on the result data key and the heading value
        //The heading is used to create the column headers and the
        //key is used to pull the data from the results
        final Map<String, String> reportMap = new LinkedHashMap<String, String>();

        //Check to see if this is an OptIn Report. The session attribute gets
        //set in optInReportResults.jsp.
        boolean optInReport;
        try {
            if (session.getAttribute("optInReport").equals("true")) {
                optInReport = true;
                exGenThread.setTitle("Opt-In Patients Detailed Report");
            } else {
                optInReport = false;
                exGenThread.setTitle("Expiring Consent Detailed Report");
            }
        } catch (NullPointerException e) {
            optInReport = false;
            exGenThread.setTitle("Expiring Consent Detailed Report");
        }

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        final LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();
        //Check if this this is an Expiring Consent report in which case the
        //start date and end dates MUST be present. Otherwise this is an Opt In
        //report and dates CANNOT be provided.
        if (!optInReport) {
            filters.put("Start Date", this.getReportHelper().getFormattedDate((Date) session.getAttribute("startDate")));
            filters.put("End Date", this.getReportHelper().getFormattedDate((Date) session.getAttribute("endDate")));
        }
        filters.put("Entered By", session.getAttribute("userId"));
        String facilityNames = ExcelExporter.getFacilitiesFilter(getFacilityDAO(), (String) session.getAttribute("stationNumbers"));
        filters.put("Authenticating Facility", facilityNames);
        filters.put("Consent Type", ExcelExporter.getConsentTypeFilter((String) session.getAttribute("consentTypeName")));
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, session.getAttribute("patientTypes").toString());
        exGenThread.setFilterMap(filterMap);

        reportMap.put("ssn", "SSN");
        reportMap.put("lastName", "Patient Last Name");
        reportMap.put("firstName", "Patient First Name");
        reportMap.put("middleName", "Patient Middle Name");
        reportMap.put("optInDate", "Opt In Date");
        reportMap.put("expirationDate", "Expiration Date");
        reportMap.put("consentTypeName", "Consent Type");
        reportMap.put("userId", "Entered By");
        reportMap.put("facilityName", "Authenticating Facility");
        exGenThread.setReportMap(reportMap);

        exGenThread.setExcelExporter(this.getExcelExporter());
        exGenThread.setReportHelper(getReportHelper());
        exGenThread.setUserId(UserHelper.getUserName(request));
        exGenThread.setDocumentHelper(getDocumentHelper());
        exGenThread.setPatientConsentDirDAO(getPatientCosnentDirDAO());
        exGenThread.setUserDocumentDao(getUserDocumentDAO());

        // Set search attributes for getting audit results
        exGenThread.setAttributes(mapSessionAttributes(session));

        exGenThread.start();

        this.forward(request, response, "show");
    }

    private Map<String, Object> mapSessionAttributes(HttpSession session) {
        Map<String, Object> attributes = new HashMap<String, Object>();

        attributes.put("userId", (String) session
                .getAttribute("userId"));

        Date startDate = (Date) session.getAttribute("startDate");
        Date endDate = (Date) session.getAttribute("endDate");
        attributes.put("startDate", startDate);
        attributes.put("endDate", endDate);

        attributes.put("stationNumbers", (String) session.getAttribute("stationNumbers"));
        attributes.put("patientTypes", (Integer) session.getAttribute("patientTypes"));
        attributes.put("consentTypeName", (String) session.getAttribute("consentTypeName"));
        attributes.put("includeUnknownVisn", Boolean.TRUE.equals(session.getAttribute("includeUnknownVisn")));
        // sort
        attributes.put("sortValue", (String) session
                .getAttribute("inboundSortValue"));
        attributes.put("sortDirection", (String) session
                .getAttribute("inboundSortDirection"));

        return attributes;
    }

    public DocumentHelper getDocumentHelper() {
        return this.getBean("adapterDocumentHelper", DocumentHelper.class);
    }

    /**
     * Get the excel exporter class from Spring.
     *
     * @return the excel exporter object
     */
    public ExcelExporter getExcelExporter() {
        final ExcelExporter excelExporter = this.getBean("excelExporter",
            ExcelExporter.class);
        return excelExporter;
    }

    public ReportHelper getReportHelper() {
        final ReportHelper reportHelper = this.getBean("reportHelper",
            ReportHelper.class);
        return reportHelper;
    }

    private PatientConsentDirDAO getPatientCosnentDirDAO() {
        return this.getBean("PatientConsentDirDAO", PatientConsentDirDAO.class);
    }

    /**
     * Gathers and assembles the information for the Expiring Consent Report.
     *
     * @param session the http session
     * @param paginator the paginator for the expiring consent report
     * @return a list of map objects with patient information for the report
     */
    private List<Map<String, Object>> getResults(HttpSession session,
        final Paginator paginator) {

        PatientConsentDirDAO dao = this.getPatientCosnentDirDAO();
        Date startDate = (Date) session.getAttribute("startDate");
        Date endDate = (Date) session.getAttribute("endDate");
        final PatientConsentDirDAO.SearchRequest searchRequest = new PatientConsentDirDAO().new SearchRequest();

        searchRequest.startDate = startDate;
        searchRequest.endDate = endDate;
        searchRequest.stationNumbers = (String) session.getAttribute("stationNumbers");
        searchRequest.includeUnknownVisn = Boolean.TRUE.equals(session.getAttribute("includeUnknownVisn"));
        searchRequest.consentTypeName = (String) session.getAttribute("consentTypeName");
        searchRequest.userId = (String) session.getAttribute("userId");

        final String inboundSortValue = (String) session
            .getAttribute("inboundSortValue");
        final String inboundSortDirection = (String) session
            .getAttribute("inboundSortDirection");

        final int patientTypes = (Integer) session.getAttribute("patientTypes");

        searchRequest.patientTypes = patientTypes;

        // For use in sorting results
        if (NullChecker.isNotEmpty(inboundSortValue)) {
            searchRequest.sortBy = inboundSortValue;
        }
        if (NullChecker.isNotEmpty(inboundSortDirection)) {
            searchRequest.sortOrder = inboundSortDirection;
        }

        searchRequest.fromPage = paginator.getCurrentPage();
        searchRequest.recordsPerPage = paginator.getRecordsPerPage();

        final List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        try {

            final PatientConsentDirDAO.SearchResponse searchResponse = dao.find(searchRequest);

            if (searchResponse != null) {

                if ((searchResponse.consents != null)
                    && (searchResponse.consents.size() > 0)) {

                    Long totalRows = searchResponse.count;
                    session.setAttribute("totalRows", totalRows);
                    session.setAttribute("expiredNotificationLettersMaxTime", Constants.getExpiredNotificationLettersMaxTime());
                    session.setAttribute("expiredNotificationPerLetterTime", Constants.getExpiredNotificationPerLetterTime());
                    int pageNumber = paginator.getCurrentPage();
                    int pageSize = paginator.getRecordsPerPage();
                    int toPage = 0;
                    boolean nextDisabled = false;

                    if (pageNumber <= 0) {
                        toPage = pageSize;
                    } else {
                        toPage = (pageNumber + 1) * pageSize;
                    }
                    if (null != totalRows) {
                        if (toPage == totalRows.intValue()) {
                            nextDisabled = true;
                        }
                    }
                    paginator.setNextDisabled(nextDisabled);
                    session.setAttribute("nextDisabled", nextDisabled);

                    for (DetailedConsentDirective detailedConsentReference : searchResponse.consents) {
                        final Map<String, Object> resultMap = new HashMap<String, Object>();
                        if (detailedConsentReference != null) {
                            ReportDataProcessor.addSsnToResultMap(resultMap, detailedConsentReference.getSsn());
                            resultMap.put("lastName", detailedConsentReference.getLastName());
                            resultMap.put("firstName", detailedConsentReference.getFirstName());
                            resultMap.put("middleName", detailedConsentReference.getMiddleName());
                            String consentTypeName = detailedConsentReference.getOptinConsentType().getName();
                            if (consentTypeName.contains("NwHIN")) {
                                consentTypeName = consentTypeName.replace("NwHIN", Constants.getOrganizationName());
                            }
                            resultMap.put("consentTypeName", consentTypeName);
                            resultMap.put("optInDate", detailedConsentReference.getOptinDate());
                            resultMap.put("expirationDate", detailedConsentReference.getExpirationDate());
                            resultMap.put("facilityName", detailedConsentReference.getFacilityName());
                            resultMap.put("userId", detailedConsentReference.getUserId());
                            results.add(resultMap);
                        }
                    }
                }
            }
        } catch (final Exception ex) {
            throw new RuntimeException(ex);
        }

        return results;
    }

    private boolean isPaginatorPresent(HttpSession session) {
        return !NullChecker.isNullOrEmpty(session.getAttribute("paginator"));
    }

    public void next(HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        HttpSession session = request.getSession(false);
        if (this.isPaginatorPresent(session)) {
            Paginator paginator = (Paginator) session
                .getAttribute("paginator");
            paginator.next(request);
            session.setAttribute("paginator", paginator);
            session.setAttribute("expiringConsentReportResults",
                this.getResults(session, paginator));

            this.forward(request, response, "show");
        } else {
            this.forward(request, response, "noshow");
        }
    }

    public void prev(HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        HttpSession session = request.getSession(false);
        if (this.isPaginatorPresent(session)) {
            Paginator paginator = (Paginator) session
                .getAttribute("paginator");
            paginator.previous(request);
            session.setAttribute("paginator", paginator);
            session.setAttribute("expiringConsentReportResults",
                this.getResults(session, paginator));
            this.forward(request, response, "show");
        } else {
            this.forward(request, response, "noshow");
        }
    }

    @Override
    protected void unspecified(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        final HttpSession session = request.getSession(false);

        // sort
        final String expiringConsentSortValue = request
            .getParameter("inboundSortValue");
        final String expiringConsentSortDirection = request
            .getParameter("inboundSortDirection");
        session.setAttribute("inboundSortValue", expiringConsentSortValue);
        session.setAttribute("inboundSortDirection",
            expiringConsentSortDirection);

        if (this.isPaginatorPresent(session)) {
            Paginator paginator = (Paginator) session.getAttribute("paginator");
            if (NullChecker.isEmpty(paginator)) {
                paginator = new Paginator(0);
                session.setAttribute("paginator", paginator);
            }
            paginator.reset();
            session.setAttribute("expiringConsentReportResults",
                this.getResults(session, paginator));
            this.forward(request, response, "show");
        } else {
            this.forward(request, response, "noshow");
        }
    }

    public void view(HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        HttpSession session = request.getSession(false);
        final String icn = request.getParameter("icn");
        final String documentUniqueId = request
            .getParameter("expiringConsentUniqueId");
        final String styleSheetViewType = request
            .getParameter("styleSheetViewType");

        String document = null;
        try {
            document = this.getDocumentHelper().getDocument(icn,
                documentUniqueId, UserHelper.getUserName(request),
                this.getCmsHelper().getCompleteHomeCommunityId());
        } catch (final AdapterException ex) {
            throw new ServletException(ex);
        }

        session.setAttribute("icn", icn);
        session.setAttribute("documentUniqueId", documentUniqueId);
        session.setAttribute("document", document);
        session.setAttribute("isXmlViewEnabled", this.getDocumentHelper()
            .isXmlViewEnabled());

        if ("xml".equals(styleSheetViewType)) {
            document = document
                .replaceAll(
                    "<?xml-stylesheet type=\"text/xsl\" href=\"CCD.xsl\"?>",
                    "");
            response.setContentType("text/xml;charset=UTF-8");
            response.getWriter().write(document);
        } else {
            // Update session in the document
            final String html = this.getDocumentHelper().getHtml(document,
                styleSheetViewType);

            session.setAttribute("updatedDocument", html);
            this.forward(request, response, "showInboundDocumentReportDocument");
        }
    }

    public void download(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {

        final HttpSession session = request.getSession(false);
        Date startDate = (Date) session.getAttribute("startDate");
        Date endDate = (Date) session.getAttribute("endDate");
        String consentTypeName = (String) session.getAttribute("consentTypeName");
        String userId = (String) session.getAttribute("userId");
        final int patientTypes = (Integer) session.getAttribute("patientTypes");

        String stationNumber = (String) session.getAttribute("stationNumbers");

        ConsentDirectiveDetailedExpirationRequest expirationRequest
            = new ConsentDirectiveDetailedExpirationRequest();
        expirationRequest.setStartDate(startDate);
        expirationRequest.setEndDate(endDate);
        expirationRequest.setFacilityStation(stationNumber);
        expirationRequest.setConsentTypeName(consentTypeName);
        expirationRequest.setUserId(userId);
        expirationRequest.setPatientTypes(patientTypes);

        String template = this.getMailTemplateDAO().getByLetterType(4L).getText();

        expirationRequest.setTemplate(template);

        final ByteArrayInputStream byteInputStream
            = this.getCmsHelper().getExpiringConsentReportLetters(expirationRequest);

        response.setContentType("application/pdf");
        response.addHeader("Content-Disposition", "attachment; filename=patient_letters.pdf");

        OutputStream responseOutStream = response.getOutputStream();

        byte[] buffer = new byte[4096];
        int bytesRead;
        int totalBytes = 0;

        while ((bytesRead = byteInputStream.read(buffer)) != -1) {
            responseOutStream.write(buffer, 0, bytesRead);
            totalBytes += bytesRead;
        }

        response.setContentLength(totalBytes);

        byteInputStream.close();
        responseOutStream.flush();

        this.forward(request, response, "success");
    }

    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }

    private MailTemplateDAO getMailTemplateDAO() {
        return this.getBean("MailTemplateDAO", MailTemplateDAO.class);
    }

    private UserDocumentDAO getUserDocumentDAO() {
        return this.getBean("UserDocumentDAO", UserDocumentDAO.class);
    }
}
