package gov.va.nvap.web.report;

import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.ConsentDirectiveDetailedExpirationRequest;
import gov.va.nvap.service.adapter.doc.AdapterException;
import gov.va.nvap.svc.consentmgmt.stub.dao.MailTemplateDAO;
import gov.va.nvap.svc.consentmgmt.stub.dao.PatientConsentDirDAO;
import gov.va.nvap.svc.consentmgmt.stub.data.DetailedConsentDirective;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.dao.UserDocumentDAO;
import gov.va.nvap.web.helper.document.DocumentHelper;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.patient.ExcelGeneratorThread;
import gov.va.nvap.web.user.UserHelper;
import gov.va.nvap.web.util.Constants;
import gov.va.nvap.web.util.date.DateUtil;
import gov.va.nvap.web.util.xls.CsvExporter;
import gov.va.nvap.web.util.xls.ExcelExporter;
import java.io.ByteArrayInputStream;

import java.io.IOException;
import java.io.OutputStream;
import java.io.StringWriter;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.poi.ss.usermodel.Workbook;
import org.json.JSONException;
import org.json.JSONObject;

public class ExpiringConsentReport extends ResponseDispatcherHttpServlet {

	/**
	 * Serial Version UID
	 */
	private static final long serialVersionUID = -1303638679062246256L;

	/**
	 * Gets the consent management helper from Spring.
	 */
	private ConsentManagementHelper getCmsHelper() {
		return getBean("cmsHelper", ConsentManagementHelper.class);
	}

	/**
	 * Gets the facility helper from Spring.
	 */
	private FacilityHelper getFacilityHelper() {
		return getBean("facilityHelper", FacilityHelper.class);
	}

    private Map<String, String> getPreLoadValues() {
        return getCmsHelper().getExpiringConsentConfiguration();
    }

	// When redirected to the page before any actions
	@Override
	protected void unspecified(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
        final HttpSession session = request.getSession(false);
        final int dayWindow = Integer.parseInt(this.getPreLoadValues().get("window"));

        session.removeAttribute("results");

        DateFormat dateFormat = new SimpleDateFormat("MM/dd/yyyy");
        Calendar calendar = Calendar.getInstance();
        String startDateString = dateFormat.format(calendar.getTime());
        calendar.add(Calendar.DATE, dayWindow);
        String endDateString = dateFormat.format(calendar.getTime());
        session.setAttribute("startDateString", startDateString);
        session.setAttribute("endDateString", endDateString);

		request.setAttribute("facilities", this.getFacilityHelper().getAllVistAFacilities());

		this.forward(request, response, "show");
	}

    /**
     * Main search function returns json results use primarily for datatables.
     * @param request
     * @param response
     * @throws IOException
     * @throws ServletException
     */
    public void doSearch(final HttpServletRequest request,
			final HttpServletResponse response) throws IOException, ServletException{

        final HttpSession session = request.getSession(false);

        session.setAttribute("results", encodeIntoJSON(getResults(request), Long.parseLong(request.getParameter("draw"))));

        this.forward(request, response, "searchResultsJSON");

    }

    /**
     * Gathers and assembles the information for the Expiring Consent Report.
     *
     * @param session the http session
     * @param paginator the paginator for the expiring consent report
     * @return a list of map objects with patient information for the report
     */
    private SearchPackage getResults(HttpServletRequest request) {

        PatientConsentDirDAO dao = this.getPatientConsentDirDAO();
        Date startDate = ReportHelper.getStartDate(request.getParameter("startDate"));
        Date endDate = ReportHelper.getEndDate(request.getParameter("endDate"));
        final PatientConsentDirDAO.SearchRequest searchRequest = new PatientConsentDirDAO().new SearchRequest();

        searchRequest.startDate = startDate;
        searchRequest.endDate = endDate;
        searchRequest.stationNumbers = request.getParameter("stationNumbers");
        searchRequest.includeUnknownVisn = "true".equals(request.getParameter("includeUnknownVisn"));
        searchRequest.consentTypeName = request.getParameter("consentTypeName");
        searchRequest.userId = request.getParameter("enteredBy");

        final String inboundSortValue = (request.getParameter("order[0][column]")!=null)
                ? getSortValue(request.getParameter("order[0][column]")) : getSortValue(request.getParameter("sortBy"));
        final String inboundSortDirection = (request.getParameter("order[0][dir]")!=null)
                ?request.getParameter("order[0][dir]") : request.getParameter("sortOrder");

        final int patientTypes = Integer.parseInt(request.getParameter("patientTypes"));

        //if the parameters dont exist, return everything. The parameters are automatically passed if called
        //from search page. If else, its from an export so provide everything.
        final int page = (request.getParameter("start") != null)
                ? Integer.parseInt(request.getParameter("start"))/ Integer.parseInt(request.getParameter("length")) : 0;
        final int pageLength = (request.getParameter("length") != null)
                ? Integer.parseInt(request.getParameter("length")) : -1;

        searchRequest.patientTypes = patientTypes;

        // For use in sorting results
        if (NullChecker.isNotEmpty(inboundSortValue)) {
            searchRequest.sortBy = inboundSortValue;
        }
        if (NullChecker.isNotEmpty(inboundSortDirection)) {
            searchRequest.sortOrder = inboundSortDirection;
        }

        searchRequest.fromPage = page;
        searchRequest.recordsPerPage = pageLength;

        final List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        Long totalQueryCount = 0L;
        Long rowTotalForLetters =0L;
        try {

            final PatientConsentDirDAO.SearchResponse searchResponse = dao.find(searchRequest);

            if (searchResponse != null) {

                if ((searchResponse.consents != null)
                    && (searchResponse.consents.size() > 0)) {

                    rowTotalForLetters = searchResponse.count;
                    int pageNumber = page;
                    int pageSize = pageLength;
                    int toPage = 0;

                    if (pageNumber <= 0) {
                        toPage = pageSize;
                    } else {
                        toPage = (pageNumber + 1) * pageSize;
                    }

                    for (DetailedConsentDirective detailedConsentReference : searchResponse.consents) {
                        final Map<String, Object> resultMap = new HashMap<String, Object>();
                        if (detailedConsentReference != null) {
                            ReportDataProcessor.addSsnToResultMap(resultMap, detailedConsentReference.getSsn());
                            resultMap.put("lastName", detailedConsentReference.getLastName());
                            resultMap.put("firstName", detailedConsentReference.getFirstName());
                            resultMap.put("middleName", detailedConsentReference.getMiddleName());
                            String consentTypeName = detailedConsentReference.getOptinConsentType().getName();
                            if (consentTypeName.contains("NwHIN")) {
                                consentTypeName = consentTypeName.replace("NwHIN", Constants.getOrganizationName());
                            }
                            resultMap.put("consentTypeName", consentTypeName);
                            resultMap.put("optInDate", detailedConsentReference.getOptinDate());
                            resultMap.put("expirationDate", detailedConsentReference.getExpirationDate());
                            resultMap.put("facilityName", detailedConsentReference.getFacilityName());
                            resultMap.put("userId", detailedConsentReference.getUserId());
                            results.add(resultMap);
                        }
                    }
                }
            }

            final PatientConsentDirDAO.SearchResponse searchCountResponse = dao.findCount(searchRequest);
            totalQueryCount = searchCountResponse.count;

        } catch (final Exception ex) {
            throw new RuntimeException(ex);
        }
        //for expiring consent we need to know if one stationNumber was selected to allow pdf generation
        final boolean oneStation = (!request.getParameter("stationNumbers").contains(",") && request.getParameter("stationNumbers")!=null
                && !request.getParameter("stationNumbers").equals("ALL"))
                ? true : false;
        return new SearchPackage(results, totalQueryCount,
                Constants.getExpiredNotificationLettersMaxTime(),
                Constants.getExpiredNotificationPerLetterTime(),
                rowTotalForLetters, oneStation);
    }

    /**
     * transforms packed results into json object string, including any additional attributes that
     * may be needed on the view such as draw, recordsTotal, etc.
     * @param packedResults
     * @param draw
     * @return
     */
    private String encodeIntoJSON(SearchPackage packedResults, Long draw){
        String jsonString = "";

        DateFormat dateFormat = new SimpleDateFormat("MM/dd/yyyy");

        try {
            StringWriter json = new StringWriter();
            JSONObject obj = new JSONObject();
            List<List> data = new ArrayList<List>();

            for (Map<String, Object> row : packedResults.getResults()) {
                List<String> dataItem = new ArrayList<String>();

                dataItem.add(row.get("ssnMasked").toString());
                dataItem.add(row.get("lastName").toString());
                dataItem.add(row.get("firstName").toString());

                if(row.get("middleName")!=null){
                    dataItem.add(row.get("middleName").toString());
                } else {
                    dataItem.add("");
                }
                dataItem.add(dateFormat.format(row.get("optInDate")));
                dataItem.add(dateFormat.format(row.get("expirationDate")));
                dataItem.add(row.get("consentTypeName").toString());
                dataItem.add(row.get("userId").toString());
                dataItem.add(row.get("facilityName").toString());

                //add unmasked ssn for linkToPatients, does not appear as table column
                dataItem.add(row.get("ssn").toString());
                data.add(dataItem);
            }

            obj.put("data", data);
            obj.put("draw", draw);
            obj.put("recordsTotal", packedResults.getCount());
            obj.put("recordsFiltered",packedResults.getCount());
            obj.put("expiredNotificationLettersMaxTime", packedResults.getExpiredNotificationLettersMaxTime());
            obj.put("expiredNotificationPerLetterTime", packedResults.getExpiredNotificationPerLetterTime());
            obj.put("rowTotalForLetters", packedResults.getRowTotalForLetters());
            obj.put("oneStationRequested",packedResults.isOnlyOneStationRequested());

            obj.write(json);

            jsonString = json.toString();
        } catch (JSONException ex) {
            Logger.getLogger(ExpiringConsentReport.class.getName()).log(Level.SEVERE, null, ex);
        }

        return jsonString;
    }

    /**
     * converts the column number to the appropriate column sort value
     * @param columnNumber
     * @return
     */
    private String getSortValue(String columnNumber){
        String sortValue = "";

        switch (Integer.parseInt(columnNumber)){
            case 0:
                sortValue = "patientSSN";
                break;
            case 1:
                sortValue = "patientLastName";
                break;
            case 2:
                sortValue = "patientGivenName";
                break;
            case 3:
                sortValue = "patientMiddleName";
                break;
            case 4:
                sortValue = "optInDate";
                break;
            case 5:
                sortValue = "expirationDate";
                break;
            case 6:
                sortValue = "optInConsentType";
                break;
            case 7:
                sortValue = "userId";
                break;
            case 8:
                sortValue = "facilityName";
                break;
        }
        return sortValue;
    }

    public void exportToExcel(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException, IOException {

        // Create a map of key which is based on the result data key and the heading value
        //The heading is used to create the column headers and the
        //key is used to pull the data from the results
        final Map<String, String> reportMap = new LinkedHashMap<String, String>();

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        final LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();

        boolean optInReport = false;
        if (!NullChecker.isNullOrEmpty(request.getParameter("optInReport"))) {
            optInReport = true;
        }

        //Check if this this is an Expiring Consent report in which case the
        //start date and end dates MUST be present. Otherwise this is an Opt In
        //report and dates CANNOT be provided.
        if (!optInReport) {
            filters.put("Start Date", this.getReportHelper().getFormattedDate(ReportHelper.getStartDate(request.getParameter("startDate"))));
            filters.put("End Date", this.getReportHelper().getFormattedDate(ReportHelper.getEndDate(request.getParameter("endDate"))));
        }
        filters.put("Entered By", request.getParameter("enteredBy"));
        String facilityNames = ExcelExporter.getFacilitiesFilter(getFacilityDAO(), request.getParameter("stationNumbers"));
        filters.put("Authenticating Facility", facilityNames);
        filters.put("Consent Type", ExcelExporter.getConsentTypeFilter(request.getParameter("consentTypeName")));
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, request.getParameter("patientTypes"));

        reportMap.put("ssn", "SSN");
        reportMap.put("lastName", "Patient Last Name");
        reportMap.put("firstName", "Patient First Name");
        reportMap.put("middleName", "Patient Middle Name");
        reportMap.put("optInDate", "Opt In Date");
        reportMap.put("expirationDate", "Expiration Date");
        reportMap.put("consentTypeName", "Consent Type");
        reportMap.put("userId", "Entered By");
        reportMap.put("facilityName", "Authenticating Facility");

        int minimumThreshold = Integer.parseInt((String) this.getServletContext().getAttribute("scheduledExportMin"));
        int total = Integer.parseInt(request.getParameter("totalRows"));

        //Start a scheduled export with its own thread if greater than the minimum threshold
        if (total >= minimumThreshold) {
            ExcelGeneratorThread exGenThread = new ExcelGeneratorThread("exGenThread");

            //Check to see if this is an OptIn Report. The request parameter
            //set in optInReport.jsp.
            exGenThread.setTitle("Expiring Consent Detail Report");
            if (optInReport) {
                exGenThread.setTitle("Opt-In Patients Detail Report");
            }
            exGenThread.setFilterMap(filterMap);
            exGenThread.setReportMap(reportMap);

            exGenThread.setExcelExporter(this.getExcelExporter());
            exGenThread.setReportHelper(getReportHelper());
            exGenThread.setUserId(UserHelper.getUserName(request));
            exGenThread.setDocumentHelper(getDocumentHelper());
            exGenThread.setPatientConsentDirDAO(getPatientConsentDirDAO());
            exGenThread.setUserDocumentDao(getUserDocumentDAO());

            // Set search attributes for getting audit results
            exGenThread.setAttributes(mapSessionAttributes(request));

            exGenThread.start();
        }
        //Otherwise generate and download the export directly
        else {
            // Create workbook
            String title = "Expiring Consent Detail Report";
            String fileName = "Expiring_Consent_Detail_Report";
            if (optInReport) {
                title = "Opt-In Patients Detail Report";
                fileName = "Opt-In_Patients_Detail_Report";
            }

            List<Map<String, Object>> results;

            ExcelGeneratorThread doNow = new ExcelGeneratorThread("doNow");
            doNow.setAttributes(mapSessionAttributes(request));
            doNow.setPatientConsentDirDAO(this.getPatientConsentDirDAO());
            // Set search attributes for getting audit results
            doNow.setAttributes(mapSessionAttributes(request));

            results = doNow.getPatientConsentDirResults();

            final Workbook wb = this.getExcelExporter().exportToExcel(title, title, reportMap, results, filterMap, null);

            // Write Excel workbook to Stream
            this.getExcelExporter().writeExcelToStream(fileName, wb, response);
        }
    }

    private Map<String, Object> mapSessionAttributes(HttpServletRequest request) {
        Map<String, Object> attributes = new HashMap<String, Object>();

        attributes.put("userId", request.getParameter("enteredBy"));

        Date startDate = ReportHelper.getStartDate(request.getParameter("startDate"));
        Date endDate = ReportHelper.getEndDate(request.getParameter("endDate"));
        attributes.put("startDate", startDate);
        attributes.put("endDate", endDate);

        attributes.put("stationNumbers", request.getParameter("stationNumbers"));
        attributes.put("patientTypes", Integer.parseInt(request.getParameter("patientTypes")));
        attributes.put("consentTypeName", request.getParameter("consentTypeName"));
        attributes.put("includeUnknownVisn", "true".equals(request.getParameter("includeUnknownVisn")));
        // sort
        final String inboundSortValue = (request.getParameter("order[0][column]")!=null)
                ? getSortValue(request.getParameter("order[0][column]")) : getSortValue(request.getParameter("sortBy"));
        final String inboundSortDirection = (request.getParameter("order[0][dir]")!=null)
                ?request.getParameter("order[0][dir]") : request.getParameter("sortOrder");

        attributes.put("sortValue", inboundSortValue);
        attributes.put("sortDirection", inboundSortDirection);

        return attributes;
    }

    public void exportToCsv(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException, IOException {

        Map<String, String> csvReportMap = new LinkedHashMap<String, String>();

        csvReportMap.put("SSN", "ssn");
        csvReportMap.put("Patient Last Name", "lastName");
        csvReportMap.put("Patient First Name", "firstName");
        csvReportMap.put("Patient Middle Name", "middleName");
        csvReportMap.put("Opt In Date", "optInDate");
        csvReportMap.put("Expiration Date", "expirationDate");
        csvReportMap.put("Consent Type", "consentTypeName");
        csvReportMap.put("Entered By", "userId");
        csvReportMap.put("Authenticating Facility", "facilityName");

        int minimumThreshold = Integer.parseInt((String) this.getServletContext().getAttribute("scheduledExportMin"));
        int total = Integer.parseInt(request.getParameter("totalRows"));

        //Start a scheduled export with its own thread if greater than the minimum threshold
        if (total >= minimumThreshold) {
            ExcelGeneratorThread csvGenThread = new ExcelGeneratorThread("csvGenThread");
            csvGenThread.setDocumentType("csv");
            csvGenThread.setReportMap(csvReportMap);

            csvGenThread.setTitle("Expiring Consent Detail Report");
            if (!NullChecker.isNullOrEmpty(request.getParameter("optInReport"))) {
                csvGenThread.setTitle("Opt-In Patients Detail Report");
            }
            csvGenThread.setExcelExporter(this.getExcelExporter());
            csvGenThread.setReportHelper(getReportHelper());
            csvGenThread.setUserId(UserHelper.getUserName(request));
            csvGenThread.setDocumentHelper(getDocumentHelper());
            csvGenThread.setPatientConsentDirDAO(getPatientConsentDirDAO());
            csvGenThread.setUserDocumentDao(getUserDocumentDAO());

            // Set search attributes for getting audit results
            csvGenThread.setAttributes(mapSessionAttributes(request));

            csvGenThread.start();

        }
        //Otherwise generate and download the export directly
        else {
            List<Map<String, Object>> results;
            SearchPackage packedResults;

            packedResults = this.getResults(request);
            results = packedResults.getResults();

            CsvExporter csvExporter = new CsvExporter();
            String title = "Expiring Consent Detail Report";
            if (!NullChecker.isNullOrEmpty(request.getParameter("optInReport"))) {
                title = "Opt-In Patients Detail Report";
            }
            csvExporter.exportToCSV(response, title, results, (LinkedHashMap<String, String>) csvReportMap);
        }
    }

    public void view(HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        HttpSession session = request.getSession(false);
        final String icn = request.getParameter("icn");
        final String documentUniqueId = request
            .getParameter("expiringConsentUniqueId");
        final String styleSheetViewType = request
            .getParameter("styleSheetViewType");

        String document = null;
        try {
            document = this.getDocumentHelper().getDocument(icn,
                documentUniqueId, UserHelper.getUserName(request),
                this.getCmsHelper().getCompleteHomeCommunityId());
        } catch (final AdapterException ex) {
            throw new ServletException(ex);
        }

        session.setAttribute("icn", icn);
        session.setAttribute("documentUniqueId", documentUniqueId);
        session.setAttribute("document", document);
        session.setAttribute("isXmlViewEnabled", this.getDocumentHelper()
            .isXmlViewEnabled());

        if ("xml".equals(styleSheetViewType)) {
            document = document
                .replaceAll(
                    "<?xml-stylesheet type=\"text/xsl\" href=\"CCD.xsl\"?>",
                    "");
            response.setContentType("text/xml;charset=UTF-8");
            response.getWriter().write(document);
        } else {
            // Update session in the document
            final String html = this.getDocumentHelper().getHtml(document,
                styleSheetViewType);

            session.setAttribute("updatedDocument", html);
            this.forward(request, response, "showInboundDocumentReportDocument");
        }
    }

    public void download(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {

        Date startDate = ReportHelper.getStartDate(request.getParameter("startDate"));
        Date endDate = ReportHelper.getStartDate(request.getParameter("endDate"));
        String consentTypeName = request.getParameter("consentTypeName");
        String userId = request.getParameter("enteredBy");
        final int patientTypes = Integer.parseInt(request.getParameter("patientTypes"));

        String stationNumber = request.getParameter("stationNumbers");

        if (endDate != null) {
            endDate = DateUtil.addTime(endDate, Calendar.HOUR, 24);
        }

        ConsentDirectiveDetailedExpirationRequest expirationRequest
            = new ConsentDirectiveDetailedExpirationRequest();
        expirationRequest.setStartDate(startDate);
        expirationRequest.setEndDate(endDate);
        expirationRequest.setFacilityStation(stationNumber);
        expirationRequest.setConsentTypeName(consentTypeName);
        expirationRequest.setUserId(userId);
        expirationRequest.setPatientTypes(patientTypes);

        String template = this.getMailTemplateDAO().getByLetterType(4L).getText();

        expirationRequest.setTemplate(template);

        final ByteArrayInputStream byteInputStream
            = this.getCmsHelper().getExpiringConsentReportLetters(expirationRequest);

        response.setContentType("application/pdf");
        response.addHeader("Content-Disposition", "attachment; filename=patient_letters.pdf");

        OutputStream responseOutStream = response.getOutputStream();

        byte[] buffer = new byte[4096];
        int bytesRead;
        int totalBytes = 0;

        while ((bytesRead = byteInputStream.read(buffer)) != -1) {
            responseOutStream.write(buffer, 0, bytesRead);
            totalBytes += bytesRead;
        }

        response.setContentLength(totalBytes);

        byteInputStream.close();
        responseOutStream.flush();

        //this.forward(request, response, "success");
    }
    private PatientConsentDirDAO getPatientConsentDirDAO() {
        return this.getBean("PatientConsentDirDAO", PatientConsentDirDAO.class);
    }

    private MailTemplateDAO getMailTemplateDAO() {
        return this.getBean("MailTemplateDAO", MailTemplateDAO.class);
    }

    public ReportHelper getReportHelper() {
        final ReportHelper reportHelper = this.getBean("reportHelper",
            ReportHelper.class);
        return reportHelper;
    }

    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }

    public DocumentHelper getDocumentHelper() {
        return this.getBean("adapterDocumentHelper", DocumentHelper.class);
    }

    private UserDocumentDAO getUserDocumentDAO() {
        return this.getBean("UserDocumentDAO", UserDocumentDAO.class);
    }

    /**
     * Get the excel exporter class from Spring.
     *
     * @return the excel exporter object
     */
    public ExcelExporter getExcelExporter() {
        final ExcelExporter excelExporter = this.getBean("excelExporter",
            ExcelExporter.class);
        return excelExporter;
    }

    /**
    Inner class for packaging parameters
    */
    private class SearchPackage {
        private final List<Map<String,Object>> results;
        private final Long count;
        private final String expiredNotificationLettersMaxTime;
        private final String expiredNotificationPerLetterTime;
        private final Long rowTotalForLetters;
        private final boolean onlyOneStationRequested;

        public SearchPackage(List<Map<String, Object>> results, Long count, String expiredNotificationLettersMaxTime,
                String expiredNotificationPerLetterTime, Long rowTotalForLetters, boolean oneStation) {
            this.results = results;
            this.count = count;
            this.expiredNotificationLettersMaxTime = expiredNotificationLettersMaxTime;
            this.expiredNotificationPerLetterTime = expiredNotificationPerLetterTime;
            this.rowTotalForLetters = rowTotalForLetters;
            this.onlyOneStationRequested = oneStation;
        }

        public List<Map<String, Object>> getResults() {
            return results;
        }

        public Long getCount() {
            return count;
        }

        public String getExpiredNotificationLettersMaxTime() {
            return expiredNotificationLettersMaxTime;
        }

        public String getExpiredNotificationPerLetterTime() {
            return expiredNotificationPerLetterTime;
        }

        public Long getRowTotalForLetters() {
            return rowTotalForLetters;
        }

        public boolean isOnlyOneStationRequested() {
            return onlyOneStationRequested;
        }

    }
}
