package gov.va.nvap.web.report;

import gov.va.med.nhin.adapter.audit.ActionType;
import gov.va.med.nhin.adapter.audit.ActionValuesType;
import gov.va.med.nhin.adapter.audit.AuditType;
import gov.va.med.nhin.adapter.audit.FieldType;
import gov.va.med.nhin.adapter.audit.GetAudits;
import gov.va.med.nhin.adapter.audit.GetAuditsResponse;
import gov.va.med.nhin.adapter.audit.PageInfoType;
import gov.va.med.nhin.adapter.audit.SortDirection;
import gov.va.med.nhin.adapter.audit.SortFieldType;
import gov.va.med.nhin.adapter.audit.SortFieldsType;
import gov.va.med.nhin.adapter.audit.StringValuesType;
import gov.va.nvap.common.date.GregorianDateUtil;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.service.adapter.audit.AdapterAuditManager;
import gov.va.nvap.service.adapter.audit.DirectAuditManager;
import gov.va.nvap.service.adapter.audit.DirectServiceEndpoint;
import gov.va.nvap.service.adapter.audit.DirectServiceException;
import gov.va.nvap.service.adapter.doc.AdapterException;
import gov.va.nvap.service.audit.AuditException;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.dao.PurposeOfUseDAO;
import gov.va.nvap.web.dao.UserDocumentDAO;
import gov.va.nvap.web.entities.PurposeOfUse;
import gov.va.nvap.web.helper.document.DocumentHelper;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.patient.ExcelGeneratorThread;
import gov.va.nvap.web.user.UserHelper;
import gov.va.nvap.web.util.Constants;
import gov.va.nvap.web.util.xls.CsvExporter;
import gov.va.nvap.web.util.xls.ExcelExporter;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.ejb.EJB;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.poi.ss.usermodel.Workbook;

import org.json.JSONException;
import org.json.JSONObject;

/**
 * Disclosures Detail Report search servlet
 *
 * @author David Vazquez
 */
public class DocumentDisclosureReport extends
    gov.va.nvap.web.app.ResponseDispatcherHttpServlet {

    /**
     * Serial UID.
     */
    private static final long serialVersionUID = -4913924881974107661L;
    private AdapterAuditManager adapterAuditManager;
    private final DirectAuditManager directAuditManager = new DirectServiceEndpoint();


    public ConsentManagementHelper getCmsHelper() {
        final ConsentManagementHelper cmsHelper = this.getBean("cmsHelper",
            ConsentManagementHelper.class);
        return cmsHelper;
    }

    /**
     * Get the facility helper from Spring.
     * @return FacilityHelper
     */
    public FacilityHelper getFacilityHelper() {
        final FacilityHelper facilityHelper = this.getBean("facilityHelper",
            FacilityHelper.class);
        return facilityHelper;
    }

    private ReportHelper getReportHelper() {
        return this.getBean("reportHelper", ReportHelper.class);
    }

    @Override
    public void unspecified(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        final HttpSession session = request.getSession(false);


        session.removeAttribute("results");

        request.setAttribute("facilities", this.getFacilityHelper()
            .getAllVistAFacilities());
        this.getReportHelper().getOrgLists(request, this.getCmsHelper());

        if (Constants.PURPOSE_OF_USE_FLAG) {
            final PurposeOfUseDAO purposeOfUseDAO = this.getBean("purposeOfUseDAO", PurposeOfUseDAO.class);
            Collection<PurposeOfUse> pous = purposeOfUseDAO.findPurposeOfUse();
            request.setAttribute("pous", pous);
        }
        request.setAttribute("pouFlag", Constants.PURPOSE_OF_USE_FLAG);
        ReportHelper.setDefaultSearchDates(session);

        this.forward(request, response, "show");
    }

    public void doSearch(final HttpServletRequest request,
        final HttpServletResponse response) throws AuditException, IOException, ServletException{

        final HttpSession session = request.getSession(false);
        session.setAttribute("results", encodeIntoJSON(getResults(request, false), request.getParameter("draw")));
        this.forward(request, response, "searchResultsJSON");
    }

    private SearchPackage getResults(HttpServletRequest request, Boolean exporting) throws AuditException, DirectServiceException {
        // Get the information entered by the user in the GUI
        final String ssn = request.getParameter("ssn");
        final String lastName = request.getParameter("lastName");
        final String firstName = request.getParameter("firstName");
        final String userId = "";
        final String facility = request.getParameter("facility");
        final String purposeOfUse = request.getParameter("purposeOfUse");

        final Date sd = ReportHelper.getStartDate(request.getParameter("startDate"));
        final Date ed = ReportHelper.getEndDate(request.getParameter("endDate"));
        final String organization = request.getParameter("organization");
        final String documentDisclosureSource = request.getParameter("source");
        final int patientTypes = Integer.parseInt(request.getParameter("patientTypes"));
        //Sorting parameters passed from table, if not its directly passed.
        String documentDisclosureSortValue = (request.getParameter("order[0][column]") != null)
            ? request.getParameter("order[0][column]") : request.getParameter("sortValue");

        documentDisclosureSortValue = getSortValue(documentDisclosureSortValue, documentDisclosureSource);
        final String documentDisclosureSortDirection = (request.getParameter("order[0][dir]") != null)
            ? request.getParameter("order[0][dir]") : request.getParameter("sortOrder");



        final GetAudits getAuditsRequest = new GetAudits();
        if (NullChecker.isNotEmpty(sd)) {
            getAuditsRequest.setFromDate(GregorianDateUtil
                .getGregorianCalendarByDate(sd));
        }
        if (NullChecker.isNotEmpty(ed)) {
            getAuditsRequest.setToDate(GregorianDateUtil
                .getGregorianCalendarByDate(ed));
        }

        // Set Patient Types
        getAuditsRequest.setPatientTypes(patientTypes);

        /*
         * // Set Patient Ids StringValuesType patientIdsType = new
         * StringValuesType(); patientIdsType.setNotIn(false);
         * patientIdsType.getValue().add(icn);
         * getAuditsRequest.setPatientIds(patientIdsType);
         */
        /*
         * // Set Patient Ids if (NullChecker.isNotEmpty(icn)) { final
         * StringValuesType patientIds = new StringValuesType();
         * patientIds.getValue().add(icn); patientIds.setNotIn(false);
         * getAuditsRequest.setPatientIds(patientIds); }
         */
        // Set SSN
        if (NullChecker.isNotEmpty(ssn)) {
            final StringValuesType patientSSNs = new StringValuesType();
            patientSSNs.getValue().add(ssn);
            patientSSNs.setNotIn(false);
            getAuditsRequest.setPatientSSNs(patientSSNs);
        }
        // Set Last Name
        if (NullChecker.isNotEmpty(lastName)) {
            final StringValuesType patientLastNames = new StringValuesType();
            patientLastNames.getValue().add(lastName);
            patientLastNames.setNotIn(false);
            getAuditsRequest.setPatientLastNames(patientLastNames);
        }
        // Set First Name
        if (NullChecker.isNotEmpty(firstName)) {
            final StringValuesType patientFirstNames = new StringValuesType();
            patientFirstNames.getValue().add(firstName);
            patientFirstNames.setNotIn(false);
            getAuditsRequest.setPatientGivenNames(patientFirstNames);
        }
        //Set User ID
        /*
        if (NullChecker.isNotEmpty(userId)) {
            final StringValuesType userIdValue = new StringValuesType();
            userIdValue.getValue().add(userId);
            userIdValue.setNotIn(false);
            getAuditsRequest.setUserIds(userIdValue);
        }
        */
        // Set Purpose of Use
        if (!NullChecker.isNullOrEmpty(purposeOfUse)) {
            final StringValuesType purposeForUses = new StringValuesType();
            purposeForUses.getValue().add(purposeOfUse);
            purposeForUses.setNotIn(false);
            getAuditsRequest.setPurposeForUses(purposeForUses);
        }

        // Set Facility Ids
        if (NullChecker.isNotEmpty(facility)) {
            final StringValuesType facilityIds = new StringValuesType();
            facilityIds.setNotIn(false);
            facilityIds.getValue().add(facility);
            getAuditsRequest.setPatientFacilityNumbers(facilityIds);
        }

        // Set Organization Types
        if (NullChecker.isNotEmpty(organization)) {
            final StringValuesType organizationIds = new StringValuesType();
            organizationIds.setNotIn(false);
            organizationIds.getValue().add(organization);
            getAuditsRequest.setOrganizationIds(organizationIds);
        } else {
            // input orgs are empty - intent is to retrieve all orgs except VA
            final StringValuesType organizationIds = new StringValuesType();
            organizationIds.setNotIn(true);
            organizationIds.getValue().add(
                this.getCmsHelper().getCompleteHomeCommunityId());
            getAuditsRequest.setOrganizationIds(organizationIds);
        }
        // sorting
        if (NullChecker.isNotEmpty(documentDisclosureSortValue)) {
            final SortFieldType sortField = new SortFieldType();
            sortField
                .setField(FieldType.fromValue(documentDisclosureSortValue));
            if ("DESC".equalsIgnoreCase(documentDisclosureSortDirection)) {
                sortField.setDirection(SortDirection.DESC);
            } else {
                sortField.setDirection(SortDirection.ASC);
            }
            final SortFieldsType sortFieldsType = new SortFieldsType();
            sortFieldsType.getSortField().add(sortField);
            getAuditsRequest.setSortFields(sortFieldsType);
        }

        /*
         * StringValuesType orgTypes = new StringValuesType(); if
         * (NullChecker.isNotEmpty(organization) &&
         * "external".equals(organization)) { orgTypes.getValue().add("%" +
         * this.getCmsHelper().getHomeCommunity().getOrgOid());
         * orgTypes.setNotIn(true); } else {
         * orgTypes.getValue().add(organization); orgTypes.setNotIn(false); }
         * getAuditsRequest.setOrganizationIds(orgTypes);
         *
         * // Set Remote Organization Types StringValuesType remoteOrgTypes =
         * new StringValuesType(); remoteOrgTypes.setNotIn(false);
         * getAuditsRequest.setRemoteOrganizationIds(remoteOrgTypes);
         */
        // Set Action Value Types
        final ActionValuesType actionsType = new ActionValuesType();
        actionsType.getValue().add(ActionType.RETRIEVE_DOCUMENT);
        actionsType.setNotIn(false);
        getAuditsRequest.setActions(actionsType);
        // Set Page information
        final PageInfoType pageInfoType = new PageInfoType();

        pageInfoType.setPageNumber((request.getParameter("start") != null) ? Integer.parseInt(request.getParameter("start"))/Integer.parseInt(request.getParameter("length")) : 0);
        pageInfoType.setPageSize((request.getParameter("length") != null) ? Integer.parseInt(request.getParameter("length")): -1);

        getAuditsRequest.setPageInfo(pageInfoType);

        final List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        String resultIcn = "";
        Long totalCount = new Long(0); //default it to zero, but we set it below
        try {
            final GetAuditsResponse queryResponse;
            if (documentDisclosureSource.equals("direct")) {
                queryResponse = this.directAuditManager.getAudits(getAuditsRequest);
                totalCount = exporting ? 0 : this.directAuditManager.getAuditsCount(getAuditsRequest);
            } else {
                queryResponse = this.adapterAuditManager.getAudits(getAuditsRequest);
                if(queryResponse.getPageInfo() != null) {
                    totalCount = exporting ? 0 : new Long(queryResponse.getPageInfo().getTotalSize());
                }
            }
            if (NullChecker.isNotEmpty(queryResponse)
                && NullChecker.isNotEmpty(queryResponse.getAudits())
                && NullChecker.isNotEmpty(queryResponse.getAudits()
                    .getAudit())) {
                final List<AuditType> auditTypeList = queryResponse.getAudits()
                    .getAudit();
                for (final AuditType auditType : auditTypeList) {
                    final Map<String, Object> resultMap = new HashMap<String, Object>();
                    // Replace SSN with Mask
                    ReportDataProcessor.addSsnToResultMap(resultMap, auditType.getPatientSSN());
                    resultMap.put("patientICN", ReportDataProcessor
                        .nullEmptyReplaceWithUnknown(auditType
                            .getPatientId()));
                    resultMap.put("auditTime", GregorianDateUtil
                        .getDateFromGregorianCalendar(auditType
                            .getAuditTime()));
                    resultMap.put("auditTimeFormatted", this.getReportHelper().getFormattedDateTime(GregorianDateUtil
                        .getDateFromGregorianCalendar(auditType
                            .getAuditTime())));

                    resultMap.put("purposeForUse", ReportDataProcessor
                        .nullEmptyReplaceWithUnknown(auditType
                            .getPurposeForUse()));

                    resultMap
                        .put("userId", ReportDataProcessor
                            .nullEmptyReplaceWithUnknown(auditType
                                .getUserId()));

                    resultMap.put("lastName", ReportDataProcessor
                        .nullEmptyReplaceWithUnknown(auditType
                            .getPatientLastName()));
                    resultMap.put("firstName", ReportDataProcessor
                        .nullEmptyReplaceWithUnknown(auditType
                            .getPatientGivenName()));
                    resultMap.put("patientFacilityName", ReportDataProcessor
                        .fixStation(auditType.getPatientFacilityName()));
                    resultMap.put("patientFacilityNumber", ReportDataProcessor
                        .nullEmptyReplaceWithUnknown(auditType
                            .getPatientFacilityNumber()));
                    resultMap.put("organizationName", this.getReportHelper().findOrganizationByOid(auditType.getOrganizationId(), auditType.getOrganizationName()));
                    resultMap.put("facilityOid", ReportDataProcessor
                        .nullEmptyReplaceWithUnknown(auditType
                            .getOrganizationId()));
                     resultMap.put("roleName", ReportDataProcessor
                        .nullEmptyReplaceWithUnknown(auditType
                            .getRoleName()));
                    /*
                     * resultMap.put( "facility", this.getCmsHelper()
                     * .getOrganizationByHomeCommunityId(
                     * auditType.getOrganizationId()));
                     */
                    resultMap.put("documentId", ReportDataProcessor
                        .nullEmptyReplaceWithUnknown(auditType
                            .getDocumentId()));
                    resultMap.put("documentTitle", ReportDataProcessor
                        .nullEmptyReplaceWithUnknown(auditType
                            .getDocumentTitle()));
                    results.add(resultMap);
                }

                // Check if any of the patient match criteria exists
                if (NullChecker.isNotEmpty(ssn)
                    || NullChecker.isNotEmpty(lastName)
                    || NullChecker.isNotEmpty(firstName)) {
                    // Default to unknown
                    if (NullChecker.isNotEmpty(auditTypeList)) {
                        final AuditType result = auditTypeList.get(0);
                        resultIcn = result.getPatientId();

                        if (NullChecker.isNotEmpty(resultIcn)) {
                            for (final AuditType auditType : auditTypeList) {
                                if (!resultIcn.equals(auditType.getPatientId())) {
                                    resultIcn = "Multiple ICNs";
                                    break;
                                }
                            }
                        } else if (NullChecker.isNotEmpty(result
                            .getPatientGivenName())
                            && NullChecker.isNotEmpty(result
                                .getPatientLastName())
                            && NullChecker.isNotEmpty(result
                                .getPatientSSN())) {
                            final String patientGivenName = result
                                .getPatientGivenName();
                            final String patientLastName = result
                                .getPatientLastName();
                            final String patientSsn = result.getPatientSSN();
                            boolean multipleIcns = false;
                            for (final AuditType auditType : auditTypeList) {
                                if (!patientGivenName.equals(auditType
                                    .getPatientGivenName())
                                    || !patientLastName.equals(auditType
                                        .getPatientLastName())
                                    || !patientSsn.equals(auditType
                                        .getPatientSSN())) {
                                    multipleIcns = true;
                                    break;
                                }
                            }
                            if (multipleIcns) {
                                resultIcn = "Multiple ICNs";
                            } else {
                                resultIcn = this.getReportHelper().resolveICN(
                                    result.getPatientGivenName(),
                                    result.getPatientLastName(),
                                    result.getPatientSSN());
                            }
                        }
                    } else if (NullChecker.isNotEmpty(ssn)
                        && NullChecker.isNotEmpty(lastName)
                        && NullChecker.isNotEmpty(firstName)) {
                        // Try to get from what was typed
                        resultIcn = this.getReportHelper().resolveICN(
                            firstName, lastName, ssn);
                    }
                    // Set in Session
                    if (NullChecker.isEmpty(resultIcn)) {
                        resultIcn = "Unknown";
                    }
                }
            }

        } catch (final AuditException ex) {
            throw new AuditException();
        } catch (final DirectServiceException ex) {
            throw new DirectServiceException();
        }

        SearchPackage sp = new SearchPackage(results,resultIcn, documentDisclosureSource);
        sp.setTotalCount(totalCount);
        return sp;
    }

    /**
     * Changes the result package into a JSON string, adds the disclosure source and patient ICN
     * @param packagedResults
     * @return
     */
    private String encodeIntoJSON(SearchPackage packagedResults, String draw){
        String returnValue = "";

        try {
            StringWriter json = new StringWriter();
            JSONObject obj = new JSONObject();
            List<List> data = new ArrayList<List>();

            for (Map<String,Object> currentRow : packagedResults.getSearchResults()) {
                List<String> dataItem = new ArrayList<String>();

                dataItem.add(currentRow.get("ssnMasked").toString());
                dataItem.add(currentRow.get("lastName").toString());
                dataItem.add(currentRow.get("firstName").toString());
                dataItem.add(currentRow.get("auditTimeFormatted").toString());
                dataItem.add(currentRow.get("documentTitle").toString());

                //Columns vary depending on source
                if("exchange".equalsIgnoreCase(packagedResults.getSource())){
                    dataItem.add(currentRow.get("patientFacilityName").toString());
                    dataItem.add(currentRow.get("patientFacilityNumber").toString());
                }
                dataItem.add(currentRow.get("organizationName").toString());
                dataItem.add(currentRow.get("userId").toString());
                dataItem.add(currentRow.get("purposeForUse").toString());
                //Columns vary depending on source
                if("exchange".equalsIgnoreCase(packagedResults.getSource())){
                    dataItem.add(currentRow.get("roleName").toString());
                    dataItem.add(currentRow.get("patientICN").toString());
                    dataItem.add(currentRow.get("documentId").toString());
                }

                //unmasked ssn added for linking to patient search
                dataItem.add(currentRow.get("ssn").toString());

                data.add(dataItem);
            }

            obj.put("data", data);

            obj.put("disclosureSource", packagedResults.getSource());
            obj.put("patientICN", packagedResults.getIcn());
            obj.put("recordsTotal", packagedResults.getTotalCount());
            obj.put("recordsFiltered", packagedResults.getTotalCount());
            obj.put("draw", draw);

            obj.write(json);

            returnValue = json.toString();
        } catch (JSONException ex) {
            Logger.getLogger(DocumentDisclosureReport.class.getName()).log(Level.SEVERE, null, ex);
        }

        return returnValue;
    }

    /**
     * sort values are passed as column numbers, based on parameters corresponding string value is returned.
     * @param colNumber
     * @param source
     * @return
     */
    private String getSortValue(String colNumber, String source){

        String sortValue = "";
        if("exchange".equalsIgnoreCase(source)){
            if(null != colNumber)switch (Integer.parseInt(colNumber)) {
                case 0:
                    sortValue = "patientSSN";
                    break;
                case 1:
                    sortValue = "patientLastName";
                    break;
                case 2:
                    sortValue = "patientGivenName";
                    break;
                case 3:
                    sortValue = "auditTime";
                    break;
                case 4:
                    sortValue = "documentTitle";
                    break;
                case 5:
                    sortValue = "patientFacilityName";
                    break;
                case 6:
                    sortValue = "patientFacilityNumber";
                    break;
                case 7:
                    sortValue = "organizationName";
                    break;
                case 8:
                    sortValue = "userId";
                    break;
                case 9:
                    sortValue = "purposeForUse";
                    break;
                default:
                    sortValue = "patientSSN";
                    break;
            }
        } else {
            if(null != colNumber) //source = direct
            switch (Integer.parseInt(colNumber)) {
                case 0:
                    sortValue = "patientSSN";
                    break;
                case 1:
                    sortValue = "patientLastName";
                    break;
                case 2:
                    sortValue = "patientGivenName";
                    break;
                case 3:
                    sortValue = "auditTime";
                    break;
                case 4:
                    sortValue = "documentTitle";
                    break;
                case 5:
                    sortValue = "organizationName";
                    break;
                case 6:
                    sortValue = "userId";
                    break;
                case 7:
                    sortValue = "purposeForUse";
                    break;
                default:
                    sortValue = "patientSSN";
                    break;
            }
        }
        return sortValue;
    }

    public void view(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        final HttpSession session = request.getSession(false);
        final String icn = request.getParameter("documentICN");
        final String documentUniqueId = request
            .getParameter("documentUniqueId");
        final String styleSheetViewType = request
            .getParameter("styleSheetViewType");

        String document = null;
        try {
            document = this.getDocumentHelper().getDocument(icn,
                documentUniqueId, UserHelper.getUserName(request),
                this.getCmsHelper().getCompleteHomeCommunityId());
        } catch (final AdapterException ex) {
            throw new ServletException(ex);
        }
        session.setAttribute("document", document);
        session.setAttribute("icn", icn);
        session.setAttribute("documentUniqueId", documentUniqueId);
        session.setAttribute("isXmlViewEnabled", this.getDocumentHelper()
            .isXmlViewEnabled());

        if ("xml".equals(styleSheetViewType)) {
            document = document
                .replaceAll(
                    "<?xml-stylesheet type=\"text/xsl\" href=\"CCD.xsl\"?>",
                    "");
            response.setContentType("text/xml;charset=UTF-8");
            response.getWriter().write(document);
        } else {
            // Update session in the document
            final String html = this.getDocumentHelper().getHtml(document,
                styleSheetViewType);
            session.setAttribute("updatedDocument", html);
            this.forward(request, response,
                "showDocumentDisclosureReportDocument");
        }
    }

    public DocumentHelper getDocumentHelper() {
        return this.getBean("adapterDocumentHelper", DocumentHelper.class);
    }

    @EJB(beanInterface = AdapterAuditManager.class, mappedName = "AdapterAuditManager")
    public void setAdapterAuditManager(
        final AdapterAuditManager adapterAuditManager) {
        this.adapterAuditManager = adapterAuditManager;
    }

    public void exportToCsv(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException, IOException {

        Map<String, String> csvReportMap = new LinkedHashMap<>();
        csvReportMap.put("SSN", "ssn");
        csvReportMap.put("Patient Last Name", "lastName");
        csvReportMap.put("Patient First Name", "firstName");
        csvReportMap.put("Date of Disclosures (CT)", "auditTimeFormatted");
        csvReportMap.put("Disclosures", "documentTitle");
        csvReportMap.put("Patient Preferred Facility", "patientFacilityName");
        csvReportMap.put("Patient Preferred Facility Station ID", "patientFacilityNumber");
        csvReportMap.put("eHealth Exchange Organization", "organizationName");
        csvReportMap.put("OID", "facilityOid");
        csvReportMap.put("User ID", "userId");
        csvReportMap.put("Purpose of Use", "purposeForUse");
        csvReportMap.put("User Role", "roleName");

        int minimumThreshold = Integer.parseInt((String) this.getServletContext().getAttribute("scheduledExportMin"));
        int total = Integer.parseInt(request.getParameter("totalRows"));

        //Start a scheduled export with its own thread if greater than the minimum threshold
        if (total >= minimumThreshold) {
            ExcelGeneratorThread csvGenThread = new ExcelGeneratorThread("csvGenThread");
            csvGenThread.setDocumentType("csv");
            csvGenThread.setReportMap(csvReportMap);

            csvGenThread.setTitle("Disclosures Detail Report");
            csvGenThread.setUserId(UserHelper.getUserName(request));
            csvGenThread.setAdapterAuditManager(adapterAuditManager);
            csvGenThread.setReportHelper(getReportHelper());
            csvGenThread.setUserDocumentDao(getUserDocumentDAO());
            csvGenThread.setCsvExport(new CsvExporter());

            // Set search attributes for getting audit results
            csvGenThread.setAttributes(mapSessionAttributes(request));

            csvGenThread.start();
        }
        //Otherwise generate and download the export directly
        else {
            List<Map<String, Object>> results;
            SearchPackage packedResults;

            packedResults = this.getResults(request, true);
            results = packedResults.getSearchResults();

            CsvExporter csvExporter = new CsvExporter();
            csvExporter.exportToCSV(response, "Disclosures_Detail_Report", results, (LinkedHashMap<String, String>) csvReportMap);
        }
    }

    /**
     * Export to excel - This method is called when the user clicks on the excel icon in the JSP. Reflection is used in the
     * ResponseDispatcherHttpServlet to call this method based on the http parameters.
     *
     * @param request HTTP Request
     * @param response HTTP Response
     * @throws ServletException
     * @throws IOException
     */
    public void exportToExcel(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {

		// Create a map of key which is based on the result data key and the
        // heading value
        // The heading is used to create the column headers and the key is used
        // to pull the data from the results
        final Map<String, String> disclosureMap = new LinkedHashMap<String, String>();

        final Date startDateStr = ReportHelper.getStartDate(request.getParameter("startDate"));
        final Date endDateStr = ReportHelper.getEndDate(request.getParameter("endDate"));

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        final LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();

        final String source = request.getParameter("source");
        filters.put("Disclosure Source", source);
        filters.put("SSN", request.getParameter("ssn"));
        filters.put("Last Name", request.getParameter("lastName"));
        filters.put("First Name", request.getParameter("firstName"));
        //filters.put("User ID", request.getParameter("userId"));
        filters.put("Start Date", this.getReportHelper().getFormattedDate(startDateStr));
        filters.put("End Date", this.getReportHelper().getFormattedDate(endDateStr));
        filters.put("Purpose of Use", ExcelExporter.getFilterValue(request.getParameter("purposeOfUse")));
        if ("exchange".equals(source.toLowerCase())) {
            filters.put("Patient Preferred Facility", ExcelExporter.getPatientPreferredFacilityFilter(getFacilityDAO(), request.getParameter("facility")));
            filters.put("eHealth Exchange Organization", ExcelExporter.getFilterValue(request.getParameter("organizationName")));
            ExcelExporter.populateFilterMapForExport(request, filters, filterMap, request.getParameter("patientTypes"));
        }
        else {
            ExcelExporter.populateFilterMapForExport(request, filters, filterMap, "");
        }

        disclosureMap.put("ssn", "SSN");
        disclosureMap.put("lastName", "Patient Last Name");
        disclosureMap.put("firstName", "Patient First Name");
        disclosureMap.put("auditTimeFormatted", "Date of Disclosures (CT)");
        disclosureMap.put("documentTitle", "Disclosures");
        disclosureMap.put("patientFacilityName", "Patient Preferred Facility");
        disclosureMap.put("patientFacilityNumber", "Patient Preferred Facility Station ID");
        disclosureMap.put("organizationName", "eHealth Exchange Organization");
        disclosureMap.put("facilityOid", "OID");
        disclosureMap.put("userId", "User ID");
        disclosureMap.put("purposeForUse", "Purpose of Use");
        disclosureMap.put("roleName", "User Role");

        int minimumThreshold = Integer.parseInt((String) this.getServletContext().getAttribute("scheduledExportMin"));
        int total = Integer.parseInt(request.getParameter("totalRows"));

        //Start a scheduled export with its own thread if greater than the minimum threshold
        if (total >= minimumThreshold) {
            ExcelGeneratorThread exGenThread = new ExcelGeneratorThread("exGenThread");

            exGenThread.setReportSource(request.getParameter("source"));
            exGenThread.setFilterMap(filterMap);
            exGenThread.setReportMap(disclosureMap);

            // Set names for Excel generator thread
            exGenThread.setTitle("Disclosures Detail Report");

            exGenThread.setExcelExporter(this.getExcelExporter());
            exGenThread.setAdapterAuditManager(adapterAuditManager);
            exGenThread.setDirectAuditManager(directAuditManager);
            exGenThread.setReportHelper(getReportHelper());
            exGenThread.setUserId(UserHelper.getUserName(request));
            exGenThread.setUserDocumentDao(getUserDocumentDAO());
            exGenThread.setCmsHelper(getCmsHelper());

            // Set search attributes for getting audit results
            exGenThread.setAttributes(mapSessionAttributes(request));

            exGenThread.start();
        }
        //Otherwise generate and download the export directly
        else {
            // Create workbook
            final String title = "Disclosures Detail Report";

            List<Map<String, Object>> results;
            SearchPackage packedResults;
            packedResults = this.getResults(request, true);
            results = packedResults.getSearchResults();

            final Workbook wb = this.getExcelExporter().exportToExcel(title, title, disclosureMap, results, filterMap, null);

            // Write Excel workbook to Stream
            this.getExcelExporter().writeExcelToStream("Disclosures_Detail_Report", wb, response);
        }

    }

    private Map<String, Object> mapSessionAttributes(HttpServletRequest request) {

        final HttpSession session = request.getSession(false);
        Map<String, Object> attributes = new HashMap<String, Object>();

        attributes.put("ssn", request.getParameter("ssn"));
        attributes.put("lastName", request.getParameter("lastName"));
        attributes.put("firstName", request.getParameter("firstName"));
        //attributes.put("userId", request.getParameter("userId"));

        Date startDate = ReportHelper.getStartDate(request.getParameter("startDate"));
        Date endDate = ReportHelper.getEndDate(request.getParameter("endDate"));
        attributes.put("startDate", startDate);
        attributes.put("endDate", endDate);

        attributes.put("organization", (String) session
                .getAttribute("documentDisclosureQueryOrganization"));
        attributes.put("facility", request.getParameter("facility"));
        attributes.put("purposeOfUse", request.getParameter("purposeOfUse"));
        attributes.put("patientTypes", Integer.parseInt(request.getParameter("patientTypes")));
        // sort
        attributes.put("sortValue", getSortValue(request.getParameter("sortValue"), request.getParameter("source")));
        attributes.put("sortDirection", request.getParameter("sortOrder"));

        // Set Action Types
        final ActionValuesType actionsType = new ActionValuesType();
        actionsType.getValue().add(ActionType.RETRIEVE_DOCUMENT);
        actionsType.setNotIn(false);
        attributes.put("actionsType", actionsType);

        return attributes;
    }

    /**
     * Get the excel exporter class from Spring.
     *
     * @return the excel exporter object
     */
    public ExcelExporter getExcelExporter() {
        final ExcelExporter excelExporter = this.getBean("excelExporter",
            ExcelExporter.class);
        return excelExporter;
    }

    private UserDocumentDAO getUserDocumentDAO() {
        return this.getBean("UserDocumentDAO", UserDocumentDAO.class);
    }

    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }

    /**
     * used to organize parameters into a one. cleaner packaging.
     */
    private class SearchPackage{
        List<Map<String,Object>> searchResults;
        String icn;
        String source;
        Long totalCount;

        public SearchPackage(List<Map<String, Object>> searchResults, String icn, String source) {
            this.searchResults = searchResults;
            this.icn = icn;
            this.source = source;
        }

        public List<Map<String, Object>> getSearchResults() {
            return searchResults;
        }

        public void setSearchResults(List<Map<String, Object>> searchResults) {
            this.searchResults = searchResults;
        }

        public String getIcn() {
            return icn;
        }

        public void setIcn(String icn) {
            this.icn = icn;
        }

        public String getSource() {
            return source;
        }

        public void setSource(String source) {
            this.source = source;
        }

        public Long getTotalCount() {
            return totalCount;
        }

        public void setTotalCount(Long totalCount) {
            this.totalCount = totalCount;
        }
    }

}
