package gov.va.nvap.web.report;

import gov.va.med.nhin.adapter.audit.ActionType;
import gov.va.med.nhin.adapter.audit.ActionValuesType;
import gov.va.med.nhin.adapter.audit.AuditSummariesType;
import gov.va.med.nhin.adapter.audit.AuditSummaryType;
import gov.va.med.nhin.adapter.audit.FieldType;
import gov.va.med.nhin.adapter.audit.GetAuditsSummary;
import gov.va.med.nhin.adapter.audit.GetAuditsSummaryResponse;
import gov.va.med.nhin.adapter.audit.GroupByFieldsType;
import gov.va.med.nhin.adapter.audit.PageInfoType;
import gov.va.med.nhin.adapter.audit.StringValuesType;
import gov.va.med.nhin.adapter.audit.SummaryFieldsType;
import gov.va.nvap.common.date.GregorianDateUtil;
import gov.va.nvap.common.sort.BubbleSortListMap;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.OrganizationType;
import gov.va.nvap.service.adapter.audit.AdapterAuditManager;
import gov.va.nvap.service.adapter.audit.DirectAuditManager;
import gov.va.nvap.service.adapter.audit.DirectServiceEndpoint;
import gov.va.nvap.service.audit.AuditException;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.helper.document.DocumentHelper;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.util.xls.ExcelExporter;

import java.io.IOException;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.ejb.EJB;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import java.util.LinkedHashMap;
import org.apache.poi.ss.usermodel.Workbook;

/**
 * Disclosures Summary Report results servlet
 */
public class DisclosureSummaryReportResults extends
    ResponseDispatcherHttpServlet {

    /**
     * Serial UID.
     */
    private static final long serialVersionUID = -7902989393412212166L;
    private AdapterAuditManager adapterAuditManager;
    private final DirectAuditManager directAuditManager = new DirectServiceEndpoint();

    /**
     * Export to excel - This method is called when the user clicks on the excel icon in the JSP. Reflection is used in the
     * ResponseDispatcherHttpServlet to call this method based on the http parameters.
     *
     * @param request HTTP Request
     * @param response HTTP Response
     * @throws ServletException
     * @throws IOException
     */
    public void exportToExcel(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        // Get the session
        final HttpSession session = request.getSession(false);
        // Get the results from the adapter
        final List<Map<String, Object>> results = this.getResults(
            request.getSession());
                                //request.getSession(), new Paginator(-1));
        // Create a map of key which is based on the result data key and the
        // heading value
        // The heading is used to create the column headers and the key is used
        // to pull the data from the results
        final Map<String, String> disclosureSummaryMap = new LinkedHashMap<String, String>();
        disclosureSummaryMap.put("partnerOrg", "eHealth Exchange Organization");
        if ("Exchange".equals((String) session.getAttribute("disclosureSummaryQuerySource"))) {
            disclosureSummaryMap.put("vaFacility", "Patient Preferred Facility");
            disclosureSummaryMap.put("vaFacilityNumber", "Patient Preferred Facility Station ID");
            disclosureSummaryMap.put("oid", "OID");
        }
        disclosureSummaryMap.put("total", "Total");

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();
        
        String orgName = (String) session.getAttribute("disclosureSummaryQueryOrgName");
        if (NullChecker.isNullOrEmpty(orgName)) {
            orgName = "ALL";
        }
        
        filters.put("Disclosure Source", session.getAttribute("disclosureSummaryQuerySource"));
        filters.put("Start Date", this.getReportHelper().getFormattedDate((Date) session.getAttribute("disclosureSummaryQueryStartDate")));
        filters.put("End Date", this.getReportHelper().getFormattedDate((Date) session.getAttribute("disclosureSummaryQueryEndDate")));
        //filters.put("User ID", session.getAttribute("disclosureSummaryQueryUserId"));
        filters.put("Patient Preferred Facility", ExcelExporter.getPatientPreferredFacilityFilter(getFacilityDAO(), (String) session.getAttribute("disclosureSummaryQueryFacility")));
        filters.put("eHealth Exchange Organization", orgName);
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, session.getAttribute("patientTypes").toString());

		// Create the optional data rows
        final Object disclosureDocumentCount = session.getAttribute("disclosureSummaryTotalCount");
        final Map<String, List<Object>> optionalRows = new LinkedHashMap<String, List<Object>>();
        final List<Object> disclosureDocumentRow = new ArrayList<Object>();
        disclosureDocumentRow.add(disclosureDocumentCount);
        optionalRows.put("Grand Total", disclosureDocumentRow);
        
        // Create the workbook
        final String title = "Disclosures Summary Report";
        final Workbook wb = this.getExcelExporter().exportToExcel(title, title, disclosureSummaryMap, results, filterMap, optionalRows);
        
        // Write Excel to Stream
        this.getExcelExporter().writeExcelToStream("Disclosures_Summary_Report", wb, response);
    }

    /**
     * Get the consent management helper from Spring.
     * @return ConsentManagementHelper
     */
    public ConsentManagementHelper getCmsHelper() {
        final ConsentManagementHelper cmsHelper = this.getBean("cmsHelper",
            ConsentManagementHelper.class);
        return cmsHelper;
    }

    /**
     * Get the document helper from Spring.
     * @return DocumentHelper
     */
    public DocumentHelper getDocumentHelper() {
        return this.getBean("adapterDocumentHelper", DocumentHelper.class);
    }

    /**
     * Get the excel exporter class from Spring.
     *
     * @return the excel exporter object
     */
    public ExcelExporter getExcelExporter() {
        final ExcelExporter excelExporter = this.getBean("excelExporter",
            ExcelExporter.class);
        return excelExporter;
    }

    /**
     * Get the facility helper from Spring.
     * @return FacilityHelper
     */
    public FacilityHelper getFacilityHelper() {
        return this.getBean("facilityHelper", FacilityHelper.class);
    }

    public ReportHelper getReportHelper() {
        final ReportHelper reportHelper = this.getBean("reportHelper",
            ReportHelper.class);
        return reportHelper;
    }

    private List<Map<String, Object>> getResults(final HttpSession session)
        throws ServletException {
        
        final String source = (String) session.getAttribute("disclosureSummaryQuerySource");
        final Date sd = (Date) session.getAttribute("disclosureSummaryQueryStartDate");
        final Date ed = (Date) session.getAttribute("disclosureSummaryQueryEndDate");
        final String userId = (String) session.getAttribute("disclosureSummaryQueryUserId");
        final String patientFacility = (String) session.getAttribute("disclosureSummaryQueryFacility");
        final String externalPartnerOrg = (String) session.getAttribute("disclosureSummaryQueryOrg");
        final int patientTypes = (Integer) session.getAttribute("patientTypes");
        
        // sort
        final String disclosureSummarySortValue = (String) session.getAttribute("disclosureSummarySortValue");
        final String disclosureSummarySortDirection = (String) session.getAttribute("disclosureSummarySortDirection");
        
        final GetAuditsSummary getAuditsSummary = new GetAuditsSummary();

        // start time
        if (NullChecker.isNotEmpty(sd)) {
            getAuditsSummary.setFromDate(GregorianDateUtil
                .getGregorianCalendarByDate(sd));
        }
        // to time
        if (NullChecker.isNotEmpty(ed)) {
            getAuditsSummary.setToDate(GregorianDateUtil
                .getGregorianCalendarByDate(ed));
        }
        //Set User ID
        if (NullChecker.isNotEmpty(userId)) {
            final StringValuesType userIdValue = new StringValuesType();
            userIdValue.getValue().add(userId);
            userIdValue.setNotIn(false);
            getAuditsSummary.setUserIds(userIdValue);
        }
        getAuditsSummary.setPatientTypes(patientTypes);
        
		// // sorting
        // if (NullChecker.isNotEmpty(disclosureSummarySortValue)) {
        // SortFieldType sortField = new SortFieldType();
        // sortField.setField(FieldType.fromValue(disclosureSummarySortValue));
        // if ("DESC".equalsIgnoreCase(disclosureSummarySortDirection)) {
        // sortField.setDirection(SortDirection.DESC);
        // } else {
        // sortField.setDirection(SortDirection.ASC);
        // }
        // SortFieldsType sortFieldsType = new SortFieldsType();
        // sortFieldsType.getSortField().add(sortField);
        // getAuditsRequest.setSortFields(sortFieldsType);
        // }
        // action type
        final ActionValuesType actionValuesType = new ActionValuesType();
        final List<ActionType> actiontype = actionValuesType.getValue();
        actiontype.add(ActionType.RETRIEVE_DOCUMENT);
        actionValuesType.setNotIn(false);
        getAuditsSummary.setActions(actionValuesType);

        // patient preferred facility
        if (!NullChecker.isNullOrEmpty(patientFacility)) {

            final StringValuesType patientFacilyNumber = new StringValuesType();
            patientFacilyNumber.setNotIn(false);
            final List<String> patientFacilyNumberList = patientFacilyNumber
                .getValue();
            patientFacilyNumberList.add(patientFacility);
            getAuditsSummary.setPatientFacilityNumbers(patientFacilyNumber);
        }

        if (!NullChecker.isNullOrEmpty(externalPartnerOrg)
            && !externalPartnerOrg.equals("external")) {
            final StringValuesType organizationIds = new StringValuesType();// getAuditsSummary.getRemoteOrganizationIds();
            organizationIds.setNotIn(false);
            final List<String> organizationIdsList = organizationIds.getValue();
            organizationIdsList
                .add(externalPartnerOrg.replace("%", "urn:oid:"));
            getAuditsSummary.setOrganizationIds(organizationIds);
        } else {
            // input orgs are empty - intent is to retrieve all orgs except VA
            final StringValuesType organizationIds = new StringValuesType();
            organizationIds.setNotIn(true);
            organizationIds.getValue().add(
                this.getCmsHelper().getCompleteHomeCommunityId());
            getAuditsSummary.setOrganizationIds(organizationIds);
        }

        // group by
        final GroupByFieldsType groupByFieldsType = new GroupByFieldsType();
        final List<FieldType> fieldTypeList = groupByFieldsType
            .getGroupByField();
        fieldTypeList.add(FieldType.ORGANIZATION_ID);
        fieldTypeList.add(FieldType.PATIENT_FACILITY_NUMBER);
        fieldTypeList.add(FieldType.PATIENT_FACILITY_NAME);
        fieldTypeList.add(FieldType.ORGANIZATION_NAME);
        getAuditsSummary.setGroupByFields(groupByFieldsType);

        // Set Page information
        final PageInfoType pageInfoType = new PageInfoType();
        pageInfoType.setPageSize(-1);
        getAuditsSummary.setPageInfo(pageInfoType);

        // web services response
        GetAuditsSummaryResponse getAuditsSummaryResponse;
        try {
            if (source.equals("Direct")) {
                getAuditsSummaryResponse = this.directAuditManager.getAuditSummary(getAuditsSummary);
            } else {
                getAuditsSummaryResponse = this.adapterAuditManager.getAuditSummary(getAuditsSummary);
            }
        } catch (final AuditException e) {
            throw new AuditException(e);
        }

        final List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        long totalCount = 0l;
        final AuditSummariesType auditSummariesType = getAuditsSummaryResponse
            .getAuditSummaries();
        List<AuditSummaryType> auditSummary = new ArrayList<AuditSummaryType>();
        if (auditSummariesType != null) {
            auditSummary = auditSummariesType.getAuditSummary();
        }
        for (final AuditSummaryType auditSummaryType2 : auditSummary) {
            final AuditSummaryType auditSummaryType = auditSummaryType2;

            final Map<String, Object> resultMap = new HashMap<String, Object>();
            final long count = auditSummaryType.getCount();
            final SummaryFieldsType summaryFields = auditSummaryType
                .getSummaryFields();
            final List<String> summaryField = summaryFields.getSummaryField();

            OrganizationType organizationType = null;
            if (NullChecker.isEmpty(summaryField)) {
                continue;
            }

            if (source.equals("Direct")) {
                if (summaryField.size() < 1) {
                    continue;
                }
                resultMap.put("partnerOrg", summaryField.get(0));
            } else {
                if (summaryField.size() < 4) {
                    continue;
                }

                if (NullChecker.isNotEmpty(summaryField.get(0))) {
                    organizationType = (this.getCmsHelper()
                        .getOrganizationByHomeCommunityId(summaryField.get(0)
                            .replace("urn:oid:", "")));
                    if (!NullChecker.isNullOrEmpty(organizationType)) {
                        resultMap.put("oid", ReportDataProcessor.nullEmptyReplaceWithUnknown(organizationType.getOrgOid()));
                    }
                    else {
                        resultMap.put("oid", "Unknown");
                    }
                }

                if (organizationType != null) {
                    resultMap.put("partnerOrg", ReportDataProcessor
                        .nullEmptyReplaceWithUnknown(organizationType
                            .getOrgName()));
                } else {
                    resultMap.put("partnerOrg", ReportDataProcessor
                        .nullEmptyReplaceWithUnknown(summaryField.get(3)));
                }

                // Extract value
                final String adapterVAFacility = summaryField.get(1);
                String vaFacility;
                if (ReportDataProcessor.isValidStation(adapterVAFacility)) {
                    vaFacility = this.getFacilityHelper()
                        .getFacilityNameByStationId(adapterVAFacility);
                    if (NullChecker.isEmpty(vaFacility)) {
                        vaFacility = summaryField.get(2);
                    }
                } else {
                    vaFacility = summaryField.get(2);
                }

                resultMap.put("vaFacility",
                    ReportDataProcessor.fixStation(vaFacility));

                final String adapterVAFacilityNumber = summaryField.get(1);
                if (!NullChecker.isNullOrEmpty(adapterVAFacilityNumber)) {
                    resultMap.put("vaFacilityNumber",
                        ReportDataProcessor.fixStation(summaryField.get(1)));
                } else {
                    resultMap.put("vaFacilityNumber",
                        ReportDataProcessor.fixStation("Unavailable"));
                }
            }
            resultMap.put("total", count);
            results.add(resultMap);
            totalCount += count;

        }
        session.setAttribute("disclosureSummaryTotalCount", totalCount);
        // sort
        boolean ascending = true;
        if (!NullChecker.isNullOrEmpty(disclosureSummarySortDirection)
            && !NullChecker.isNullOrEmpty(disclosureSummarySortValue)) { // first
            // time
            // through,
            // no
            // sort
            // specified
            if (disclosureSummarySortDirection.equalsIgnoreCase("DESC")) {
                ascending = false;
            }
            final BubbleSortListMap bSorter = new BubbleSortListMap();
            return bSorter.sortByColumn(results, disclosureSummarySortValue,
                ascending);
        }
        return results;
    }

    @EJB(beanInterface = AdapterAuditManager.class, mappedName = "AdapterAuditManager")
    public void setAdapterAuditManager(
        final AdapterAuditManager adapterAuditManager) {
        this.adapterAuditManager = adapterAuditManager;
    }

    @Override
    public void unspecified(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        
        final HttpSession session = request.getSession(false);
        final String sortValue = request.getParameter("disclosureSummarySortValue");
        final String sortDirection = request.getParameter("disclosureSummarySortDirection");
        
        session.setAttribute("disclosureSummarySortValue", sortValue);
        session.setAttribute("disclosureSummarySortDirection", sortDirection);
        session.setAttribute("disclosureSummaryResults", this.getResults(session));
        
        this.forward(request, response, "show");
    }

    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }
    
}
