package gov.va.nvap.web.report;

import gov.va.med.nhin.adapter.audit.ActionType;
import gov.va.med.nhin.adapter.audit.ActionValuesType;
import gov.va.med.nhin.adapter.audit.AuditSummariesType;
import gov.va.med.nhin.adapter.audit.AuditSummaryType;
import gov.va.med.nhin.adapter.audit.FieldType;
import gov.va.med.nhin.adapter.audit.GetAuditsSummary;
import gov.va.med.nhin.adapter.audit.GetAuditsSummaryResponse;
import gov.va.med.nhin.adapter.audit.GroupByFieldsType;
import gov.va.med.nhin.adapter.audit.PageInfoType;
import gov.va.med.nhin.adapter.audit.StringValuesType;
import gov.va.med.nhin.adapter.audit.SummaryFieldsType;
import gov.va.nvap.common.date.GregorianDateUtil;
import gov.va.nvap.common.sort.BubbleSortListMap;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.OrganizationType;
import gov.va.nvap.service.adapter.audit.AdapterAuditManager;
import gov.va.nvap.service.adapter.audit.DirectAuditManager;
import gov.va.nvap.service.adapter.audit.DirectServiceEndpoint;
import gov.va.nvap.service.adapter.audit.DirectServiceException;
import gov.va.nvap.service.audit.AuditException;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.util.xls.ExcelExporter;

import java.io.IOException;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Date;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.ejb.EJB;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.poi.ss.usermodel.Workbook;
import org.json.JSONException;
import org.json.JSONObject;

/**
 * Disclosures Summary Report search servlet
 */
public class DisclosureSummaryReport extends ResponseDispatcherHttpServlet {
    /**
     * Serial UID.
     */
    private static final long serialVersionUID = 3721503355079491829L;
    
    private AdapterAuditManager adapterAuditManager;
    private final DirectAuditManager directAuditManager = new DirectServiceEndpoint();

    @Override
    protected void unspecified(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {

        final HttpSession session = request.getSession(false);
        session.removeAttribute("results");
        ReportHelper.setDefaultSearchDates(session);
        request.setAttribute("facilities", getFacilityHelper().getAllVistAFacilities());
        getReportHelper().getOrgLists(request, this.getCmsHelper());

        this.forward(request, response, "show");
    }

    
    private ConsentManagementHelper getCmsHelper() {
        return this.getBean("cmsHelper", ConsentManagementHelper.class);
    }

    private FacilityHelper getFacilityHelper() {
        return this.getBean("facilityHelper", FacilityHelper.class);
    }

    private ReportHelper getReportHelper() {
        return this.getBean("reportHelper", ReportHelper.class);
    }
    
    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }

    private ExcelExporter getExcelExporter() {
        return this.getBean("excelExporter", ExcelExporter.class);
    }

    @EJB(beanInterface = AdapterAuditManager.class, mappedName = "AdapterAuditManager")
    public void setAdapterAuditManager(
            final AdapterAuditManager adapterAuditManager) {
        this.adapterAuditManager = adapterAuditManager;
    }

    /**
     *
     * @param request
     * @param response
     * @throws ServletException
     * @throws IOException
     */
    public void doSearch(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
        final HttpSession session = request.getSession(false);

        session.setAttribute("results", encodeIntoJSON(getResults(request)));

        this.forward(request, response, "searchResultsJSON");
    }

    /**
     * Returns search results map and total count packaged together in SearchResultsPackage class. if request parameters are not passed,
     * default values are used for search results.
     *
     * @param request
     * @return
     * @throws ServletException
     */
    private SearchResultsPackage getResults(final HttpServletRequest request)
            throws ServletException {

        final String source = (request.getParameter("source") != null)
                ? request.getParameter("source") : "Exchange";
        final Date sd = ReportHelper.getStartDate(request.getParameter("startDate"));
        final Date ed = ReportHelper.getEndDate(request.getParameter("endDate"));
        final String userId = "";
        final String organization = (request.getParameter("organization") != null)
                ? request.getParameter("organization") : "external";
        final String patientFacility = (request.getParameter("facility") != null)
                ? request.getParameter("facility") : "";
        final int patientTypes = (request.getParameter("patientTypes") != null)
                ? Integer.parseInt(request.getParameter("patientTypes")) : 1;

        final GetAuditsSummary getAuditsSummary = new GetAuditsSummary();

        // start time
        if (NullChecker.isNotEmpty(sd)) {
            getAuditsSummary.setFromDate(GregorianDateUtil
                    .getGregorianCalendarByDate(sd));
        }
        // to time
        if (NullChecker.isNotEmpty(ed)) {
            getAuditsSummary.setToDate(GregorianDateUtil
                    .getGregorianCalendarByDate(ed));
        }
        //Set User ID
        if (NullChecker.isNotEmpty(userId)) {
            final StringValuesType userIdValue = new StringValuesType();
            userIdValue.getValue().add(userId);
            userIdValue.setNotIn(false);
            getAuditsSummary.setUserIds(userIdValue);
        }
        getAuditsSummary.setPatientTypes(patientTypes);

        // action type
        final ActionValuesType actionValuesType = new ActionValuesType();
        final List<ActionType> actiontype = actionValuesType.getValue();
        actiontype.add(ActionType.RETRIEVE_DOCUMENT);
        actionValuesType.setNotIn(false);
        getAuditsSummary.setActions(actionValuesType);

        // patient preferred facility
        if (!NullChecker.isNullOrEmpty(patientFacility)) {

            final StringValuesType patientFacilyNumber = new StringValuesType();
            patientFacilyNumber.setNotIn(false);
            final List<String> patientFacilyNumberList = patientFacilyNumber
                    .getValue();
            patientFacilyNumberList.add(patientFacility);
            getAuditsSummary.setPatientFacilityNumbers(patientFacilyNumber);
        }

        if (!NullChecker.isNullOrEmpty(organization)
                && !organization.equals("external")) {
            final StringValuesType organizationIds = new StringValuesType();// getAuditsSummary.getRemoteOrganizationIds();
            organizationIds.setNotIn(false);
            final List<String> organizationIdsList = organizationIds.getValue();
            organizationIdsList
                    .add(organization.replace("%", "urn:oid:"));
            getAuditsSummary.setOrganizationIds(organizationIds);
        } else {
            // input orgs are empty - intent is to retrieve all orgs except VA
            final StringValuesType organizationIds = new StringValuesType();
            organizationIds.setNotIn(true);
            organizationIds.getValue().add(
                    this.getCmsHelper().getCompleteHomeCommunityId());
            getAuditsSummary.setOrganizationIds(organizationIds);
        }

        // group by
        final GroupByFieldsType groupByFieldsType = new GroupByFieldsType();
        final List<FieldType> fieldTypeList = groupByFieldsType
                .getGroupByField();
        fieldTypeList.add(FieldType.ORGANIZATION_ID);
        fieldTypeList.add(FieldType.PATIENT_FACILITY_NUMBER);
        fieldTypeList.add(FieldType.PATIENT_FACILITY_NAME);
        fieldTypeList.add(FieldType.ORGANIZATION_NAME);
        getAuditsSummary.setGroupByFields(groupByFieldsType);

        // Set Page information
        final PageInfoType pageInfoType = new PageInfoType();
        pageInfoType.setPageSize(-1);
        getAuditsSummary.setPageInfo(pageInfoType);

        // web services response
        GetAuditsSummaryResponse getAuditsSummaryResponse;
        try {
            if (source.equals("Direct")) {
                getAuditsSummaryResponse = this.directAuditManager.getAuditSummary(getAuditsSummary);
            } else {
                getAuditsSummaryResponse = this.adapterAuditManager.getAuditSummary(getAuditsSummary);
            }
        } catch (final AuditException e) {
            throw new AuditException();
        } catch (final DirectServiceException ex) {
            throw new DirectServiceException();
        }

        final List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        long totalCount = 0l;
        final AuditSummariesType auditSummariesType = getAuditsSummaryResponse
                .getAuditSummaries();
        List<AuditSummaryType> auditSummary = new ArrayList<AuditSummaryType>();
        if (auditSummariesType != null) {
            auditSummary = auditSummariesType.getAuditSummary();
        }
        for (final AuditSummaryType auditSummaryType2 : auditSummary) {
            final AuditSummaryType auditSummaryType = auditSummaryType2;

            final Map<String, Object> resultMap = new LinkedHashMap<String, Object>();
            final long count = auditSummaryType.getCount();
            final SummaryFieldsType summaryFields = auditSummaryType
                    .getSummaryFields();
            final List<String> summaryField = summaryFields.getSummaryField();

            OrganizationType organizationType = null;
            if (NullChecker.isEmpty(summaryField)) {
                continue;
            }

            if (source.equals("Direct")) {
                if (summaryField.size() < 1) {
                    continue;
                }
                resultMap.put("partnerOrg", summaryField.get(0));
            } else {
                if (summaryField.size() < 4) {
                    continue;
                }

                if (NullChecker.isNotEmpty(summaryField.get(0))) {
                    organizationType = (this.getCmsHelper()
                            .getOrganizationByHomeCommunityId(summaryField.get(0)
                                    .replace("urn:oid:", "")));
                }

                if (!NullChecker.isNullOrEmpty(organizationType)) {

                    resultMap.put("partnerOrg", ReportDataProcessor
                            .nullEmptyReplaceWithUnknown(organizationType
                                    .getOrgName()));
                } else {
                    resultMap.put("partnerOrg", ReportDataProcessor
                            .nullEmptyReplaceWithUnknown(summaryField.get(3)));
                }

                // Extract value
                final String adapterVAFacility = summaryField.get(1);
                String vaFacility;
                if (ReportDataProcessor.isValidStation(adapterVAFacility)) {
                    vaFacility = this.getFacilityHelper()
                            .getFacilityNameByStationId(adapterVAFacility);
                    if (NullChecker.isEmpty(vaFacility)) {
                        vaFacility = summaryField.get(2);
                    }
                } else {
                    vaFacility = summaryField.get(2);
                }

                resultMap.put("vaFacility",
                        ReportDataProcessor.fixStation(vaFacility));
                
                final String adapterVAFacilityNumber = summaryField.get(1);
                if (!NullChecker.isNullOrEmpty(adapterVAFacilityNumber)) {
                    resultMap.put("vaFacilityNumber",
                        ReportDataProcessor.fixStation(summaryField.get(1)));
                } else {
                    resultMap.put("vaFacilityNumber",
                        ReportDataProcessor.fixStation("Unavailable"));
                }                
            }
            resultMap.put("total", count);
            results.add(resultMap);
            totalCount += count;

        }
        //sort and return packaged data
        return new SearchResultsPackage( sortResults(results, request), totalCount, source);
    }
    
    
    /**
     * sorts values of List<Map<String,Object>> keys based on request parameters
     * @param unsortedList
     * @param request
     * @return 
     */
    private List<Map<String,Object>> sortResults(List<Map<String,Object>> unsortedList, HttpServletRequest request){
        
        if(unsortedList.isEmpty()){
            return unsortedList;
        }
        
        
        String selectedSource = (request.getParameter("source") != null) 
                ? request.getParameter("source") : "Exchange";
        String sortValue = request.getParameter("order[0][column]") == null ? request.getParameter("sortBy") : request.getParameter("order[0][column]");
        boolean sortAscending = request.getParameter("order[0][dir]") == null ? !"desc".equalsIgnoreCase(request.getParameter("sortOrder")) :
            !"desc".equalsIgnoreCase(request.getParameter("order[0][dir]"));
        final BubbleSortListMap bSorter = new BubbleSortListMap();
        
        if("Exchange".equals(selectedSource)){
            if("0".equals(sortValue)){
                sortValue = "partnerOrg";
            } else if ("1".equals(sortValue)){
                sortValue = "vaFacility";
            } else if ("2".equals(sortValue)){
                sortValue = "vaFacilityNumber";
            } else if ("3".equals(sortValue)){
                sortValue = "total";
            } else {
                sortValue = "partnerOrg";
            }
        } else {
            if("0".equals(sortValue)){
                sortValue = "partnerOrg";
            } else if ("1".equals(sortValue)){
                sortValue = "total";
            } else {
                sortValue = "partnerOrg";
            }
        }
        
        return bSorter.sortByColumn(unsortedList, sortValue, sortAscending);
    }
    

    /**
     * Searches its own copy of search results, and converts into excel download file.
     *
     * @param request
     * @param response
     * @throws ServletException
     * @throws IOException
     */
    public void exportToExcel(final HttpServletRequest request,
            final HttpServletResponse response) throws ServletException,
            IOException {

        SearchResultsPackage results = getResults(request);

        // Create a map of key which is based on the result data key and the
        // heading value
        // The heading is used to create the column headers and the key is used
        // to pull the data from the results
        final Map<String, String> disclosureSummaryMap = new LinkedHashMap<String, String>();
        if ("Exchange".equals(results.getSource())) {
            disclosureSummaryMap.put("partnerOrg", "eHealth Exchange Organization");
            disclosureSummaryMap.put("vaFacility", "Patient Preferred Facility");
            disclosureSummaryMap.put("vaFacilityNumber", "Patient Preferred Facility Station ID");
        } else {
            disclosureSummaryMap.put("partnerOrg", "Direct Endpoint");
        }
        disclosureSummaryMap.put("total", "Total");

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();

        String orgName = (String) request.getParameter("organizationName");
        if (NullChecker.isNullOrEmpty(orgName)) {
            orgName = "ALL";
        }
        final String source = results.getSource();
        filters.put("Disclosure Source", source);
        filters.put("Start Date", request.getParameter("startDate"));
        filters.put("End Date",request.getParameter("endDate"));
        if ("exchange".equals(source.toLowerCase())) {
            filters.put("Patient Preferred Facility", ExcelExporter.getPatientPreferredFacilityFilter(getFacilityDAO(), request.getParameter("facility")));
            filters.put("eHealth Exchange Organization", orgName);
            ExcelExporter.populateFilterMapForExport(request, filters, filterMap, request.getParameter("patientTypes"));
        } else {
            ExcelExporter.populateFilterMapForExport(request, filters, filterMap, "");
        }
        
        // Create the optional data rows
        Map<String, List<Object>> optionalRows = null;
        if ( results.getTotalCount() > 0) {
            final Object disclosureDocumentCount = results.getTotalCount();
            optionalRows = new LinkedHashMap<String, List<Object>>();
            final List<Object> disclosureDocumentRow = new ArrayList<Object>();
            if ("exchange".equals(source.toLowerCase())) {
                disclosureDocumentRow.add("");
                disclosureDocumentRow.add("");
            }
            disclosureDocumentRow.add(disclosureDocumentCount);
            optionalRows.put("Total", disclosureDocumentRow);
        }
        
        // Create the workbook
        final String title = "Disclosures Summary Report";
        final Workbook wb = this.getExcelExporter().exportToExcel(title, title, disclosureSummaryMap, results.getSearchResults(), filterMap, optionalRows);

        // Write Excel to Stream
        this.getExcelExporter().writeExcelToStream("Disclosures_Summary_Report", wb, response);
    }

    /**
     * Converts SearchResutlsPackge to json string Changing List<Map<String,Object>> to a json array with each map being its own array.
     * Object Name is "data" Adds object "totalCount" for total number of audits.
     *
     * @param searchResults
     * @return
     */
    private String encodeIntoJSON(SearchResultsPackage searchResults) {
        String jsonString = "";

        try {
            StringWriter json = new StringWriter();
            JSONObject obj = new JSONObject();
            List<List> data = new ArrayList<List>();

            for (Map<String, Object> row : searchResults.getSearchResults()) {
                List<String> dataItem = new ArrayList<String>();

                for (String key : row.keySet()) {
                    dataItem.add(row.get(key).toString());
                }
                data.add(dataItem);
            }

            obj.put("data", data);

            obj.put("source", searchResults.getSource());

            obj.write(json);

            jsonString = json.toString();
        } catch (JSONException ex) {
            Logger.getLogger(PatientDiscoverySummaryReport.class.getName()).log(Level.SEVERE, null, ex);
        }
        return jsonString;
    }

    /**
     * Class specific to this report format, putting the searchresult list and total count long together
     * Allowing ease of parameter passing.
     */
    private class SearchResultsPackage {

        private final List<Map<String, Object>> searchResults;
        private final long totalCount;
        private final String source;

        public SearchResultsPackage(List<Map<String, Object>> searchResults, long totalCount, String sourceParam) {
            this.searchResults = searchResults;
            this.totalCount = totalCount;
            this.source = sourceParam;
        }

        public List<Map<String, Object>> getSearchResults() {
            return searchResults;
        }

        public long getTotalCount() {
            return totalCount;
        }

        public String getSource() {
            return source;
        }
        
    }
}