package gov.va.nvap.web.report;

import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.svc.consentmgmt.stub.dao.DelayReasonDAO;
import gov.va.nvap.svc.consentmgmt.stub.dao.DelayedConsentReportDAO;
import gov.va.nvap.svc.consentmgmt.stub.data.DelayReason;
import gov.va.nvap.svc.consentmgmt.stub.data.DelayedConsentRpt;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.facility.Facility;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.util.xls.CsvExporter;
import gov.va.nvap.web.util.xls.ExcelExporter;
import java.io.IOException;
import java.io.StringWriter;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.poi.ss.usermodel.Workbook;
import org.json.JSONException;
import org.json.JSONObject;

/**
 *
 * @since 04/05/2016
 * @author Irakli Kakushadze
 */
public class DelayedConsentSummaryReport extends gov.va.nvap.web.app.ResponseDispatcherHttpServlet {

    private DelayedConsentReportDAO getDelayedConsentReportDAO() {
        return this.getBean("DelayedConsentReportDAO", DelayedConsentReportDAO.class);
    }

    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }

    private DelayReasonDAO getDelayReasonDAO() {
        return this.getBean("DelayReasonDAO", DelayReasonDAO.class);
    }

    private static Integer draw = 0; // "global" variable to count up so every draw of the Delayed Consent Summary Report for DataTables is new
    
    public void doSearch(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException, ParseException {
        final HttpSession session = request.getSession(false);
        
        draw++;
        
        // Perform the search.
        DelayedConsentReportDAO.SearchAllResponse searchResponse = this.getResults(request, response);

        // Generate JSON results.
        session.setAttribute("results", encodeIntoJSON(searchResponse, draw.toString()));
        
        // Forward the response.
        this.forward(request, response, "searchResultsJSON");
    }

    @Override
    protected void unspecified(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException {
        final HttpSession session = request.getSession(false);

        List<Facility> facilities = this.getFacilityDAO().findAll();
        List<DelayReason> reasons = this.getDelayReasonDAO().findAll();

        session.setAttribute("facilities", facilities);
        session.setAttribute("reasons", reasons);

        this.forward(request, response, "searchForm");
    }

    public void exportToCSV(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        DelayedConsentReportDAO.SearchAllResponse searchResponse = this.getResults(request, response);
        List<DelayedConsentRpt> dc = searchResponse.delayedConsents;
        final List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        CsvExporter csvExporter = new CsvExporter();

        //loop over all of the delayed consents and put them in a form the csv exporter likes 
        for (DelayedConsentRpt r : dc) {
            final Map<String, Object> resultMap = new HashMap<String, Object>();

            resultMap.put("authenticatingFacility", r.getFacilityName());
            resultMap.put("consentType", ReportHelper.normalizeConsnentTypeName(r.getConsentTypeName()));
            resultMap.put("total", r.getTotal().toString());

            results.add(resultMap);
        }

        Map<String, String> csvReportMap = new LinkedHashMap<String, String>();

        if (NullChecker.isNullOrEmpty(results)) {
            csvReportMap.put("No records were found.", "");
        } else {
            csvReportMap.put("Authenticating Facility", "authenticatingFacility");
            csvReportMap.put("Consent Type", "consentType");
            csvReportMap.put("Total", "total");
        }

        csvExporter.exportToCSV(response, "Delayed_Consent_Summary_Report", results, (LinkedHashMap<String, String>) csvReportMap);
    }

    public void exportToExcel(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
        DelayedConsentReportDAO.SearchAllResponse searchResponse = this.getResults(request, response);
        Long totalCount = 0L;
        
        // Generate report data.
        final List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        for (DelayedConsentRpt r : searchResponse.delayedConsents) {
            final Map<String, Object> resultMap = new HashMap<String, Object>();

            resultMap.put("authenticatingFacility", r.getFacilityName());
            resultMap.put("consentType", ReportHelper.normalizeConsnentTypeName(r.getConsentTypeName()));
            resultMap.put("total", r.getTotal());

            totalCount = totalCount + (NullChecker.isNullOrEmpty(r.getTotal()) ? 0 : r.getTotal());
            
            results.add(resultMap);
        }
        
        //Create Optional Rows (the totals row)
        final Map<String, List<Object>> optionalRows = new LinkedHashMap<String, List<Object>>();
        List<Object> typePlaceHolder;
        
        typePlaceHolder = new ArrayList<Object>();
        typePlaceHolder.add("");
        typePlaceHolder.add(totalCount);
        
        optionalRows.put("Total", typePlaceHolder);
        
        // Generate column headers.
        final Map<String, String> columnHeaders = new LinkedHashMap<String, String>();
        if (NullChecker.isNullOrEmpty(results)) {
            columnHeaders.put("", "No records were found.");
        } else {
            columnHeaders.put("authenticatingFacility", "Authenticating Facility");
            columnHeaders.put("consentType", "Consent Type");
            columnHeaders.put("total", "Total");
        }

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        final LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();
        
        String daysSinceDelayed = request.getParameter("daysSinceDelayed");
        if (!NullChecker.isNullOrEmpty(daysSinceDelayed) && !"ALL".equals(daysSinceDelayed)) {
            if (daysSinceDelayed.equals("0")) {
                daysSinceDelayed = "0-5";
            }
            else {
                daysSinceDelayed = ">" + daysSinceDelayed;
            }
        }
        
        filters.put("Authenticating Facility", ExcelExporter.getFacilitiesFilter(this.getFacilityDAO(), request.getParameter("stationNumbers")));
        filters.put("Reason(s) for Delay", ExcelExporter.getReasonsForDelayFilter(getDelayReasonDAO(), request.getParameter("reasonsForDelay")));
        filters.put("Days Since Delayed", daysSinceDelayed);
        filters.put("Consent Type", ExcelExporter.getConsentTypeFilter(request.getParameter("consentType")));
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, request.getParameter("patientTypes"));
        
        // Create workbook.
        String title = "Delayed Consent Summary Report";        
        final Workbook wb = this.getExcelExporter().exportToExcel(title, title, columnHeaders, results, filterMap, optionalRows);

        // Write Excel to Stream.
        this.getExcelExporter().writeExcelToStream("Delayed_Consent_Summary_Report", wb, response);
    }

    /**
     * Get the excel exporter class from Spring.
     */
    private ExcelExporter getExcelExporter() {
        final ExcelExporter excelExporter = this.getBean("excelExporter", ExcelExporter.class);
        return excelExporter;
    }

    private DelayedConsentReportDAO.SearchAllResponse getResults(HttpServletRequest request, HttpServletResponse response) {
        DelayedConsentReportDAO.SearchAllRequest searchRequest = this.getDelayedConsentReportDAO().new SearchAllRequest();
        
        int patientType = NullChecker.isNullOrEmpty(request.getParameter("patientTypes")) ? 1 : Integer.parseInt(request.getParameter("patientTypes"));
        
        // Get parameters for the search from  the request.
        searchRequest.stationNumbers = ReportHelper.getStationNumbersFromRequest(request);
        searchRequest.aggregateAtFacilityLevel = "true".equals(request.getParameter("aggregateAtFacilityLevel"));
        searchRequest.includeUnknownVisn = (request.getParameter("includeUnknownVisn") != null) ? "true".equals(request.getParameter("includeUnknownVisn")) : false;
        searchRequest.reasonsForDelay = request.getParameter("reasonsForDelay");
        searchRequest.daysSinceDelayed = request.getParameter("daysSinceDelayed");
        searchRequest.consentType = request.getParameter("consentType");
        searchRequest.start = 0; // this is a summary report, so we want it all and always start at the first record (there is no paging)
        searchRequest.length = -1; // this is a summary report, so we want it all
        
        String sortValue = request.getParameter("order[0][column]") == null ? "0" : request.getParameter("order[0][column]");
        String sortDirection = request.getParameter("order[0][dir]") == null ? "asc" : request.getParameter("order[0][dir]");
        searchRequest.sortBy = sortValue;
        searchRequest.sortOrder = sortDirection;
        searchRequest.patientTypes = patientType;

        // When exporting, we'll receive sort field and sort order explicitly in the corresponding params.
        if (searchRequest.sortBy == null) {
            searchRequest.sortBy = request.getParameter("sortBy");
        }
        if (searchRequest.sortOrder == null) {
            searchRequest.sortOrder = request.getParameter("sortOrder");
        }

        // Perform the search.
        DelayedConsentReportDAO.SearchAllResponse searchResponse = this.getDelayedConsentReportDAO().searchAllForSummary(searchRequest);

        return searchResponse;
    }

    private String encodeIntoJSON(DelayedConsentReportDAO.SearchAllResponse searchResponse, String draw) throws ParseException {
        String returnValue = "";

        try {
            StringWriter json = new StringWriter();
            JSONObject obj = new JSONObject();
            List<List> data = new ArrayList<List>();

            for (DelayedConsentRpt delayedConsent : searchResponse.delayedConsents) {
                List<String> dataItem = new ArrayList<String>();

                dataItem.add(delayedConsent.getFacilityName());
                dataItem.add(ReportHelper.normalizeConsnentTypeName(delayedConsent.getConsentTypeName()));
                dataItem.add(delayedConsent.getTotal().toString());

                data.add(dataItem);
            }

            obj.put("data", data);
            obj.put("draw", Integer.parseInt(draw));
            obj.put("recordsTotal", searchResponse.count);
            obj.put("recordsFiltered", searchResponse.count);

            obj.write(json);

            returnValue = json.toString();
        } catch (JSONException ex) {
            Logger.getLogger(DelayedConsentSummaryReport.class.getName()).log(Level.SEVERE, null, ex);
        }

        return returnValue;
    }

}
