package gov.va.nvap.web.report;

import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.svc.consentmgmt.stub.dao.DelayReasonDAO;
import gov.va.nvap.svc.consentmgmt.stub.dao.DelayedConsentReportDAO;
import gov.va.nvap.svc.consentmgmt.stub.dao.MailNotificationDAO;
import gov.va.nvap.svc.consentmgmt.stub.data.DelayReason;
import gov.va.nvap.svc.consentmgmt.stub.data.DelayedConsentRpt;
import gov.va.nvap.svc.consentmgmt.stub.data.MailNotification;
import gov.va.nvap.web.admin.reports.ServiceAuditReport;
import gov.va.nvap.web.dao.FacilityDAO;
import gov.va.nvap.web.dao.UserDocumentDAO;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.patient.ExcelGeneratorThread;
import gov.va.nvap.web.user.UserHelper;
import gov.va.nvap.web.util.xls.CsvExporter;
import gov.va.nvap.web.util.xls.ExcelExporter;
import java.io.IOException;
import java.io.StringWriter;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.poi.ss.usermodel.Workbook;
import org.json.JSONException;
import org.json.JSONObject;

/**
 *
 * @since 04/05/2016
 * @author Irakli Kakushadze
 */
public class DelayedConsentReport extends gov.va.nvap.web.app.ResponseDispatcherHttpServlet {

    private DelayedConsentReportDAO getDelayedConsentReportDAO() {
        return this.getBean("DelayedConsentReportDAO", DelayedConsentReportDAO.class);
    }

    private DelayReasonDAO getDelayReasonDAO() {
        return this.getBean("DelayReasonDAO", DelayReasonDAO.class);
    }

    private FacilityDAO getFacilityDAO() {
        return this.getBean("FacilityDAO", FacilityDAO.class);
    }

    private ReportHelper getReportHelper() {
        return this.getBean("reportHelper", ReportHelper.class);
    }

    private MailNotificationDAO getMailNotificationDAO() {
        return this.getBean("MailNotificationDAO", MailNotificationDAO.class);
    }

    public void doSearch(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException, ParseException {
        final HttpSession session = request.getSession(false);

        // Perform the search.
        DelayedConsentReportDAO.SearchAllResponse searchResponse = this.getResults(request, response);

        // Generate JSON results.
        session.setAttribute("results", encodeServiceAuditIntoJSON(searchResponse, request.getParameter("draw"), request.getParameter("reasonsForDelay")));

        // Forward the response.
        this.forward(request, response, "searchResultsJSON");
    }

    @Override
    protected void unspecified(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException {
        final HttpSession session = request.getSession(false);

        List<DelayReason> reasons = this.getDelayReasonDAO().findAll();

        session.setAttribute("reasons", reasons);

        this.forward(request, response, "searchForm");
    }

    public void exportToCsv(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

        Map<String, String> columnHeaders = new LinkedHashMap<String, String>();

        columnHeaders.put("Date Entered (CT)", "dateAdded");
        columnHeaders.put("ICN", "icn");
        columnHeaders.put("SSN", "ssn");
        columnHeaders.put("Last Name", "lastName");
        columnHeaders.put("First Name", "firstName");
        columnHeaders.put("Middle Name", "middleName");
        columnHeaders.put("Consent Type", "consentType");
        columnHeaders.put("Reason(s) for Delay", "reasonsForDelay");
        columnHeaders.put("Entered By", "enteredBy");
        columnHeaders.put("Authenticating Facility", "authenticatingFacility");
        columnHeaders.put("Mailed Dates", "mailNotifications");

        int minimumThreshold = Integer.parseInt((String) this.getServletContext().getAttribute("scheduledExportMin"));
        int total = Integer.parseInt(request.getParameter("totalRows"));

        //Start a scheduled export with its own thread if greater than the minimum threshold
        if (total >= minimumThreshold) {
            ExcelGeneratorThread csvGenThread = new ExcelGeneratorThread("csvGenThread");
            csvGenThread.setDocumentType("csv");

            csvGenThread.setReportMap(columnHeaders);

            // Set names for Excel generator thread
            csvGenThread.setTitle("Delayed Consent Detail Report");

            csvGenThread.setExcelExporter(this.getExcelExporter());
            csvGenThread.setReportHelper(getReportHelper());
            csvGenThread.setUserId(UserHelper.getUserName(request));
            csvGenThread.setDelayedConsentReportDAO(getDelayedConsentReportDAO());
            csvGenThread.setMailNotificationDAO(getMailNotificationDAO());
            csvGenThread.setUserDocumentDao(getUserDocumentDAO());

            // Set search attributes for getting audit results
            csvGenThread.setAttributes(mapSessionAttributes(request));

            csvGenThread.start();
        }
        //Otherwise generate and download the export directly
        else {
            List<Map<String, Object>> results;

            ExcelGeneratorThread doNow = new ExcelGeneratorThread("doNow");
            doNow.setAttributes(mapSessionAttributes(request));
            doNow.setDelayedConsentReportDAO(this.getDelayedConsentReportDAO());
            doNow.setReportHelper(this.getReportHelper());
            doNow.setMailNotificationDAO(this.getMailNotificationDAO());
            results = doNow.getDelayedConsentResults();

            CsvExporter csvExporter = new CsvExporter();
            csvExporter.exportToCSV(response, "Delayed_Consent_Detail_Report", results, (LinkedHashMap<String, String>) columnHeaders);
        }
    }

    public void exportToExcel(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {

        // Generate column headers.
        final Map<String, String> columnHeaders = new LinkedHashMap<String, String>();
        columnHeaders.put("dateAdded", "Date Entered (CT)");
        columnHeaders.put("icn", "ICN");
        columnHeaders.put("ssn", "SSN");
        columnHeaders.put("lastName", "Last Name");
        columnHeaders.put("firstName", "First Name");
        columnHeaders.put("middleName", "Middle Name");
        columnHeaders.put("consentType", "Consent Type");
        columnHeaders.put("reasonsForDelay", "Reason(s) for Delay");
        columnHeaders.put("enteredBy", "Entered By");
        columnHeaders.put("authenticatingFacility", "Authenticating Facility");
        columnHeaders.put("mailNotifications", "Mailed Dates");

        // Generate filters.
        final Map<String, List<Object>> filterMap = new LinkedHashMap<String, List<Object>>();
        final LinkedHashMap<String, Object> filters = new LinkedHashMap<String, Object>();

        String daysSinceDelayed = request.getParameter("daysSinceDelayed");
        if (!NullChecker.isNullOrEmpty(daysSinceDelayed) && !"ALL".equals(daysSinceDelayed)) {
            if (daysSinceDelayed.equals("0")) {
                daysSinceDelayed = "0-5";
            }
            else {
                daysSinceDelayed = ">" + daysSinceDelayed;
            }
        }

        filters.put("SSN", request.getParameter("ssn"));
        filters.put("Last Name", request.getParameter("lastName"));
        filters.put("First Name", request.getParameter("firstName"));
        filters.put("Authenticating Facility", ExcelExporter.getFacilitiesFilter(this.getFacilityDAO(), request.getParameter("stationNumbers")));
        filters.put("Reason(s) for Delay", ExcelExporter.getReasonsForDelayFilter(getDelayReasonDAO(), request.getParameter("reasonsForDelay")));
        filters.put("Days Since Delayed", daysSinceDelayed);
        filters.put("Consent Type", ExcelExporter.getConsentTypeFilter(request.getParameter("consentType")));
        filters.put("Entered By", ExcelExporter.getConsentTypeFilter(request.getParameter("enteredBy")));
        ExcelExporter.populateFilterMapForExport(request, filters, filterMap, request.getParameter("patientTypes"));

        int minimumThreshold = Integer.parseInt((String) this.getServletContext().getAttribute("scheduledExportMin"));
        int total = Integer.parseInt(request.getParameter("totalRows"));

        //Start a scheduled export with its own thread if greater than the minimum threshold
        if (total >= minimumThreshold) {
            ExcelGeneratorThread exGenThread = new ExcelGeneratorThread("exGenThread");
            exGenThread.setReportMap(columnHeaders);
            exGenThread.setFilterMap(filterMap);

            // Set names for Excel generator thread
            exGenThread.setTitle("Delayed Consent Detail Report");

            exGenThread.setExcelExporter(this.getExcelExporter());
            exGenThread.setReportHelper(getReportHelper());
            exGenThread.setUserId(UserHelper.getUserName(request));
            exGenThread.setDelayedConsentReportDAO(getDelayedConsentReportDAO());
            exGenThread.setMailNotificationDAO(getMailNotificationDAO());
            exGenThread.setUserDocumentDao(getUserDocumentDAO());

            // Set search attributes for getting audit results
            exGenThread.setAttributes(mapSessionAttributes(request));

            exGenThread.start();
        }
        //Otherwise generate and download the export directly
        else {
            List<Map<String, Object>> results;
            String title = "Delayed Consent Detail Report";

            ExcelGeneratorThread doNow = new ExcelGeneratorThread("doNow");
            doNow.setAttributes(mapSessionAttributes(request));
            doNow.setDelayedConsentReportDAO(this.getDelayedConsentReportDAO());
            doNow.setReportHelper(this.getReportHelper());
            doNow.setMailNotificationDAO(this.getMailNotificationDAO());
            results = doNow.getDelayedConsentResults();

            final Workbook wb = this.getExcelExporter().exportToExcel(title, title, columnHeaders, results, filterMap, null);

            // Write Excel workbook to Stream
            this.getExcelExporter().writeExcelToStream("Delayed_Detail_Report", wb, response);
        }
    }

    private Map<String, Object> mapSessionAttributes(HttpServletRequest request) {
        Map<String, Object> attributes = new HashMap<String, Object>();

        int patientType = NullChecker.isNullOrEmpty(request.getParameter("patientTypes")) ? 1 : Integer.parseInt(request.getParameter("patientTypes"));

        // Get parameters for the search from the request.
        String sortBy = request.getParameter("order[0][column]");
        String sortOrder = request.getParameter("order[0][dir]");

        // When exporting, we'll receive sort field and sort order explicitly in the corresponding params.
        if (sortBy == null) {
            sortBy = request.getParameter("sortBy");
        }
        if (sortOrder == null) {
            sortOrder = request.getParameter("sortOrder");
        }

        // Get parameters for the search from the request.
        attributes.put("ssn", request.getParameter("ssn"));
        attributes.put("lastName", request.getParameter("lastName"));
        attributes.put("firstName", request.getParameter("firstName"));
        attributes.put("userId", request.getParameter("enteredBy"));

        attributes.put("stationNumbers", ReportHelper.getStationNumbersFromRequest(request));
        attributes.put("patientTypes", patientType);
        attributes.put("consentType", request.getParameter("consentType"));
        attributes.put("reasonsForDelay", request.getParameter("reasonsForDelay"));
        attributes.put("daysSinceDelayed", request.getParameter("daysSinceDelayed"));
        attributes.put("includeUnknownVisn", "true".equals(request.getParameter("includeUnknownVisn")));
        // sort
        attributes.put("sortValue", sortBy);
        attributes.put("sortDirection", sortOrder);

        return attributes;
    }

    /**
     * Get the excel exporter class from Spring.
     */
    private ExcelExporter getExcelExporter() {
        final ExcelExporter excelExporter = this.getBean("excelExporter", ExcelExporter.class);
        return excelExporter;
    }

    private DelayedConsentReportDAO.SearchAllResponse getResults(HttpServletRequest request, HttpServletResponse response) {
        DelayedConsentReportDAO.SearchAllRequest searchRequest = this.getDelayedConsentReportDAO().new SearchAllRequest();

        int patientType = NullChecker.isNullOrEmpty(request.getParameter("patientTypes")) ? 1 : Integer.parseInt(request.getParameter("patientTypes"));
        int start = NullChecker.isNullOrEmpty(request.getParameter("start")) ? 0 : Integer.parseInt(request.getParameter("start"));
        int length = NullChecker.isNullOrEmpty(request.getParameter("length")) ? -1 : Integer.parseInt(request.getParameter("length"));

        // Get parameters for the search from the request.
        searchRequest.patientSsn = request.getParameter("ssn");
        searchRequest.patientLastName = request.getParameter("lastName");
        searchRequest.patientFirstName = request.getParameter("firstName");
        searchRequest.stationNumbers = ReportHelper.getStationNumbersFromRequest(request);
        searchRequest.aggregateAtFacilityLevel = false;
        searchRequest.includeUnknownVisn = "true".equals(request.getParameter("includeUnknownVisn"));
        searchRequest.reasonsForDelay = request.getParameter("reasonsForDelay");
        searchRequest.daysSinceDelayed = request.getParameter("daysSinceDelayed");
        searchRequest.consentType = request.getParameter("consentType");
        searchRequest.start = start;
        searchRequest.length = length;
        searchRequest.sortBy = request.getParameter("order[0][column]");
        searchRequest.sortOrder = request.getParameter("order[0][dir]");
        searchRequest.patientTypes = patientType;
        searchRequest.enteredBy = request.getParameter("enteredBy");

        // When exporting, we'll receive sort field and sort order explicitly in the corresponding params.
        if (searchRequest.sortBy == null) {
            searchRequest.sortBy = request.getParameter("sortBy");
        }
        if (searchRequest.sortOrder == null) {
            searchRequest.sortOrder = request.getParameter("sortOrder");
        }

        // Perform the search.
        DelayedConsentReportDAO.SearchAllResponse searchResponse = this.getDelayedConsentReportDAO().searchAll(searchRequest);

        //return unavailable if the ssn is null or empty or mask the ssn if we have one
        for(DelayedConsentRpt dc : searchResponse.delayedConsents){
            String ssn = dc.getPatientSsn();

            if (NullChecker.isNullOrEmpty(ssn)) {
                dc.setPatientSsn("Unavailable");
            } else {
                dc.setPatientSsn(ReportDataProcessor.maskSsn(ssn));
            }
        }

        return searchResponse;
    }

    private Collection<MailNotification> getMailNotificationsByDelayedConsent(Long delayedConsentId) {
        Collection<MailNotification> notifications = this.getMailNotificationDAO().findByDelayedConsentId(delayedConsentId.toString());

        return notifications;
    }

    private String handleDelayReasons(Collection<DelayReason> delayedReasons, String searchedForReasons) {
        if (!NullChecker.isNullOrEmpty(delayedReasons)) {
            StringBuilder sb = new StringBuilder();

            if (NullChecker.isNullOrEmpty(searchedForReasons) || "ALL".equals(searchedForReasons)) {
                //get em all
                for (DelayReason dr : delayedReasons) {
                    sb.append(dr.getName());
                    sb.append(", ");
                }
            } else {
                //concatenate applicable reasons only
                List<String> reasons = Arrays.asList(searchedForReasons.split("\\s*,\\s*"));

                for (DelayReason dr : delayedReasons) {
                    if (reasons.contains(dr.getDelayReasonId().toString())) {
                        sb.append(dr.getName());
                        sb.append(", ");
                    }
                }
            }

            return sb.length() > 0 ? sb.substring(0, sb.length() - 2) : ""; //get rid of the last comma and space (yeah its clunky and I don't particularly like it)
        } else {
            return "";
        }
    }

    private String handleMailNotifications(Collection<MailNotification> notifications, String delimiter) {
        StringBuilder result = new StringBuilder();
        boolean isFirstItem = true;

        for (MailNotification mn : notifications) {
            result.append(isFirstItem ? "" : delimiter);
            result.append(this.getReportHelper().getFormattedDate(mn.getSentDate()));
            isFirstItem = false;
        }

        return result.toString();
    }

    private String handleConsentType(DelayedConsentRpt delayedConsent) {
        String consentType = "";
        if (!NullChecker.isNullOrEmpty(delayedConsent.getConsentTypeId())) {
            consentType = delayedConsent.getConsentTypeId().getName();
            consentType = ReportHelper.normalizeConsnentTypeName(consentType);
        }
        return consentType;
    }

    private String encodeServiceAuditIntoJSON(DelayedConsentReportDAO.SearchAllResponse searchResponse, String draw, String reasonsForDelay)
        throws ParseException {
        String returnValue = "";

        try {
            StringWriter json = new StringWriter();
            JSONObject obj = new JSONObject();
            List<List> data = new ArrayList<List>();

            for (DelayedConsentRpt delayedConsent : searchResponse.delayedConsents) {
                List<String> dataItem = new ArrayList<String>();
                dataItem.add(this.getReportHelper().getFormattedDateTime(delayedConsent.getDateAdded()));
                dataItem.add(delayedConsent.getPatientIen());
                dataItem.add(ReportDataProcessor.generatePatientLink(delayedConsent.getPatientSsn(), delayedConsent.getPatientFirstName(), delayedConsent.getPatientLastName()));
                dataItem.add(delayedConsent.getPatientLastName());
                dataItem.add(delayedConsent.getPatientFirstName());
                dataItem.add(delayedConsent.getPatientMiddleName());
                dataItem.add(handleConsentType(delayedConsent));
                dataItem.add(handleDelayReasons(delayedConsent.getDelayReasonCollection(), reasonsForDelay));
                dataItem.add(delayedConsent.getUserId());
                dataItem.add(delayedConsent.getFacilityName());
                dataItem.add(handleMailNotifications(this.getMailNotificationsByDelayedConsent(delayedConsent.getDelayedConsentId()),"<br />"));

                data.add(dataItem);
            }

            obj.put("data", data);
            obj.put("draw", Integer.parseInt(draw));
            obj.put("recordsTotal", searchResponse.count);
            obj.put("recordsFiltered", searchResponse.count);

            obj.write(json);

            returnValue = json.toString();
        } catch (JSONException ex) {
            Logger.getLogger(ServiceAuditReport.class.getName()).log(Level.SEVERE, null, ex);
        }

        return returnValue;
    }

    private UserDocumentDAO getUserDocumentDAO() {
        return this.getBean("UserDocumentDAO", UserDocumentDAO.class);
    }
}
