package gov.va.nvap.web.report;

import gov.va.nvap.audit.BulkDownloadDocs;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.helper.report.ReportHelper;

import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.Properties;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.commons.io.IOUtils;
import weblogic.security.internal.SerializedSystemIni;
import weblogic.security.internal.encryption.ClearOrEncryptedService;
import weblogic.security.internal.encryption.EncryptionService;

/**
 * Servlet implementation class AdapterDateRangeReport
 * 
 * @author Sricharan Reddy
 */
public class AdapterDateRangeReport extends ResponseDispatcherHttpServlet {
	
	//Begin Constant Declarations for Class
	private static final long   serialVersionUID = 1L;
	
	private static final int    BYTES_DOWNLOAD = 1024;
	
	private static final String MOCK_FLAG = "mock.bulk.download";
	
	private static final String REPORT_HELPER_CLASS= "reportHelper";
	
	private static final String CMS_HELPER_CLASS = "cmsHelper";
	
	private static final String INPUT_DATE_FORMAT = "MM/dd/yyyy";
	
	private static final String OUTPUT_DATE_FORMAT = "yyyy-MM-dd";
	
	private static final String START_DATE_PARAM = "startDate";
	
	private static final String END_DATE_PARAM ="endDate";

	private static final String PARTNER_PARAM = "item";
	
	private static final String PATIENT_IDENTIFIER_PARAM = "patientIdentifier";
	
	private static final String CONTENT_TYPE = "application/zip";
	
	private static final String RESPONSE_HEADER =  "attachment; filename=auditedDocsExport.zip"; 
    private static final String PROP_FILE_LOC_SYSPROP  = "vap.bulkDownload.config";
    private static final String PROP_FILE_LOC_DFTL_VAL = "config/vap/bulkDownload.properties";
    private static final String VAP_ID_PROP = "vap.identifier";
    
	// End of Constants Declaration

    //***
    // Set by init()
    //***
    private String vapId;
    private ClearOrEncryptedService encSvc;

    /**
     * Default constructor. 
     */
    public AdapterDateRangeReport()
    {
    	
    }

    @Override
    public void init() throws ServletException
    {
        super.init();
        try
        {
            vapId = loadVAPId();
        }
        catch(IOException ex)
        {
            throw new ServletException(
                "Unable to load configuration for bulk download.", ex);
        }
        
        // works off convention that the current working directory is the domain home
        String domainHome = new File("dummy").getAbsoluteFile().getParent();
        EncryptionService encryptSrv = SerializedSystemIni.getEncryptionService(domainHome);
        encSvc = new ClearOrEncryptedService(encryptSrv);
    }

    /**
     * Helper method to get the CmsHelper instance
     * @return instance of ConsentManagementHelper
     */
    public ConsentManagementHelper getCmsHelper()
    {
        final ConsentManagementHelper cmsHelper = this.getBean(CMS_HELPER_CLASS,ConsentManagementHelper.class);
        return cmsHelper;
    }
    
    /**
     *  Method to get an instance of bulk download docs
     * @return instance of BulkDownloadDocs
     */
    public BulkDownloadDocs getbulkDownloadDocs()
    {
    	String useMock =""; 
    	useMock = System.getProperty(MOCK_FLAG)!=null?System.getProperty(MOCK_FLAG):"";
    	String mockClass="";    
    	 if(useMock.equalsIgnoreCase("true"))
         {
    	      mockClass = "bulkDownloadDocs-mock";     		 
    	 }    	   
    	 else
    	 {
    	      mockClass = "bulkDownloadDocs";    		
    	 }
		final BulkDownloadDocs bulkDownloadBean = getBean(mockClass, BulkDownloadDocs.class);
    	return bulkDownloadBean;    	
    }
    
    /**
     * Method  for the Report Helper class
     * @return instance of ReportHelper
     */
    private ReportHelper getReportHelper()
    {
        return this.getBean(REPORT_HELPER_CLASS, ReportHelper.class);
    }

	/**
	 * Servlet logic for getting the dates and generating the output file
	 */
	protected void doGet(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException
	{
		// Get relevant session variables
		String startDate= request.getParameter(START_DATE_PARAM)!=null?request.getParameter(START_DATE_PARAM):"";
		String endDate = request.getParameter(END_DATE_PARAM)!=null?request.getParameter(END_DATE_PARAM):"";
		String organizationName= request.getParameter(PARTNER_PARAM)!=null?request.getParameter(PARTNER_PARAM):"";
		String patientIdentifier= request.getParameter(PATIENT_IDENTIFIER_PARAM)!=null?request.getParameter(PATIENT_IDENTIFIER_PARAM):"";
		
		//Initialize page with Default values
		 if(startDate.equalsIgnoreCase("") || endDate.equalsIgnoreCase(""))
		 {
			 unspecified(request, response);
			 return;
		 }
		 
		 if(!startDate.equalsIgnoreCase("") && !endDate.equalsIgnoreCase(""))			 
		 {
            String startAudit      = formatDateForAdapter(startDate);
            String endAudit        = formatDateForAdapter(endDate);
            String docGenPartnerId = formatStringforPartner(organizationName);			    
            InputStream is;
            try
            {
                String vapIdClear = encSvc.decrypt(vapId);
                is = getbulkDownloadDocs().downloadRetrievedDocs(vapIdClear, startAudit, endAudit, docGenPartnerId, patientIdentifier); // Get input stream from BulkDownload Class
            }catch(Exception e)
            {
                throw new ServletException(
                    "An exception occurred while attempting to communicate with Adapter Bulk Document Download.", e);
            }

            BufferedOutputStream bos = null;
            try
            {
                response.setContentType(CONTENT_TYPE);
                response.setHeader("Content-Disposition",RESPONSE_HEADER);		    	
                int read=0;
                byte[] bytes = new byte[BYTES_DOWNLOAD];
                bos = new BufferedOutputStream(response.getOutputStream());                
                while((read = is.read(bytes))!= -1)
                {
                    bos.write(bytes, 0, read);		    		
                }
                bos.flush();
            }
            finally
            {
                IOUtils.closeQuietly(bos);
                IOUtils.closeQuietly(is);
            }
		 }
     }

	// Method to Format the partner string for input to client function
	private String formatStringforPartner(String organizationName)
	{
        String partnerId= organizationName.replace('%', ' '); // Remove % signs from Partner ID
        partnerId = partnerId.replaceAll("\\s+", ""); // Replace empty characters
		return partnerId!=null?partnerId:organizationName;
	}

	//Method to format dates for Webservice client
	private String formatDateForAdapter(String inputDate) 
	{
		SimpleDateFormat sdf= new SimpleDateFormat(INPUT_DATE_FORMAT);
		SimpleDateFormat dt = new SimpleDateFormat(OUTPUT_DATE_FORMAT);
		Date date = null;
		try {
			date = sdf.parse(inputDate);
		} catch (ParseException e) {
            throw new RuntimeException("Exception in date format.", e);
		}
		String returnDate=dt.format(date);
		return returnDate!=null?returnDate:inputDate;
	}

	/**
	 * @see HttpServlet #doPost(HttpServletRequest request, HttpServletResponse response)
	 */
	protected void doPost(HttpServletRequest request, HttpServletResponse response) throws ServletException, IOException {
		doGet(request, response);
	}

	@Override
	protected void unspecified(HttpServletRequest request, HttpServletResponse response)
			throws ServletException, IOException
	{
		final HttpSession session = request.getSession(false);
		ReportHelper.setDefaultSearchDates(session);
		this.getReportHelper().getOrgLists(request,this.getCmsHelper());
	    this.forward(request, response, "show");		
	}
    
    private String loadVAPId() throws IOException
    {
        File file;
        if (System.getProperty(PROP_FILE_LOC_SYSPROP) != null)
        {
            file = new File(System.getProperty(PROP_FILE_LOC_SYSPROP));
        }
        else
        {
            file = new File(PROP_FILE_LOC_DFTL_VAL);
        }

        InputStream stream = null;
        Properties props = new Properties();
        try
        {
            stream = new FileInputStream(file);
            props.load(stream);
        }
        finally
        {
            IOUtils.closeQuietly(stream);
        }
        return props.getProperty(VAP_ID_PROP);
    } 
}
