package gov.va.nvap.web.patient;

import gov.va.nvap.common.transformer.TransformerException;
import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.common.xpath.XPathException;
import gov.va.nvap.privacy.ConsentDirectiveOptOutReasonType;
import gov.va.nvap.privacy.ConsentDirectiveReferenceType;
import gov.va.nvap.privacy.ConsentType;
import gov.va.nvap.privacy.OrganizationType;
import gov.va.nvap.service.adapter.doc.AdapterException;
import gov.va.nvap.service.pdq.Facility;
import gov.va.nvap.service.pdq.PatientCorrelationsQuery;
import gov.va.nvap.service.pdq.PatientCorrelationsResponse;
import gov.va.nvap.service.pdq.PatientDemographics;
import gov.va.nvap.service.pdq.PatientDemographicsQuery;
import gov.va.nvap.service.pdq.PatientDemographicsResponse;
import gov.va.nvap.service.pdq.PdqException;
import gov.va.nvap.service.pdq.PdqService;
import gov.va.nvap.service.pdq.RemovePatientCorrelationRequest;
import gov.va.nvap.svc.consentmgmt.jpa.ConsentDirectiveJpaController;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.Announcement;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.PatientAnnouncer;
import gov.va.nvap.svc.consentmgmt.stub.dao.DelayReasonDAO;
import gov.va.nvap.svc.consentmgmt.stub.dao.DelayedConsentDAO;
import gov.va.nvap.svc.consentmgmt.stub.dao.LetterTypeDAO;
import gov.va.nvap.svc.consentmgmt.stub.dao.MailLogDAO;
import gov.va.nvap.svc.consentmgmt.stub.dao.MailNotificationDAO;
import gov.va.nvap.svc.consentmgmt.stub.dao.MailTemplateDAO;
import gov.va.nvap.svc.consentmgmt.stub.data.ConsentDirective;
import gov.va.nvap.svc.consentmgmt.stub.data.DelayReason;
import gov.va.nvap.svc.consentmgmt.stub.data.DelayedConsent;
import gov.va.nvap.svc.consentmgmt.stub.data.MailLog;
import gov.va.nvap.svc.consentmgmt.stub.data.MailNotification;
import gov.va.nvap.web.consent.audit.AuditedConsentEx;
import gov.va.nvap.web.consent.audit.dao.AuditedConsentDAO;
import gov.va.nvap.web.consent.comment.ConsentComment;
import gov.va.nvap.web.consent.comment.ConsentCommentDAO;
import gov.va.nvap.web.helper.document.DocumentHelper;
import gov.va.nvap.web.helper.document.MediaType;
import gov.va.nvap.web.helper.document.RepresentationType;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.user.UserHelper;
import gov.va.nvap.web.util.Constants;
import gov.va.nvap.web.util.DASUtil;
import gov.va.nvap.web.util.date.DateUtil;
import gov.va.nvap.web.util.file.FileUploadUtil;
import gov.va.nvap.web.util.file.PdfGenerator;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.ejb.EJB;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.io.IOUtils;
import static org.apache.commons.lang.StringEscapeUtils.escapeXml;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.json.JSONException;
import org.json.JSONObject;
import weblogic.net.http.HttpsURLConnection;

/**
 * Get the detailed view of the patient.
 *
 * @author Asha Amritraj
 * @author Irakli Kakushadze
 * @author Zack Peterson
 */
public class PatientDetails extends
		gov.va.nvap.web.app.ResponseDispatcherHttpServlet {
	static private final Log LOG = LogFactory.getLog(PatientDetails.class);

	/**
	 * Serial UID.
	 */
	private static final long serialVersionUID = 960355389572259403L;

    private static final int bufferSize = 8192;

	/**
	 * Announce patients to the NwHIN.
	 */
	@EJB(beanInterface = PatientAnnouncer.class, mappedName = "PatientAnnouncer")
	private PatientAnnouncer patientAnnouncer;
	/**
	 * Patient Demographics Service to get the demographics and correlations
	 * from the MPI.
	 */
	@EJB(beanInterface = PdqService.class, mappedName = "PdqService")
	PdqService pdqService;

	/**
	 * Announce is called when the user clicked on Announce on the Patient
	 * Details page.
     *
     * @param request http servlet request
     * @param response http servlet response
     *
     * @throws javax.servlet.ServletException
     * @throws java.io.IOException
	 */
	public void announce(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		// Get the ICN
		final String icn = request.getParameter("icn");
		Assert.assertNotEmpty(icn, "ICN cannot be null!");
		if (this.validate(request, response, "announce")) {
            PatientAnnounceThread announceThread =
                    new PatientAnnounceThread("announcerThread");
            announceThread.setCmsHelper(this.getCmsHelper());
            announceThread.setFacilityHelper(this.getFacilityHelper());
            announceThread.setPatientAnnouncer(this.patientAnnouncer);
            announceThread.setPatientIcn(icn);
            announceThread.setPdqService(this.pdqService);
            announceThread.setRemoteUserId(UserHelper.getUserName(request));
            announceThread.start();

            final HttpSession session = request.getSession(false);
            session.setAttribute("message", "Patient announcement started. You may move on to another announcement or activity.");

            this.forward(request, response, "search");
		} else {
			// Validation
			this.forward(request, response, "validate");
		}
	}

    public void delayAuthorization(final List<FileItem> fileItems,
			final HttpServletRequest request, final HttpServletResponse response)
			throws ServletException, IOException {

        final HttpSession session = request.getSession(false);

		// Get the fields
		final Map<String, String> formFields = FileUploadUtil
				.getFormFields(fileItems);

        Long consentId = null;

        //Get all the delay reasons
        Collection<DelayReason> delayReasonCollection = this.getDelayReasonDAO().findAll();

        //Get the delay reason(s) selected
        final ArrayList<String> delayReasons = new ArrayList<String>(Arrays.asList(formFields.get("collectedReasons").split(",")));

        ArrayList<DelayReason> theseReasons = new ArrayList<DelayReason>();

        //Convert the list of delay reason id strings into DelayReason objects
        for (DelayReason reason : delayReasonCollection) {
            if (delayReasons.contains(reason.getDelayReasonId().toString())) {
                theseReasons.add(reason);
            }
        }

		// Get other fields from the form.
		final String consentType = formFields.get("consentType");
        final String comments = formFields.get("comments");
		String userFacilityStationId = formFields.get("userFacility");

        if (this.validate(formFields, request, response, "delayAuthorization")) {

			final String userId = UserHelper.getUserName(request);

			if (NullChecker.isEmpty(userFacilityStationId)) {
				userFacilityStationId = this.getFacilityHelper()
						.getFaciltyStationIdByUserId(userId);
			}

            PatientDemographics patientDemographics = (PatientDemographics)session.getAttribute("patientDemographics");

            gov.va.nvap.svc.consentmgmt.stub.data.ConsentType consentTypeObject = this.getConsentDirectiveJpaController()
                .findConsentTypeByName(consentType);
            Date today = new Date();
            DelayedConsent da = new DelayedConsent();
            da.setDateAdded(today);
            da.setDelayReasonCollection(theseReasons);
            da.setPatientFirstName(patientDemographics.getFirstName());
            da.setPatientLastName(patientDemographics.getLastName());
            da.setPatientMiddleName(patientDemographics.getMiddleName());
            da.setPatientSsn(patientDemographics.getSsn());
            da.setPatientIen((String)session.getAttribute("patientIen"));
            da.setConsentTypeId(consentTypeObject);
            da.setStationNumber(userFacilityStationId);
            da.setUserId(userId);

            try {
                consentId = this.getDelayedConsentDAO().create(da);
            } catch (Exception ex) {
                Logger.getLogger(PatientDetails.class.getName()).log(Level.SEVERE, null, ex);
            }

            //Handle comments
            if(!NullChecker.isEmpty(comments)) {
                ConsentComment comment = new ConsentComment();
                comment.setComments(comments);
                comment.setConsentType(consentType);
                comment.setUserId(userId);
                comment.setDateAdded(new Date());
                comment.setDelayedConsentId(consentId);

                this.getConsentCommentDAO().create(comment);
            }
            // Success
            this.forward(request, response, "success");
		} else {
			// Validate
			this.forward(request, response, "validate");
		}

    }

    public ConsentDirectiveJpaController getConsentDirectiveJpaController() {
        final ConsentDirectiveJpaController consentDirectiveJpaController = this.getBean("consentDirectiveJpaController",
            ConsentDirectiveJpaController.class);
        return consentDirectiveJpaController;
    }

	/**
	 * Get the consent management helper from Spring.
	 */
	public ConsentManagementHelper getCmsHelper() {
		final ConsentManagementHelper cmsHelper = this.getBean("cmsHelper",
				ConsentManagementHelper.class);
		return cmsHelper;
	}

    public DelayedConsentDAO getDelayedConsentDAO() {
        final DelayedConsentDAO delayedConsentDAO = this.getBean("DelayedConsentDAO", DelayedConsentDAO.class);
        return delayedConsentDAO;
    }

    public DelayReasonDAO getDelayReasonDAO() {
        final DelayReasonDAO delayReasonDAO = this.getBean("DelayReasonDAO", DelayReasonDAO.class);
        return delayReasonDAO;
    }

	/**
	 * Get the document helper from Spring.
	 */
	public DocumentHelper getDocumentHelper() {
		return this.getBean("adapterDocumentHelper", DocumentHelper.class);
	}

	/**
	 * Get the facility helper from Spring.
	 */
	public FacilityHelper getFacilityHelper() {
		final FacilityHelper facilityHelper = this.getBean("facilityHelper",
				FacilityHelper.class);
		return facilityHelper;
	}

    public ConsentCommentDAO getConsentCommentDAO() {
        final ConsentCommentDAO consentCommentDAO = this.getBean("consentCommentDAO", ConsentCommentDAO.class);

        return consentCommentDAO;
    }

    public PatientCommentDAO getPatientCommentDAO() {
        final PatientCommentDAO patientCommentDAO = this.getBean("patientCommentDAO", PatientCommentDAO.class);

        return patientCommentDAO;
    }

    public AuditedConsentDAO getAuditedConsentDAO() {
        final AuditedConsentDAO auditedConsentDAO = this.getBean("auditedConsentDAO", AuditedConsentDAO.class);

        return auditedConsentDAO;
    }

    public LetterTypeDAO getLetterTypeDAO() {
        final LetterTypeDAO letterTypeDAO = this.getBean("letterTypeDAO", LetterTypeDAO.class);

        return letterTypeDAO;
    }

    public MailNotificationDAO getMailNotificationDAO() {
        final MailNotificationDAO mailNotificationDAO = this.getBean("MailNotificationDAO", MailNotificationDAO.class);

        return mailNotificationDAO;
    }

    public MailLogDAO getMailLogDAO() {
        final MailLogDAO mailLogDAO = this.getBean("MailLogDAO", MailLogDAO.class);

        return mailLogDAO;
    }

    public MailTemplateDAO getMailTemplateDAO() {
        final MailTemplateDAO mailTemplateDAO = this.getBean("MailTemplateDAO", MailTemplateDAO.class);

        return mailTemplateDAO;
    }

	/**
	 * Show the consent management history.
     *
     * @param request http servlet request
     * @param response http servlet response
     *
     * @throws javax.servlet.ServletException
     * @throws java.io.IOException
	 */
	public void history(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		final HttpSession session = request.getSession(false);
		// Document type = OPT-IN/OPT-OUT
		final String documentType = request.getParameter("documentType");
		final String consentDirId = request.getParameter("consentDirId");

		Assert.assertNotEmpty(documentType, "Document Type cannot be null!");
		final String document = this.getCmsHelper().retrieveDocument(
				consentDirId, documentType);
		// Set the document for further queries
		session.setAttribute("document", document);
		session.setAttribute("documentType", documentType);
		session.setAttribute("consentDirId", consentDirId);
		// Return the PDF
		try {
			if ( DASUtil.isDASDocument(document)){
                // Get from DAS and show
                // Commented out code that should only be invoked
                // in dev/test environment anyway to get around
                // fixing code that will not be used in production.
				/*if ( Constants.isDasUseMockRetrieve() ){
					//mockGetDocumentFromDAS(response.getOutputStream());
				}else{*/
					getDocumentFromDAS(response.getOutputStream(), document);
				//}
			}else{
				final String attachment = this.getDocumentHelper()
						.getPrivacyConsentDirectiveAttachment(document);
				if (NullChecker.isNotEmpty(attachment)) {
					this.getDocumentHelper()
							.writeAttachmentToStream(
									response,
									attachment,
									MediaType.fromValue(this.getDocumentHelper()
											.getPrivacyConsentDirectiveMediaType(
													document)),
									RepresentationType.B64);
				} else {
					session.setAttribute("informationMessage",
							"There are no attachments associated with this consent directive.");
					this.forward(request, response, "noattachment");
				}
			}
		} catch (final TransformerException ex) {
			throw new ServletException(ex);
		} catch (final XPathException ex) {
			throw new ServletException(ex);
		}

	}

	/**
	 * User selected opt-in in the Patient details page.
     *
     * @param fileItems list of uploaded file items
     * @param request http servlet request
     * @param response http servlet response
     *
     * @throws javax.servlet.ServletException
     * @throws java.io.IOException
	 */
	public void optIn(final List<FileItem> fileItems,
			final HttpServletRequest request, final HttpServletResponse response)
			throws ServletException, IOException {
		final HttpSession session = request.getSession(false);

		// Get the fields
		final Map<String, String> formFields = FileUploadUtil
				.getFormFields(fileItems);

		// Get the uploaded files
		final List<FileItem> filesUploaded = FileUploadUtil
				.getFilesUploaded(fileItems);

        String consentId = "";

		// Get the icn
		final String icn = formFields.get("icn");
		final String consentType = formFields.get("consentType");
		final String purposeOfUse = formFields.get("purposeOfUse");
		final String expirationYears = formFields.get("expirationYears");
                final String comments = formFields.get("comments");
		// Get the organizations selected by the user
		final String newRight = formFields.get("newRight");
		// Get the date signed
		final String signatureDateStr = formFields.get("signatureDate");
		String userFacilityStationId = formFields.get("userFacility");

		String[] organizationNumbers = null;
		if (NullChecker.isNotEmpty(newRight)) {
			organizationNumbers = newRight.split(",");

		}

		if (this.validate(formFields, request, response, "optIn")) {
			// Get the date after validation
			final Date signatureDate = DateUtil
					.parseDateMMddyyyy(signatureDateStr);
			// Maybe go to MPI
			final PatientDemographics patientDemographics = (PatientDemographics) session
					.getAttribute("patientDemographics");
			// Get the files uploaded
			final String userId = UserHelper.getUserName(request);
			// Do it as part of the GUI if possible

			if (NullChecker.isEmpty(userFacilityStationId)) {
				userFacilityStationId = this.getFacilityHelper()
						.getFaciltyStationIdByUserId(userId);
			}
			if ("NwHIN Organization Restriction Modification"
					.equals(consentType)) {
				this.getCmsHelper().optOut(
						icn,
						patientDemographics,
						filesUploaded,
						"Revoked",
						userId,
						userFacilityStationId,
						signatureDate,
						ConsentType.NW_HIN_ORGANIZATION_RESTRICTION_REVOCATION
								.value(), purposeOfUse);
				consentId = this.getCmsHelper()
						.optIn(icn,
								organizationNumbers,
								patientDemographics,
								filesUploaded,
								userId,
								userFacilityStationId,
								signatureDate,
								ConsentType.NW_HIN_ORGANIZATION_RESTRICTION_AUTHORIZATION
										.value(), purposeOfUse, expirationYears);

			} else {
				// Call opt-in with files uploaded
				consentId = this.getCmsHelper().optIn(icn, organizationNumbers,
						patientDemographics, filesUploaded, userId,
						userFacilityStationId, signatureDate, consentType,
						purposeOfUse, expirationYears);
			}

            // Update corresponding delayed authorization (if one exists) after opt-in
            boolean delayedConsentUpdated = false;
            if (consentType.equals(ConsentType.NW_HIN_AUTHORIZATION.value()) || consentType.equals(ConsentType.SSA_AUTHORIZATION.value())) {
                List<DelayedConsent> delayedConsents = this.getDelayedConsentDAO().findByPatientIen(icn);
                if (delayedConsents.size() > 0) {
                    gov.va.nvap.svc.consentmgmt.stub.data.ConsentType consentTypeObject = this.getConsentDirectiveJpaController().findConsentTypeByName(consentType);
                    if (consentTypeObject != null) {
                       for (DelayedConsent dc : delayedConsents) {
                            long delayedId = dc.getConsentTypeId().getConsentTypeId();
                            if ((consentTypeObject.getConsentTypeId() == delayedId) && dc.getStatus().equals("PENDING")) {
                                delayedConsentUpdated = true;
                                try {
                                    dc.setStatus("APPROVED");
                                    dc.setResolutionDate(new Date());
                                    this.getDelayedConsentDAO().update(dc);

                                    //Store any comments with the delayed authorzation
                                    if(!NullChecker.isEmpty(comments)) {
                                        ConsentComment comment = new ConsentComment();
                                        comment.setComments(comments);
                                        comment.setConsentType(consentType);
                                        comment.setUserId(userId);
                                        comment.setDateAdded(new Date());
                                        comment.setDelayedConsentId(dc.getDelayedConsentId());

                                        this.getConsentCommentDAO().create(comment);
                                    }
                                } catch (Exception ex) {
                                    Logger.getLogger(PatientDetails.class.getName()).log(Level.SEVERE, null, ex);
                                }
                            }
                        }
                    }
                }
            }
            if (!delayedConsentUpdated) {
                //Handle comments normally if not approving a delayed authorization
                if(!NullChecker.isEmpty(comments)) {
                    ConsentComment comment = new ConsentComment();
                    comment.setComments(comments);
                    comment.setConsentType(consentType);
                    //There is no Restriction Modification consent type so we need to change it to Authorization in order to be
                    //populated correctly in the jsp.
                    if (consentType.equals("NwHIN Organization Restriction Modification")) {
                        comment.setConsentType(ConsentType.NW_HIN_ORGANIZATION_RESTRICTION_AUTHORIZATION.value());
                    }
                    comment.setUserId(userId);
                    comment.setDateAdded(new Date());
                    comment.setConsentId(Long.valueOf(consentId));

                    this.getConsentCommentDAO().create(comment);
                }
            }
			// Success
			this.forward(request, response, "success");
			// announce(request, response);
		} else {
			// Validate
			this.forward(request, response, "validate");
		}
	}

	/**
	 * The user selected opt-out from the Patient details page.
     *
     * @param request http servlet request
     * @param response http servlet response
     *
     * @throws javax.servlet.ServletException
     * @throws java.io.IOException
	 */
	public void optOut(final List<FileItem> fileItems,
			final HttpServletRequest request, final HttpServletResponse response)
			throws ServletException, IOException {
		final HttpSession session = request.getSession(false);

		// Get the fields
		final Map<String, String> formFields = FileUploadUtil
				.getFormFields(fileItems);
		// Get the uploaded files
		final List<FileItem> filesUploaded = FileUploadUtil
				.getFilesUploaded(fileItems);
		final String consentType = formFields.get("consentType");

                String consentId = "";

		// Get the icn
		final String icn = formFields.get("icn");
		// Get the opt out reason
		final String reason = formFields.get("reason");
		String signatureDateStr = formFields.get("signatureDate");
                String comments = formFields.get("comments");
		String deceasedDateStr = formFields.get("deceasedDate");
		String userFacilityStationId = formFields.get("userFacility");
		final String purposeOfUse = formFields.get("purposeOfUse");
		final String expiryYears = formFields.get("expirationYears");
		if (ConsentDirectiveOptOutReasonType.PATIENT_DECEASED.value().equals(
				reason)) {
			signatureDateStr = deceasedDateStr;
		}

		if (this.validate(formFields, request, response, "optOut")) {
			final Date signatureDate = DateUtil
					.parseDateMMddyyyy(signatureDateStr);
			// Maybe go to MPI
			final PatientDemographics patientDemographics = (PatientDemographics) session
					.getAttribute("patientDemographics");
			// Get the files uploaded
			final String userId = UserHelper.getUserName(request);
			if (NullChecker.isEmpty(userFacilityStationId)) {
				userFacilityStationId = this.getFacilityHelper()
						.getFaciltyStationIdByUserId(userId);
			}

			// Opt-out without files
			consentId = this.getCmsHelper().optOut(icn, patientDemographics, filesUploaded,
					reason, userId, userFacilityStationId, signatureDate,
					consentType, purposeOfUse);
			if (ConsentDirectiveOptOutReasonType.NEW_AUTHORIZATION.value()
					.equals(reason)) {
				if (ConsentType.NW_HIN_REVOCATION.value().equals(consentType)) {
					consentId = this.getCmsHelper().optIn(icn, null, patientDemographics,
							null, userId, userFacilityStationId, signatureDate,
							ConsentType.NW_HIN_AUTHORIZATION.value(),
							purposeOfUse, expiryYears);
				} else if (ConsentType.SSA_REVOCATION.value().equals(
						consentType)) {
					consentId = this.getCmsHelper().optIn(icn, null, patientDemographics,
							null, userId, userFacilityStationId, signatureDate,
							ConsentType.SSA_AUTHORIZATION.value(),
							purposeOfUse, expiryYears);
				}
			}

                        //handle comments
                        if(!NullChecker.isEmpty(comments)) {
                            ConsentComment comment = new ConsentComment();
                            comment.setComments(comments);
                            comment.setConsentType(consentType);
                            comment.setUserId(userId);
                            comment.setDateAdded(new Date());
                            comment.setConsentId(Long.valueOf(consentId));

                            this.getConsentCommentDAO().create(comment);
                        }

			// Remove attributes before exiting
			// Success
			this.forward(request, response, "success");
		} else {
			// Validate
			this.forward(request, response, "validate");
		}
	}

	/**
	 * If a system administrator needs to remove all correlations from the MPI.
     *
     * @param request http servlet request
     * @param response http servlet response
     *
     * @throws javax.servlet.ServletException
     * @throws java.io.IOException
	 */
	public void revoke(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		// Get the icn
		final String icn = request.getParameter("icn");
                final String comments = request.getParameter("comments");
                final String consentType = request.getParameter("consentType");
		if (this.validate(request, response, "revoke")) {
			// Make the correlations query
			final PatientCorrelationsQuery correlationsQuery = new PatientCorrelationsQuery();
			correlationsQuery.setPatientId(icn);
			try {
				// Get the correlations response
				final PatientCorrelationsResponse correlationsResponse = this.pdqService
						.getCorrelatedFacilities(correlationsQuery);
				// Get the demographics
				final PatientDemographicsQuery demographicsQuery = new PatientDemographicsQuery();
				demographicsQuery.setPatientId(icn);
				final PatientDemographicsResponse demographicsResponse = this.pdqService
						.getPatientDemographics(demographicsQuery);
				final PatientDemographics demographics = demographicsResponse
						.getPatientDemographics();
				// Get all the facilities
				final List<Facility> facilities = correlationsResponse
						.getFacilities();
				// For each NwHIN facilities other than DoD, create a request to
				// remove the correlations on the MPI.
				if (NullChecker.isNotEmpty(facilities)) {
					for (final Facility facility : facilities) {
						// Get the organization
						final OrganizationType stdOrganization = this
								.getCmsHelper().getOrganizationByNumber(
										facility.getFacilityId());
						// You really cant remove the DoD correlations
						if (NullChecker.isEmpty(stdOrganization)
								|| (NullChecker.isNotEmpty(stdOrganization) && stdOrganization
										.getOrgNumber().equals("200DOD"))) {
						} else {
							// Get the facility to be removed from the MPI
							// correlation
							String correlatedAssigningAuthorityName = facility
									.getAssigningAuthorityName();
							final String correlatedAssigningAuthorityOid = stdOrganization
									.getOrgOid();
							final String firstName = demographics
									.getFirstName();
							final String middleName = demographics
									.getMiddleName();
							final String lastName = demographics.getLastName();
							final RemovePatientCorrelationRequest removePatientCorrelationRequest = new RemovePatientCorrelationRequest();
							if (NullChecker
									.isEmpty(correlatedAssigningAuthorityName)) {
								correlatedAssigningAuthorityName = stdOrganization
										.getOrgNumber();
							}
							// Set values
							removePatientCorrelationRequest
									.setCorrelatedAssigningAuthorityName(correlatedAssigningAuthorityName);
							removePatientCorrelationRequest
									.setCorrelatedAssigningAuthorityOid(correlatedAssigningAuthorityOid);
							removePatientCorrelationRequest
									.setCorrelatedPatientId(facility
											.getCorrelatedPatientId());
							removePatientCorrelationRequest.setPatientId(icn);
							removePatientCorrelationRequest
									.setPatientFirstName(firstName);
							removePatientCorrelationRequest
									.setPatientMiddleName(middleName);
							removePatientCorrelationRequest
									.setPatientLastName(lastName);
							removePatientCorrelationRequest.setUserId(UserHelper.getUserName(request));
							// Call the service to remove correlations.
							RemovePatientCorrelationRequest r = this.pdqService
									.removePatientCorrelation(removePatientCorrelationRequest);

                                                        //handle comments
                                                        if(!NullChecker.isEmpty(comments)) {
                                                            ConsentComment comment = new ConsentComment();
                                                            comment.setComments(comments);
                                                            comment.setConsentType(consentType);
                                                            comment.setUserId(UserHelper.getUserName(request));
                                                            comment.setDateAdded(new Date());
                                                            //TODO: comment.setConsentId(Long.valueOf(consentId));

                                                            this.getConsentCommentDAO().create(comment);
                                                        }
						}
					}
				}
			} catch (final PdqException ex) {
				throw new ServletException(ex);
			}
			this.forward(request, response, "success");
		} else {
			this.forward(request, response, "validate");
		}
	}
    /*
    Method that updates delayed authorization upon cancelation.
    */
    public void cancelDelay(final HttpServletRequest request,
            final HttpServletResponse response) throws ServletException,
            IOException {

        final HttpSession session = request.getSession(false);

        //Find the delayed authorization by the id sent in from the jsp
        DelayedConsent dc = this.getDelayedConsentDAO().findByDelayedConsentId((Long)session.getAttribute("delayedConsentId"));
        String comments = request.getParameter("cancelComments");

        //Create comment if applicable
        if (!NullChecker.isNullOrEmpty(comments)) {
            ConsentComment comment = new ConsentComment();
            comment.setComments(comments);
            comment.setDateAdded(new Date());
            comment.setUserId(UserHelper.getUserName(request));
            comment.setDelayedConsentId(dc.getDelayedConsentId());
            comment.setConsentType(dc.getConsentTypeId().getName());
            this.getConsentCommentDAO().create(comment);
        }

        dc.setStatus("CANCELED");
        dc.setResolutionDate(new Date());

        try {
            this.getDelayedConsentDAO().update(dc);
        } catch (Exception e) {
            throw new ServletException("There was an issue updating the delayed authorization with id" + dc.getDelayedConsentId());
        }

        this.forward(request, response, "success");
    }
    /*
    Method that passes attributes to cancelDelay method and calls the cancel jsp
    */
    public void cancelDelayedAuthorization(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {

        final HttpSession session = request.getSession(false);

		final String icn = (String) session.getAttribute("icn");
        final long delayedConsentId = (Long.parseLong(request.getParameter("delayedConsentId")));
        session.setAttribute("delayedConsentId", delayedConsentId);
        this.forward(request,response, "cancelDelay");
    }

    public void printLetter(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {

        final HttpSession session = request.getSession(false);

        session.removeAttribute("delayedConsent");
        session.removeAttribute("consent");

		final String icn = (String) session.getAttribute("icn");
        if (request.getParameter("delayedConsentId") != null) {
            if (request.getParameter("ssaDelay") != null) {
                session.setAttribute("ssaDelay", true);
            }
            else if (request.getParameter("eheDelay") != null) {
                session.setAttribute("eheDelay", true);
            }
            DelayedConsent dc = this.getDelayedConsentDAO().findByDelayedConsentId(Long.parseLong(request.getParameter("delayedConsentId")));
            session.setAttribute("delayedConsent", dc);
        }
        else if (request.getParameter("consentId") != null) {
            ConsentDirective consent = this.getConsentDirectiveJpaController().findConsentDirective(Long.parseLong(request.getParameter("consentId")));
            session.setAttribute("consent", consent);
        }

		this.forward(request, response, "printLetter");

    }

    public class ConsentWrapper {

        private ConsentDirectiveReferenceType normalConsent;
        private DelayedConsent delayedConsent;
        private Boolean isRevoked;
        private String user;
        private List<ConsentComment> comments = new ArrayList<ConsentComment>();
        private Boolean hasOptInDocumentAttachment;
        private Boolean hasOptOutDocumentAttachment;
        private List<MailNotification> mailNotifications = new ArrayList<MailNotification>();

        public ConsentWrapper(ConsentDirectiveReferenceType normalConsent, boolean isRevoked) {
            this.normalConsent = normalConsent;
            this.isRevoked = isRevoked;
        }

        public ConsentWrapper(DelayedConsent delayedConsent) {
            this.delayedConsent = delayedConsent;
            this.user = delayedConsent.getUserId();
        }

        public Date entryDate() {
            if (normalConsent == null) {
                return delayedConsent.getDateAdded();
            } else if (isRevoked) {
                return normalConsent.getOptoutTS();
            } else {
                return normalConsent.getOptinTS();
            }
        }

        public List<ConsentComment> getComments() {
            return this.comments;
        }

        public List<MailNotification> getMailNotifications() {
            return this.mailNotifications;
        }

        public DelayedConsent getDelayedConsent() {
            return this.delayedConsent;
        }

        public ConsentDirectiveReferenceType getNormalConsent() {
            return this.normalConsent;
        }


        public void setOptInDocumentAttachment(boolean hasAttachment) {
            this.hasOptInDocumentAttachment = hasAttachment;
        }

        public boolean getOptInDocumentAttachment() {
            return this.hasOptInDocumentAttachment;
        }

        public void hasOptOutDocumentAttachment(boolean hasAttachment) {
            this.hasOptOutDocumentAttachment = hasAttachment;
        }

        public boolean getOptOutDocumentAttachment() {
            return this.hasOptOutDocumentAttachment;
        }

        public void setUser(String user) {
            this.user = user;
        }

        public String getUser() {
            return this.user;
        }

        public boolean getIsRevoked(){
            return this.isRevoked;
        }

        public String getConsentDisplayName() {
            String displayName;
            if (normalConsent == null) {
                displayName = delayedConsent.getConsentTypeId().getName();
            } else if (isRevoked) {
                displayName = normalConsent.getOptoutConsentType().value();
            } else {
                displayName = normalConsent.getOptinConsentType().value();
            }
            return displayName.replace("NwHIN", Constants.getOrganizationName());
        }

    }

    /**
     * Base method that handles requests that don't specify another method
     *
     * @param request http servlet request
     * @param response http servlet response
     *
     * @throws javax.servlet.ServletException
     * @throws java.io.IOException
     */
	@Override
	protected void unspecified(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		final HttpSession session = request.getSession(false);

		final String icn = (String) session.getAttribute("icn");

		if (!NullChecker.isNullOrEmpty(icn)) {
			// Store the request id in the session
			this.setRequestId(request, response);
			try {
                List<ConsentWrapper> consents = new ArrayList<ConsentWrapper>();
                List<Long> consentIds = new ArrayList<Long>();

                // Get consent directives for current patient
                List<ConsentDirectiveReferenceType> consentDirectiveReferences = this.getCmsHelper().getConsentDirectiveHistory(icn);

                // get CONSENT_AUDIT info for this patient
                List<Object> auditedConsentsUnordered = this.getAuditedConsentDAO().getResultsByPatientId(icn);

                // make a copy of auditedConsentsUnordered so that the copy will match the consentDirectiveReferences
                // according to the UserIds in auditedConsentsUnordered
                // but first make sure there is one auditedConsent for each consentDirectiveReferences
                ArrayList<AuditedConsentEx> auditedConsents = new ArrayList<AuditedConsentEx>();
                for (ConsentDirectiveReferenceType cdr : consentDirectiveReferences) {
                    // find the userId in auditedConsentsUnordered for this consentDirectiveReference
                    ConsentWrapper wrapperOut = null, wrapperIn = null;
                    //check whether already found optin/optout for this
                    Boolean foundOptIn = false;
                    Boolean foundOptOut = false;
                    //check to see if consent is opted out
                    if (cdr.getOptoutTS() != null) {
                        wrapperOut = new ConsentWrapper(cdr, true);
                    }else{
                        foundOptOut = true;
                    }
                    wrapperIn = new ConsentWrapper(cdr, false);
                    //go through audits
                    for (int i = 0; i < auditedConsentsUnordered.size(); i++) {
                        AuditedConsentEx ac = (AuditedConsentEx)auditedConsentsUnordered.get(i);
                        String consentType = ac.getConsentType();
                        String cdrOptInConsentType =
                            (cdr.getOptinConsentType() != null) ? cdr.getOptinConsentType().value() : "null";
                        String cdrOptOutConsentType =
                            (cdr.getOptoutConsentType() != null) ? cdr.getOptoutConsentType().value() : "null";
                        if (consentType.equals(cdrOptInConsentType) && !foundOptIn){
                             wrapperIn.setUser(ac.getUserId());
                             foundOptIn= true;
                             auditedConsentsUnordered.remove(i);
                             i--;
                        }
                        else if (consentType.equals(cdrOptOutConsentType) && !foundOptOut && wrapperOut != null) {
                            foundOptOut = true;
                            wrapperOut.setUser(ac.getUserId());
                            auditedConsentsUnordered.remove(i);
                             i--;
                        }
                        if(foundOptIn && foundOptOut){
                            break;
                        }
                    }
                    if(wrapperOut != null){
                        consents.add(wrapperOut);
                    }
                    consents.add(wrapperIn);
                    consentIds.add(Long.parseLong(cdr.getConsentDirId()));
                }

                //now that we have the consent Ids for this patient, let's get the comments for them
                if (!consentIds.isEmpty()) {
                    List<ConsentComment> comments = this.getConsentCommentDAO().getCommentsByConsentIds(consentIds);
                    for (ConsentComment comment : comments) {
                        //There is no Restriction Modification consent type so we need to change it to Authorization in order to be
                        //populated correctly in the jsp.
                        if (comment.getConsentType().equals("NwHIN Organization Restriction Modification")) {
                            comment.setConsentType(ConsentType.NW_HIN_ORGANIZATION_RESTRICTION_AUTHORIZATION.value());
                        }
                        for (ConsentWrapper consentWrapper : consents) {
                            if (comment.getConsentId().toString().equals(consentWrapper.normalConsent.getConsentDirId())) {
                                consentWrapper.comments.add(comment);
                            }
                        }
                    }
                }

                //get mail_notification data and add it to the ConsentWrapper for "normal" eHealth Exchange Revocations
                if (!consentIds.isEmpty()) {
                    List<MailNotification> mails = this.getMailNotificationDAO().findByConsentDirIds(consentIds);
                    for (MailNotification mail : mails) {
                        for (ConsentWrapper consentWrapper : consents) {
                            if(mail.getConsentDirId().getConsentDirId().toString().equals(consentWrapper.normalConsent.getConsentDirId())) {
                                consentWrapper.mailNotifications.add(mail);
                            }
                        }
                    }
                }

                //Get any delayed consent authorizations for current patient
                List<DelayedConsent> delayedConsentReferences = this.getDelayedConsentDAO().findByPatientIen(icn);

                DelayedConsent NwHINactiveDelay = null;
                DelayedConsent SSAactiveDelay = null;
                List<Long> delayedConsentIds = new ArrayList<Long>();

                for (DelayedConsent dc : delayedConsentReferences) {
                    consents.add(new ConsentWrapper(dc));
                    delayedConsentIds.add(dc.getDelayedConsentId());

                    //Populate any active delays
                    if (dc.getStatus().equals("PENDING")) {
                        if (dc.getConsentTypeId().getName().contains("NwHIN")) {
                            NwHINactiveDelay = dc;
                        }
                        else if (dc.getConsentTypeId().getName().contains("SSA")) {
                            SSAactiveDelay = dc;
                        }
                    }
                }
                if (delayedConsentIds.size() > 0) {
                    List<ConsentComment> delayedComments = this.getConsentCommentDAO().getCommentsByDelayedConsentIds(delayedConsentIds);
                    for (ConsentComment comment : delayedComments) {
                        for (ConsentWrapper consentWrapper : consents) {
                            if (consentWrapper.delayedConsent != null) {
                                if (comment.getDelayedConsentId().equals(consentWrapper.delayedConsent.getDelayedConsentId())) {
                                    consentWrapper.comments.add(comment);
                                }
                            }
                        }
                    }
                }

                //get mail_notification data and add it to the ConsentWrapper for Delayed Consents
                try {
                    List<MailNotification> mails = this.getMailNotificationDAO().findByDelayedConsentIds(delayedConsentIds);
                    for (MailNotification mail : mails) {
                        for (ConsentWrapper consentWrapper : consents) {
                            if(consentWrapper.delayedConsent != null) {
                                if(mail.getDelayedConsentId().getDelayedConsentId().equals(consentWrapper.delayedConsent.getDelayedConsentId()))
                                consentWrapper.mailNotifications.add(mail);
                            }
                        }
                    }
                } catch (Exception e) {
                    throw new ServletException(e);
                }

                session.setAttribute("NwHINactiveDelay", NwHINactiveDelay);
                session.setAttribute("SSAactiveDelay", SSAactiveDelay);

                // Irakli Kakushadze 11/25/2014 JAZZ 442142 -
                // Get the list of announcements for this patient by consent directive
                // ID list so that we can change caption of "Announce" button to "Re-Announce"
                // if the patient has already been announced.
                String lastAnnounceMessage = "";
                if (consentIds.isEmpty()) {
                    session.setAttribute("hasBeenAnnounced", false);
                } else {
                    List<Announcement> announcements = this.patientAnnouncer.getAnnouncementsByConsentDirectiveId(consentIds);
                    session.setAttribute("hasBeenAnnounced", !announcements.isEmpty());

                    if (!announcements.isEmpty()) {
                        // Check to see if this patient was announced successfully.
                        for (int i = announcements.size() - 1; i >= 0; i--) {
                            if (announcements.get(i).getAnnouncementResult() != null) {
                                if (Integer.parseInt(announcements.get(i).getAnnouncementResult()) > 0) {
                                    if (announcements.get(i).getCompletedTs() != null) {
                                        lastAnnounceMessage = "Last successful announcement: " +
                                            new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(announcements.get(i).getCompletedTs());
                                        break;
                                    }
                                }
                            }
                        }
                        if (lastAnnounceMessage.length() == 0) {
                            // There are no successful announcements, let's show the last failed announcement, if any.
                            for (int i = announcements.size() - 1; i >= 0; i--) {
                                if (announcements.get(i).getAnnouncementResult() != null) {
                                    if (announcements.get(i).getCompletedTs() != null) {
                                        lastAnnounceMessage = "No successful announcements. Last failed announcement: " +
                                            new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(announcements.get(i).getCompletedTs());
                                        break;
                                    }
                                }
                            }
                        }
                        if (lastAnnounceMessage.length() == 0) {
                            // There are no completed announcements, let's show the last scheduled announcement, if any.
                            for (int i = announcements.size() - 1; i >= 0; i--) {
                                if (announcements.get(i).getCreatedTs()!= null) {
                                    lastAnnounceMessage = "No completed announcements. Last scheduled announcement: " +
                                        new SimpleDateFormat("yyyy-MM-dd HH:mm:ss").format(announcements.get(i).getCreatedTs());
                                    break;
                                }
                            }
                        }
                    }
                }
                session.setAttribute("lastAnnouncement", lastAnnounceMessage);

				// Get the demographic details section
				final PatientDemographicsQuery patientDemographicsQuery = new PatientDemographicsQuery();
				patientDemographicsQuery.setPatientId(icn);
				final PatientDemographicsResponse patientDemographicsResponse = this.pdqService
						.getPatientDemographics(patientDemographicsQuery);
				final PatientDemographics demographics = patientDemographicsResponse
						.getPatientDemographics();
				session.setAttribute("patientDemographics", demographics);
				session.setAttribute("isOrganizationExclusionsAllowed", this
						.getCmsHelper().isOrganizationExclusionsAllowed());
				session.setAttribute("isFormUploadAllowed", this.getCmsHelper()
						.isFormUploadEnabled());

				session.setAttribute("maxFileUploadSize", this
						.getDocumentHelper().getMaxUploadFileSize());
				// Get the allowed organizations to list them on the patient
				// details page
                final ArrayList<OrganizationType> allowedOrganizations = new ArrayList<OrganizationType>(this.getCmsHelper().getAllowedNonConsumerOnlyOrganizations());
				//final Collection<OrganizationType> allowedOrganizations = this.getCmsHelper().getAllowedNonConsumerOnlyOrganizations();
				session.setAttribute("allowedOrganizations",allowedOrganizations);

                final ArrayList<gov.va.nvap.svc.facility.data.Facility> allowedFacilities = new ArrayList<gov.va.nvap.svc.facility.data.Facility>(this.getFacilityHelper().getAllowedVistAFacilities());
				//final Collection<gov.va.nvap.svc.facility.data.Facility> allowedFacilities = this.getFacilityHelper().getAllowedVistAFacilities();
				session.setAttribute("allowedFacilities", allowedFacilities);
				// Get the Default Facility by user id
				/*
				final gov.va.nvap.svc.facility.data.Facility defaultUserFacility = this
						.getFacilityHelper().getFacilityStationByVHAUserId(
								UserHelper.getUserName(request));
				if (NullChecker.isNotEmpty(defaultUserFacility)) {
					session.setAttribute("defaultUserFacility",
							defaultUserFacility);
				}
				*/
				final gov.va.nvap.svc.facility.data.Facility defaultUserFacility =
					(gov.va.nvap.svc.facility.data.Facility) session.getAttribute("defaultUserFacility");

				// Get all the consent directives - needed to populate the
				// consent management history
//				final List<ConsentDirectiveReferenceType> consentDirectiveReferences = this
//						.getCmsHelper().getConsentDirectiveHistory(icn);
				// Get the active consent directive to display before history
				ConsentDirectiveReferenceType activeConsentDirectiveReferenceType = null;
				ConsentDirectiveReferenceType ssaActiveConsentDirectiveReferenceType = null;
				ConsentDirectiveReferenceType nwhinOrgRestrictionsActiveConsentDirectiveReferenceType = null;
				if (NullChecker.isNotEmpty(consentDirectiveReferences)) {
					for (final ConsentDirectiveReferenceType type : consentDirectiveReferences) {
						if (NullChecker.isEmpty(type.getOptoutTS())
								&& ConsentType.NW_HIN_AUTHORIZATION.value()
										.equals(type.getOptinConsentType()
												.value())
								&& type.getExpirationDate().after(new Date())) {
							activeConsentDirectiveReferenceType = type;
						}
						if (NullChecker.isEmpty(type.getOptoutTS())
								&& ConsentType.SSA_AUTHORIZATION.value()
										.equals(type.getOptinConsentType()
												.value())
								&& type.getExpirationDate().after(new Date())) {
							ssaActiveConsentDirectiveReferenceType = type;
						}
						if (NullChecker.isEmpty(type.getOptoutTS())
								&& ConsentType.NW_HIN_ORGANIZATION_RESTRICTION_AUTHORIZATION
										.value().equals(
												type.getOptinConsentType()
														.value())) {
							nwhinOrgRestrictionsActiveConsentDirectiveReferenceType = type;
						}
					}
				}

                //Populate any attachments for normal consents.
                for (final ConsentWrapper consent : consents) {
                    if (consent.delayedConsent == null) {
                        try {
                            final String optinDocument = this.getCmsHelper().retrieveDocument(consent.normalConsent.getConsentDirId(), "Authorize");

                            if (DASUtil.isDASDocument(optinDocument)) {
                                consent.setOptInDocumentAttachment(NullChecker.isNotEmpty(optinDocument));
                            } else {
                                final String optinAttachment = optinDocument == null ? null : this.getDocumentHelper().getPrivacyConsentDirectiveAttachment(optinDocument);
                                consent.setOptInDocumentAttachment(NullChecker.isNotEmpty(optinAttachment));
                            }

                            if (NullChecker.isNotEmpty(consent.normalConsent.getOptoutTS())) {
                                final String optoutDocument = this.getCmsHelper().retrieveDocument(consent.normalConsent.getConsentDirId(), "Revoke");

                                if (DASUtil.isDASDocument(optoutDocument)) {
                                    consent.hasOptOutDocumentAttachment(NullChecker.isNotEmpty(optoutDocument));
                                } else {
                                    final String optoutAttachment = optoutDocument == null ? null : this.getDocumentHelper().getPrivacyConsentDirectiveAttachment(optoutDocument);
                                    consent.hasOptOutDocumentAttachment(NullChecker.isNotEmpty(optoutAttachment));
                                }

                            } else {
                                consent.hasOptOutDocumentAttachment(false);
                            }
                        } catch (final TransformerException ex) {
                            throw new ServletException(ex);
                        } catch (final XPathException ex) {
                            throw new ServletException(ex);
                        }
                    }
                }

//                //Sort the combined consent references by date added.
//                Collections.sort(consents, new Comparator<ConsentWrapper>() {
//                    @Override
//                    public int compare(ConsentWrapper o1, ConsentWrapper o2) {
//                        return o2.entryDate().compareTo(o1.entryDate());
//                    }
//                });

				session.setAttribute("consents", consents);

				// if there is a active consent directive, then get the excluded
				// organizations
				if (nwhinOrgRestrictionsActiveConsentDirectiveReferenceType != null) {
					final Collection<OrganizationType> excludedOrganizations = nwhinOrgRestrictionsActiveConsentDirectiveReferenceType
							.getExcludedOrganizations();
					final List<OrganizationType> authorizedOrganizations = new ArrayList<OrganizationType>();
					for (final OrganizationType organization : allowedOrganizations) {
						boolean isExcluded = false;
						for (final OrganizationType excludedOrganization : excludedOrganizations) {
							if (organization.getOrgNumber().equals(
									excludedOrganization.getOrgNumber())) {
								isExcluded = true;
							}
						}
						if (!isExcluded) {
							authorizedOrganizations.add(organization);
						}
					}
                    //cast it to ArrayList because that is serializable and Fortify won't complain
                    ArrayList<OrganizationType> exOrg = new ArrayList<OrganizationType>(excludedOrganizations);

					// Set in session
					session.setAttribute("excludedOrganizations", exOrg);
					session.setAttribute("authorizedOrganizations",
							authorizedOrganizations);
				}
				// Must be opt-out
				session.setAttribute("consentDirective",
						activeConsentDirectiveReferenceType);
				session.setAttribute("ssaConsentDirective",
						ssaActiveConsentDirectiveReferenceType);

				session.setAttribute("isNewAuthorizationAllowed", true);
				session.setAttribute("isSSANewAuthorizationAllowed", true);
				session.setAttribute("nwhinOrganizationRestrictionsDirective",
						nwhinOrgRestrictionsActiveConsentDirectiveReferenceType);

				session.setAttribute("optOutReasons", this.getCmsHelper()
						.getOptoutReasons());

				// Get the correlations section
				final PatientCorrelationsQuery patientCorrelationsQuery = new PatientCorrelationsQuery();
				patientCorrelationsQuery.setPatientId(icn);
				// Get the correlations
				final PatientCorrelationsResponse patientCorrelationsResponse = this.pdqService
						.getCorrelatedFacilities(patientCorrelationsQuery);
				final List<Facility> facilities = patientCorrelationsResponse
						.getFacilities();
				final List<Map<String, String>> correlations = new ArrayList<Map<String, String>>();
				if (NullChecker.isNotEmpty(facilities)) {
					for (final Facility facility : facilities) {
						final String correlatedPatientId = facility
								.getCorrelatedPatientId();
						final String correlatedFacilityId = facility
								.getFacilityId();
						final OrganizationType organization = this
								.getCmsHelper().getOrganizationByNumber(
										correlatedFacilityId);

						if (NullChecker.isNotEmpty(organization)) {
							final Map<String, String> correlation = new HashMap<String, String>();
							final String facilityName = organization
									.getOrgName();
							final String facilityNumber = organization
									.getOrgNumber();
							final String orgOid = organization.getOrgOid();
							correlation.put("organizationOid", orgOid);

							correlation.put("facilityName", facilityName);
							final String oid = facility
									.getAssigningAuthorityNumber();
							correlation.put("assigningAuthorityOid", oid);
							correlation.put("facilityNumber", facilityNumber);
							correlation.put("correlatedPatientId",
									correlatedPatientId);
							correlations.add(correlation);
						}
					}
				}

				// Set the correlations int he session
				session.setAttribute("correlations", correlations);

                                // get the comments for the patient detail section
                                final String patientIen = demographics.getIcn();
                                session.setAttribute("patientIen", patientIen);
                                List<PatientComment> patientComments = this.getPatientCommentDAO().getCommentsByPatientIen(patientIen);
                                session.setAttribute("patientComments", patientComments);

			} catch (final PdqException ex) {
				throw new ServletException(ex);
			}
			// SUCCESS
			this.forward(request, response, "show");
		} else {
			this.forward(request, response, "noshow");
		}
	}

    /**
     * Handles consent directive file upload
     *
     * @param fileItems list of uploaded file items
     * @param request http servlet request
     * @param response http servlet response
     *
     * @throws javax.servlet.ServletException
     * @throws java.io.IOException
     */
	public void upload(final List<FileItem> fileItems,
			final HttpServletRequest request, final HttpServletResponse response)
			throws ServletException, IOException {
		// Get the fields
		final Map<String, String> formFields = FileUploadUtil
				.getFormFields(fileItems);
		// Get the uploaded files
		final List<FileItem> filesUploaded = FileUploadUtil
				.getFilesUploaded(fileItems);
		if (NullChecker.isNotEmpty(filesUploaded)) {
			// Get the document type and the consent directive Id
			final String documentType = formFields.get("documentType");
			final String consentDirId = formFields.get("consentDirId");
			final String userId = UserHelper.getUserName(request);
			final String userFacilityStationId = this.getFacilityHelper()
					.getFaciltyStationIdByUserId(userId);
			this.getCmsHelper().updateDocument(consentDirId, documentType,
					filesUploaded, userId, userFacilityStationId);
		}
		this.forward(request, response, "success");
	}

	/**
	 * View the C32 from the adapter. It makes a webservice request to the
	 * adapter, pulls the data from local VistA systems and displays it to the
	 * user.
     *
     * @param request http servlet request
     * @param response http servlet response
     *
     * @throws javax.servlet.ServletException
     * @throws java.io.IOException
	 */
	public void view(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		final HttpSession session = request.getSession(false);

		final String icn = (String) session.getAttribute("icn");
		if (NullChecker.isNotEmpty(icn)) {
			try {
				// Get the C32 document from the adapter
				String document = this.getDocumentHelper().getC32Document(icn,
						UserHelper.getUserName(request));
				if (NullChecker.isEmpty(document)) {
					throw new ServletException(
							"There was an issue retrieving health summary for this patient with ICN:"
									+ icn);
				}
				// Remove xml-stylesheet
				document = document
						.replaceAll(
								"<?xml-stylesheet type=\"text/xsl\" href=\"CCD.xsl\"?>",
								"");
				// Get the style sheets
				final String styleSheetViewType = request
						.getParameter("styleSheetViewType");
				if ("xml".equals(styleSheetViewType)) {
					response.setContentType("text/xml;charset=UTF-8");
					response.getWriter().write(document);
				} else {
					// Update session in the document
					final String html = this.getDocumentHelper().getHtml(
							document, styleSheetViewType);
					session.setAttribute("updatedDocument", html);
					session.setAttribute("document", document);
					session.setAttribute("isXmlViewEnabled", this
							.getDocumentHelper().isXmlViewEnabled());
					this.forward(request, response, "showdoc");
				}
			} catch (final AdapterException ex) {
				throw new ServletException(ex);
			}
		} else {
			this.forward(request, response, "noshow");
		}

	}

        public void addComment(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
            final String comments = request.getParameter("comments");
            final String consentType = request.getParameter("consentType");
            final String consentId = request.getParameter("consentId");
            final String delayedConsentId = request.getParameter("delayedConsentId");
            final String userId = UserHelper.getUserName(request);

            if(NullChecker.isNotEmpty(comments)) {
                //handle comments
                ConsentComment comment = new ConsentComment();
                comment.setComments(comments);
                comment.setConsentType(consentType);
                comment.setUserId(userId);
                comment.setDateAdded(new Date());

                //Determine if comment is for a delayed or normal authorization
                if (NullChecker.isNullOrEmpty(consentId)) {
                    comment.setDelayedConsentId(Long.parseLong(delayedConsentId));
                }
                else {
                    comment.setConsentId(Long.parseLong(consentId));
                }


                this.getConsentCommentDAO().create(comment);

                // Success
                this.forward(request, response, "success");
            } else {
                // Validate
                this.forward(request, response, "validate");
            }
        }

        public void addPatientComment(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
            final String comments = request.getParameter("comments");
            final String patientIen = request.getParameter("patientIen");
            final String userId = UserHelper.getUserName(request);

            if(NullChecker.isNotEmpty(comments)) {
                //handle comments
                PatientComment comment = new PatientComment();
                comment.setUserId(userId);
                comment.setDateAdded(new Date());
                comment.setComments(comments);
                comment.setPatientIen(patientIen);

                this.getPatientCommentDAO().create(comment);

                // Success
                this.forward(request, response, "success");
            } else {
                // Validate
                this.forward(request, response, "validate");
            }
        }

	public void authorizeNwHIN(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		final HttpSession session = request.getSession(false);

		final String icn = (String) session.getAttribute("icn");
		if (NullChecker.isNotEmpty(icn)) {
		} else {
		}
        session.setAttribute("delayedNwHINAuth", null);
		this.forward(request, response, "authorizeNwHIN");
	}

    public void authorizeDelayNwHIN(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		final HttpSession session = request.getSession(false);

		final String icn = (String) session.getAttribute("icn");
		session.setAttribute("delayedNwHINAuth", true);
		this.forward(request, response, "authorizeDelayNwHIN");
	}

    public void authorizeDelaySSA(final HttpServletRequest request,
        final HttpServletResponse response) throws ServletException,
        IOException {
        final HttpSession session = request.getSession(false);
        final String icn = (String) session.getAttribute("icn");
        session.setAttribute("delayedSSAAuth", true);
        this.forward(request, response, "authorizeDelaySSA");
    }

	public void revokeNwHIN(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		final HttpSession session = request.getSession(false);

		final String icn = (String) session.getAttribute("icn");
		if (NullChecker.isNotEmpty(icn)) {
		} else {
		}
		this.forward(request, response, "revokeNwHIN");
	}

	public void authorizeSSA(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		final HttpSession session = request.getSession(false);

		final String icn = (String) session.getAttribute("icn");
		if (NullChecker.isNotEmpty(icn)) {
		} else {
		}
        session.setAttribute("delayedSSAAuth", null);
		this.forward(request, response, "authorizeSSA");
	}

	public void revokeSSA(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		final HttpSession session = request.getSession(false);

		final String icn = (String) session.getAttribute("icn");
		if (NullChecker.isNotEmpty(icn)) {
		} else {
		}
		this.forward(request, response, "revokeSSA");
	}

	public void authorizeNwHINOrganizationRestrictions(
			final HttpServletRequest request, final HttpServletResponse response)
			throws ServletException, IOException {
		final HttpSession session = request.getSession(false);

		final String icn = (String) session.getAttribute("icn");
		if (NullChecker.isNotEmpty(icn)) {
		} else {
		}
		this.forward(request, response,
				"authorizeNwHINOrganizationRestrictions");
	}

	public void revokeNwHINOrganizationRestrictions(
			final HttpServletRequest request, final HttpServletResponse response)
			throws ServletException, IOException {
		final HttpSession session = request.getSession(false);

		final String icn = (String) session.getAttribute("icn");
		if (NullChecker.isNotEmpty(icn)) {
		} else {
		}
		this.forward(request, response, "revokeNwHINOrganizationRestrictions");
	}

	public void modifyNwHINOrganizationRestrictions(
			final HttpServletRequest request, final HttpServletResponse response)
			throws ServletException, IOException {
		final HttpSession session = request.getSession(false);

		final String icn = (String) session.getAttribute("icn");
		if (NullChecker.isNotEmpty(icn)) {
		} else {
		}
		this.forward(request, response, "modifyNwHINOrganizationRestrictions");
	}

        public void addCommentForm(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
            final HttpSession session = request.getSession(false);

            final String consentId = request.getParameter("consentId");
            final String delayedConsentId = request.getParameter("delayedConsentId");
            final String consentType = request.getParameter("consentType");
            session.setAttribute("consentId", consentId);
            session.setAttribute("delayedConsentId", delayedConsentId);
            session.setAttribute("consentType", consentType);

            this.forward(request, response, "addCommentForm");
	}

        public void addPatientCommentForm(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException {
            final HttpSession session = request.getSession(false);

            final String patientIen = request.getParameter("patientIen");
            session.setAttribute("patientIen", patientIen);

            this.forward(request, response, "addPatientCommentForm");
	}

    public void editMailDatesForm(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException, ParseException {
        final HttpSession session = request.getSession(false);

        final String delayedConsentId = request.getParameter("delayedConsentId");
        final String consentDirId = request.getParameter("consentDirId");
        List<MailNotification> mailNotifications = new ArrayList<MailNotification>();

        if(NullChecker.isNullOrEmpty(delayedConsentId)) {
            //get mail notifications by consentDirId
            mailNotifications = this.getMailNotificationDAO().findByConsentDirId(consentDirId);
        } else {
            //get mail notifications by delayedConsentId
            mailNotifications = this.getMailNotificationDAO().findByDelayedConsentId(delayedConsentId);
        }

        //TODO: JPS - get the logs for this by consentDirId or delayedConsentId

        session.setAttribute("mailNotifications", mailNotifications);
        session.setAttribute("delayedConsentId", delayedConsentId);
        session.setAttribute("consentDirId", consentDirId);

        this.forward(request, response, "editMailDatesForm");
	}

    public void editMailDates(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException, ParseException, Exception {
        final HttpSession session = request.getSession(false);

        //get everything from the form
        final String delayedConsentId = request.getParameter("delayedConsentId");
        final String consentDirId = request.getParameter("consentDirId");
        final String[] newMailDates = NullChecker.isNullOrEmpty(request.getParameterValues("mailDate")) ? new String[0] : request.getParameterValues("mailDate");
        SimpleDateFormat formatter = new SimpleDateFormat("MM/dd/yyyy");
        final String userId = UserHelper.getUserName(request);

        List<MailNotification> existingMailNotifications = new ArrayList<MailNotification>();

        if(NullChecker.isNullOrEmpty(delayedConsentId)) {
            //get mail notifications by consentDirId
            existingMailNotifications = this.getMailNotificationDAO().findByConsentDirId(consentDirId);
        } else {
            //get mail notifications by delayedConsentId
            existingMailNotifications = this.getMailNotificationDAO().findByDelayedConsentId(delayedConsentId);
        }

        //make a list of the old dates for the log and do the delete of all of these at the same time
        ArrayList<Date> oldDates = new ArrayList<Date>();
        for(MailNotification em : existingMailNotifications) {
            oldDates.add(em.getSentDate());

            this.getMailNotificationDAO().delete(em);
        }

        //make a list of all the new dates for the log and do the insert of all of these at the same time
        ArrayList<Date> newDates = new ArrayList<Date>();
        for(String nmd : newMailDates) {
            if(nmd.length() > 0) {
                newDates.add(formatter.parse(nmd));

                MailNotification mn = new MailNotification();
                Date dt = formatter.parse(nmd);
                mn.setSentDate(dt);
                mn.setUserId(userId);

                if(NullChecker.isNullOrEmpty(delayedConsentId)) {
                    ConsentDirective cd = this.getConsentDirectiveJpaController().findConsentDirective(Long.parseLong(consentDirId));
                    mn.setConsentDirId(cd);
                } else {
                    DelayedConsent dc = this.getDelayedConsentDAO().findByDelayedConsentId(Long.parseLong(delayedConsentId));
                    mn.setDelayedConsentId(dc);
                }

                this.getMailNotificationDAO().create(mn);
            }
        }

        //get the log entry
        String logEntry = this.getLogEntry(oldDates, newDates, userId);

        //save the log entry
        if(!logEntry.isEmpty()) {
            MailLog ml = new MailLog();
            ml.setLogEntry(logEntry);
            if(NullChecker.isNullOrEmpty(delayedConsentId)) {
                Long conDir = Long.parseLong(consentDirId);
                ml.setConsentDirId(conDir);
            } else {
                Long delDir = Long.parseLong(delayedConsentId);
                ml.setDelayedConsentId(delDir);
            }

            this.getMailLogDAO().create(ml);
        }

        this.forward(request, response, "success");
    }

    public void getLogEntries(final HttpServletRequest request, final HttpServletResponse response) throws ServletException, IOException, ParseException, Exception {
        final HttpSession session = request.getSession(false);

        final String delayedConsentId = request.getParameter("delayedConsentId");
        final String consentDirId = request.getParameter("consentDirId");

        List<MailLog> logs = new ArrayList<MailLog>();

        if(NullChecker.isNullOrEmpty(delayedConsentId)) {
            //get mail logs by consentDirId
            logs = this.getMailLogDAO().findByConsentDirId(consentDirId);
        } else {
            //get mail logs by delayedConsentId
            logs = this.getMailLogDAO().findByDelayedConsentId(delayedConsentId);
        }

        session.setAttribute("logs", logs);

        this.forward(request, response, "listMailLogs");
    }

    private String getLogEntry(ArrayList<Date> oldDates, ArrayList<Date> newDates, String userId) throws ParseException {
        String log = "";
        boolean added = false;
        boolean deleted = false;
        SimpleDateFormat finalDate = new SimpleDateFormat("MM/dd/yyyy");
        SimpleDateFormat occuredDateFormatter = new SimpleDateFormat("MM/dd/yyyy HH:mm:ss");
        List<Date> oldDatesToUse = new ArrayList<Date>(oldDates);
        List<Date> newDatesToUse = new ArrayList<Date>(newDates);

        for (int i = oldDatesToUse.size() - 1; i >= 0; i--) {
            for (int j = newDatesToUse.size() - 1; j >=0; j--) {
                String oldDate = finalDate.format(oldDatesToUse.get(i));
                String newDate = finalDate.format(newDatesToUse.get(j));
                if (oldDate.equals(newDate)) {
                    oldDates.remove(i);
                    newDates.remove(j);
                }
            }
        }

        if (oldDates.size() == 0 && newDates.size() > 0) {
            // Some new date(s) were added.
            added = true;
        } else if (oldDates.size() > 0 && newDates.size() == 0) {
            // Some date(s) were deleted.
            deleted = true;
        } else if (oldDates.size() > 0 && newDates.size() > 0 ) {
            // Some date(s) were added and some deleted.
            added = true;
            deleted = true;
        } else {
            // Nothing has changed.
        }

        if (added) {
            log += "added notification date" + (newDates.size() > 1 ? "s " : " ");

            List<String> newDatesAsStrings = new ArrayList<String>();
            for(Date date : newDates) {
                newDatesAsStrings.add(finalDate.format(date));
            }

            log += newDatesAsStrings.toString().replaceAll("\\[|\\]", "");
        }

        if (added && deleted) {
            log += " and ";
        }

        if (deleted) {
            // Some date(s) were deleted.
            log += "deleted notification date" + (oldDates.size() > 1 ? "s " : " ");

            List<String> oldDatesAsStrings = new ArrayList<String>();
            for(Date date : oldDates) {
                oldDatesAsStrings.add(finalDate.format(date));
            }
            log += oldDatesAsStrings.toString().replaceAll("\\[|\\]", "");
        }

        if (log.length() > 0) {
            Date occuredOn = new Date();
            String occuredOnFormatted = occuredDateFormatter.format(occuredOn);
            log = "On " + occuredOnFormatted + " CT, user " + userId + " " + log + ".";
        }

        return log;
    }

    /*
        This method commented out because it should only be used in dev/test.
        These types of test harnesses should be implemented using interfaces
        rather than conditional logic that will always return false in production.
    */
    /*
    private void mockGetDocumentFromDAS(ServletOutputStream ostream) throws IOException {
        InputStream is =
   			 this.getClass().getClassLoader().getResourceAsStream("gov/va/nvap/web/das.txt");

        final char[] buffer = new char[bufferSize];
        final StringBuilder out = new StringBuilder();
        try {
            Reader in = new InputStreamReader(is, "UTF-8");
            for (;;) {
            int rsz = in.read(buffer, 0, buffer.length);
            if (rsz < 0)
                break;
            out.append(buffer, 0, rsz);
          }
        }
        catch (UnsupportedEncodingException ex) {
        }
        catch (IOException ex) {
        }
        String jsonString = out.toString();


        ostream.close();
        is.close();

        String xmlId = "";
        int xmlLength = 0;
        String pdfId = "";
        int pdfLength = 0;
        try {
            JSONObject jObject = new JSONObject(jsonString);
            xmlId = jObject.getString("gridfsID");
            xmlLength = jObject.getInt("gridfsLength");
            pdfId = ((JSONObject)jObject.getJSONArray("gridfsAttachments").get(0)).getString("gridfsID");
            pdfLength = ((JSONObject)jObject.getJSONArray("gridfsAttachments").get(0)).getInt("gridfsLength");
        } catch (JSONException ex) {
            Logger.getLogger(PatientDetails.class.getName()).log(Level.SEVERE, null, ex);
        }

        is = this.getClass().getClassLoader().getResourceAsStream("gov/va/nvap/web/das.pdf");
        byte[] buf = new byte[bufferSize];
        int c = 0;
        while ((c = is.read(buf, 0, buf.length)) > 0) {
        	ostream.write(buf, 0, c);
        	ostream.flush();
        }
    }
    */

    public void generateLetter(final List<FileItem> fileItems, final HttpServletRequest request, final HttpServletResponse response)
        throws ServletException, IOException {

        final SimpleDateFormat letterFormat = new SimpleDateFormat("MM/dd/yyyy");
        final HttpSession session = request.getSession(false);
        final String type = FileUploadUtil.getFormFields(fileItems).get("type");
        InputStream inputStream = null;
        String templateString = "";

        // Get the fields
		final HashMap<String, String> formFields = (HashMap<String, String>) FileUploadUtil.getFormFields(fileItems);
        ArrayList<HashMap<String,String>> patientDetails = new ArrayList<HashMap<String,String>>();
        Boolean mailed = Boolean.parseBoolean(FileUploadUtil.getFormFields(fileItems).get("markMailed"));
        gov.va.nvap.svc.facility.data.Facility facility = this.getFacilityHelper().getFacilityByStationId(formFields.get("authenticatingFacility"));
        //Get and parse delayed fields.
        if (type.equals("delayed")) {
            if (session.getAttribute("ssaDelay") != null) {
                session.removeAttribute("ssaDelay");
                inputStream = IOUtils.toInputStream(this.getMailTemplateDAO().getByLetterType(3L).getText(), "UTF-8");
            }
            else if (session.getAttribute("eheDelay") != null) {
                session.removeAttribute("eheDelay");
                inputStream = IOUtils.toInputStream(this.getMailTemplateDAO().getByLetterType(2L).getText(), "UTF-8");
            }

            DelayedConsent dc = (DelayedConsent)session.getAttribute("delayedConsent");
            try {

                templateString = new Scanner(inputStream,"UTF-8").useDelimiter("\\A").next();
            }
            finally {
                if(inputStream != null) {
                    inputStream.close();
                }
            }
            formFields.put("[entryDate]", letterFormat.format(dc.getDateAdded()));
            //Parse the delayed reasons.
            String delayReasonString = "";
            String[] delayedReasons = formFields.get("delayCollection").split(",");
            for (int i = 0; i < delayedReasons.length; i++) {
                delayedReasons[i] = delayedReasons[i].replaceAll("[^0-9]+", "");
                delayedReasons[i] = this.getDelayReasonDAO().findByDelayReasonId(Long.parseLong(delayedReasons[i])).getName();
            }
            for (int i = 0; i < delayedReasons.length; i++) {
                if (i == 4) {
                    delayReasonString += ",\n                    ";
                } else if (i > 0) {
                    delayReasonString += ", ";
                }
                delayReasonString += delayedReasons[i];
            }
            templateString = templateString.replace("[reasonsForDelay]", delayReasonString);
        }

        //Get and parse revoked fields.
        else if (type.equals("revoked")) {
            inputStream = IOUtils.toInputStream(this.getMailTemplateDAO().getByLetterType(1L).getText(), "UTF-8");
            try {
                templateString = new Scanner(inputStream,"UTF-8").useDelimiter("\\A").next();
            }
            finally {
                if(inputStream != null) {
                    inputStream.close();
                }
            }
            ConsentDirective consent = (ConsentDirective)session.getAttribute("consent");
            formFields.put("[purpose]", consent.getPurposeOfUse().getPouValue());
            formFields.put("[optoutDate]", letterFormat.format(consent.getOptoutDate()));
            formFields.put("[expirationDate]", letterFormat.format(consent.getExpirationDate()));
        }

        //Get and parse expiring fields.
        else if (type.equals("expiring")) {
            inputStream = IOUtils.toInputStream(this.getMailTemplateDAO().getByLetterType(4L).getText(), "UTF-8");
            try {
                templateString = new Scanner(inputStream, "UTF-8").useDelimiter("\\A").next();
            }
            finally {
                if(inputStream != null) {
                    inputStream.close();
                }
            }
            ConsentDirective consent = (ConsentDirective)session.getAttribute("consent");
            if (!NullChecker.isNullOrEmpty(consent)) {
                if (consent.getOptinConsentType().getName().equals("SSA Authorization")) {
                    formFields.put("[consentDescription]", "Authorized access to Social Security Administration");
                    formFields.put("[consentName]", consent.getOptinConsentType().getName());
                } else if (consent.getOptinConsentType().getName().equals("NwHIN Authorization")) {
                    formFields.put("[consentDescription]", "Authorized access to Providers and Organizations");
                    formFields.put("[consentName]", "eHealth Exchange Authorization");
                }
                formFields.put("[purpose]", consent.getPurposeOfUse().getPouValue());
                formFields.put("[entryDate]", letterFormat.format(consent.getOptinTS()));
                formFields.put("[expirationDate]", letterFormat.format(consent.getExpirationDate()));
            }
        }

        if(inputStream != null) {
            inputStream.close();
        }

        //Get patient details
        PatientDemographics patientDemographics = (PatientDemographics) session.getAttribute("patientDemographics");
        formFields.put("[date]", new SimpleDateFormat("dd MMM, yyyy").format(new Date()));
        formFields.put("[firstName]", patientDemographics.getFirstName());
        formFields.put("[lastName]", patientDemographics.getLastName());
        formFields.put("[middleName]", patientDemographics.getMiddleName());
        formFields.put("[facilityName]", facility.getFacilityName());
        String address1 = (facility.getAddress() != null) ? facility.getAddress() : "";
        formFields.put("[facilityAddress1]", address1);
        String address2 = (facility.getCity() != null) ? facility.getCity() + ", " : "";
        address2 += (facility.getState() != null) ? facility.getState() + " " : "";
        address2 += (facility.getPostalCode() != null) ? facility.getPostalCode() : "";
        formFields.put("[facilityAddress2]", address2);
        String phone = (facility.getPhone() != null) ? facility.getPhone() : "";
        if (!NullChecker.isNullOrEmpty(phone)) {
            phone = "(" + phone.substring(0, 3) + ") " + phone.substring(3, 6) + "-" + phone.substring(6);
        }
        formFields.put("[facilityPhone]", phone);

        //Get and format patient address.
        String address = patientDemographics.getStreetAddressLine1() + "\n";
        if (!NullChecker.isNullOrEmpty(patientDemographics.getStreetAddressLine2())) {
            address += patientDemographics.getStreetAddressLine2() + "\n";
        }
        if (!NullChecker.isNullOrEmpty(patientDemographics.getStreetAddressLine3())) {
            address += patientDemographics.getStreetAddressLine3() + "\n";
        }

        address += patientDemographics.getResidenceCity() + ", " + patientDemographics.getResidenceState() + " " +
            patientDemographics.getResidenceZip4();

        templateString = templateString.replace("[patientAddress]", address);

        String sigFields = formFields.get("signature");
        templateString = templateString.replace("[signature]", sigFields);

        patientDetails.add(formFields);

        PdfGenerator pdfGenerator = new PdfGenerator();
        ByteArrayOutputStream pdfStream = pdfGenerator.create(templateString, patientDetails, type);

        response.setContentType("application/pdf");
        response.setContentLength(pdfStream.size());
        response.setHeader("Content-Transfer-Encoding", "binary");
        response.addHeader("Content-Disposition", "attachment; filename=patient_letter.pdf");
        response.setHeader("pragma", "empty");
        response.setHeader("Cache-Control", "must-revalidate");

        try {
            final ServletOutputStream os = response.getOutputStream();
            Assert.assertNotEmpty(os, "Output Stream cannot be null!");
            os.write(pdfStream.toByteArray());
            os.flush();
            os.close();
            if (mailed) {
                MailNotification mN = new MailNotification();
                MailLog mL = new MailLog();
                mN.setSentDate(new Date());
                mN.setUserId(UserHelper.getUserName(request));
                ArrayList<Date> date = new ArrayList<Date>();
                date.add(new Date());
                try {
                    String log = getLogEntry(new ArrayList<Date>(), date, UserHelper.getUserName(request));
                    mL.setLogEntry(log);
                }
                catch (ParseException ex) {
                    Logger.getLogger(PatientDetails.class.getName()).log(Level.SEVERE, null, ex);
                }
                if (type.equals("delayed")) {
                    DelayedConsent dc = (DelayedConsent)session.getAttribute("delayedConsent");
                    mN.setDelayedConsentId(dc);
                    mL.setDelayedConsentId(dc.getDelayedConsentId());
                    try {
                        this.getMailLogDAO().create(mL);
                        this.getMailNotificationDAO().create(mN);
                    }
                    catch (Exception ex) {
                    }
                }
                else {
                    ConsentDirective consent = (ConsentDirective)session.getAttribute("consent");
                    mN.setConsentDirId(consent);
                    mL.setConsentDirId(consent.getConsentDirId());
                    try {
                        this.getMailLogDAO().create(mL);
                        this.getMailNotificationDAO().create(mN);
                    }
                    catch (Exception ex) {
                    }
                }

            }
        }
        catch (final IOException ex) {
			throw new ServletException(ex);
        }
    }

 private void getDocumentFromDAS(ServletOutputStream ostream, String externalDocumentId) throws IOException{
    // Build and send initial DAS request
    String dasRetrieveDocumentEndPoint =  Constants.getDasRetrieveDocumentEndPoint();
    String dasRetrieveOneDocumentEndPoint =  Constants.getDasRetrieveOneDocumentEndPoint();
    String dasRetrieveDocumentUrlWithDocId =  dasRetrieveDocumentEndPoint + externalDocumentId;

    //HttpClient client = HttpClientBuilder.create().build();
    //HttpGet request = new HttpGet(dasRetrieveDocumentUrlWithDocId);
    URL url = new URL(dasRetrieveDocumentUrlWithDocId);
    Logger.getLogger(PatientDetails.class.getName()).log(Level.INFO, "DAS URL: "+dasRetrieveDocumentUrlWithDocId);
    HttpsURLConnection conn = null;
    conn = (HttpsURLConnection) url.openConnection();
    conn.connect();

    //added the response message on failure as well, because it can help with debugging potentially
    if (conn.getResponseCode() != 200) {
        throw new RuntimeException("\nFailed : HTTP error code : " + conn.getResponseCode() + "\nHTTP response : " + conn.getResponseMessage());
    }
    InputStream inputStream =  conn.getInputStream();
    // HttpResponse response = client.execute(request);
    // HttpEntity entity = response.getEntity();

    // Read DAS response to JSON string
    if (inputStream != null) {

        final char[] buffer = new char[bufferSize];
        final StringBuilder out = new StringBuilder();
        try {
            Reader in = new InputStreamReader(inputStream, "UTF-8");
            for (;;) {
            int rsz = in.read(buffer, 0, buffer.length);
            if (rsz < 0)
                break;
            out.append(buffer, 0, rsz);
          }
        }
        catch (UnsupportedEncodingException ex) {
            Logger.getLogger(PatientDetails.class.getName()).log(Level.SEVERE, null, ex);
        }
        catch (IOException ex) {
            Logger.getLogger(PatientDetails.class.getName()).log(Level.SEVERE, null, ex);
        }
        String jsonString = out.toString();
        inputStream.close();

        // Extract eAuth PDF ID from JSON DAS response
        String pdfId = "";
        int pdfLength = 0;
        try {
            JSONObject jObject = new JSONObject(jsonString);
            pdfId = ((JSONObject)jObject.getJSONArray("gridfsAttachments").get(0)).getString("gridfsID");
            pdfLength = ((JSONObject)jObject.getJSONArray("gridfsAttachments").get(0)).getInt("gridfsLength");
        } catch (JSONException ex) {
            Logger.getLogger(PatientDetails.class.getName()).log(Level.SEVERE, null, ex);
        }

        if (pdfLength <= 0) {
            Logger.getLogger(PatientDetails.class.getName()).log(Level.WARNING,
                    "eAuth PDF length is not greater than zero");
        }

        dasRetrieveDocumentUrlWithDocId =  dasRetrieveOneDocumentEndPoint + pdfId;
         Logger.getLogger(PatientDetails.class.getName()).log(Level.INFO, "DAS Doc URL: "+dasRetrieveDocumentUrlWithDocId);
        url = new URL(dasRetrieveDocumentUrlWithDocId);
        conn = (HttpsURLConnection) url.openConnection();
        conn.connect();

        //added the response message on failure as well, because it can help with debugging potentially
        if (conn.getResponseCode() != 200) {
            throw new RuntimeException("\nFailed : HTTP error code : " + conn.getResponseCode() + "\nHTTP response : " + conn.getResponseMessage());
        }
        InputStream is =  conn.getInputStream();
        if (is != null) {
            // Print PDF
            byte[] buf = new byte[bufferSize];
            int c;
            while ((c = is.read(buf, 0, buf.length)) > 0) {
                ostream.write(buf, 0, c);
                ostream.flush();
            }
            is.close();
        } else {
            Logger.getLogger(PatientDetails.class.getName()).log(Level.WARNING, "No inner PDF returned");
        }
    } else {
        Logger.getLogger(PatientDetails.class.getName()).log(Level.WARNING, "No documents returned");
    }
 }
}