package gov.va.nvap.web.patient;

import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.OrganizationType;
import gov.va.nvap.service.pdq.PatientDemographics;
import gov.va.nvap.service.pdq.PatientDemographicsQuery;
import gov.va.nvap.service.pdq.PatientDemographicsResponse;
import gov.va.nvap.service.pdq.PdqException;
import gov.va.nvap.service.pdq.PdqService;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.PatientAnnouncer;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.AnnouncePatientRequest;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.AnnouncePatientResponse;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.AuthorizedOrganizationType;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.PatientType;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.UserType;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import java.util.Collection;
import java.util.Date;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 * Class to create and run a separate thread for single patient announcements
 * 
 * @author Zack Peterson
 */
public class PatientAnnounceThread implements Runnable{
    
    private static final Logger logger = 
            Logger.getLogger(PatientAnnounceThread.class.getName());
    
    private ConsentManagementHelper cmsHelper;
    private FacilityHelper facilityHelper;
    private PdqService pdqService;
    private PatientAnnouncer patientAnnouncer;
    private String remoteUserId;
    private String patientIcn;
    
    private Thread thread;
    private final String threadName;
    
    PatientAnnounceThread(String name) {
        this.threadName = name;
    }

    /**
     * Patient announce code for the new thread
     */
    @Override
    public void run() {
        // Get the opted-in organizations
        final Collection<OrganizationType> organizations = 
                this.cmsHelper.getAllowedNonConsumerOnlyOrganizations();
        // Announce to each one of them
        if (NullChecker.isNotEmpty(organizations)) {
            try {
                boolean announceError = false;
                final String userId = this.remoteUserId;
                final String userFacilityStationId = 
                        this.facilityHelper.getFaciltyStationIdByUserId(userId);
                // Add the authorized organization
                String message = "Announcement to Veteran with VA identifier "
                        + patientIcn
                        + " initiated by user id "
                        + userId
                        + " from facility "
                        + userFacilityStationId
                        + ".\n\nAnnounce Results:\n ";

                final AnnouncePatientRequest announcePatientRequest = new AnnouncePatientRequest();
                announcePatientRequest.setCreatedDate(new Date());
                final PatientDemographicsQuery query = new PatientDemographicsQuery();
                query.setPatientId(patientIcn);
                final PatientDemographicsResponse patientDemographicsResponse = this.pdqService
                        .getPatientDemographics(query);
                final PatientDemographics demogprahics = patientDemographicsResponse
                        .getPatientDemographics();
                final PatientType patientType = new PatientType();
                patientType.setPatientId(patientIcn);
                patientType.setGivenName(demogprahics.getFirstName());
                patientType.setMiddleName(demogprahics.getMiddleName());
                patientType.setLastName(demogprahics.getLastName());
                patientType.setSsn(demogprahics.getSsn());
                announcePatientRequest.setPatient(patientType);

                final UserType userType = new UserType();
                userType.setUserId(userId);
                userType.setFacilityNumber(userFacilityStationId);
                announcePatientRequest.setUser(userType);

                for (final OrganizationType organization : organizations) {
                    final AuthorizedOrganizationType authorizedOrganization = new AuthorizedOrganizationType();
                    authorizedOrganization.setOrgNumber(organization
                            .getOrgNumber());
                    authorizedOrganization.setOrgName(organization
                            .getOrgName());
                    authorizedOrganization.setOrgOid(organization
                            .getOrgOid());
                    announcePatientRequest.getAuthorizedOrganizations()
                            .add(authorizedOrganization);
                }

                try {
                    final AnnouncePatientResponse announcePatientResponse = 
                            this.patientAnnouncer.announceAll(announcePatientRequest);
                    if (announcePatientResponse.getResult() > 0) {
                        // Announcement to Veteran with identifier :
                        // <id>
                        message = message + "SUCCCESS! [Response Code:"
                                + announcePatientResponse.getResult()
                                + "]";

                    } else {
                        message = message + "FAILURE! [Response Code:"
                                + announcePatientResponse.getResult()
                                + "]";
                        announceError = true;
                    }

                } catch (final Exception ex) {
                    announceError = true;
                    message = message
                            + "FAILURE! Exception Thrown - Reason: "
                            + ex.getMessage();
                }

                if (announceError) {
                    // Throw an exception with the message
                    logger.log(Level.INFO, "Announce Error: {0}", message);
                }

            } catch (final PdqException ex) {
                throw new RuntimeException(ex);
            }
        }
    }

    /**
     * Creates new thread and starts it
     */
    public void start() {
        if (thread == null) {
            thread = new Thread(this, threadName);
            thread.start();
        }
    }
    
    public void setCmsHelper(ConsentManagementHelper cmsHelper) {
        this.cmsHelper = cmsHelper;
    }

    public void setFacilityHelper(FacilityHelper facilityHelper) {
        this.facilityHelper = facilityHelper;
    }

    public void setPdqService(PdqService pdqService) {
        this.pdqService = pdqService;
    }

    public void setPatientAnnouncer(PatientAnnouncer patientAnnouncer) {
        this.patientAnnouncer = patientAnnouncer;
    }

    public void setRemoteUserId(String remoteUserId) {
        this.remoteUserId = remoteUserId;
    }

    public void setPatientIcn(String patientIcn) {
        this.patientIcn = patientIcn;
    }
    
}
