package gov.va.nvap.web.patient;

import gov.va.med.nhin.adapter.audit.ActionValuesType;
import gov.va.med.nhin.adapter.audit.AuditSummariesType;
import gov.va.med.nhin.adapter.audit.AuditSummaryType;
import gov.va.med.nhin.adapter.audit.AuditType;
import gov.va.med.nhin.adapter.audit.FieldType;
import gov.va.med.nhin.adapter.audit.GetAudits;
import gov.va.med.nhin.adapter.audit.GetAuditsResponse;
import gov.va.med.nhin.adapter.audit.GetAuditsSummary;
import gov.va.med.nhin.adapter.audit.GetAuditsSummaryResponse;
import gov.va.med.nhin.adapter.audit.GroupByFieldsType;
import gov.va.med.nhin.adapter.audit.PageInfoType;
import gov.va.med.nhin.adapter.audit.SortDirection;
import gov.va.med.nhin.adapter.audit.SortFieldType;
import gov.va.med.nhin.adapter.audit.SortFieldsType;
import gov.va.med.nhin.adapter.audit.StringValuesType;
import gov.va.med.nhin.adapter.audit.SummaryFieldsType;
import gov.va.nvap.common.date.GregorianDateUtil;
import gov.va.nvap.common.sort.BubbleSortListMap;
import gov.va.nvap.common.transformer.TransformerException;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.common.xpath.XPathException;
import gov.va.nvap.privacy.OrganizationType;
import gov.va.nvap.service.adapter.audit.AdapterAuditManager;
import gov.va.nvap.service.adapter.audit.DirectAuditManager;
import gov.va.nvap.service.adapter.audit.DirectServiceException;
import gov.va.nvap.service.audit.AuditException;
import gov.va.nvap.svc.consentmgmt.stub.dao.DelayedConsentReportDAO;
import gov.va.nvap.svc.consentmgmt.stub.dao.MailNotificationDAO;
import gov.va.nvap.svc.consentmgmt.stub.dao.PatientConsentDirDAO;
import gov.va.nvap.svc.consentmgmt.stub.data.DelayReason;
import gov.va.nvap.svc.consentmgmt.stub.data.DelayedConsentRpt;
import gov.va.nvap.svc.consentmgmt.stub.data.DetailedConsentDirective;
import gov.va.nvap.svc.consentmgmt.stub.data.MailNotification;
import gov.va.nvap.web.consent.audit.AuditedConsentEx;
import gov.va.nvap.web.consent.audit.dao.AuditedConsentDAO;
import gov.va.nvap.web.dao.UserDocumentDAO;
import gov.va.nvap.web.helper.document.DocumentHelper;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.helper.report.ReportHelper;
import gov.va.nvap.web.report.ReportDataProcessor;
import gov.va.nvap.web.user.document.UserDocument;
import gov.va.nvap.web.util.Constants;
import gov.va.nvap.web.util.FieldChecks;
import gov.va.nvap.web.util.xls.CsvExporter;
import gov.va.nvap.web.util.xls.ExcelExporter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.servlet.ServletException;
import org.apache.commons.lang.StringUtils;
import org.apache.poi.ss.usermodel.Workbook;

/**
 * Class to create and run a separate thread for generating and storing Excel and CSV files
 *
 * @author Zack Peterson
 */
public class ExcelGeneratorThread implements Runnable{

    private static final Logger logger =
            Logger.getLogger(ExcelGeneratorThread.class.getName());

    private String title;
    private String userId;
    private Map<String, String> reportMap;
    private Map<String, List<Object>> filterMap;
    private boolean hasOptionalRows = false;
    private boolean isSummaryReport = false;
    private ExcelExporter exExport;
    private CsvExporter csvExport;
    private AdapterAuditManager adapterAuditManager;
    private DirectAuditManager directAuditManager;
    private ReportHelper reportHelper;
    private FacilityHelper facilityHelper;
    private ConsentManagementHelper cmsHelper;
    private DocumentHelper documentHelper;
    private UserDocumentDAO userDocDao;
    private AuditedConsentDAO auditedConsentDAO;
    private PatientConsentDirDAO patientConsentDirDAO;
    private DelayedConsentReportDAO delayedConsentReportDAO;
    private MailNotificationDAO mailNotificationDAO;
    private Map<String, Object> attributes;
    private Map<String, String> optionalRowsMap;
    private String reportSource;
    private String documentType;

    // Total counters
    private int realPatientMessages = 0;
    private int realPatientFails = 0;
    private int realPatientMatches = 0;
    private int testPatientMessages = 0;
    private int testPatientFails = 0;
    private int testPatientMatches = 0;
    private long totalCount = 0l;
    private int uniqueRealPatients = 0;
    private int uniqueTestPatients = 0;

    private Thread thread;
    private final String threadName;

    public ExcelGeneratorThread(String name) {
        this.threadName = name;
    }

    /**
     * Excel generation code for the new thread
     */
    @Override
    public void run() {
        if ("csv".equals(documentType)) {
            logger.log(Level.SEVERE, "EXPORT_CSV_SCHEDULED_BEGIN_" + Thread.currentThread().getId() + " " + System.currentTimeMillis());
        } else {
            logger.log(Level.SEVERE, "EXPORT_EXCEL_SCHEDULED_BEGIN_" + Thread.currentThread().getId() + " " + System.currentTimeMillis());
        }
        // Create initial incomplete user document row
        // Using document date to later retrieve document since ID is generated
        Date documentDate = new Date();
        UserDocument userDoc = new UserDocument(title, documentDate, userId, null, "In Progress");
        if (documentType != null && documentType.equals("csv")) {
            userDoc.setFormat("CSV");
        } else {
            userDoc.setFormat("Excel");
        }
        try {
            userDocDao.create(userDoc);
        } catch (Exception ex) {
            logger.log(Level.SEVERE, "User document already exists: ", ex);
        }

        List<Map<String, Object>> results;
        try {
            // Get the results from the adapter
            if (isSummaryReport) {
                results = getSummaryResults();
            } else if (!NullChecker.isNullOrEmpty(auditedConsentDAO)) {
                try {
                    results = getAuditedConsentResults();
                } catch (ServletException ex) {
                    logger.log(Level.SEVERE, "Failed to retrieve Audited Consent data: ", ex);
                    markDocumentFailure(documentDate);
                    return;
                }
            } else if (!NullChecker.isNullOrEmpty(patientConsentDirDAO)) {
                results = getPatientConsentDirResults();
            } else if (!NullChecker.isNullOrEmpty(delayedConsentReportDAO)) {
                results = getDelayedConsentResults();
            } else {
                results = getResults();
            }
        } catch (AuditException ex) {
            logger.log(Level.SEVERE, "Error getting report results: ", ex);
            markDocumentFailure(documentDate);
            return;
        } catch (DirectServiceException ex) {
            logger.log(Level.SEVERE, "Error getting report results: ", ex);
            markDocumentFailure(documentDate);
            return;
        }

        if (documentType != null && documentType.equals("csv")) {
            createAndStoreCsvDocument(results, documentDate);
        } else {
            createAndStoreExcelDocument(results, documentDate);
        }
        if ("csv".equals(documentType)) {
            logger.log(Level.SEVERE, "EXPORT_CSV_SCHEDULED_END_" + Thread.currentThread().getId() + " " + System.currentTimeMillis());
        } else {
            logger.log(Level.SEVERE, "EXPORT_EXCEL_SCHEDULED_END_" + Thread.currentThread().getId() + " " + System.currentTimeMillis());
        }
    }

    private void createAndStoreExcelDocument(List<Map<String, Object>> results, Date documentDate) {
        // Clear report map if there are no results
        if (NullChecker.isNullOrEmpty(results)) {
            reportMap.clear();
            reportMap.put("", "No records were found.");
        }

        Map<String, List<Object>> optionalRows = null;
        if (hasOptionalRows) {
            try {
                optionalRows = generateOptionalRows();
            } catch (NoSuchFieldException | IllegalArgumentException | IllegalAccessException ex) {
                logger.log(Level.SEVERE, "An optional row does not have a corresponding accessible field: ", ex);
            }
        }

        // Create workbook
        final Workbook wb = exExport.exportToExcel(title, title, reportMap, results, filterMap, optionalRows);

        try {
            byte[] docBytes = exExport.getExcelAsBytes(wb);

            // Get incomplete user doc row to fill in content
            UserDocument blankDoc = userDocDao.findByCreationDate(documentDate);
            if (blankDoc != null) {
                blankDoc.setContent(docBytes);
                blankDoc.setStatus("COMPLETED");
                userDocDao.edit(blankDoc);
            } else {
                UserDocument newDoc = new UserDocument(title, new Date(), userId, docBytes, "COMPLETED", "Excel");
                userDocDao.create(newDoc);
            }
        } catch (Exception ex) {
            logger.log(Level.SEVERE, "Error storing Excel export: ", ex);
            markDocumentFailure(documentDate);
        }
    }

    private void createAndStoreCsvDocument(List<Map<String, Object>> results, Date documentDate) {

        StringBuilder sbOutput;
        if (NullChecker.isNullOrEmpty(results)) {
            sbOutput = new StringBuilder();
            sbOutput.append("No records were found.");
        } else {
            SimpleDateFormat dt = new SimpleDateFormat("MM/dd/yyyy");
            // Allocate string builder based on the number of rows, but no larger than 20MB.
            if (results.size() > 100000) {
                sbOutput =  new StringBuilder(20000000);
            } else {
                sbOutput =  new StringBuilder(results.size() * 200);
            }

            // Append column headers
            for (String heading : reportMap.keySet()) {
                sbOutput.append(heading);
                sbOutput.append(",");
            }
            sbOutput.append("\n");

            // Append report data
            for (Map<String, Object> row : results) {
                // Create the row of data
                String separator = "";
                for (String dataField : reportMap.values()) {
                    sbOutput.append(separator);
                    separator = ",";
                    // Include escape character.
                    sbOutput.append("\"");

                    if (NullChecker.isNullOrEmpty(row.get(dataField))) {
                        sbOutput.append("");
                    } else {
                        if (dataField.contains("Date")) {
                            // Format dates. patientDateFormatted is an exception - comes pre-formatted
                            if (dataField.equals("patientDateFormatted")) {
                                sbOutput.append(row.get(dataField));
                            } else {
                                sbOutput.append(row.get(dataField));
                            }
                        } else if (dataField.toLowerCase().contains("oid") && !NullChecker.isNullOrEmpty(row.get(dataField))) {
                            sbOutput.append(ReportHelper.trimOrgOid((String)row.get(dataField)));
                        } else {
                            // Else print the data as-is.
                            sbOutput.append(row.get(dataField));
                        }
                    }
                    // Include closing escape character.
                    sbOutput.append("\"");
                }
                sbOutput.append("\n");
            }
        }

        byte[] docBytes = String.valueOf(sbOutput).getBytes();

        try {
            // Get incomplete user doc row to fill in content
            UserDocument blankDoc = userDocDao.findByCreationDate(documentDate);
            if (blankDoc != null) {
                blankDoc.setContent(docBytes);
                blankDoc.setStatus("COMPLETED");
                userDocDao.edit(blankDoc);
            } else {
                UserDocument newDoc = new UserDocument(title, new Date(), userId, docBytes, "COMPLETED", "CSV");
                userDocDao.create(newDoc);
            }
        } catch (Exception ex) {
            logger.log(Level.SEVERE, "Error storing CSV export: ", ex);
            markDocumentFailure(documentDate);
        }
    }

    /**
     * Creates new thread and starts it
     */
    public void start() {
        if (thread == null) {
            thread = new Thread(this, threadName);
            thread.start();
        }
    }

    private void markDocumentFailure(Date documentDate) {
        UserDocument failedDoc = userDocDao.findByCreationDate(documentDate);
        if (failedDoc != null) {
            failedDoc.setStatus(title);
            try {
                userDocDao.edit(failedDoc);
            } catch (Exception ex) {
                logger.log(Level.SEVERE, "Failed user document does not exist: ", ex);
            }
        }
    }

    // Handle the calculation of optional row totals here.
    private Map<String, List<Object>> generateOptionalRows() throws NoSuchFieldException, IllegalArgumentException, IllegalAccessException {
        Map<String, List<Object>> optionalRows = new LinkedHashMap<>();

        for(Map.Entry<String, String> entry : optionalRowsMap.entrySet()) {
            if(!entry.getKey().isEmpty()) {
                List<Object> optionalRowList = new ArrayList<>();
                optionalRowList.add(getClass().getDeclaredField(entry.getKey()).get(this));
                optionalRows.put(entry.getValue(), optionalRowList);
            } else {
                List<Object> blank = new ArrayList<>();
                optionalRows.put("", blank);
            }
        }

        return optionalRows;
    }

    private List<Map<String, Object>> getResults() throws AuditException, DirectServiceException {
        final String icn = (String) attributes.get("icn");
        final String ssn = (String) attributes.get("ssn");
        final String lastName = (String) attributes.get("lastName");
        final String firstName = (String) attributes.get("firstName");
        final String queryUserId = (String) attributes.get("userId");
        final Date sd = (Date) attributes.get("startDate");
        final Date ed = (Date) attributes.get("endDate");
        final String purposeOfUse = (String) attributes.get("purposeOfUse");
        final String facility = (String) attributes.get("facility");
        final String organization = (String) attributes.get("organization");
        final String remoteOrganization = (String) attributes.get("remoteOrganization");
        final int patientTypes = (Integer) attributes.get("patientTypes");
        String mpiMatch = (String) attributes.get("mpiMatch");
        // sort
        final String sortValue = (String) attributes.get("sortValue");
        final String sortDirection = (String) attributes.get("sortDirection");
        // action values
        final ActionValuesType actionsType = (ActionValuesType) attributes.get("actionsType");

        if (NullChecker.isNotEmpty(mpiMatch) && mpiMatch.equalsIgnoreCase("all")) {
            mpiMatch = null;
        }

        final GetAudits getAuditsRequest = new GetAudits();
        if (NullChecker.isNotEmpty(sd)) {
            getAuditsRequest.setFromDate(GregorianDateUtil
                    .getGregorianCalendarByDate(sd));
        }
        if (NullChecker.isNotEmpty(ed)) {
            getAuditsRequest.setToDate(GregorianDateUtil
                    .getGregorianCalendarByDate(ed));
        }

        // Set Patient Types
        getAuditsRequest.setPatientTypes(patientTypes);

        // Set Patient Ids
        if (NullChecker.isNotEmpty(icn)) {
            final StringValuesType patientIds = new StringValuesType();
            patientIds.getValue().add(icn);
            patientIds.setNotIn(false);
            getAuditsRequest.setPatientIds(patientIds);
        }
        // Set SSN
        if (NullChecker.isNotEmpty(ssn)) {
            String[] ssns = ssn.split(",");
            final StringValuesType patientSSNs = new StringValuesType();
            patientSSNs.getValue().addAll(Arrays.asList(ssns));
            patientSSNs.setNotIn(false);
            getAuditsRequest.setPatientSSNs(patientSSNs);
        }
        // Set Last Name
        if (NullChecker.isNotEmpty(lastName)) {
            final StringValuesType patientLastNames = new StringValuesType();
            patientLastNames.getValue().add(lastName);
            patientLastNames.setNotIn(false);
            getAuditsRequest.setPatientLastNames(patientLastNames);
        }
        // Set First Name
        if (NullChecker.isNotEmpty(firstName)) {
            final StringValuesType patientFirstNames = new StringValuesType();
            patientFirstNames.getValue().add(firstName);
            patientFirstNames.setNotIn(false);
            getAuditsRequest.setPatientGivenNames(patientFirstNames);
        }
        // Set User ID
        if (NullChecker.isNotEmpty(queryUserId)) {
            final StringValuesType userIds = new StringValuesType();
            userIds.getValue().add(queryUserId);
            userIds.setNotIn(false);
            getAuditsRequest.setUserIds(userIds);
        }
        // Set Purpose of Use
		if (!NullChecker.isNullOrEmpty(purposeOfUse) ) {
			final StringValuesType purposeForUses = new StringValuesType();
			purposeForUses.getValue().add(purposeOfUse);
			purposeForUses.setNotIn(false);
			getAuditsRequest.setPurposeForUses(purposeForUses);
		}
        // Set Facility Ids
		if (NullChecker.isNotEmpty(facility)) {
			final StringValuesType facilityIds = new StringValuesType();
			facilityIds.setNotIn(false);
			facilityIds.getValue().add(facility);
			getAuditsRequest.setPatientFacilityNumbers(facilityIds);
		}
        // Set Organization Ids
        if (NullChecker.isNotEmpty(organization)) {
			final StringValuesType organizationIds = new StringValuesType();
			organizationIds.setNotIn(false);
			organizationIds.getValue().add(remoteOrganization);
			getAuditsRequest.setOrganizationIds(organizationIds);
		} else if ("Disclosures Detail Report".equals(title)) {
            // For Disclosure Detailed report:
            // input orgs are empty - intent is to retrieve all orgs except VA
            final StringValuesType organizationIds = new StringValuesType();
            organizationIds.setNotIn(true);
            organizationIds.getValue().add(cmsHelper.getCompleteHomeCommunityId());
            getAuditsRequest.setOrganizationIds(organizationIds);
        }
        // Set Remote Organization Ids
		if (NullChecker.isNotEmpty(remoteOrganization)) {
			final StringValuesType remoteOrganizationIds = new StringValuesType();
			remoteOrganizationIds.setNotIn(false);
			remoteOrganizationIds.getValue().add(remoteOrganization);
			getAuditsRequest.setRemoteOrganizationIds(remoteOrganizationIds);
		}

        // sorting
        if (NullChecker.isNotEmpty(sortValue)) {
            final SortFieldType sortField = new SortFieldType();
            sortField.setField(FieldType.fromValue(sortValue));
            if ("DESC".equalsIgnoreCase(sortDirection)) {
                sortField.setDirection(SortDirection.DESC);
            } else {
                sortField.setDirection(SortDirection.ASC);
            }
            final SortFieldsType sortFieldsType = new SortFieldsType();
            sortFieldsType.getSortField().add(sortField);
            getAuditsRequest.setSortFields(sortFieldsType);
        }

        // Set Action Types
        getAuditsRequest.setActions(actionsType);
        // Set MPI Match
        if ("all".equals(mpiMatch) || NullChecker.isEmpty(mpiMatch)) {
            getAuditsRequest.setDetails(null);
        } else {
            getAuditsRequest.setDetails(mpiMatch);
        }
        // Set Page Information for all results
        final PageInfoType pageInfoType = new PageInfoType();
        pageInfoType.setPageNumber(0);
        if ("direct".equals(reportSource)) {
            pageInfoType.setPageSize(Integer.MAX_VALUE);
        } else {
            pageInfoType.setPageSize(-1);
        }
        getAuditsRequest.setPageInfo(pageInfoType);

        final List<Map<String, Object>> results = new ArrayList<>();
        try {
            GetAuditsResponse queryResponse;
            if ("direct".equals(reportSource)) {
                queryResponse = this.directAuditManager.getAudits(getAuditsRequest);
            } else {
                queryResponse = this.adapterAuditManager.getAudits(getAuditsRequest);
            }

            if (NullChecker.isNotEmpty(queryResponse)
                    && NullChecker.isNotEmpty(queryResponse.getAudits())
                    && NullChecker.isNotEmpty(queryResponse.getAudits().getAudit())) {
                final List<AuditType> auditTypeList = queryResponse.getAudits().getAudit();
                HashSet<String> uniqueRealPatientsSet = new HashSet<>();
                HashSet<String> uniqueTestPatientsSet = new HashSet<>();
                for (final AuditType auditType : auditTypeList) {
                    final Map<String, Object> resultMap = new HashMap<>();
                    String explanationOfFailure = "N/A";
                    String patientFirstName = "";
                    String patientLastName = "";
                    String patientMiddleName = "";
                    String patientSSN = "";
                    Boolean mviFindMatchFailed = null;

                    String details = ReportDataProcessor.nullEmptyReplaceWithUnknown(auditType.getDetails());
                    if (details.startsWith("QUERY FAILED")) {
                        explanationOfFailure = "MVI: Query Failed";
                        mviFindMatchFailed = true;
                    } else if (details.startsWith("MATCH FAILED")) {
                        explanationOfFailure = "MVI: Match Not Found";
                        mviFindMatchFailed = true;
                    } else if (details.startsWith("AMBIGUOUS FAILED")) {
                        explanationOfFailure = "MVI: Ambiguous Match";
                        mviFindMatchFailed = true;
                    } else if (details.startsWith("MATCH FOUND")) {
                        mviFindMatchFailed = false;
                    }
                    resultMap.put("failureExplanation", explanationOfFailure);

                    if (Boolean.TRUE.equals(mviFindMatchFailed)) {
                        patientFirstName = reportHelper.extractFromDetails(details, "FIRSTNAME");
                        patientLastName = reportHelper.extractFromDetails(details, "LASTNAME");
                        patientMiddleName = reportHelper.extractFromDetails(details, "MIDDLENAME");
                        patientSSN = reportHelper.extractFromDetails(details, "SSN");
                    }
                    if (patientSSN.length() == 0) {
                        patientSSN = auditType.getPatientSSN();
                    }

                    // SSN
                    ReportDataProcessor.addSsnToResultMap(resultMap, patientSSN);

                    boolean isTestPatient = reportHelper.isTestPatient(resultMap.get("ssn").toString());

                    if (NullChecker.isNotEmpty(patientSSN)) {
                        if (isTestPatient) {
                            if (!uniqueTestPatientsSet.contains(patientSSN)) {
                                uniqueTestPatientsSet.add(patientSSN);
                            }
                        } else {
                            if (!uniqueRealPatientsSet.contains(patientSSN)) {
                                uniqueRealPatientsSet.add(patientSSN);
                            }
                        }
                    }

                    if (isTestPatient) {
                        testPatientMessages++;
                        if (Boolean.TRUE.equals(mviFindMatchFailed)) {
                            testPatientFails++;
                        } else if (Boolean.FALSE.equals(mviFindMatchFailed)) {
                            testPatientMatches++;
                        }
                    } else {
                        realPatientMessages++;
                        if (Boolean.TRUE.equals(mviFindMatchFailed)) {
                            realPatientFails++;
                        } else if (Boolean.FALSE.equals(mviFindMatchFailed)) {
                            realPatientMatches++;
                        }
                    }

                    // First name.
                    if (patientFirstName.length() == 0) {
                        patientFirstName = ReportDataProcessor.nullEmptyReplaceWithUnknown(auditType.getPatientGivenName());
                    }
                    resultMap.put("firstName", patientFirstName);

                    // Middle name.
                    if (patientMiddleName.length() == 0) {
                        patientMiddleName =  ReportDataProcessor.nullEmptyReplaceWithUnknown(auditType.getPatientMiddleName());
                    }
                    if ("Unknown".equalsIgnoreCase(patientMiddleName)) {
                        patientMiddleName = "";
                    }
                    resultMap.put("middleName", patientMiddleName);

                    // Last name.
                    if (patientLastName.length() == 0) {
                        patientLastName = ReportDataProcessor.nullEmptyReplaceWithUnknown(auditType.getPatientLastName());
                    }
                    resultMap.put("lastName", patientLastName);

                    // Patient preferred facility
                    resultMap.put("patientFacilityName", ReportDataProcessor
							.fixStation(auditType.getPatientFacilityName()));
					resultMap.put("patientFacilityNumber", ReportDataProcessor
							.nullEmptyReplaceWithUnknown(auditType
									.getPatientFacilityNumber()));

                    resultMap.put("organizationName", reportHelper.findOrganizationByOid(auditType.getOrganizationId(), auditType.getOrganizationName()));
                    resultMap.put("facilityOid", ReportDataProcessor
                        .nullEmptyReplaceWithUnknown(auditType
                            .getOrganizationId()));

                    resultMap.put("remoteOrganizationName", reportHelper.findOrganizationByOid(auditType.getRemoteOrganizationId(),
                            auditType.getRemoteOrganizationName()));

                    resultMap.put("remoteFacilityOid", ReportDataProcessor.nullEmptyReplaceWithUnknown(
                            auditType.getRemoteOrganizationId()));

                    resultMap.put("icn", ReportDataProcessor.nullEmptyReplaceWithUnknown(auditType.getPatientId()));
                    resultMap.put("auditTime", GregorianDateUtil.getDateFromGregorianCalendar(auditType.getAuditTime()));
                    resultMap.put("auditTimeFormatted",
                            reportHelper.getFormattedDateTime(
                            GregorianDateUtil
                            .getDateFromGregorianCalendar(auditType
                            .getAuditTime())));

                    String userIdString = ReportDataProcessor.nullEmptyReplaceWithUnknown(auditType.getUserId()).replace(" ? ", " ");
                    resultMap.put("userId", userIdString);

                    //Parse the user id field for facility code and user name only if this is Received Documents Detailed Report
                    if ("RECEIVED EHEALTH EXCHANGE DOCUMENTS DETAIL REPORT".equals(this.title.toUpperCase())) {
                       if (userIdString.length() > 4 && userIdString.substring(0,4).matches("^\\d{3}:")){
                            resultMap.put("userFacilityCode", userIdString.substring(0,3));
                            if (facilityHelper != null) {
                                resultMap.put("userFacilityName", facilityHelper.getFacilityNameByStationId(userIdString.substring(0,3)));
                            }
                        }

                        if (userIdString.indexOf("CN=") > 0){
                            resultMap.put("userName",userIdString.substring((userIdString.indexOf("CN=") + 3),
                                userIdString.indexOf(",", userIdString.indexOf("CN="))));
                        } 
                    }
                    
                    resultMap.put("purposeForUse", ReportDataProcessor
                            .nullEmptyReplaceWithUnknown(auditType
                            .getPurposeForUse()));

                    if (NullChecker.isNotEmpty(auditType.getAction())) {
                        resultMap.put("message", auditType.getAction().name());
                    }

                    details = processDetails(details);

                    resultMap.put("details", details);

                    resultMap.put("documentId", ReportDataProcessor
							.nullEmptyReplaceWithUnknown(auditType
									.getDocumentId()));
					resultMap.put("documentTitle", ReportDataProcessor
							.nullEmptyReplaceWithUnknown(auditType
									.getDocumentTitle()));
                    resultMap.put("roleName", ReportDataProcessor
							.nullEmptyReplaceWithUnknown(auditType
									.getRoleName()));

                    results.add(resultMap);
                }
                uniqueRealPatients = uniqueRealPatientsSet.size();
                uniqueTestPatients = uniqueTestPatientsSet.size();
            }

            return results;

        } catch (final AuditException ex) {
            throw new AuditException();
        } catch (final DirectServiceException ex) {
            throw new DirectServiceException();
        }

    }

    private List<Map<String, Object>> getSummaryResults() throws AuditException {
        final String queryUserId = (String) attributes.get("userId");
        final Date sd = (Date) attributes.get("startDate");
        final Date ed = (Date) attributes.get("endDate");
        final String patientFacility = (String) attributes.get("facility");
        final String organization = (String) attributes.get("organization");
        final String remoteOrganization = (String) attributes.get("remoteOrganization");
        final int patientTypes = (Integer) attributes.get("patientTypes");
        // sort
        final String sortValue = (String) attributes.get("sortValue");
        final String sortDirection = (String) attributes.get("sortDirection");
        // action values
        final ActionValuesType actionsType = (ActionValuesType) attributes.get("actionsType");
        // group by values
        final GroupByFieldsType groupByFieldsType = (GroupByFieldsType) attributes.get("groupByFieldsType");

        final GetAuditsSummary getAuditsSummaryRequest = new GetAuditsSummary();

        // Set start date
        if (NullChecker.isNotEmpty(sd)) {
            getAuditsSummaryRequest.setFromDate(GregorianDateUtil.getGregorianCalendarByDate(sd));
        }

        // Set end date
        if (NullChecker.isNotEmpty(ed)) {
            getAuditsSummaryRequest.setToDate(GregorianDateUtil.getGregorianCalendarByDate(ed));
        }

        // Set user ID
        if (NullChecker.isNotEmpty(queryUserId)) {
            final StringValuesType userIdValue = new StringValuesType();
            userIdValue.getValue().add(queryUserId);
            userIdValue.setNotIn(false);
            getAuditsSummaryRequest.setUserIds(userIdValue);
        }

        // Set patient preferred facility
        if (!NullChecker.isNullOrEmpty(patientFacility)) {
            final StringValuesType patientFacilyNumber = new StringValuesType();
            patientFacilyNumber.setNotIn(false);
            final List<String> patientFacilyNumberList = patientFacilyNumber.getValue();
            patientFacilyNumberList.add(patientFacility);
            getAuditsSummaryRequest.setPatientFacilityNumbers(patientFacilyNumber);
        }

        // Set partner organization
        if (NullChecker.isNotEmpty(organization)) {
            final StringValuesType orgTypes = new StringValuesType();
            orgTypes.getValue().add(organization);
            getAuditsSummaryRequest.setOrganizationIds(orgTypes);
        }

        if (NullChecker.isNotEmpty(remoteOrganization)) {
            final StringValuesType remoteOrgTypes = new StringValuesType();
            remoteOrgTypes.getValue().add(remoteOrganization);
            getAuditsSummaryRequest.setRemoteOrganizationIds(remoteOrgTypes);
        }

        // Set patient types
        getAuditsSummaryRequest.setPatientTypes(patientTypes);

        // Set action types
        getAuditsSummaryRequest.setActions(actionsType);

        // Group by
        getAuditsSummaryRequest.setGroupByFields(groupByFieldsType);

        List<Map<String, Object>> results = null;
        try {
            // Get summary list from Exchange
            GetAuditsSummaryResponse getAuditsSummaryResponse = this.adapterAuditManager.getAuditSummary(getAuditsSummaryRequest);

            // Construct results for the jsp page
            results = new ArrayList<>();
            final AuditSummariesType auditSummariesType = getAuditsSummaryResponse.getAuditSummaries();

            List<AuditSummaryType> auditSummary = new ArrayList<>();
            if (auditSummariesType != null) {
                auditSummary = auditSummariesType.getAuditSummary();
            }

            for (final AuditSummaryType auditSummaryType : auditSummary) {
                final Map<String, Object> resultMap = new HashMap<>();
                final long count = auditSummaryType.getCount();
                final SummaryFieldsType summaryFields = auditSummaryType.getSummaryFields();
                final List<String> summaryField = summaryFields.getSummaryField();

                if (NullChecker.isEmpty(summaryField)) {
                    continue;
                }

                if (summaryField.size() < 4) {
                    continue;
                }

                // Partner organization
                OrganizationType organizationType = null;
                if (NullChecker.isNotEmpty(summaryField.get(0))) {
                    organizationType = (cmsHelper.getOrganizationByHomeCommunityId(summaryField.get(0).replace("urn:oid:", "")));
                }
                if (organizationType != null) {
                    resultMap.put("partnerOrg", ReportDataProcessor.nullEmptyReplaceWithUnknown(organizationType.getOrgName()));
                } else {
                    resultMap.put("partnerOrg", ReportDataProcessor.nullEmptyReplaceWithUnknown(summaryField.get(3)));
                }

                // Patient preferred facility
                final String adapterVAFacility = summaryField.get(1);
                String vaFacility;
                if (ReportDataProcessor.isValidStation(adapterVAFacility)) {
                    vaFacility = facilityHelper.getFacilityNameByStationId(adapterVAFacility);
                    if (NullChecker.isEmpty(vaFacility)) {
                        vaFacility = summaryField.get(2);
                    }
                } else {
                    vaFacility = summaryField.get(2);
                }
                resultMap.put("vaFacility", ReportDataProcessor.fixStation(vaFacility));

                // Audits count
                resultMap.put("auditsCount", count);

                // The rest of the values are simply copied from auditSummary
                resultMap.put("countUniqueRealPatients", StringUtils.isNumeric(summaryField.get(4)) ? Long.parseLong(summaryField.get(4)) : 0);
                resultMap.put("countUniqueTestPatients", StringUtils.isNumeric(summaryField.get(5)) ? Long.parseLong(summaryField.get(5)) : 0);
                resultMap.put("matchesFoundRealPatients", StringUtils.isNumeric(summaryField.get(6)) ? Long.parseLong(summaryField.get(6)) : 0);
                resultMap.put("matchesFoundTestPatients", StringUtils.isNumeric(summaryField.get(7)) ? Long.parseLong(summaryField.get(7)) : 0);
                resultMap.put("matchFailsRealPatients", StringUtils.isNumeric(summaryField.get(8)) ? Long.parseLong(summaryField.get(8)) : 0);
                resultMap.put("matchFailsTestPatients", StringUtils.isNumeric(summaryField.get(9)) ? Long.parseLong(summaryField.get(9)) : 0);

                results.add(resultMap);
                totalCount += count;
            }

            // Sort
            if (NullChecker.isNotEmpty(sortValue) && NullChecker.isNotEmpty(sortDirection)) {
                boolean ascending = !sortDirection.equalsIgnoreCase("DESC");
                final BubbleSortListMap bSorter = new BubbleSortListMap();
                results = bSorter.sortByColumn(results, sortValue, ascending);
            }
        } catch (final AuditException ex) {
            throw new AuditException(ex);
        }

        return results;
    }

    private List<Map<String, Object>> getAuditedConsentResults() throws ServletException {
        final AuditedConsentDAO.DetailRequest req = auditedConsentDAO.new DetailRequest();

        req.patientFirstName = (String) attributes.get("firstName");
        req.patientLastName = (String) attributes.get("lastName");
        req.patientSsn = (String) attributes.get("ssn");
        req.stationNumbers = (String) attributes.get("stationNumbers");
        req.consentType = (String) attributes.get("consentType");
        req.inactivationReason = (String) attributes.get("inactivationReason");
        req.startDate = (Date) attributes.get("startDate");
        req.endDate = (Date) attributes.get("endDate");
        req.patientTypes = (Integer) attributes.get("patientTypes");
        req.userId = (String) attributes.get("userId");
        req.includeUnknownVisn = (Boolean) attributes.get("includeUnknownVisn");
        req.sortField = (String) attributes.get("sortValue");
        req.sortDirection = (String) attributes.get("sortDirection");
        req.currentPage = 0;
        req.pageSize = -1;

        List<String> actionTypes = (List<String>) attributes.get("actionTypes");

        // The result of the query is a list of arrays.
        // Each array is a [AuditedConsent, Facility, Visn]
        List<Object> auditedConsentExList = auditedConsentDAO.getEvents(req);

        final List<Map<String, Object>> results = new ArrayList<>();
        if (NullChecker.isNotEmpty(auditedConsentExList)) {
            for (final Object o : auditedConsentExList) {
                AuditedConsentEx auditType = (AuditedConsentEx) o;

                final Map<String, Object> resultMap = new HashMap<>();

                // Replace SSN with Mask
                ReportDataProcessor.addSsnToResultMap(resultMap, auditType.getPatientSsn());

                resultMap.put("icn", auditType.getPatientId());
                resultMap.put("auditTime", auditType.getTimeOfEvent());
                resultMap.put("auditTimeFormatted", reportHelper
                    .getFormattedDateTime(auditType.getTimeOfEvent()));
                resultMap.put("userId", auditType.getUserId());
                resultMap.put("actionType", auditType.getActionType());

                String consentTypeRow = auditType.getConsentType();
                if (consentTypeRow.contains("NwHIN")) {
                    consentTypeRow = consentTypeRow.replace("NwHIN", Constants.getOrganizationName());
                }
                resultMap.put("consentType", consentTypeRow);
                resultMap.put("firstName", auditType.getPatientGivenName());
                resultMap.put("lastName", auditType.getPatientLastName());
                resultMap.put("purposeOfUse", auditType.getPouValue());

                String inactivationReasonRow = auditType.getOptoutReason();
                resultMap.put("inactivationReason", inactivationReasonRow);
                resultMap.put("patientDate", auditType.getCreatedDate());
                resultMap.put("patientDateFormatted", reportHelper
                    .getFormattedDate(auditType.getCreatedDate()));

                resultMap.put("facilityName", ReportDataProcessor
                    .fixStation(auditType.getFacilityName()));
                resultMap.put("facilityId", auditType.getFacility());
                resultMap.put("visnName", NullChecker.isNullOrEmpty(auditType.getVisnName()) ? "Unknown" : auditType.getVisnName());

                if (actionTypes.contains(auditType.getActionType())) {
                    resultMap.put("auditId", auditType.getConsentAuditId());
                }
                results.add(FieldChecks.replaceEmptyOrNullWithSpace(resultMap));
            }
        }

        return results;
    }

    public List<Map<String, Object>> getPatientConsentDirResults() {

        Date startDate = (Date) attributes.get("startDate");
        Date endDate = (Date) attributes.get("endDate");
        final PatientConsentDirDAO.SearchRequest searchRequest = patientConsentDirDAO.new SearchRequest();

        searchRequest.startDate = startDate;
        searchRequest.endDate = endDate;
        searchRequest.stationNumbers = (String) attributes.get("stationNumbers");
        searchRequest.includeUnknownVisn = (Boolean) attributes.get("includeUnknownVisn");
        searchRequest.consentTypeName = "";
        if (!NullChecker.isNullOrEmpty(attributes.get("consentTypeName"))) {
            searchRequest.consentTypeName = (String) attributes.get("consentTypeName");
        }
        searchRequest.userId = (String) attributes.get("userId");

        final String inboundSortValue = (String) attributes.get("sortValue");
        final String inboundSortDirection = (String) attributes.get("sortDirection");

        final int patientTypes = (Integer) attributes.get("patientTypes");

        searchRequest.patientTypes = patientTypes;

        // For use in sorting results
        if (NullChecker.isNotEmpty(inboundSortValue)) {
            searchRequest.sortBy = inboundSortValue;
        }
        if (NullChecker.isNotEmpty(inboundSortDirection)) {
            searchRequest.sortOrder = inboundSortDirection;
        }

        searchRequest.fromPage = 0;
        searchRequest.recordsPerPage = -1;

        final List<Map<String, Object>> results = new ArrayList<>();
        try {

            final PatientConsentDirDAO.SearchResponse searchResponse = patientConsentDirDAO.find(searchRequest);

            if (searchResponse != null) {

                if ((searchResponse.consents != null)
                    && (searchResponse.consents.size() > 0)) {

                    for (DetailedConsentDirective detailedConsentReference : searchResponse.consents) {
                        final Map<String, Object> resultMap = new HashMap<>();
                        if (detailedConsentReference != null) {
                            ReportDataProcessor.addSsnToResultMap(resultMap, detailedConsentReference.getSsn());
                            resultMap.put("lastName", detailedConsentReference.getLastName());
                            resultMap.put("firstName", detailedConsentReference.getFirstName());
                            resultMap.put("middleName", detailedConsentReference.getMiddleName());
                            String consentTypeName = detailedConsentReference.getOptinConsentType().getName();
                            if (consentTypeName.contains("NwHIN")) {
                                consentTypeName = consentTypeName.replace("NwHIN", Constants.getOrganizationName());
                            }
                            resultMap.put("consentTypeName", consentTypeName);

                            SimpleDateFormat dt = new SimpleDateFormat("MM/dd/yyyy");
                            resultMap.put("optInDate", dt.format(detailedConsentReference.getOptinDate()));
                            resultMap.put("expirationDate", dt.format(detailedConsentReference.getExpirationDate()));

                            resultMap.put("facilityName", detailedConsentReference.getFacilityName());
                            resultMap.put("userId", detailedConsentReference.getUserId());
                            results.add(resultMap);
                        }
                    }
                }
            }
        } catch (final Exception ex) {
            throw new RuntimeException(ex);
        }

        return results;
    }

    public List<Map<String, Object>> getDelayedConsentResults() {
        DelayedConsentReportDAO.SearchAllRequest searchRequest = delayedConsentReportDAO.new SearchAllRequest();

        // Get parameters for the search from the request.
        searchRequest.patientSsn = (String) attributes.get("ssn");
        searchRequest.patientLastName = (String) attributes.get("lastName");
        searchRequest.patientFirstName = (String) attributes.get("firstName");
        searchRequest.stationNumbers = (String) attributes.get("stationNumbers");
        searchRequest.aggregateAtFacilityLevel = false;
        searchRequest.includeUnknownVisn = (Boolean) attributes.get("includeUnknownVisn");
        searchRequest.reasonsForDelay = (String) attributes.get("reasonsForDelay");
        searchRequest.daysSinceDelayed = (String) attributes.get("daysSinceDelayed");
        searchRequest.consentType = (String) attributes.get("consentType");
        searchRequest.start = 0;
        searchRequest.length = -1;
        searchRequest.sortBy = (String) attributes.get("sortValue");
        searchRequest.sortOrder = (String) attributes.get("sortDirection");
        searchRequest.patientTypes = (Integer) attributes.get("patientTypes");
        searchRequest.enteredBy = (String) attributes.get("userId");

        // Perform the search.
        DelayedConsentReportDAO.SearchAllResponse searchResponse = delayedConsentReportDAO.searchAll(searchRequest);

        // Generate report data.
        final List<Map<String, Object>> results = new ArrayList<>();
        for (DelayedConsentRpt r : searchResponse.delayedConsents) {
            final Map<String, Object> resultMap = new HashMap<>();

            resultMap.put("dateAdded", reportHelper.getFormattedDateTime(r.getDateAdded()));
            resultMap.put("icn", r.getPatientIen());
            resultMap.put("ssn", r.getPatientSsn());
            resultMap.put("lastName", r.getPatientLastName());
            resultMap.put("firstName", r.getPatientFirstName());
            resultMap.put("middleName", r.getPatientMiddleName());
            resultMap.put("consentType", this.handleConsentType(r));
            resultMap.put("reasonsForDelay", this.handleDelayReasons(r.getDelayReasonCollection(), (String) attributes.get("reasonsForDelay")));
            resultMap.put("enteredBy", r.getUserId());
            resultMap.put("authenticatingFacility", r.getFacilityName());
            resultMap.put("mailNotifications", this.handleMailNotifications(this.getMailNotificationsByDelayedConsent(r.getDelayedConsentId()),"\r\n"));

            results.add(resultMap);
        }

        return results;
    }

    private String handleConsentType(DelayedConsentRpt delayedConsent) {
        String consentType = "";
        if (!NullChecker.isNullOrEmpty(delayedConsent.getConsentTypeId())) {
            consentType = delayedConsent.getConsentTypeId().getName();
            consentType = ReportHelper.normalizeConsnentTypeName(consentType);
        }
        return consentType;
    }

    private String handleDelayReasons(Collection<DelayReason> delayedReasons, String searchedForReasons) {
        if (!NullChecker.isNullOrEmpty(delayedReasons)) {
            StringBuilder sb = new StringBuilder();

            if (NullChecker.isNullOrEmpty(searchedForReasons) || "ALL".equals(searchedForReasons)) {
                //get em all
                for (DelayReason dr : delayedReasons) {
                    sb.append(dr.getName());
                    sb.append(", ");
                }
            } else {
                //concatenate applicable reasons only
                List<String> reasons = Arrays.asList(searchedForReasons.split("\\s*,\\s*"));

                for (DelayReason dr : delayedReasons) {
                    if (reasons.contains(dr.getDelayReasonId().toString())) {
                        sb.append(dr.getName());
                        sb.append(", ");
                    }
                }
            }

            return sb.length() > 0 ? sb.substring(0, sb.length() - 2) : ""; //get rid of the last comma and space (yeah its clunky and I don't particularly like it)
        } else {
            return "";
        }
    }

    private String handleMailNotifications(Collection<MailNotification> notifications, String delimiter) {
        StringBuilder result = new StringBuilder();
        boolean isFirstItem = true;

        for (MailNotification mn : notifications) {
            result.append(isFirstItem ? "" : delimiter);
            result.append(reportHelper.getFormattedDate(mn.getSentDate()));
            isFirstItem = false;
        }

        return result.toString();
    }

    private Collection<MailNotification> getMailNotificationsByDelayedConsent(Long delayedConsentId) {
        Collection<MailNotification> notifications = this.mailNotificationDAO.findByDelayedConsentId(delayedConsentId.toString());

        return notifications;
    }

    /**
     * Processes Details property as follows:
     * - Masks SSN
     * - Replaces "MOTHERSMAIDENNAME" with "MMN"
     * @param details Details property of AuditType
     * @return Processed Details property string
     */
    private String processDetails(String details) {
        // Mask SSN
        int pos1 = details.indexOf("SSN=");
        int pos2;
        String ssn;
        if (pos1 !=  -1) {
            pos2 = details.indexOf(",", pos1);
            if (pos2 != -1) {
                ssn = details.substring(pos1 + 4, pos2).trim();
                if (ssn.length() > 0) {
                    details = details.substring(0, pos1 + 4) + ReportDataProcessor.maskSsn(ssn) + details.substring(pos2);
                }
            }
        }
        // Replace "MOTHERSMAIDENNAME" with "MMN"
        if (details.contains("MOTHERSMAIDENNAME=")) {
            details = details.replace("MOTHERSMAIDENNAME=", "MMN=");
        }
        return details;
    }

    public void setTitle(String title) {
        this.title = title;
    }

    public void setReportMap(Map<String, String> patientDiscoveryMap) {
        this.reportMap = patientDiscoveryMap;
    }

    public void setFilterMap(Map<String, List<Object>> filterMap) {
        this.filterMap = filterMap;
    }

    public void setExcelExporter(ExcelExporter exExport) {
        this.exExport = exExport;
    }

    public void setAdapterAuditManager(AdapterAuditManager adapterAuditManager) {
        this.adapterAuditManager = adapterAuditManager;
    }

    public void setReportHelper(ReportHelper reportHelper) {
        this.reportHelper = reportHelper;
    }

    public void setAttributes(Map<String, Object> attributes) {
        this.attributes = attributes;
    }

    public void setHasOptionalRows(boolean hasOptionalRows) {
        this.hasOptionalRows = hasOptionalRows;
    }

    public void setFacilityHelper(FacilityHelper facilityHelper) {
        this.facilityHelper = facilityHelper;
    }

    public void setCmsHelper(ConsentManagementHelper cmsHelper) {
        this.cmsHelper = cmsHelper;
    }

    public void setIsSummaryReport(boolean isSummaryReport) {
        this.isSummaryReport = isSummaryReport;
    }

    public void setOptionalRowsMap(Map<String, String> optionalRowsMap) {
        this.optionalRowsMap = optionalRowsMap;
    }

    public void setUserId(String userId) {
        this.userId = userId;
    }

    public void setUserDocumentDao(UserDocumentDAO userDocDao) {
        this.userDocDao = userDocDao;
    }

    public void setDirectAuditManager(DirectAuditManager directAuditManager) {
        this.directAuditManager = directAuditManager;
    }

    public void setReportSource(String reportSource) {
        this.reportSource = reportSource;
    }

    public void setAuditedConsentDAO(AuditedConsentDAO auditedConsentDAO) {
        this.auditedConsentDAO = auditedConsentDAO;
    }

    public void setDocumentHelper(DocumentHelper documentHelper) {
        this.documentHelper = documentHelper;
    }

    public void setPatientConsentDirDAO(PatientConsentDirDAO patientConsentDirDAO) {
        this.patientConsentDirDAO = patientConsentDirDAO;
    }

    public void setDelayedConsentReportDAO(DelayedConsentReportDAO delayedConsentReportDAO) {
        this.delayedConsentReportDAO = delayedConsentReportDAO;
    }

    public void setMailNotificationDAO(MailNotificationDAO mailNotificationDAO) {
        this.mailNotificationDAO = mailNotificationDAO;
    }

    public void setDocumentType(String documentType) {
        this.documentType = documentType;
    }

    public void setCsvExport(CsvExporter csvExport) {
        this.csvExport = csvExport;
    }
}