package gov.va.nvap.web.helper.report;

import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.service.pdq.PatientMatchQuery;
import gov.va.nvap.service.pdq.PatientMatchResponse;
import gov.va.nvap.service.pdq.PatientProfile;
import gov.va.nvap.service.pdq.PdqException;
import gov.va.nvap.service.pdq.PdqService;
import gov.va.nvap.svc.consentmgmt.stub.data.Organization;
import gov.va.nvap.svc.facility.data.Facility;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.util.Constants;
import gov.va.nvap.web.util.date.DateUtil;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import org.springframework.beans.factory.annotation.Required;

public class ReportHelper {
    
    @PersistenceContext
    private EntityManager em;
    
	private PdqService pdqService;
	private String disclosuresDetailedReportCap;
	private String recvdDocsDetailedReportCap;
	private String consentDirectivesReportCap;
	private String pdDetailedReportCap;
		
	public String resolveICN(final String firstName, final String lastName,
			final String ssn) {

		if (NullChecker.isNotEmpty(ssn)
				&& NullChecker.isNotEmpty(firstName)
				&& NullChecker.isNotEmpty(lastName)) {

			final List<PatientProfile> patients = getMatchedPatients(ssn, firstName, lastName);
			if(patients != null) {
				if (patients.isEmpty()) {
					return "Unknown";
				} else if (patients.size() >= 2) {
					return "Multiple ICNs";
				} else {
					final String icn = patients.get(0).getIcn();
					return icn;
				}
			}
		} else {
			return "Unknown";
		}

		return "Unknown";
	}

	/**
	 * This method returns list of Matched Patients.
	 * 
	 * @param ssn
	 * @param firstName
	 * @param lastName
	 * @return List of Matched Patients
	 */
	public List<PatientProfile> getMatchedPatients(String ssn, String firstName, String lastName) {

		List<PatientProfile> patients = null;
		try {

			// Create a patient profile and search for patient
			final PatientProfile patient = new PatientProfile();
			patient.setFirstName(firstName);
			patient.setLastName(lastName);
			patient.setSsn(ssn);
			final PatientMatchQuery patientSearchRequest = new PatientMatchQuery();
			patientSearchRequest.setPatientProfile(patient);
			// Call PDQ service to get the matching patient
			final PatientMatchResponse patientSearchResponse = this.pdqService
					.searchPatient(patientSearchRequest);
			patients = patientSearchResponse
					.getPatients().getPatientProfile();

		} catch (final PdqException ex) {
			// TODO: Maybe enter a message that MPI is not available to resolve
			// the ICN??
			// throw new ServletException(ex);
			return patients;
		}

		return patients;
	}
		
	/**
	 * Checks whether the given SSN is for a test patient.
	 * 
	 * @param ssn SSN to check
	 * @return true or false
	 */
	public boolean isTestPatient(String ssn) {
        return NullChecker.isNotEmpty(ssn) &&        // Ignore null entries
                ssn.length() == 9 && (               // Ignore entries that aren't exactly 9 digits
                ssn.startsWith("000") ||             // Area number 000 denotes a test patient
                ssn.startsWith("666") ||             // Area number 666 denotes a test patient
                ssn.startsWith("9") ||               // Area number in 900-999 range denotes a test patient
                ssn.substring(3, 5).equals("00") ||  // Group number 00 denotes a test patient
                ssn.substring(5, 9).equals("0000")); // Serial number 0000 denotes a test patient
	}
	
	public String getFormattedDateTime(Date date) {
		return getFormattedDateString(date,
				"MM/dd/yyyy hh:mm a");
	}
	
	public String getFormattedDate(Date date) {
		return getFormattedDateString(date,
				"MM/dd/yyyy");
	}
	
	public String getFormattedDate(String paramDate) {
        if (NullChecker.isNullOrEmpty(paramDate)) {
            return "";
        } else {
            Date date = DateUtil.parseDateMMddyyyy(paramDate);
            return getFormattedDate(date);
        }
	}
    
	public int getDisclosuresDetailedReportCapAsInt() {
		return Integer.parseInt(this.disclosuresDetailedReportCap);
	}

	public int getRecvdDocsDetailedReportCapAsInt() {
		return Integer.parseInt(this.recvdDocsDetailedReportCap);
	}

	public int getConsentDirectivesReportCapAsInt() {
		return Integer.parseInt(this.consentDirectivesReportCap);
	}

	public int getPdDetailedReportCapAsInt() {
		return Integer.parseInt(this.pdDetailedReportCap);
	}

	@Required
	public void setPdqService(final PdqService pdqService) {
		this.pdqService = pdqService;
	}

	@Required
	public void setDisclosuresDetailedReportCap(final String disclosuresDetailedReportCap) {
		this.disclosuresDetailedReportCap = disclosuresDetailedReportCap;
	}

	@Required
	public void setRecvdDocsDetailedReportCap(final String recvdDocsDetailedReportCap) {
		this.recvdDocsDetailedReportCap = recvdDocsDetailedReportCap;
	}

	@Required
	public void setConsentDirectivesReportCap(final String consentDirectivesReportCap) {
		this.consentDirectivesReportCap = consentDirectivesReportCap;
	}

	@Required
	public void setPdDetailedReportCap(final String pdDetailedReportCap) {
		this.pdDetailedReportCap = pdDetailedReportCap;
	}
	
	private String getFormattedDateString(final Date date, final String datePattern) {
		if (date == null) {
			return "n/a";
		}

		final SimpleDateFormat formatter = new SimpleDateFormat(
				datePattern);
		
		return formatter.format(date);

	}
	
	  /**
     * Extracts value of key=value pair from details property
     * @param details Details property of AuditType
     * @param key Key of the key/value pair
     * @return Value of the key/value pair
     */
	public String extractFromDetails(String details, String key) {
        int pos1 = details.indexOf(key + "=");
        int pos2;
        
        if (pos1 != -1) {
            pos2 = details.indexOf(",", pos1);
            if (pos2 != -1) {
                return details.substring(pos1 + key.length() + 1, pos2);
            }
        }
        return "";
    }
    
    public String findOrganizationByOid(String orgOId, String orgName) {
        String retValue = "Unknown";

        if (!NullChecker.isNullOrEmpty(orgName)) {
            retValue = orgName;
        }

        if (!NullChecker.isNullOrEmpty(orgOId)) {
            final Query q = this.em.createNamedQuery("Organization.findByOrgOid");
            if (orgOId.startsWith("urn:oid:")) {
                orgOId = orgOId.substring("urn:oid:".length());
            }
            q.setParameter("orgOid", orgOId.trim());
            List results = q.getResultList();
            if (!results.isEmpty()) {
                Organization o = (Organization) results.get(0);
                retValue = o.getOrgName();
            }
        }
        
        return retValue;
    }

    /**
     * Creates the active and inactive organization lists
     * @param request 
     */
    public void getOrgLists(HttpServletRequest request, ConsentManagementHelper cmsHelper) {
        final Query getActive = this.em.createNamedQuery("Organization.findByActive");
        final Query getInactive = this.em.createNamedQuery("Organization.findByActive");

        getActive.setParameter("active", 'Y');
        getInactive.setParameter("active", 'N');
        
        ArrayList<Organization> activeOrgs = (ArrayList<Organization>)getActive.getResultList();
        ArrayList<Organization> inactiveOrgs = (ArrayList<Organization>)getInactive.getResultList();

        Organization va = new Organization();
        va.setOrgName(cmsHelper.getHomeCommunity().getOrgName());
        va.setOrgOid(cmsHelper.getHomeCommunity().getOrgOid());
        activeOrgs.add(va);
        
        request.setAttribute("activeOrgs", activeOrgs);
        request.setAttribute("inactiveOrgs", inactiveOrgs);
    }
    
    /**
     * Creates the active and inactive organization lists for the Received Documents reports
     * @param request 
     */
    public void getReceivedDocsOrgLists(HttpServletRequest request) {
        final Query getActive = this.em.createNamedQuery("Organization.findByActiveOrgConsumerOnly");
        final Query getInactive = this.em.createNamedQuery("Organization.findByActiveOrgConsumerOnly");

        getActive.setParameter("active", 'Y');
        getInactive.setParameter("active", 'N');

        getActive.setParameter("orgConsumerOnly", 'N');
        getInactive.setParameter("orgConsumerOnly", 'N');
        
        ArrayList<Organization> activeOrgs = (ArrayList<Organization>)getActive.getResultList();
        ArrayList<Organization> inactiveOrgs = (ArrayList<Organization>)getInactive.getResultList();
        
        request.setAttribute("activeOrgs", activeOrgs);
        request.setAttribute("inactiveOrgs", inactiveOrgs);    
    }
    
    public static String getStationNumbersFromRequest(HttpServletRequest request) {
        String stationNumbers = request.getParameter("stationNumbers");
        if (stationNumbers == null) {
            // This is the initial report request, user hasn't picked anything in the Facilities dialog yet.
            stationNumbers = "";
            try {
                Facility facility = (Facility) request.getSession().getAttribute("defaultUserFacility");
                if (facility != null) {
                    stationNumbers = facility.getFacilityStation();
                }
            } catch (Exception e) {
                // TODO log error
            }
        }
        return stationNumbers;
    }
    
    public static String normalizeConsnentTypeName(String consentType) {
        if (consentType.contains("NwHIN")) {
            consentType = consentType.replace("NwHIN", Constants.getOrganizationName());
        }
        return consentType;
    }
    
    /**
     * Accepts an OID and trims the urn prefix
     * @param oid
     * @return oid without urn prefix
     */
    public static String trimOrgOid(String oid) {
        if (oid.contains("urn:oid:")) {
            oid = oid.replace("urn:oid:", "");
        }
        return oid;
    }
    
    /**
     * Changes string parameter start date to Date object.
     *
     * @param paramStart Start date parameter
     * @return Date object
     */
    public static Date getStartDate(String paramStartDate) {
        Date startDate = null;

        if (!NullChecker.isNullOrEmpty(paramStartDate)) {
            startDate = DateUtil.parseDateMMddyyyy(paramStartDate);
        }

        return startDate;
    }

    /**
     * Changes string parameter end date to Date object with 23:59:59 timestamp.
     *
     * @param paramEndDate End date parameter
     * @return Date object
     */
    public static Date getEndDate(String paramEndDate) {
        Date endDate = null;

        if (!NullChecker.isNullOrEmpty(paramEndDate)) {
            endDate = DateUtil.parseDateMMddyyyy(paramEndDate);
            endDate = DateUtil.voidTime(endDate);
            endDate = DateUtil.addTime(endDate, Calendar.HOUR_OF_DAY, 23);
            endDate = DateUtil.addTime(endDate, Calendar.MINUTE, 59);
            endDate = DateUtil.addTime(endDate, Calendar.SECOND, 59);
        }

        return endDate;
    }
    
    /**
     * sets the default start date and end date to the session as strings to be used in the search filters.
     * the end date is current day with the start date 30 days prior
     * startDateString and endDateString
     * @param session 
     */
    public static void setDefaultSearchDates(HttpSession session){
        
        DateFormat dateFormat = new SimpleDateFormat("MM/dd/yyyy");
        Calendar calendar = Calendar.getInstance();
        String endDateString = dateFormat.format(calendar.getTime());
        calendar.add(Calendar.MONTH, -1);
        String startDateString = dateFormat.format(calendar.getTime());
        session.setAttribute("startDateDefault", startDateString);
        session.setAttribute("endDateDefault", endDateString);
    }
    
}
