package gov.va.nvap.web.helper.report;

import gov.va.nvap.audit.BulkDownloadDocs;
import java.io.Closeable;
import java.io.FilterInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.text.MessageFormat;
import org.apache.http.HttpStatus;


import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.slf4j.LoggerFactory;
        
/**
 * Connects to the ReST service provided by NHIN Adapter for bulk download of
 * (audited) retrieved documents. This service provides a binary
 * zip stream of all retrieved (disclosure or receipt) documents meeting the
 * given filter criteria.
 * @author cbarber
 */
public class BulkDownloadDocsClient implements BulkDownloadDocs
{
    public static final int MS_PER_SEC = 1000;
    private static final String RQST_HDR_VAP_ID =  "VAP_ID";

    private static final org.slf4j.Logger LOGGER =
        LoggerFactory.getLogger(BulkDownloadDocsClient.class);
    
    //****
    // Properties set by Spring configuration.
    //****
    private String downloadUrl;
    private int connectionTimeoutSec;
    private int readTimeoutSec;
    
    @Override
    public InputStream downloadRetrievedDocs(String vapId, String startAudit, String endAudit, String docGenPartnerId, String patientIdentifier) throws IOException
    {
        //Define URI to hit Adapter BulkDownloadDocs
        URI uri;
        try {
            // Validate patientIdentifier
            if (patientIdentifier != null && patientIdentifier.isEmpty() == false)
            {
                uri = new URIBuilder(downloadUrl)
                    .setParameter("patientIdentifier", patientIdentifier)
                    .setParameter("start", startAudit)
                    .setParameter("end", endAudit)
                    .setParameter("creator", docGenPartnerId)
                    .build();
            }
            else
            {
                uri = new URIBuilder(downloadUrl)
                    .setParameter("start", startAudit)
                    .setParameter("end", endAudit)
                    .setParameter("creator", docGenPartnerId)
                    .build();
            }
        } catch (URISyntaxException ex) {
            throw new IOException("Exception thrown while creating The URI.", ex);
        }
        
        HttpGet httpGet = new HttpGet(uri);
        
        //Define and Set Connection Timeouts
        RequestConfig.Builder requestBuilder = RequestConfig.custom();
        requestBuilder = requestBuilder.setConnectTimeout(connectionTimeoutSec * MS_PER_SEC);
        requestBuilder = requestBuilder.setConnectionRequestTimeout(connectionTimeoutSec * MS_PER_SEC);
        requestBuilder = requestBuilder.setSocketTimeout(readTimeoutSec * MS_PER_SEC);

        //Create HttpClient 
        HttpClientBuilder builder = HttpClientBuilder.create();
        builder.setDefaultRequestConfig(requestBuilder.build());

        CloseableHttpClient httpClient = null;
        CloseableHttpResponse response = null;
        InputStream contentStream = null;
        try
        {
            // httpClient actually maintains an HTTP conn pool, and would be 
            // reusable if this bulk download feature ever sees heavier usage.
            httpClient = builder.build();

            // Execute the get request and set the response.
            httpGet.addHeader(RQST_HDR_VAP_ID, vapId);
            response = httpClient.execute(httpGet);
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != HttpStatus.SC_OK)
            {
                String msg = MessageFormat.format(
                    "Adapter Bulk Download failed with HTTP error code : {0}",
                    statusCode);
                throw new IOException(msg);
            }
            else
            {
                LOGGER.debug("Adapter Bulk Download response Status Code: " +
                    statusCode );
            }

            contentStream = response.getEntity().getContent();

            //***
            // Special close method to close the underlying socket connection
            // and HTTPClient instance, too. Note need for separate final
            // copies of the two vars below, since they must be initialized to
            // null outside the try block for the finally block to work.
            //***
            final CloseableHttpClient httpClientF = httpClient;
            final CloseableHttpResponse responseF = response;
            return new FilterInputStream(contentStream)
            {
                @Override
                public void close() throws IOException
                {
                    closeQuietly(responseF);
                    closeQuietly(httpClientF);
                    super.close();
                }
            };
        }
        finally
        {
            if(contentStream == null)
            {
                // otherwise, caller must close the returned InputStream
                // subclass, which will close these
                closeQuietly(response);
                closeQuietly(httpClient);
            }
        }
    }

    public void setDownloadUrl(String s)
    {
        downloadUrl = s;
    }

    public void setConnectionTimeoutSec(int connectionTimeoutSec)
    {
        this.connectionTimeoutSec = connectionTimeoutSec;
    }

    public void setReadTimeoutSec(int readTimeoutSec)
    {
       this.readTimeoutSec = readTimeoutSec;
    }
    
   /**
     * Unconditionally close a <code>Closeable</code>.
     * This is from commons-io....WebLogic seems to be packing an old version
     * somewhere that is giving NoSuchMethodError.
     * <p>
     * Equivalent to {@link Closeable#close()}, except any exceptions will be ignored.
     * This is typically used in finally blocks.

     * @param closeable the object to close, may be null or already closed
     */
    public static void closeQuietly(Closeable closeable)
    {
        try
        {
            if (closeable != null) {
                closeable.close();
            }
        }
        catch (IOException ioe)
        {
            // ignore
        }
    }    
}
