package gov.va.nvap.web.helper.privacy;

import gov.va.nvap.common.date.hl7.HL7DateUtil;
import gov.va.nvap.common.jaxb.JaxbUtil;
import gov.va.nvap.common.transformer.Transformer;
import gov.va.nvap.common.transformer.TransformerException;
import gov.va.nvap.common.transformer.xml.StringToXML;
import gov.va.nvap.common.transformer.xml.XMLToString;
import gov.va.nvap.common.transformer.xsl.XSLTransformer;
import gov.va.nvap.common.uuid.UUIDUtil;
import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.ConsentDirectiveAuthorizationRequest;
import gov.va.nvap.privacy.ConsentDirectiveAuthorizationResponse;
import gov.va.nvap.privacy.ConsentDirectiveDetailedExpirationRequest;
import gov.va.nvap.privacy.ConsentDirectiveDetailedExpirationResponse;
import gov.va.nvap.privacy.ConsentDirectiveDocumentRetrieveRequest;
import gov.va.nvap.privacy.ConsentDirectiveDocumentRetrieveResponse;
import gov.va.nvap.privacy.ConsentDirectiveDocumentType;
import gov.va.nvap.privacy.ConsentDirectiveOptOutReasonType;
import gov.va.nvap.privacy.ConsentDirectiveQueryParamType;
import gov.va.nvap.privacy.ConsentDirectiveQueryRequest;
import gov.va.nvap.privacy.ConsentDirectiveQueryResponse;
import gov.va.nvap.privacy.ConsentDirectiveReferenceType;
import gov.va.nvap.privacy.ConsentDirectiveRevocationRequest;
import gov.va.nvap.privacy.ConsentDirectiveUpdateRequest;
import gov.va.nvap.privacy.ConsentDirectiveUpdateResponse;
import gov.va.nvap.privacy.ConsentType;
import gov.va.nvap.privacy.ExpiringConsentConfigurationRequest;
import gov.va.nvap.privacy.ExpiringConsentConfigurationResponse;
import gov.va.nvap.privacy.ExpiringConsentLetterResponse;
import gov.va.nvap.privacy.OrganizationType;
import gov.va.nvap.privacy.OrganizationsQueryRequest;
import gov.va.nvap.privacy.OrganizationsQueryResponse;
import gov.va.nvap.privacy.PurposeOfUseType;
import gov.va.nvap.privacy.ServiceConsumer;
import gov.va.nvap.privacy.ServiceConsumerContextType;
import gov.va.nvap.privacy.data.ConsentDirectiveData;
import gov.va.nvap.privacy.data.Participant;
import gov.va.nvap.privacy.data.ScannedConsentDirectiveDocumentBase64Text;
import gov.va.nvap.service.pdq.PatientDemographics;
import gov.va.nvap.service.privacy.ConsentManagementService;
import gov.va.nvap.service.privacy.ConsentManagementServiceException;
import gov.va.nvap.service.privacy.ExpiringConsentService;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.xml.bind.JAXBException;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.dom.DOMSource;
import org.apache.commons.fileupload.FileItem;
import org.apache.xerces.impl.dv.util.Base64;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import org.w3c.dom.Document;

/**
 * Helper class to deal with consent management.
 *
 * @author Asha Amritraj
 * @author Zack Peterson
 */
public class ConsentManagementHelper implements InitializingBean {

    /**
     * Default expiring consent configuration values
     */
    private static final String DEFAULT_FREQUENCY = "MONTHLY";
    private static final Long DEFAULT_WINDOW = 30L;
    private static final String DEFAULT_EMAILS = "";

    /**
     * Consent Management Service.
     */
    private ConsentManagementService cms;
    /**
     * Expiring Consent Service.
     */
    private ExpiringConsentService ecs;
    /**
     * To convert from ConsentDirectiveData to XML for transformation into the CDA XML.
     */
    private JaxbUtil dataJaxbHelper;
    /**
     * Transformer to transform from the value object to CDA R2 XML document.
     */
    private Transformer<Document, Document> dataToConsentDirectiveDocument;
    /**
     * VA Organization.
     */
    private OrganizationType homeCommunity;
    /**
     * Form Upload Allowed.
     */
    private boolean isFormUploadEnabled;
    /**
     * Organizational Exclusions - Enabled.
     */
    private boolean isOrganizationExclusionsAllowed;
    private final List<String> optOutReasons = new ArrayList<String>();

    /**
     * String to XML Transformer.
     */
    private StringToXML stringToXml;

    /**
     * Stylesheet to update the consent directive document.
     */
    private XSLTransformer updateConsentDirectiveDocumentStylesheet;

    /**
     * XML to String Transformer.
     */
    private XMLToString xmlToString;

    /*
	 * ConsumerOnlyOID
     */
    private String consumerOnlyOrgOID;

    @Override
    public void afterPropertiesSet() throws Exception {
        // Keep the optout reasons in the cache, so you dont have to hit the
        // server everytime
        for (final ConsentDirectiveOptOutReasonType types : ConsentDirectiveOptOutReasonType
            .values()) {
            this.optOutReasons.add(types.value());
        }
    }

    /**
     * Construct the patient demographics data for the CDA R2 XML.
     */
    private ConsentDirectiveData constructConsentDirectiveDocumentDemographics(
        final ConsentDirectiveData data, final String icn,
        final PatientDemographics demographics,
        final List<FileItem> fileItems) {
        Assert.assertNotEmpty(data, "ConsentDirectiveData cannot be NULL!");

        // Set the demographics information
        data.setIcn(icn);
        if (NullChecker.isNotEmpty(demographics)) {
            data.setPatientRoleCity(demographics.getResidenceCity());
            data.setPatientRoleState(demographics.getResidenceState());
            data.setPatientRoleGivenName(demographics.getFirstName());
            data.setPatientRoleFamilyName(demographics.getLastName());
            data.setPatientRoleFamilyNameAlias(demographics.getAlias1());
            data.setPatientRoleMiddleName(demographics.getMiddleName());
            data.setPatientRoleEthnicGroupCodeDisplayText(demographics
                .getEthnicityDescription());
            data.setPatientRoleSsn(demographics.getSsn());
            data.setPatientRoleGenderCode(demographics.getGender());
            data.setPatientRoleGenderDisplayText(demographics
                .getGenderDescription());
            data.setPatientRoleMaritalStatusCode(demographics
                .getMaritalStatus());
            data.setPatientRoleMaritalStatusDisplayText(demographics
                .getMaritalStatusDescription());
            data.setPatientRolePostalCode(demographics.getResidenceZip4());
            data.setPatientRolePrefix(demographics.getPrefix());
            data.setPatientRoleStreetAddressLine(demographics
                .getStreetAddressLine1());
            data.setPatientRoleSuffix(demographics.getSuffix());
            data.setPatientRoleTelecom(demographics.getResidencePhoneNumber());
            data.setPatientRoleProviderOrganizationName(demographics
                .getFacilityName());
            data.setPatientRoleProviderOrganizationNumber(demographics
                .getFacilityNumber());
            data.setPatientRoleProviderOrganizationOid(this.getHomeCommunity()
                .getOrgOid());
            if (NullChecker.isNotEmpty(demographics.getDob())) {
                try {
                    // Need to use the HL7 date, because the CDA R2 needs it
                    data.setPatientRoleDob(HL7DateUtil
                        .yyyyMMddhhmmssZ(demographics.getDob()));
                } catch (final ParseException ex) {
                    throw new RuntimeException(ex);
                }
            }
        }
        // For each file item, create the media type
        if (NullChecker.isNotEmpty(fileItems)) {
            for (final FileItem fileItem : fileItems) {
                if (fileItem.getSize() > 0) {
                    final ScannedConsentDirectiveDocumentBase64Text scannedConsentDirectiveDocumentBase64Text = new ScannedConsentDirectiveDocumentBase64Text();

                    final byte[] bytes = fileItem.get();
                    // JAXB automatically decodes the document when making it
                    // into XML
                    // String encodedString = new sun.misc.BASE64Encoder()
                    // .encode(bytes);
                    // byte[] encodedBytes = encodedString.getBytes();
                    final String contentType = fileItem.getContentType();
                    // Set the type and representation
                    scannedConsentDirectiveDocumentBase64Text
                        .setMediaType(contentType);
                    scannedConsentDirectiveDocumentBase64Text
                        .setRepresentation("B64");
                    scannedConsentDirectiveDocumentBase64Text
                        .setDocument(bytes);
                    // add the document to ConsentDirectiveData
                    data.getScannedConsentDirectiveDocumentBase64Text().add(
                        scannedConsentDirectiveDocumentBase64Text);
                }
            }
        }
        // Set a unique id per document
        data.setDocId(UUIDUtil.generateUUID());
        // return response
        return data;

    }

    private ServiceConsumerContextType createServiceConsumerContext(
        final String userId, final String userFacilityStationId) {
        final ServiceConsumerContextType consumerContext = new ServiceConsumerContextType();
        consumerContext.setUser(userId);
        consumerContext.setFacility(userFacilityStationId);
        consumerContext
            .setServiceConsumerType(ServiceConsumer.ADMINISTRATOR_PORTAL);
        return consumerContext;
    }

    /**
     * Anand - 06/06/2012 - Day before 2.0.0.7 code cutoff getActiveConsentDirectiveForAnnouncements is a duplicate of
     * getActiveConsentDirective with minor changes 1) The consent type is set on the request, prior to dispatching it.
     *
     * BEfore creating this method getActiveConsentDirective was changed to incorporate the afore-mentioned change This method is invoked
     * both during Announce and OptOut Scenarios. The afore-mentioned change caused issues during opt-outs
     *
     * The framework for consent management is buggy and has to be reevaluated in Increment 3. At that point this method will be
     * reconsolidated with getActiveConsentDirective, in addition to other refactoring
     */
    public ConsentDirectiveReferenceType getActiveConsentDirectiveForAnnouncements(
        final String patientIcn, final ConsentType consentType) {
        Assert.assertNotEmpty(patientIcn, "ICN cannot be empty!");
        try {
            // Construct request
            final ConsentDirectiveQueryRequest consentDirectiveQueryRequest = new ConsentDirectiveQueryRequest();
            // Set the patient id
            consentDirectiveQueryRequest.setPatientId(patientIcn);

            ServiceConsumerContextType type = new ServiceConsumerContextType();
            type.setConsentType(consentType);
            consentDirectiveQueryRequest.setServiceConsumerContext(type);
            // Get the response

            // Query only for active
            consentDirectiveQueryRequest
                .setQueryParam(ConsentDirectiveQueryParamType.ACTIVE);
            final ConsentDirectiveQueryResponse response = this.cms
                .getConsentDirectives(consentDirectiveQueryRequest);
            if (NullChecker.isEmpty(response.getConsentDirectiveReference())) {
                return null;
            }
            // Return response
            return response.getConsentDirectiveReference().get(0);
        } catch (final ConsentManagementServiceException ex) {
            throw new RuntimeException(ex);
        }
    }

    /**
     * Get the active consent directive from the Consent management service.
     */
    public ConsentDirectiveReferenceType getActiveConsentDirective(
        final String patientIcn, final ConsentType consentType) {
        Assert.assertNotEmpty(patientIcn, "ICN cannot be empty!");
        try {
            // Construct request
            final ConsentDirectiveQueryRequest consentDirectiveQueryRequest = new ConsentDirectiveQueryRequest();
            // Set the patient id
            consentDirectiveQueryRequest.setPatientId(patientIcn);
            // Query only for active
            consentDirectiveQueryRequest
                .setQueryParam(ConsentDirectiveQueryParamType.ACTIVE);
            final ConsentDirectiveQueryResponse response = this.cms
                .getConsentDirectives(consentDirectiveQueryRequest);
            ServiceConsumerContextType type = new ServiceConsumerContextType();
            type.setConsentType(consentType);
            consentDirectiveQueryRequest.setServiceConsumerContext(type);
            // Get the response
            if (NullChecker.isEmpty(response.getConsentDirectiveReference())) {
                return null;
            }
            // Return response
            //Need to differentiate between active consent types and return the correct active consent.
            for (ConsentDirectiveReferenceType cdr : response.getConsentDirectiveReference()) {
                if (consentType.equals(ConsentType.NW_HIN_REVOCATION) && (cdr.getOptinConsentType() == ConsentType.NW_HIN_AUTHORIZATION)) {
                    return cdr;
                }
                if (consentType.equals(ConsentType.SSA_REVOCATION) && (cdr.getOptinConsentType() == ConsentType.SSA_AUTHORIZATION)) {
                    return cdr;
                }
                if (consentType.equals(ConsentType.NW_HIN_ORGANIZATION_RESTRICTION_REVOCATION) && (cdr.getOptinConsentType() == ConsentType.NW_HIN_ORGANIZATION_RESTRICTION_AUTHORIZATION)) {
                    return cdr;
                }
            }
			return response.getConsentDirectiveReference().get(0);
		} catch (final ConsentManagementServiceException ex) {
			throw new RuntimeException(ex);
		}
	}

	/**
	 * Get all the allowed organizations from the consent management service.
	 * This basically returns the list of organizations that are supported by
	 * the NHIN including the VA.
	 */
	public Collection<OrganizationType> getAllOrganizations() {
		try {
			// Call CMS
			final OrganizationsQueryResponse response = this.cms
					.getOrganizations(new OrganizationsQueryRequest());
			// Add VA to the list
			response.getOrganization().add(this.homeCommunity);
			return response.getOrganization();
		} catch (final ConsentManagementServiceException ex) {
			throw new RuntimeException(ex);
		}
	}

	/**
	 * Get all the allowed organizations from the consent management service.
	 * This basically returns the list of organizations that are supported by
	 * the NHIN excluding the VA.
	 */
	public Collection<OrganizationType> getAllowedOrganizations() {
		try {
			final OrganizationsQueryResponse response = this.cms
					.getOrganizations(new OrganizationsQueryRequest());
			return response.getOrganization();
		} catch (final ConsentManagementServiceException ex) {
			throw new RuntimeException(ex);
		}
	}

	/**
	 * Get all the Non Consumer only allowed organizations from the consent management service.
	 * This basically returns the list of organizations that are supported by
	 * the NHIN excluding the VA, excluding SSA
	 */
	public Collection<OrganizationType> getAllowedNonConsumerOnlyOrganizations() {
		// Get the allowed organizations from the consent management service
		final Collection<OrganizationType> allowedOrganizations = this
				.getAllowedOrganizations();
		final List<OrganizationType> nonConsumerOnlyOrgs = new ArrayList<OrganizationType>();
		for (final OrganizationType allowedOrganization : allowedOrganizations) {
			if (!this.consumerOnlyOrgOID.endsWith(allowedOrganization.getOrgOid())) {
				nonConsumerOnlyOrgs.add(allowedOrganization);
			}
		}
		return nonConsumerOnlyOrgs;
	}

	/**
	 * Get the organizations that the patient is currently sharing information
	 * with. NOTE: SHOULD BE USED ONLY FOR ANNOUNCEMENTS
	 * @deprecated as of VAP 2.1 - Please do not use - AMS
	 */

	private Collection<OrganizationType> getAuthorizedOrganizations(
			final String patientId) {
		// Get the active consent directive for that patient
		final ConsentDirectiveReferenceType consentDirective = this
				.getActiveConsentDirectiveForAnnouncements(patientId,
						ConsentType.NW_HIN_AUTHORIZATION);
		// Get the allowed organizations from the consent management service
		final Collection<OrganizationType> allowedOrganizations = this
				.getAllowedNonConsumerOnlyOrganizations();
		final List<OrganizationType> optedInOrganizations = new ArrayList<OrganizationType>();
		// If no consent directive - select only to DoD.
		if (NullChecker.isEmpty(consentDirective)) {
			//optedInOrganizations.add(this.getOrganizationByNumber("200DOD"));
			optedInOrganizations.addAll(allowedOrganizations);
		} else {
			final ConsentDirectiveReferenceType restriction = this
					.getActiveConsentDirectiveForAnnouncements(
							patientId,
							ConsentType.NW_HIN_ORGANIZATION_RESTRICTION_AUTHORIZATION);

			// select only to authorized organizations
			for (final OrganizationType allowedOrganization : allowedOrganizations) {
				boolean toExclude = false;
				if (NullChecker.isNotEmpty(restriction)
						&& NullChecker.isNotEmpty(restriction
								.getExcludedOrganizations())) {
					for (final OrganizationType excludedOrganization : restriction
							.getExcludedOrganizations()) {
						if (allowedOrganization.getOrgNumber().equals(
								excludedOrganization.getOrgNumber())) {
							toExclude = true;
							break;
						}
					}
					if (!toExclude) {
						optedInOrganizations.add(allowedOrganization);
					}
				} else {
					optedInOrganizations.add(allowedOrganization);
				}
			}
		}
		// Return the opted in organizations
		return optedInOrganizations;
	}

	/**
	 * The complete home community id with the prefix for VA.
	 */
	public String getCompleteHomeCommunityId() {
		return this.homeCommunity.getOrgCommunityIdPrefix() + ":"
				+ this.homeCommunity.getOrgOid();
	}

	/**
	 * Get all the consent directives for a patient from the consent management
	 * service.
	 */
	public List<ConsentDirectiveReferenceType> getConsentDirectiveHistory(
			final String patientIcn) {
		Assert.assertNotEmpty(patientIcn, "ICN cannot be empty!");
		try {
			final ConsentDirectiveQueryRequest consentDirectiveQueryRequest = new ConsentDirectiveQueryRequest();
			consentDirectiveQueryRequest.setPatientId(patientIcn);
			// Set to ALL
			consentDirectiveQueryRequest
					.setQueryParam(ConsentDirectiveQueryParamType.ALL);
			final ConsentDirectiveQueryResponse response = this.cms
					.getConsentDirectives(consentDirectiveQueryRequest);
			// Return list
			return response.getConsentDirectiveReference();
		} catch (final ConsentManagementServiceException ex) {
			throw new RuntimeException(ex);
		}
	}
	
	/**
	 * Get all expiring consent directives.
	 */
	public ConsentDirectiveDetailedExpirationResponse getAllExpiringConsentDirectives(
			final ConsentDirectiveDetailedExpirationRequest request) {
		try {
			
			final ConsentDirectiveDetailedExpirationResponse response = this.cms
					.getAllExpiringConsentDirectives(request);
			// If empty, return null
			if (NullChecker.isEmpty(response.getConsentDirectiveReference())) {
				return null;
			}
			// Return list
			return response;
		} catch (final ConsentManagementServiceException ex) {
			throw new RuntimeException(ex);
		}
	}
    
    /**
	 * Get all expiring consent directives with some demographics.
	 */
	public ConsentDirectiveDetailedExpirationResponse getAllExpiringDetailedConsentDirectives(
			final ConsentDirectiveDetailedExpirationRequest request) {
		try {
			final ConsentDirectiveDetailedExpirationResponse response = this.cms
					.getAllExpiringDetailedConsentDirectives(request);
			// If empty, return null
			if (NullChecker.isEmpty(response.getDetailedConsentDirectiveReference())) {
				return null;
			}
			// Return list
			return response;
		} catch (final ConsentManagementServiceException ex) {
			throw new RuntimeException(ex);
		}
	}
    
    /**
	 * Get all expiring consents by facility for summary.
	 */
	public ConsentDirectiveDetailedExpirationResponse getAuthorizedConsentForSummary(
			final ConsentDirectiveDetailedExpirationRequest request) {
		try {
			final ConsentDirectiveDetailedExpirationResponse response = this.cms
					.getAuthorizedConsentForSummary(request);
			// If empty, return null
			if (NullChecker.isEmpty(response.getFacilityOptInConsentReference())) {
				return null;
			}
			// Return list
			return response;
		} catch (final ConsentManagementServiceException ex) {
			throw new RuntimeException(ex);
		}
	}

	/**
	 * Get the VA organization.
	 */
	public OrganizationType getHomeCommunity() {
		return this.homeCommunity;
	}

	/**
	 * Get the list of opt-out reasons to populate the GUI.
	 */
	public List<String> getOptoutReasons() {
		return this.optOutReasons;
	}

	/**
	 * Helper method to get the organization by domain.
	 */
	public OrganizationType getOrganizationByDomain(final String facilityDomain) {
		final Collection<OrganizationType> allOrganizationTypes = this
				.getAllOrganizations();
		for (final OrganizationType OrganizationType : allOrganizationTypes) {
			if (OrganizationType.getOrgDomain().equals(facilityDomain)) {
				return OrganizationType;
			}
		}
		return null;
	}
    
	/**
	 * Helper method to get the organization by Oid.
	 */
	public OrganizationType getOrganizationByHomeCommunityId(
			final String homeCommunityId) {
		final Collection<OrganizationType> allOrganizationTypes = this
				.getAllOrganizations();
		for (final OrganizationType OrganizationType : allOrganizationTypes) {
			if (OrganizationType.getOrgOid().equals(homeCommunityId)) {
				return OrganizationType;
			}
		}
		return null;
	}

	/**
	 * Helper method to get the organization by number.
	 */
	public OrganizationType getOrganizationByNumber(
			final String OrganizationTypeNumber) {
		final Collection<OrganizationType> allOrganizationTypes = this
				.getAllOrganizations();

		for (final OrganizationType OrganizationType : allOrganizationTypes) {
			if (OrganizationType.getOrgNumber().equals(OrganizationTypeNumber)) {
				return OrganizationType;
			}
		}
		return null;
	}

	public boolean isFormUploadEnabled() {
		return this.isFormUploadEnabled;
	}

	public boolean isOrganizationExclusionsAllowed() {
		return this.isOrganizationExclusionsAllowed;
	}

	/**
	 * Convert from the ConsentDirectiveData to the CDA R2 XML Privacy consent
	 * directive document and then convert that to string.
	 */
	private byte[] makeConsentDirectiveDocumentString(
			final ConsentDirectiveData data) {
		try {
			// Convert the ConsentDirectiveData to XML document
			final Document consentDirectiveDataDoc = this.dataJaxbHelper
					.marshal(data);
			// Convert ConsentDirectiveData XML to CDA R2 XML
			final Document consentDirectiveDocument = this.dataToConsentDirectiveDocument
					.transform(consentDirectiveDataDoc);
			// Convert CDA R2 XML to string
			final String consentDirectiveDocumentString = this.xmlToString
					.transform(consentDirectiveDocument);
			return consentDirectiveDocumentString.getBytes();

		} catch (final TransformerException ex) {
			throw new RuntimeException(ex);
		} catch (final JAXBException ex) {
			throw new RuntimeException(ex);
		}
	}

	/**
	 * Opt-in Patients!
	 */
	public String optIn(final String icn, final String[] organizationNumbers,
			final PatientDemographics demographics,
			final List<FileItem> fileItems, final String userId,
			final String userFacilityStationId, final Date signatureDate,
			final String consentType, final String purposeOfUse,
			final String expirationYears) {
		Assert.assertNotEmpty(icn, "ICN cannot be empty!");
		Assert.assertNotEmpty(signatureDate, "Signature Date cannot be empty!");
		// Get the excluded organizations
		final List<OrganizationType> excludedOrganizations = new ArrayList<OrganizationType>();
		if (NullChecker.isNotEmpty(organizationNumbers)) {
			final Collection<OrganizationType> foreignOrganizations = this
					.getAllowedOrganizations();

			for (final OrganizationType organization : foreignOrganizations) {
				for (final String selection : organizationNumbers) {
					if (organization.getOrgNumber().equals(selection)) {
						excludedOrganizations.add(organization);
					}
				}
			}
		}
		// Create a ConsentDirectiveData object
		ConsentDirectiveData data = new ConsentDirectiveData();
		data = this.constructConsentDirectiveDocumentDemographics(data, icn,
				demographics, fileItems);
		// Set author
		data.setAuthorPersonOid(userId);
		// data.setAuthorPersonOrgOid(this.homeCommunity.getOrgOid());
		data.setAuthorPersonOrgOid(userFacilityStationId);
		// Fill up the excluded organizations
		if (NullChecker.isNotEmpty(excludedOrganizations)) {
			for (final OrganizationType organizationType : excludedOrganizations) {
				final Participant participant = new Participant();
				participant.setComponentParticipantOid(organizationType
						.getOrgOid());
				participant
						.setComponentParticipantPlayingEntityName(organizationType
								.getOrgName());
				data.getParticipant().add(participant);
			}
		}
		// Set the status to active
		data.setComponentStatusCode("active");
		data.setComponentPurposeOfUseDisplayName(PurposeOfUseType.fromValue(
				purposeOfUse).value());

		try {
			final String signatureDateString = HL7DateUtil
					.yyyyMMddhhmmssZ(signatureDate);
			// Set the effective date
			data.setEffectiveDateTime(HL7DateUtil.yyyyMMddhhmmssZ(new Date()));
			// Create the begin and end date
			data.setDocumentationBeginTime(signatureDateString);
			if (NullChecker.isNotEmpty(expirationYears)) {
				Calendar dateCal = Calendar.getInstance();
				dateCal.setTime(signatureDate);
				int expiryYears = Integer.parseInt(expirationYears);
				dateCal.add(Calendar.YEAR, expiryYears);
				data.setDocumentationEndTime(HL7DateUtil
						.yyyyMMddhhmmssZ(dateCal.getTime()));
			}
		} catch (final ParseException ex) {
			throw new RuntimeException(ex);
		}
		// Make the attachment into bytes
		final byte[] consentDirectiveDocumentBytes = this
				.makeConsentDirectiveDocumentString(data);
		try {
			// Create the request and send to the consent management service
			final ConsentDirectiveAuthorizationRequest request = new ConsentDirectiveAuthorizationRequest();
			request.setDocument(consentDirectiveDocumentBytes);
			final ServiceConsumerContextType sc = this
					.createServiceConsumerContext(userId, userFacilityStationId);
			sc.setConsentType(ConsentType.fromValue(consentType));
			request.setServiceConsumerContext(sc);
			// Process
			ConsentDirectiveAuthorizationResponse response = this.cms.processConsentDirectiveAuthorization(request);
                        
                        return response.getConsentDirectiveReference().getConsentDirId();
		} catch (final ConsentManagementServiceException ex) {
			throw new RuntimeException(ex);
		}
	}

	/**
	 * Opt out the patient.
	 */
	public String optOut(final String icn,
			final PatientDemographics demographics,
			final List<FileItem> fileItems, final String optoutReason,
			final String userId, final String userFacilityStationId,
			final Date signatureDate, String consentType,
			final String purposeOfUse) {
		Assert.assertNotEmpty(icn, "ICN cannot be empty!");
		Assert.assertNotEmpty(optoutReason, "Opt-out reason cannot be empty!");
		Assert.assertNotEmpty(userId, "User id cannot be empty!");
		Assert.assertNotEmpty(signatureDate, "Signature Date cannot be empty!");
		// look up the record that needs to be opted out.
		final ConsentDirectiveReferenceType consentDirective = this
				.getActiveConsentDirective(icn,
						ConsentType.fromValue(consentType));
		if (NullChecker.isEmpty(consentDirective)) {
			throw new RuntimeException(
					"There has to be one active consent directive to process revocation!");
		}
		// Create a consent directive data object
		ConsentDirectiveData data = new ConsentDirectiveData();
		// Create Demographics
		data = this.constructConsentDirectiveDocumentDemographics(data, icn,
				demographics, fileItems);
		// Set the status code to aborted
		data.setComponentStatusCode("aborted");
		try {
			final String signatureDateString = HL7DateUtil
					.yyyyMMddhhmmssZ(signatureDate);
			// Set the effective date
			data.setEffectiveDateTime(HL7DateUtil.yyyyMMddhhmmssZ(new Date()));
			// Create the begin and end date
			data.setDocumentationBeginTime(signatureDateString);
			// Start and end time are required
			data.setDocumentationEndTime(signatureDateString);
		} catch (final ParseException ex) {
			throw new RuntimeException(ex);
		}
		// Set the consent directive
		data.setPreviousConsentDirectiveId(consentDirective.getConsentDirId());
		data.setOptoutReason(optoutReason);
		data.setAuthorPersonOid(userId);
		data.setAuthorPersonOrgOid(userFacilityStationId);
		// Make the request to revoke
		final ServiceConsumerContextType sc = this
				.createServiceConsumerContext(userId, userFacilityStationId);
		data.setComponentPurposeOfUseDisplayName(PurposeOfUseType.fromValue(
				purposeOfUse).value());
		sc.setConsentType(ConsentType.fromValue(consentType));
		// Convert the PDF into byte string
		final byte[] consentDirectiveDocumentBytes = this
				.makeConsentDirectiveDocumentString(data);
		final ConsentDirectiveRevocationRequest request = new ConsentDirectiveRevocationRequest();
		request.setServiceConsumerContext(sc);
		request.setOptoutReason(ConsentDirectiveOptOutReasonType
				.fromValue(optoutReason));
		request.setDocument(consentDirectiveDocumentBytes);

		try {
			// Process revoke
			this.cms.processConsentDirectiveRevocation(request);
		} catch (final ConsentManagementServiceException ex) {
			throw new RuntimeException(ex);
		}
            return consentDirective.getConsentDirId();
	}

	/**
	 * Retrieve a CDA R2 XML document based on the consent directive Id and
	 * document type (OPT-IN/OPT-OUT).
     * @param consentDirId
     * @param documentType
     * @return
     */
    public String retrieveDocument(final String consentDirId,
        final String documentType) {

        Assert.assertNotEmpty(consentDirId,
            "Consent directive cannot be empty!");
        Assert.assertNotEmpty(documentType, "Document type cannot be empty!");

        final ConsentDirectiveDocumentRetrieveRequest request = new ConsentDirectiveDocumentRetrieveRequest();
        final ConsentDirectiveReferenceType reference = new ConsentDirectiveReferenceType();
        reference.setConsentDirId(consentDirId);
        // Set the reference and the document type and retrieve the CDA R2 XML
        // document
        request.setDocumentType(ConsentDirectiveDocumentType
            .fromValue(documentType));
        request.setConsentDirectiveReference(reference);
        try {
            final ConsentDirectiveDocumentRetrieveResponse response = this.cms
                .getConsentDirectiveDocuments(request);
            // There would be only one document based on the document type
            if (NullChecker.isNotEmpty(response)) {
                if (NullChecker.isNotEmpty(response.getDocuments())) {
                    final byte[] document = response.getDocuments().get(0);
                    if (NullChecker.isNotEmpty(document)) {
                        return new String(document);
                    }
                }
            }
            // Empty if no document is found
            return null;
        } catch (final ConsentManagementServiceException ex) {
            throw new RuntimeException(ex);
        }
    }

    @Required
    public void setCms(final ConsentManagementService cms) {
        this.cms = cms;
    }

    @Required
    public void setEcs(final ExpiringConsentService ecs) {
        this.ecs = ecs;
    }

    @Required
    public void setDataJaxbHelper(final JaxbUtil dataJaxbHelper) {
        this.dataJaxbHelper = dataJaxbHelper;
    }

    @Required
    public void setDataToConsentDirectiveDocument(
        final Transformer<Document, Document> dataToConsentDirectiveDocument) {
        this.dataToConsentDirectiveDocument = dataToConsentDirectiveDocument;
    }

    @Required
    public void setFormUploadEnabled(final boolean isFormUploadEnabled) {
        this.isFormUploadEnabled = isFormUploadEnabled;
    }

    @Required
    public void setHomeCommunity(final OrganizationType homeCommunity) {
        this.homeCommunity = homeCommunity;
    }

    @Required
    public void setOrganizationExclusionsAllowed(
        final boolean isOrganizationExclusionsAllowed) {
        this.isOrganizationExclusionsAllowed = isOrganizationExclusionsAllowed;
    }

    @Required
    public void setStringToXml(final StringToXML stringToXml) {
        this.stringToXml = stringToXml;
    }

    @Required
    public void setUpdateConsentDirectiveDocumentStylesheet(
        final XSLTransformer updateConsentDirectiveDocumentStylesheet) {
        this.updateConsentDirectiveDocumentStylesheet = updateConsentDirectiveDocumentStylesheet;
    }

    @Required
    public void setXmlToString(final XMLToString xmlToString) {
        this.xmlToString = xmlToString;
    }

    @Required
    public void setConsumerOnlyOrgOID(final String consumerOnlyOrgOID) {
        this.consumerOnlyOrgOID = consumerOnlyOrgOID;
    }

    public void updateDocument(final String consentDirId,
        final String documentType, final List<FileItem> fileItems,
        final String userId, final String userFacilityStationId) {
        if (NullChecker.isNotEmpty(fileItems)) {
            final String document = this.retrieveDocument(consentDirId,
                documentType);
            Assert.assertNotEmpty(document, "Document cannot be empty!");
            // For each file item, create the media type
            try {
                Document result = null;
                final Document doc = this.stringToXml.transform(document);
                DOMSource source = new DOMSource(doc);
                for (final FileItem fileItem : fileItems) {
                    if (fileItem.getSize() > 0) {
                        final byte[] bytes = fileItem.get();
                        // JAXB automatically decodes the document when making
                        // it
                        // into XML
                        final String encodedString = Base64.encode(bytes);
                        final String contentType = fileItem.getContentType();
                        final Map<String, Object> dynamicParameters = new HashMap<String, Object>();
                        dynamicParameters.put("mediaType", contentType);
                        dynamicParameters.put("document", encodedString);
                        dynamicParameters.put("representation", "B64");
                        dynamicParameters.put("userId", userId);
                        dynamicParameters.put("userFacilityNumber",
                            userFacilityStationId);
                        DOMResult target = new DOMResult();
                        target = (DOMResult) this.updateConsentDirectiveDocumentStylesheet
                            .transform(source, target, dynamicParameters);
                        result = (Document) target.getNode();
                        source = new DOMSource(result);
                    }
                }

                if (NullChecker.isNotEmpty(result)) {
                    final String updatedDoc = this.xmlToString
                        .transform(result);
                    final ConsentDirectiveUpdateRequest consentDirectiveUpdateRequest = new ConsentDirectiveUpdateRequest();
                    final ServiceConsumerContextType sc = this
                        .createServiceConsumerContext(userId,
                            userFacilityStationId);
                    consentDirectiveUpdateRequest.setServiceConsumerContext(sc);
                    consentDirectiveUpdateRequest.setDocument(updatedDoc
                        .getBytes());
                    final ConsentDirectiveUpdateResponse response = this.cms
                        .processConsentDirectiveUpdate(consentDirectiveUpdateRequest);
                    response.getConsentDirectiveReference();
                }
            } catch (final javax.xml.transform.TransformerException ex) {
                throw new RuntimeException(ex);
            } catch (final ConsentManagementServiceException ex) {
                throw new RuntimeException(ex);
            } catch (final TransformerException ex) {
                throw new RuntimeException(ex);
            }
        }
    }

    public void updateExpiringConsentConfiguration(String frequency, Long window, String emailAddresses) {
        ExpiringConsentConfigurationRequest configRequest
            = new ExpiringConsentConfigurationRequest(frequency, window, emailAddresses);
        this.ecs.updateConfiguration(configRequest);
    }

    /**
     * Gets the current expiring consent report configuration from the database. If the configuration is not set, default values are
     * returned and these defaults are then saved to the database for the configuration.
     *
     * @return A map of strings that specify the expiring consent configuration
     */
    public Map<String, String> getExpiringConsentConfiguration() {
        ExpiringConsentConfigurationResponse configResponse
            = this.ecs.getConfiguration(new ExpiringConsentConfigurationRequest());

        final Map<String, String> resultMap = new HashMap<String, String>();
        if (configResponse != null) {
            resultMap.put("frequency", configResponse.getConfigurationReference().getFrequency());
            resultMap.put("window", configResponse.getConfigurationReference().getWindow().toString());
            resultMap.put("emailAddresses", configResponse.getConfigurationReference().getEmailAddresses());
        } else {
            // Use default configuration values
            resultMap.put("frequency", DEFAULT_FREQUENCY);
            resultMap.put("window", DEFAULT_WINDOW.toString());
            resultMap.put("emailAddresses", DEFAULT_EMAILS);

            // Set configuration to default values
            this.updateExpiringConsentConfiguration(DEFAULT_FREQUENCY,
                DEFAULT_WINDOW, DEFAULT_EMAILS);
        }

        return resultMap;
    }

    public ByteArrayInputStream getExpiringConsentReportLetters(Date startDate, Date endDate) throws IOException {
        ConsentDirectiveDetailedExpirationRequest expirationRequest
            = new ConsentDirectiveDetailedExpirationRequest();
        expirationRequest.setStartDate(startDate);
        expirationRequest.setEndDate(endDate);
        ExpiringConsentLetterResponse response = this.ecs.getReportLetters(expirationRequest);

        byte[] inBytes;
        if (response != null) {
            inBytes = response.getDownloadByteArray();
        } else {
            inBytes = new byte[0];
        }

        return new ByteArrayInputStream(inBytes);
    }

    public ByteArrayInputStream getExpiringConsentReportLetters(ConsentDirectiveDetailedExpirationRequest expirationRequest) throws IOException {
        ExpiringConsentLetterResponse response = this.ecs.getReportLetters(expirationRequest);

        byte[] inBytes;
        if (response != null) {
            inBytes = response.getDownloadByteArray();
        } else {
            inBytes = new byte[0];
        }

        return new ByteArrayInputStream(inBytes);
    }

}
