package gov.va.nvap.web.helper.document;

import gov.va.nvap.common.transformer.Transformer;
import gov.va.nvap.common.transformer.TransformerException;
import gov.va.nvap.common.transformer.xml.StringToXML;
import gov.va.nvap.common.uuid.UUIDUtil;
import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.common.xpath.XPathException;
import gov.va.nvap.common.xpath.XPathUtil;
import gov.va.nvap.service.adapter.doc.AdapterDocQueryRetrieve;
import gov.va.nvap.service.adapter.doc.AdapterException;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Date;
import java.util.HashMap;

import javax.servlet.http.HttpServletResponse;

import org.apache.xerces.impl.dv.util.Base64;
import org.springframework.beans.factory.annotation.Required;
import org.w3c.dom.Document;
import org.w3c.dom.Node;

/**
 * Helper class to help query and retrieve of documents.
 * 
 * @author Asha Amritraj
 * 
 */
public class DocumentHelper {
	/**
	 * The adapter document query bean.
	 */
	private AdapterDocQueryRetrieve adapterDocQueryRetrieve;
	/**
	 * The default stylesheet the would be displayed to the user.
	 */
	private String defaultStylesheet;
	/**
	 * Flag for showing XML view.
	 */
	private boolean isXmlViewEnabled;
	/**
	 * A default document repository id to get the data from. Adapter hard codes
	 * a default id to decide whether the adapter doc retrieve would go to the
	 * NwHIN or local.
	 */
	private String localDocumentRepositoryId;
	/**
	 * The maximum size for a file upload.
	 */
	private long maxUploadFileSize;

	/**
	 * For transforming String to XML;
	 */
	private StringToXML stringToXML;

	/**
	 * A bunch of style sheet transformer map, to show different views on the
	 * GUI.
	 */
	private HashMap<String, Transformer<Document, String>> stylesheetTransformers;

	/**
	 * Get the C32 document by calling AdapterDocQueryRetrieve on the NHIN
	 * adapter.
	 */
	public String getC32Document(final String icn, final String userName)
			throws AdapterException {
		// Use current dates to get the VA C32
		final String document = this.adapterDocQueryRetrieve.getDocument(icn,
				new Date(), new Date(), userName);
		return document;
	}

	public String getC62Attachment(final String document)
			throws TransformerException, XPathException {
		final Document xmlDoc = this.stringToXML.transform(document);
		// TODO: Move to Spring
		final Node node = XPathUtil.getNodeByPath(xmlDoc,
				"//*[local-name()='nonXMLBody']/*[local-name()='text']/text()");

		String textContent = null;
		if (NullChecker.isNotEmpty(node)) {
			// Get the text
			textContent = node.getNodeValue().trim();
			if (NullChecker.isNotEmpty(textContent)) {
				textContent = textContent.replaceAll("\\s+", "");
			}
		}

		return textContent;

	}

	/**
	 * Get the document by calling NHIN AdapterDocQueryRetrieve on the NHIN
	 * adapter.
	 */
	public String getDocument(final String icn, final String documentUniqueId,
			final String userName, final String homeCommunityId)
			throws AdapterException {
		// Pass the local document repository ID which is hardcoded in the
		// spring properties
		final String document = this.adapterDocQueryRetrieve.getDocument(icn,
				documentUniqueId, this.localDocumentRepositoryId,
				homeCommunityId, userName);
		return document;

	}

	/**
	 * Convert XML to Html by using the appropriate style sheet.
	 */
	public String getHtml(final String cdaDocument, final String styleSheetType) {
		try {
			// Convert to XML
			final Document doc = new StringToXML().transform(cdaDocument);
			// Check if it is a known style sheet
			if (NullChecker.isNotEmpty(styleSheetType)
					&& this.stylesheetTransformers.containsKey(styleSheetType)) {
				// Apply the transformer
				final Transformer<Document, String> transformer = this.stylesheetTransformers
						.get(styleSheetType);
				final String html = transformer.transform(doc);
				return html;
			} else {
				Assert.assertTrue(this.stylesheetTransformers
						.containsKey(this.defaultStylesheet),
						"Default stylesheet for view must be set");
				// Use a default style sheet defined in the spring properties
				final Transformer<Document, String> defaultTransformer = this.stylesheetTransformers
						.get(this.defaultStylesheet);
				final String html = defaultTransformer.transform(doc);
				return html;
			}
		} catch (final TransformerException ex) {
			throw new RuntimeException(ex);
		}
	}

	public long getMaxUploadFileSize() {
		return this.maxUploadFileSize;
	}

	public String getPrivacyConsentDirectiveAttachment(final String document)
			throws TransformerException, XPathException {

		final Document xmlDoc = this.stringToXML.transform(document);
		// TODO: Move to Spring
		final Node node = XPathUtil
				.getNodeByPath(xmlDoc,
						"//*[local-name()='observationMedia']/*[local-name()='value']/text()");
		String textContent = null;
		if (NullChecker.isNotEmpty(node)) {
			// Get the text
			textContent = node.getNodeValue().trim();
			if (NullChecker.isNotEmpty(textContent)) {
				textContent = textContent.replaceAll("\\s+", "");
			}
		}
		return textContent;
	}

	public String getPrivacyConsentDirectiveMediaType(final String document)
			throws TransformerException, XPathException {

		final Document xmlDoc = this.stringToXML.transform(document);
		// TODO: Move to Spring
		final Node node = XPathUtil
				.getNodeByPath(xmlDoc,
						"//*[local-name()='observationMedia']/*[local-name()='value']/@mediaType");
		String mediaType = null;
		if (NullChecker.isNotEmpty(node)) {
			// Get the text
			mediaType = node.getNodeValue().trim();
			if (NullChecker.isNotEmpty(mediaType)) {
				mediaType = mediaType.replaceAll("\\s+", "");
			}
		}
		return mediaType;
	}

	public boolean isXmlViewEnabled() {
		return this.isXmlViewEnabled;
	}

	@Required
	public void setAdapterDocQueryRetrieve(
			final AdapterDocQueryRetrieve adapterDocQueryRetrieve) {
		this.adapterDocQueryRetrieve = adapterDocQueryRetrieve;
	}

	@Required
	public void setDefaultStylesheet(final String defaultStylesheet) {
		this.defaultStylesheet = defaultStylesheet;
	}

	@Required
	public void setLocalDocumentRepositoryId(
			final String localDocumentRepositoryId) {
		this.localDocumentRepositoryId = localDocumentRepositoryId;
	}

	@Required
	public void setMaxUploadFileSize(final long maxUploadFileSize) {
		// Multiply to convert from MB to bytes
		this.maxUploadFileSize = maxUploadFileSize;
	}

	@Required
	public void setStringToXML(final StringToXML stringToXML) {
		this.stringToXML = stringToXML;
	}

	@Required
	public void setStylesheetTransformers(
			final HashMap<String, Transformer<Document, String>> stylesheetTransformers) {
		this.stylesheetTransformers = stylesheetTransformers;
	}

	@Required
	public void setXmlViewEnabled(final boolean isXmlViewEnabled) {
		this.isXmlViewEnabled = isXmlViewEnabled;
	}

	public void writeAttachmentToStream(final HttpServletResponse response,
			final String content, final MediaType mediaType,
			final RepresentationType representation) throws IOException {
		Assert.assertNotEmpty(content, "Content cannot be empty!");

		// Decode if encoded with Base 64
		final byte decoded[] = RepresentationType.B64.equals(representation) ? Base64
				.decode(content) : content.getBytes();

		// Set the appropriate header in the response
		response.setContentType(mediaType.mediaType());
		response.setHeader("Content-Disposition", "attachment; filename=\""
				+ UUIDUtil.generateMessageId() + "." + mediaType.extension()
				+ "\"");
		response.setHeader("Cache-Control", "must-revalidate");
		// Fill up the response and send it back
		final OutputStream outStream = response.getOutputStream();
		outStream.write(decoded, 0, decoded.length);
		outStream.close();
	}
}
