package gov.va.nvap.web.dao;

import gov.va.nvap.svc.consentmgmt.jpa.exceptions.NonexistentEntityException;
import gov.va.nvap.svc.consentmgmt.jpa.exceptions.PreexistingEntityException;
import gov.va.nvap.web.user.document.UserDocument;
import java.util.ArrayList;
import java.util.Calendar;

import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import org.springframework.transaction.annotation.Transactional;

/**
 *
 * @author Zack Peterson
 */
public class UserDocumentDAO {

	@PersistenceContext
	private EntityManager em;

    /**
     * Set the entity manager (Injected by Spring).
     * @param em
     */
    public void setEntityManager(final EntityManager em) {
        this.em = em;
    }

    @Transactional
	public void create(final UserDocument userDocument)
			throws PreexistingEntityException, Exception {
		try {
			this.em.persist(userDocument);
		} catch (final Exception ex) {
			if (this.findUserDocument(userDocument.getDocumentId()) != null) {
				throw new PreexistingEntityException("UserDocument "
						+ userDocument + " already exists.", ex);
			}
			throw ex;
		}
	}

    @Transactional
	public void destroy(final Long id) throws NonexistentEntityException {
		UserDocument userDocument;
		try {
			userDocument = this.em.getReference(
					UserDocument.class, id);
			userDocument.getDocumentId();
		} catch (final EntityNotFoundException enfe) {
			throw new NonexistentEntityException("The userDocument with id "
					+ id + " no longer exists.", enfe);
		}
		this.em.remove(userDocument);
	}

    @Transactional
	public void edit(UserDocument userDocument)
			throws NonexistentEntityException, Exception {
		try {
			this.em.merge(userDocument);
		} catch (final Exception ex) {
			final String msg = ex.getLocalizedMessage();
			if ((msg == null) || (msg.length() == 0)) {
				final Long id = userDocument.getDocumentId();
				if (this.findUserDocument(id) == null) {
					throw new NonexistentEntityException(
							"The userDocument with id " + id
									+ " no longer exists.");
				}
			}
			throw ex;
		}
	}
    
    @Transactional
    public void removeUserDocuments() {
        Calendar cal = Calendar.getInstance();
        cal.add(Calendar.MONTH, -1);
        Date expiredDate = cal.getTime();
        
        final Query q = this.em.createNamedQuery("UserDocument.findByBeforeDate");
        q.setParameter("cutoffDate", expiredDate);
        
        ArrayList<UserDocument> docs = (ArrayList<UserDocument>)q.getResultList();
        
        for (UserDocument doc : docs) {
            this.em.remove(doc);
        }
        Logger.getLogger(UserDocumentDAO.class.getName()).log(Level.INFO, "Scheduled exports created before " + expiredDate.toLocaleString() + " removed.");
    }

	public UserDocument findUserDocument(final Long id) {
		return this.em.find(UserDocument.class, id);
	}

	private List<UserDocument> findUserDocumentEntities(
			final boolean all, final int maxResults, final int firstResult) {
		final Query q = this.em
				.createNamedQuery("UserDocument.findAll");
		if (!all) {
			q.setMaxResults(maxResults);
			q.setFirstResult(firstResult);
		}
		return q.getResultList();
	}

	public List<UserDocument> findUserDocumentEntities(
			final EntityManager em) {
		return this.findUserDocumentEntities(true, -1, -1);
	}

	public List<UserDocument> findUserDocumentEntities(
			final int maxResults, final int firstResult) {
		return this.findUserDocumentEntities(false, maxResults, firstResult);
	}

	public HashMap findByUserId(final String userId, final String sortBy, String sortOrder, final Integer maxRows, final Integer startRow) {

        String query = "SELECT d FROM UserDocument d WHERE d.userId = :userId";
        String countQuery = "SELECT COUNT(d.documentId) FROM UserDocument d WHERE d.userId = :userId";
        String queryParams = "";
        Integer sort = sortBy.isEmpty() ? 1 : Integer.parseInt(sortBy);
        
        switch (sort) {
            case 0:
                queryParams += " ORDER BY d.title";
                break;
            case 1:
                queryParams += " ORDER BY d.format";
                break;
            case 2: 
                queryParams += " ORDER BY d.creationDate";
                break; 
            case 3: 
                queryParams += " ORDER BY d.status";
                break; 
            default:
                queryParams += " ORDER BY d.creationDate";
                break;
        }
        
        sortOrder = sortOrder.isEmpty() ? " DESC" : " " + sortOrder;
        queryParams = queryParams + sortOrder;

        query = query + queryParams;
        Query q = em.createQuery(query).setFirstResult(startRow);

        if( maxRows != -1 ) //set the max rows if we have it
        {
            q.setMaxResults(maxRows);
        } 
        
		q.setParameter("userId", userId);
        
        Query cq = em.createQuery(countQuery);
        cq.setParameter("userId", userId);
        
        HashMap hm = new HashMap();
        hm.put("results", (List<UserDocument>) q.getResultList());
        hm.put("count", (Long) cq.getSingleResult());
        
        return hm;
	}

	public Collection<UserDocument> findByTitle(
			final String title) {
		final Query q = this.em
				.createNamedQuery("UserDocument.findByTitle");
		q.setParameter("title", title);
		return q.getResultList();
	}

	public Collection<UserDocument> findByUserIdAndTitle(
			final String userId, final String title) {
		final Query q = this.em
				.createNamedQuery("UserDocument.findByUserIdAndTitle");
		q.setParameter("userId", userId);
        q.setParameter("title", title);
		return q.getResultList();
	}

    public UserDocument findByCreationDate(final Date date) {
		final Query q = this.em.createNamedQuery("UserDocument.findByCreationDate");
		q.setParameter("creationDate", date);
		return (UserDocument) q.getSingleResult();
	}

	public int getUserDocumentCount(final EntityManager em) {
		final Query q = em.createQuery("select count(ud.*) from UserDocument ud");
		return ((Long) q.getSingleResult()).intValue();
	}
}
