package gov.va.nvap.web.dao;

import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.web.consent.audit.ActiveConsentSummary;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;

/**
 * This DAO handles summary Opt-In Patients report requests.
 * 
 * @author Irakli Kakushadze
 * @since 07/16/2016
 */
public class ActiveConsentSummaryDAO {
    
    @PersistenceContext
    private EntityManager em;

    // Public methods

    public List<ActiveConsentSummary> find(final SummaryRequest req) {
        final HashMap<String, Object> queryParams = new HashMap<String, Object>();
        final String stationNumberColumn = req.aggregateAtFacilityLevel ? "PARENT_STATION_NUMBER" : "FACILITY" ;
        String queryString = ActiveConsentSummary.activeConsentSummaryNativeSQL;
        String facilityFilter = "";
        
		if (req.stationNumbers == null || "ALL".equals(req.stationNumbers)) {
            if (req.includeUnknownVisn) {
                // User chose to include everything, so there's no need for the WHERE clause
            } else {
                // User chose to see everything BUT unknown VISN
                facilityFilter = "and ca.VISN_NUMBER is not null";
            }
        } else if (req.stationNumbers.length() == 0) {
            if (req.includeUnknownVisn) {
                // User chose to see unknown VISN only
                facilityFilter = "and ca.VISN_NUMBER is null";
            } else {
                // This should never be the case
            }
        } else {
			if (req.includeUnknownVisn) {
                // User chose to see selected station numbers AND unknown VISNs
				facilityFilter = "and (ca." + stationNumberColumn + " in (:stationNumbers) or ca.VISN_NUMBER is null)";
			} else {
                // User chose to see selected stations only
				facilityFilter = "and ca." + stationNumberColumn + " in (:stationNumbers)";
			}
            List<String> items = Arrays.asList(req.stationNumbers.split(","));
            queryParams.put("stationNumbers", items);
		}
        
        queryString = queryString.replace("/*facility_filter*/", facilityFilter);
        
        if (req.aggregateAtFacilityLevel) {
            queryString = queryString.replace("ca.FACILITY_NAME", "ca.PARENT_FACILITY_NAME");
        }        
        
        // Create query object
        Query query = this.em.createNativeQuery(queryString, ActiveConsentSummary.class);
        
        // Set query parameters, if any
        for (final Map.Entry<String, Object> entry : queryParams.entrySet()) {
            query.setParameter(entry.getKey(), entry.getValue());
        }
        
        // Execute the query
        List<ActiveConsentSummary> result = query.getResultList();

        return result;
    }

    public void setEntityManager(final EntityManager entityManager) {
        this.em = entityManager;
    }
    
    // Inner classes
    
    public class SummaryRequest {

        public boolean aggregateAtFacilityLevel;
        public boolean includeUnknownVisn;
        public String stationNumbers;

    }    

}
