package gov.va.nvap.web.consent.audit.dao;

import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.web.consent.audit.AuditedConsentSummary;
import java.io.IOException;
import java.io.StringReader;
import java.security.InvalidParameterException;
import java.util.Arrays;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import javax.persistence.EntityManager;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import javax.persistence.TemporalType;
import org.hibernate.CacheMode;
import org.hibernate.FlushMode;

/**
 * DAO for getting the events from the CONSENT_AUDIT table.
 *
 * @author Asha Amritraj
 * edited by Stephen Miller
 * edited by Irakli Kakushadze
 */
public class AuditedConsentDAO {

	/**
	 * Read buffer size.
	 */
	private static final int READ_BUFFER_SIZE = 4096;

	/**
	 * Entity Manager instance.
	 */
	@PersistenceContext
	private EntityManager em;

        private void appendGroupByClause(final StringBuffer buffer, final String str) {
		this.appendGroupByClause(buffer, str, " Group By ", " , ");
	}

	private void appendGroupByClause(final StringBuffer buffer,
			final String str, final String empty, final String notEmpty) {
		if (buffer.length() > 0) {
			buffer.append(notEmpty);
		} else {
			buffer.append(empty);
		}
		buffer.append(str);
	}
        
	private void appendWhereClause(final StringBuffer buffer, final String str) {
		this.appendWhereClause(buffer, str, " WHERE ", " AND ");
	}

	private void appendWhereClause(final StringBuffer buffer, final String str,
			final String empty, final String notEmpty) {
		if (buffer.length() > 0) {
			buffer.append(notEmpty);
		} else {
			buffer.append(empty);
		}
		buffer.append(str);
	}

    private void handleCommonFilters(final Date startDate, final Date endDate,
        final String consentType, final int patientTypes,
        final StringBuffer whereClause, final HashMap<String, Object> setParams) {
        // Start date
        if (startDate != null) {
            this.appendWhereClause(whereClause, "a.timeOfEvent >= :startDate");
            setParams.put("startDate", startDate);
        }

        // End date
        if (endDate != null) {
            this.appendWhereClause(whereClause, "a.timeOfEvent <= :endDate");
            setParams.put("endDate", endDate);
        }

        // Consent type
        if (!NullChecker.isNullOrEmpty(consentType)) {
            if (consentType.equalsIgnoreCase("allAuthorizations")) {
                this.appendWhereClause(whereClause, "a.actionType = :actionType");
                setParams.put("actionType", "OPT-IN");
            } else if (consentType.equalsIgnoreCase("allRevocations")) {
                this.appendWhereClause(whereClause, "a.actionType = :actionType");
                setParams.put("actionType", "OPT-OUT");
            } else if (consentType.equalsIgnoreCase("NwHINAuthorizations")) {
                this.appendWhereClause(whereClause, "a.consentType = :consentType");
                setParams.put("consentType", "NwHIN Authorization");
            } else if (consentType.equalsIgnoreCase("SSAAuthorizations")) {
                this.appendWhereClause(whereClause, "a.consentType = :consentType");
                setParams.put("consentType", "SSA Authorization");
            } else if (consentType.equalsIgnoreCase("NwHINRevocations")) {
                this.appendWhereClause(whereClause, "a.consentType = :consentType");
                setParams.put("consentType", "NwHIN Revocation");
            } else if (consentType.equalsIgnoreCase("SSARevocations")) {
                this.appendWhereClause(whereClause, "a.consentType = :consentType");
                setParams.put("consentType", "SSA Revocation");
            } else if (consentType.equalsIgnoreCase("NwHINRestrictions")) {
                this.appendWhereClause(whereClause, "a.consentType = :consentType");
                setParams.put("consentType", "NwHIN Organization Restriction Authorization");
            } else if (consentType.equalsIgnoreCase("NwHINRestrictionRevocations")) {
                this.appendWhereClause(whereClause, "a.consentType = :consentType");
                setParams.put("consentType", "NwHIN Organization Restriction Revocation");
            } else {
				//TODO throw an error
                //program did not receive expected value
            }
        }

        // Patient types
        if (patientTypes == 1) {
            this.appendWhereClause(whereClause, "a.isTestPatient = 0");
        }
        if (patientTypes == 2) {
            this.appendWhereClause(whereClause, "a.isTestPatient = 1");
        }
    }

    /**
     * Get the events from the database table by building and calling a query. Use this method for complex queries of details. It accepts
     * all parameters and builds the where clause based on values supplied.
     *
     * @param req Request object containing search field values
     * @return List of objects
     */
    public final List<Object> getEvents(DetailRequest req) {

        final StringBuffer whereClause = new StringBuffer();
        final StringBuffer sortClause = new StringBuffer();
        final HashMap<String, Object> setParams = new HashMap<String, Object>();

        handleCommonFilters(req.startDate, req.endDate, req.consentType, req.patientTypes, whereClause, setParams);

        if (!NullChecker.isNullOrEmpty(req.patientSsn)) {
            this.appendWhereClause(whereClause, "a.patientSsn = :patientSsn");
            setParams.put("patientSsn", req.patientSsn);
        }

        if (!NullChecker.isNullOrEmpty(req.patientFirstName)) {
            this.appendWhereClause(whereClause, "upper(a.patientGivenName) = :firstName");
            setParams.put("firstName", req.patientFirstName.toUpperCase());
        }

        if (!NullChecker.isNullOrEmpty(req.patientLastName)) {
            this.appendWhereClause(whereClause, "upper(a.patientLastName) = :lastName");
            setParams.put("lastName", req.patientLastName.toUpperCase());
        }

        if ("ALL".equals(req.stationNumbers)) {
            if (req.includeUnknownVisn) {
                // User chose to include everything, so there's no need for the WHERE clause
            } else {
                // User chose to see everything BUT unknown VISN
                this.appendWhereClause(whereClause, "a.visnNumber IS NOT NULL");
            }
        } else if (NullChecker.isNullOrEmpty(req.stationNumbers)) {
            if (req.includeUnknownVisn) {
                // User chose to see unknown VISN only
                this.appendWhereClause(whereClause, "a.visnNumber IS NULL");
            } else {
                // This should never be the case
            }
        } else {
            if (req.includeUnknownVisn) {
                // User chose to see selected station numbers AND unknown VISNs
                this.appendWhereClause(whereClause, "(a.facility IN (:stationNumbers) OR a.visnNumber IS NULL)");
            } else {
                // User chose to see selected stations only
                this.appendWhereClause(whereClause, "a.facility IN (:stationNumbers)");
            }
            List<String> items = Arrays.asList(req.stationNumbers.split(","));
            setParams.put("stationNumbers", items);
        }

        if (!NullChecker.isNullOrEmpty(req.inactivationReason)) {
            this.appendWhereClause(whereClause, "a.optoutReason = :optoutReason");
            setParams.put("optoutReason", req.inactivationReason);
        }

        if (!NullChecker.isNullOrEmpty(req.userId)) {
            if (req.userId.toUpperCase(Locale.ENGLISH).equals("EMPLOYEE")) {
                this.appendWhereClause(whereClause, "lower(a.userId) not like 'ebenefits'");
                this.appendWhereClause(whereClause, "lower(a.userId) not like 'automatic service'");
                this.appendWhereClause(whereClause, "lower(a.userId) not like 'system'");
            } else {
                this.appendWhereClause(whereClause, "lower(a.userId) like lower(:userId)");
                setParams.put("userId", "%" + req.userId + "%");
            }
        }

        String validatedSortField = validateSortField(req.sortField);
        String validatedSortDirection = validateSortDirection(req.sortDirection);
        if (!NullChecker.isNullOrEmpty(validatedSortField) && !NullChecker.isNullOrEmpty(validatedSortDirection)) {
            if ("timeOfEvent".equals(validatedSortField)) {
                sortClause.append("a.timeOfEvent ");
                sortClause.append(validatedSortDirection);
            } else if ("createdDate".equals(validatedSortField)) {
                sortClause.append("a.createdDate ");
                sortClause.append(validatedSortDirection);
            } else if ("visnName".equals(validatedSortField)) {
                sortClause.append("a.visnNumber ");
                sortClause.append(validatedSortDirection);
                sortClause.append(", a.timeOfEvent DESC");
            } else {
                sortClause.append("upper(a.");
                sortClause.append(validatedSortField);
                sortClause.append(") ");
                sortClause.append(validatedSortDirection);
                sortClause.append(", a.timeOfEvent DESC");
            }
        } else {
            sortClause.append("a.timeOfEvent DESC"); //default sort
        }

        final Query query = this.makeAuditedConsentQuery(whereClause, sortClause);

        for (final Map.Entry<String, Object> entry : setParams.entrySet()) {
            String pKey = entry.getKey();
            Object pValue = entry.getValue();
            if (pKey.equalsIgnoreCase("startDate") || pKey.equalsIgnoreCase("endDate")) {
                query.setParameter(pKey, (Date) pValue, TemporalType.TIMESTAMP);
            } else {
                query.setParameter(pKey, pValue);
            }
        }

        final Query updatedQuery = this.setAdditionalQueryParamaters(query);
        return this.getResults(updatedQuery, req.currentPage, req.pageSize);
    }
    
    public final long getEventsCount(DetailRequest req) {

        final StringBuffer whereClause = new StringBuffer();
        final HashMap<String, Object> setParams = new HashMap<>();

        handleCommonFilters(req.startDate, req.endDate, req.consentType, req.patientTypes, whereClause, setParams);

        if (!NullChecker.isNullOrEmpty(req.patientSsn)) {
            this.appendWhereClause(whereClause, "a.patientSsn = :patientSsn");
            setParams.put("patientSsn", req.patientSsn);
        }

        if (!NullChecker.isNullOrEmpty(req.patientFirstName)) {
            this.appendWhereClause(whereClause, "upper(a.patientGivenName) = :firstName");
            setParams.put("firstName", req.patientFirstName.toUpperCase());
        }

        if (!NullChecker.isNullOrEmpty(req.patientLastName)) {
            this.appendWhereClause(whereClause, "upper(a.patientLastName) = :lastName");
            setParams.put("lastName", req.patientLastName.toUpperCase());
        }

        if ("ALL".equals(req.stationNumbers)) {
            if (req.includeUnknownVisn) {
                // User chose to include everything, so there's no need for the WHERE clause
            } else {
                // User chose to see everything BUT unknown VISN
                this.appendWhereClause(whereClause, "a.visnNumber IS NOT NULL");
            }
        } else if (NullChecker.isNullOrEmpty(req.stationNumbers)) {
            if (req.includeUnknownVisn) {
                // User chose to see unknown VISN only
                this.appendWhereClause(whereClause, "a.visnNumber IS NULL");
            } else {
                // This should never be the case
            }
        } else {
            if (req.includeUnknownVisn) {
                // User chose to see selected station numbers AND unknown VISNs
                this.appendWhereClause(whereClause, "(a.facility IN (:stationNumbers) OR a.visnNumber IS NULL)");
            } else {
                // User chose to see selected stations only
                this.appendWhereClause(whereClause, "a.facility IN (:stationNumbers)");
            }
            List<String> items = Arrays.asList(req.stationNumbers.split(","));
            setParams.put("stationNumbers", items);
        }

        if (!NullChecker.isNullOrEmpty(req.inactivationReason)) {
            this.appendWhereClause(whereClause, "a.optoutReason = :optoutReason");
            setParams.put("optoutReason", req.inactivationReason);
        }

        if (!NullChecker.isNullOrEmpty(req.userId)) {
            if (req.userId.toLowerCase().equals("employee")) {
                this.appendWhereClause(whereClause, "lower(a.userId) not like 'ebenefits'");
                this.appendWhereClause(whereClause, "lower(a.userId) not like 'automatic service'");
                this.appendWhereClause(whereClause, "lower(a.userId) not like 'system'");
            } else if (req.consentType.equalsIgnoreCase("SSAAuthorizations") && req.userId.toLowerCase().equals("automated")) {
                this.appendWhereClause(whereClause, "lower(a.userId) like '%ssa.gov%'");
            } else {
                this.appendWhereClause(whereClause, "lower(a.userId) like lower(:userId)");
                setParams.put("userId", "%" + req.userId + "%");
            }
        }

        final Query query = this.makeAuditedConsentCountQuery(whereClause);

        for (final Map.Entry<String, Object> entry : setParams.entrySet()) {
            String pKey = entry.getKey();
            Object pValue = entry.getValue();
            if (pKey.equalsIgnoreCase("startDate") || pKey.equalsIgnoreCase("endDate")) {
                query.setParameter(pKey, (Date) pValue, TemporalType.TIMESTAMP);
            } else {
                query.setParameter(pKey, pValue);
            }
        }

        final Query updatedQuery = this.setAdditionalQueryParamaters(query);
        
        return (long) updatedQuery.getSingleResult();
    }

	/**
	 * Get the event summaries from the database table by building and calling a
	 * query. Use this method for complex queries resulting in a summary. It
	 * accepts all parameters and builds the where and order by clauses by the
	 * values supplied.
	 */
	public final List<AuditedConsentSummary> getEventsSummary(final SummaryRequest req) {

        final StringBuffer whereClause = new StringBuffer();
        final StringBuffer groupByClause = new StringBuffer();
        final HashMap<String, Object> setParams = new HashMap<String, Object>();
        final String facilityNameColumn = req.aggregateAtFacilityLevel ? "parentFacilityName" : "facilityName";
        final String stationNumberColumn = req.aggregateAtFacilityLevel ? "parentStationNumber" : "facility";

        // Construct WHERE clause
        handleCommonFilters(req.startDate, req.endDate, req.consentType, req.patientTypes, whereClause, setParams);

        if ("ALL".equals(req.stationNumbers)) {
            if (req.includeUnknownVisn) {
                // User chose to include everything, so there's no need for the WHERE clause
            } else {
                // User chose to see everything BUT unknown VISN
                this.appendWhereClause(whereClause, "a.visnNumber IS NOT NULL");
            }
        } else if (NullChecker.isNullOrEmpty(req.stationNumbers)) {
            if (req.includeUnknownVisn) {
                // User chose to see unknown VISN only
                this.appendWhereClause(whereClause, "a.visnNumber IS NULL");
            } else {
                // This should never be the case
            }
        } else {
            if (req.includeUnknownVisn) {
                // User chose to see selected station numbers AND unknown VISNs
                this.appendWhereClause(whereClause, "(a." + stationNumberColumn + " IN (:stationNumbers) OR a.visnNumber IS NULL)");
            } else {
                // User chose to see selected stations only
                this.appendWhereClause(whereClause, "a." + stationNumberColumn + " IN (:stationNumbers)");
            }
            List<String> items = Arrays.asList(req.stationNumbers.split(","));
            setParams.put("stationNumbers", items);
        }

        if (!NullChecker.isNullOrEmpty(req.userId)) {
            if (req.userId.toUpperCase(Locale.ENGLISH).equals("EMPLOYEE")) {
                this.appendWhereClause(whereClause, "lower(a.userId) not like 'ebenefits'");
                this.appendWhereClause(whereClause, "lower(a.userId) not like 'automatic service'");
                this.appendWhereClause(whereClause, "lower(a.userId) not like 'system'");
            } else {
                this.appendWhereClause(whereClause, "lower(a.userId) like lower(:userId)");
                setParams.put("userId", "%" + req.userId + "%");
            }
        }

        // Construct GROUP BY clause
        this.appendGroupByClause(groupByClause, "a." + facilityNameColumn + ", a.consentType");

        // Construct ORDER BY clause
        String orderBy = null;
        final Query query;
        String validatedSortField = validateSortFieldForSummary(req.sortField);
        String validatedSortDirection = validateSortDirection(req.sortDirection);
        if (!NullChecker.isNullOrEmpty(validatedSortField) && !NullChecker.isNullOrEmpty(validatedSortDirection)) {
            if (validatedSortField.equalsIgnoreCase("facilityName")) {
                orderBy = "LOWER(a." + facilityNameColumn + ") " + validatedSortDirection + ", a.consentType ASC";
            } else if (validatedSortField.equalsIgnoreCase("consentType")) {
                orderBy = "a.consentType " + validatedSortDirection + ", LOWER(a." + facilityNameColumn + ") ASC";
            } else {
                orderBy = "COUNT(*) " + validatedSortDirection + ", LOWER(a." + facilityNameColumn + ") ASC";
            }
        } else {
            orderBy = "LOWER(a." + facilityNameColumn + ") ASC, a.consentType ASC";
        }
        
        // Create the query
        final String queryString =
            new String(" SELECT NEW gov.va.nvap.web.consent.audit.AuditedConsentSummary " +
            " (a.consentType, a." + facilityNameColumn + ", COUNT(a.consentType)) " +
            " FROM AuditedConsentEx a " +
            whereClause.toString() +
            groupByClause.toString() +
            " ORDER BY " + orderBy.toString());
        
        query = this.em.createQuery(queryString);
        
        // Set query parameters
        for (final Map.Entry<String, Object> entry : setParams.entrySet()) {
                String pKey = entry.getKey();
                Object pValue = entry.getValue();
                if (pKey.equalsIgnoreCase("startDate") || pKey.equalsIgnoreCase("endDate")) {
                        query.setParameter(pKey, (Date)pValue, TemporalType.TIMESTAMP);
                } else {
                        query.setParameter(pKey, pValue);
                }
        }

        // Set additional Hibernate parameters
        final Query updatedQuery = this.setAdditionalQueryParamaters(query);
        Assert.assertNotEmpty(updatedQuery, "Query cannot be empty");
        
		// Return the result set
		return updatedQuery.getResultList();
	}

	/**
	 * Get the message content from the table. We do not want hibernate to pull
	 * all the records with the clob content. So have an independent query to
	 * just pull messages from the clob.
	 */
	public final String getMessageContent(final Long consentAuditId) {
		Assert.assertNotEmpty(consentAuditId,
				"Audited event id cannot be empty");
		// Query to get the individual clob content
		final Query query = this.em
				.createQuery("SELECT ca.detail from ConsentAudit ca where ca.consentAuditId = :consentAuditId");
		query.setParameter("consentAuditId", consentAuditId);

		final Query updatedQuery = this.setAdditionalQueryParamaters(query);
		//if(Clob.class.isInstance(Clob.class)) {
			//final Clob clob = (Clob) updatedQuery.getSingleResult();
		try {
			final String mydetail = (String) updatedQuery.getSingleResult();	
			if (NullChecker.isNotEmpty(consentAuditId) && !NullChecker.isNullOrEmpty(mydetail)) {
				final StringReader reader = new StringReader(mydetail);
				if (NullChecker.isEmpty(reader)) {
					return null;
				}
				final StringBuffer sb = new StringBuffer();
				final char[] charbuf = new char[AuditedConsentDAO.READ_BUFFER_SIZE];
				for (int i = reader.read(charbuf); i > 0; i = reader
						.read(charbuf)) {
					sb.append(charbuf, 0, i);
				}
				// Return the CDA R2 XML
				return sb.toString();
			}
		} catch (final IOException e) {
			throw new RuntimeException(e);
		}
		//}
		return null;
	}
        
	/**
	 * Get the results from the database table based on the patient id.
	 */
        public List<Object> getResultsByPatientId(String patientId) {
            Query query = this.em.createQuery("SELECT a FROM AuditedConsentEx a WHERE a.patientId = :patientId ORDER BY a.createdDate DESC");
            query.setParameter("patientId", patientId);

            return query.getResultList();
        }
   
	/**
	 * Get the results from the database table.
	 */
        public final List<Object> getResults(final Query query,
			final int currentPage, final int pageSize) {
		Assert.assertNotEmpty(query, "Query cannot be empty");
                List<Object> auditedEventEntityList;
		if (pageSize > 0) {
			auditedEventEntityList = query
					.setFirstResult(currentPage * pageSize)
					.setMaxResults(pageSize).getResultList();
		} else {
			auditedEventEntityList = query.getResultList();
		}
		return auditedEventEntityList;
	}

	private Query makeAuditedConsentQuery(final StringBuffer whereClause,
			final StringBuffer sortClause) {
		if (NullChecker.isNullOrEmpty(sortClause)) {
                        return this.em.createQuery("SELECT a FROM AuditedConsentEx a "
                                        + whereClause.toString());
		} else {
			return this.em.createQuery("SELECT a FROM AuditedConsentEx a "
                                        + whereClause.toString() + " ORDER BY " + sortClause.toString());
		}
	}
    
    /**
     * Counts the entries for the where clause parameters
     * @param whereClause
     * @return 
     */
    private Query makeAuditedConsentCountQuery(final StringBuffer whereClause){
        return this.em.createQuery("SELECT COUNT(*) FROM AuditedConsentEx a " + whereClause.toString());
    }

	/**
	 * Optimize the query parameters for fetching the result.
	 */
	private Query setAdditionalQueryParamaters(final Query theQuery) {
		theQuery.setHint("org.hibernate.cacheable", Boolean.TRUE)
				.setHint("org.hibernate.fetchSize", Integer.valueOf(50))
				.setHint("org.hibernate.cacheMode", CacheMode.GET)
				.setHint("org.hibernate.readOnly", Boolean.TRUE)
				.setHint("org.hibernate.flushMode", FlushMode.MANUAL);
		return theQuery;
	}

	/**
	 * Set the entity manager (Injected by Spring).
	 */
	public void setEntityManager(final EntityManager em) {
		this.em = em;
	}

	private String validateSortField(final String sortField) {
        if (NullChecker.isNullOrEmpty(sortField)) return "";
        if (sortField.equalsIgnoreCase("actionType")) return "actionType"; // Not used in UI?
        if (sortField.equalsIgnoreCase("consentAuditId")) return "consentAuditId"; // Not used in UI?
        if (sortField.equalsIgnoreCase("consentType")) return "consentType";
        if (sortField.equalsIgnoreCase("createdDate")) return "createdDate";
        if (sortField.equalsIgnoreCase("facility")) return "facility";
        if (sortField.equalsIgnoreCase("facilityName")) return "facilityName";
        if (sortField.equalsIgnoreCase("optoutReason")) return "optoutReason";
        if (sortField.equalsIgnoreCase("patientGivenName")) return "patientGivenName";
        if (sortField.equalsIgnoreCase("patientId")) return "patientId"; // Not used in UI?
        if (sortField.equalsIgnoreCase("patientLastName")) return "patientLastName";
        if (sortField.equalsIgnoreCase("patientMiddleName")) return "patientMiddleName"; // Not used in UI?
        if (sortField.equalsIgnoreCase("patientSsn")) return "patientSsn";
        if (sortField.equalsIgnoreCase("pouValue")) return "pouValue";
        if (sortField.equalsIgnoreCase("timeOfEvent")) return "timeOfEvent";
        if (sortField.equalsIgnoreCase("userId")) return "userId";
        if (sortField.equalsIgnoreCase("visnName")) return "visnName";
		
		throw new InvalidParameterException("Wrong column header to sort on: " + sortField);
	}

	/**
	 * Checks validity of columns to sort on for Summary report
	 */
    private String validateSortFieldForSummary(final String sortField) {
        if (NullChecker.isNullOrEmpty(sortField)) return "";
        if (sortField.equalsIgnoreCase("consentType")) return "consentType";
        if (sortField.equalsIgnoreCase("facilityName")) return "facilityName";
        if (sortField.equalsIgnoreCase("total")) return "total";

        throw new InvalidParameterException("Wrong column header to sort on in summary report: " + sortField);
    }

	private String validateSortDirection(final String sortDirection) {
        if ("ASC".equalsIgnoreCase(sortDirection)) return "ASC";
        if ("DESC".equalsIgnoreCase(sortDirection)) return "DESC";
		return "";
	}
    
    // Inner classes
    
    public class DetailRequest {

        public String consentType;
        public int currentPage;
        public Date endDate;
        public String inactivationReason;
        public boolean includeUnknownVisn;
        public int pageSize;
        public String patientFirstName;
        public String patientLastName;
        public String patientSsn;
        public int patientTypes;
        public String sortField;
        public String sortDirection;
        public Date startDate;
        public String stationNumbers;
        public String userId;

    }
    
    public class SummaryRequest {

        public boolean aggregateAtFacilityLevel;
        public String consentType;
        public Date endDate;
        public boolean includeUnknownVisn;
        public int patientTypes;
        public String sortField;
        public String sortDirection;
        public Date startDate;
        public String stationNumbers;
        public String userId;

    }
}
