package gov.va.nvap.web.app;

import gov.va.nvap.common.exception.StackTraceUtil;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.web.helper.document.DocumentHelper;
import gov.va.nvap.web.helper.document.MediaType;

import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServlet;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.FileUploadBase.SizeLimitExceededException;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.commons.validator.Validator;
import org.apache.commons.validator.ValidatorException;
import org.apache.commons.validator.ValidatorResources;
import org.apache.commons.validator.ValidatorResults;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;
import org.xml.sax.SAXException;

/**
 * The abstract response dispatcher servlet. It calls the appropriate method by
 * using reflection.
 * 
 * @author Asha Amritraj
 */
abstract public class ResponseDispatcherHttpServlet extends HttpServlet {
	/**
	 * Logger.
	 */
	static private final Log LOG = LogFactory
			.getLog(ResponseDispatcherHttpServlet.class);
	/**
	 * Serial UID.
	 */
	private static final long serialVersionUID = 8771859243653648687L;
	/**
	 * The validation file for validating fields.
	 */
	static private final String VALIDATION_FILE_PATH = "WEB-INF/validation.xml";
	/**
	 * Generic validation based on validation.xml file.
	 */
	private ValidatorResources validatorResources;
	/**
	 * The Spring web application context.
	 */
	protected WebApplicationContext webApplicationContext;

	@Override
	protected void doGet(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		try {
			this.processRequest(request, response);
		} catch (final Exception ex) {
			request.getSession(false).setAttribute("exception", ex);
			request.getSession(false).setAttribute("exceptionTrace",
					StackTraceUtil.getStackTraceUtilAsString(ex));
			if (RuntimeException.class.isInstance(ex)) {
				throw (RuntimeException) ex;
			} else {
				throw new ServletException(ex);
			}
		}
	}

	@Override
	protected void doPost(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		try {
			this.processRequest(request, response);
		} catch (final Exception ex) {
			request.getSession(false).setAttribute("exception", ex);
			request.getSession(false).setAttribute("exceptionTrace",
					StackTraceUtil.getStackTraceUtilAsString(ex));
			if (RuntimeException.class.isInstance(ex)) {
				throw (RuntimeException) ex;
			} else {
				throw new ServletException(ex);
			}
		}
	}

	/**
	 * Call a method using reflection based on a parameter name passed in the
	 * URL.
	 */
	private boolean executeMethod(final String parameterName,
			final HttpServletRequest request, final HttpServletResponse response)
			throws ServletException {
		try {
			final Method method = this.getClass().getDeclaredMethod(
					parameterName, HttpServletRequest.class,
					HttpServletResponse.class);
			method.invoke(this, request, response);
			return true;
		} catch (final NoSuchMethodException nsme) {
			return false;
		} catch (final InvocationTargetException ex) {
			throw new ServletException(ex.getTargetException());
		} catch (final IllegalAccessException ex) {
			throw new ServletException(ex);
		}
	}

	/**
	 * Special parameter name when uploading files in our GUI.
	 */
	private boolean executeMethod(final String parameterName,
			final List<FileItem> fileItem, final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException {
		try {
			final Method method = this.getClass().getDeclaredMethod(
					parameterName, List.class, HttpServletRequest.class,
					HttpServletResponse.class);
			method.invoke(this, fileItem, request, response);
			return true;
		} catch (final NoSuchMethodException nsme) {
			return false;
		} catch (final InvocationTargetException ex) {
			throw new ServletException(ex.getTargetException());
		} catch (final IllegalAccessException ex) {
			throw new ServletException(ex);
		}
	}

	/**
	 * If the URL starts with redirect: then redirect to a different location
	 * with the same parameters.
	 */
	protected void forward(final HttpServletRequest request,
			final HttpServletResponse response, final String actionName)
			throws IOException, ServletException {
		String url = this.getServletConfig().getInitParameter(actionName);

		if (!NullChecker.isNullOrEmpty(url)) {
			if (url.startsWith("redirect:")) {
				url = url.substring(url.indexOf(":") + 1);
				final String[] parts = url.split(",");
				if (parts.length > 1) {
					request.setAttribute("action", parts[0]);
					request.setAttribute("actionParams",
							Arrays.copyOfRange(parts, 1, parts.length));
					request.getRequestDispatcher("WEB-INF/web/redirect.jsp")
							.forward(request, response);
				} else {
					response.sendRedirect(parts[0]);
				}
			} else {
				request.getRequestDispatcher(url).forward(request, response);
			}
		} else {
			throw new ServletException("Action, " + actionName
					+ ", does not exist. Check the servlet's init parameters.");
		}
	}

	/**
	 * Convenience method to get the Spring Bean.
	 */
	public <E> E getBean(final String beanID, final Class<E> clazz) {
		return clazz.cast(this.webApplicationContext.getBean(beanID));
	}

	@Override
	public void init() throws ServletException {
		super.init();
		// Get the Spring Web application context
		this.webApplicationContext = WebApplicationContextUtils
				.getRequiredWebApplicationContext(this.getServletContext());
		// Load the validation file
		InputStream is =null;
		try {
			 is = this.getServletContext()
					.getResourceAsStream(
							ResponseDispatcherHttpServlet.VALIDATION_FILE_PATH);
			if (is != null) {
				this.validatorResources = new ValidatorResources(is);
			} else {
				throw new RuntimeException("Could not find "
						+ ResponseDispatcherHttpServlet.VALIDATION_FILE_PATH
						+ ".");
			}

		} catch (final SAXException t) {
			throw new ServletException(t);
		} catch (final IOException t) {
			throw new ServletException(t);
		}finally{
			try {
				is.close();
			} catch (IOException e) {
				// TODO Auto-generated catch block
				e.printStackTrace();
			}	
		}
	}

	/**
	 * The main entry point in this class. Looks at the parameter names and uses
	 * reflection to call a method with that name.
	 */
	protected void processRequest(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException,
			IOException {
		// Automatically call the method based on the button name.
		// The code, looks for methods with the same name specified in the
		// submit button and executes that method.

		final DocumentHelper helper = this.getBean("adapterDocumentHelper",
				DocumentHelper.class);
		if (NullChecker.isNullOrEmpty(request.getAttribute("formErrors"))) {
			boolean invokedMethod = false;
			if (!ServletFileUpload.isMultipartContent(request)) {
				for (final Enumeration<?> e = request.getParameterNames(); e
						.hasMoreElements();) {
					final String parameterName = (String) e.nextElement();
					invokedMethod = this.executeMethod(parameterName, request,
							response);
					if (invokedMethod) {
						break;
					}
				}
			} else {
				try {
					final ServletFileUpload servletFileUpload = new ServletFileUpload(
							new DiskFileItemFactory());
					// Get the MB size from Spring properties
					servletFileUpload.setSizeMax(helper.getMaxUploadFileSize());
					final List<?> fileItemsList = servletFileUpload
							.parseRequest(request);
					if (NullChecker.isNotEmpty(fileItemsList)) {
						final Iterator<?> it = fileItemsList.iterator();
						while (it.hasNext()) {
							final FileItem fileItemTemp = (FileItem) it.next();
							// Get the form fields to execute a specific method
							if (fileItemTemp.isFormField()) {
								final String fieldName = fileItemTemp
										.getFieldName();
								invokedMethod = this.executeMethod(fieldName,
										(List<FileItem>) fileItemsList,
										request, response);
								if (invokedMethod) {
									break;
								}
							} else {
								// If there is a file, then check if its a PDF.
								// Only PDFs are supported.
								if (fileItemTemp.getSize() > 0) {
									if (NullChecker.isEmpty(fileItemTemp
											.getContentType())) {
										throw new RuntimeException(
												"There has to be a content type for the file!");
									}
									if (MediaType.isSupported(fileItemTemp
											.getContentType())) {
										//
									} else {
										throw new ServletException(
												"Only the following file types "
														+ MediaType
																.getSupportedTypes()
														+ " are supported when uploading documents!");
									}
								}
							}
						}
					}
				} catch (final SizeLimitExceededException ex) {
					throw new ServletException(
							"The file size exceeded the allowed "
									+ helper.getMaxUploadFileSize() + " bytes.",
							ex);
				} catch (final FileUploadException ex) {
					throw new ServletException(ex);
				}
			}
			// If no matching method, then just call unspecified() method
			if (!invokedMethod) {
				this.unspecified(request, response);
			}
		} else {
			this.unspecified(request, response);
		}
	}

	/**
	 * Create a unique requestId for each request.
	 */
	protected void setRequestId(final HttpServletRequest request,
			final HttpServletResponse response) {
		final HttpSession session = request.getSession(false);
		// Set a request Id for each request.
		if (session != null) {
			session.setAttribute("requestId", UUID.randomUUID().toString());
		}
	}

	/**
	 * All servlets should implement the unspecified method!
	 */
	abstract protected void unspecified(HttpServletRequest request,
			HttpServletResponse response) throws ServletException, IOException;

	/**
	 * Validate the page if there are matching fields in the validation.xml
	 * file.
	 */
	protected boolean validate(final HttpServletRequest request,
			final HttpServletResponse response) throws ServletException {
		return this.validate(request, response, null);
	}

	/**
	 * Validate using the apache commons validator engine.
	 */
	protected boolean validate(final HttpServletRequest request,
			final HttpServletResponse response, final String subaction)
			throws ServletException {
		final HashMap<String, String> bean = new HashMap<String, String>();
		for (final Enumeration<?> e = request.getParameterNames(); e
				.hasMoreElements();) {
			final String key = (String) e.nextElement();
			bean.put(key, request.getParameter(key));
		}

		return this.validate(bean, request, response, subaction);
	}

	/**
	 * Validate using the apache commons validator engine.
	 */
	protected boolean validate(final Map<String, String> bean,
			final HttpServletRequest request,
			final HttpServletResponse response, final String subaction)
			throws ServletException {
		try {
			final String contextPath = request.getContextPath();
			String action = request.getRequestURI();
			if (!NullChecker.isNullOrEmpty(contextPath)) {
				action = action.substring(contextPath.length());
			}

			if (!NullChecker.isNullOrEmpty(subaction)) {
				action += ":" + subaction;
			}
			// Call the validator with the paramters
			final Validator validator = new Validator(this.validatorResources,
					action);
			validator.setUseContextClassLoader(true);
			validator.setParameter(Validator.BEAN_PARAM, bean);
			final ArrayList<?> messages = new ArrayList();
			validator.setParameter("java.util.ArrayList", messages);
			validator.setParameter("javax.servlet.http.HttpServletRequest",
					request);
			validator.setOnlyReturnErrors(true);
			// Validate
			final ValidatorResults results = validator.validate();
			request.setAttribute("formErrors", messages);
			return results.isEmpty();
		} catch (final ValidatorException ex) {
			throw new ServletException(ex);
		}
	}
}
