/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */

package gov.va.nvap.web.app;

import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.svc.consentmgmt.stub.dao.DescriptorDAO;
import gov.va.nvap.svc.consentmgmt.stub.data.Descriptor;
import gov.va.nvap.svc.facility.data.Facility;
import gov.va.nvap.web.dao.AllowedRoleDAO;
import gov.va.nvap.web.dao.UserRoleDAO;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.user.User;
import gov.va.nvap.web.user.role.UserNameException;
import gov.va.nvap.web.user.role.UserRoleException;

import java.io.IOException;
import java.io.PrintStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.util.HashMap;
import java.util.List;

import javax.servlet.Filter;
import javax.servlet.FilterChain;
import javax.servlet.FilterConfig;
import javax.servlet.ServletException;
import javax.servlet.ServletRequest;
import javax.servlet.ServletResponse;
import javax.servlet.http.HttpServletResponse;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpSession;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

/**
 * A filter to process errors.
 *
 * @author David Vazquez
 */
public class PermissionAppFilter implements Filter {
    private static final Log LOG = LogFactory.getLog(PermissionAppFilter.class);
    private static final String VA_USERNAME = "adSamAccountName";
    private static final String LOGOUT_PATH = "Logout.do_sec";

    protected WebApplicationContext webApplicationContext;

	public static String getStackTrace(final Throwable t) {
		String stackTrace = null;
		try {
			final StringWriter sw = new StringWriter();
			final PrintWriter pw = new PrintWriter(sw);
			t.printStackTrace(pw);
			pw.close();
			sw.close();
			stackTrace = sw.getBuffer().toString();
		} catch (final Exception ex) {
		}
		return stackTrace;
	}

	public PermissionAppFilter() {
	}

	@Override
	public void destroy() {
	}

	private void doAfterProcessing(final ServletRequest request,
			final ServletResponse response) throws IOException,
			ServletException {
	}

	private boolean doBeforeProcessing(final ServletRequest request,
			final ServletResponse response) throws IOException,
			ServletException {
		// Do not cache any pages
		boolean ret;
		final HttpServletResponse resp = (HttpServletResponse) response;
		resp.setHeader("Cache-control", "no-cache");
		resp.setHeader("Expires", "0");
		ret = true;
		return ret;
	}

	@Override
	public void doFilter(final ServletRequest request,
			final ServletResponse response, final FilterChain chain)
			throws IOException, ServletException {

        // If attempting to logout, skip filter
        if (((HttpServletRequest)request).getServletPath().contains(LOGOUT_PATH)) {
            chain.doFilter(request, response);
            return;
        }

		if (this.doBeforeProcessing(request, response)) {
			Throwable problem = null;
			try {
                // Get session from request for user and check if permitted
                HttpServletRequest httpRequest = (HttpServletRequest) request;
                HttpSession session = httpRequest.getSession();

                User user = (User) session.getAttribute("user");
                if (user == null) {
                    user = new User();
                }

                user.setName(httpRequest.getHeader(VA_USERNAME));
                if (!user.hasName()) throw new UserNameException("UserNameException");
                user.setName(user.getName().toLowerCase());

                if (!user.hasRoles()) {
                    // Attempt to use user name to get user roles
                    List<String> userRoles = this.getUserRoleDAO().findUserRoleNames(user.getName());
                    user.setRoles(userRoles);

                    if (!user.hasRoles()) {
                        throw new UserRoleException("UserRoleException");
                    }
                }

                if (!this.getAllowedRoleDAO().containsAllowedRole(user.getRoles())) {
                    throw new UserRoleException("UserRoleException");
                }

                // Set User object in session
                session.setAttribute("user", user);

                //okay we've made it this far, so we had better put the descriptors back into the session
                this.addDescriptors(session);

                // Set the default facility session atrribute and cookie
                this.setDefaultFacility((HttpServletRequest)request, (HttpServletResponse)response);

                // If user role is permitted, continue with the filter chain
                chain.doFilter(request, response);
			} catch (final IOException | ServletException t) {
				// If an exception is thrown somewhere down the filter chain,
				// we still want to execute our after processing, and then
				// rethrow the problem after that.
				problem = t;
				t.printStackTrace();
			}

			this.doAfterProcessing(request, response);

			// If there was a problem, we want to rethrow it if it is
			// a known type, otherwise log it.
			if (problem != null) {
				if (problem instanceof ServletException) {
					throw (ServletException) problem;
				}
				if (problem instanceof IOException) {
					throw (IOException) problem;
				}
				this.sendProcessingError(problem, response);
			}
		}
	}

	/**
	 * Init method for this filter
	 */
	@Override
	public void init(final FilterConfig filterConfig) {
        this.webApplicationContext =
                WebApplicationContextUtils.getRequiredWebApplicationContext(filterConfig.getServletContext());
	}

	private void sendProcessingError(final Throwable t,
			final ServletResponse response) {
		final String stackTrace = PermissionAppFilter.getStackTrace(t);

		if (NullChecker.isNotEmpty(stackTrace)) {
			try {
				response.setContentType("text/html");
				final PrintStream ps = new PrintStream(
						response.getOutputStream());
				final PrintWriter pw = new PrintWriter(ps);
				pw.print("<html>\n<head>\n<title>Error</title>\n</head>\n<body>\n"); // NOI18N

				// PENDING! Localize this for next official release
				pw.print("<h1>The resource did not process correctly</h1>\n<pre>\n");
				pw.print(stackTrace);
				pw.print("</pre></body>\n</html>"); // NOI18N
				pw.close();
				ps.close();
				response.getOutputStream().close();
			} catch (final Exception ex) {
			}
		} else {
			try {
				final PrintStream ps = new PrintStream(
						response.getOutputStream());
//				t.printStackTrace(ps);
				Logger.getLogger(PermissionAppFilter.class.getName()).log(Level.INFO, "The resource processed correctly");
				ps.close();
				response.getOutputStream().close();
			} catch (final Exception ex) {
			}
		}
	}

	@Override
	public String toString() {
		final StringBuffer sb = new StringBuffer("PermissionAppFilter()");
		return (sb.toString());
	}

    private void setDefaultFacility(HttpServletRequest request, HttpServletResponse response) {
        // get User's default facility from cookie or from VHAUserId and set it for the session
        Facility defaultUserFacility = null;
        Cookie[] vapCookies = request.getCookies();
        boolean cookieExists = false;
        if (vapCookies != null){
            for (Cookie vapCookie : vapCookies){
                if (vapCookie.getName().equals("facility")){
                    if (NullChecker.isNotEmpty(vapCookie.getValue())) {
                        defaultUserFacility = new Facility();
                        try {
                            Facility facility = this.getFacilityHelper().getFacilityByStationId(vapCookie.getValue());
                            if (facility != null) {
                                defaultUserFacility.setFacilityStation(facility.getFacilityStation());
                                defaultUserFacility.setFacilityName(facility.getFacilityName());
                                defaultUserFacility.setVisnId(facility.getVisnId());

                                cookieExists = true;
                            }
                        } catch (Exception e) {
                            // Cookie must contain an invalid station number. We'll ignore the error and set the default facility
                            // based on the user ID.
                        }
                    }
                }
            }
        }

        if (NullChecker.isEmpty(defaultUserFacility)) {
            // Get the Default Facility by user id
            defaultUserFacility = this.getFacilityHelper().getFacilityStationByVHAUserId(
                        request.getHeader(VA_USERNAME));
        }

        if (NullChecker.isNotEmpty(defaultUserFacility)) {
            request.getSession().setAttribute("defaultUserFacility", defaultUserFacility);

            if (!cookieExists) {
                response.addCookie(setFacilityCookie(defaultUserFacility));
            }
        }
    }

    private Cookie setFacilityCookie(Facility newDefaultFacility) {

		Cookie vapCookie = null;
		if (newDefaultFacility != null) {
			if (newDefaultFacility.getFacilityStation().equals("")) {
				vapCookie = new Cookie("facility", "");
				vapCookie.setMaxAge(0);
			} else if (isAlphaNumeric(newDefaultFacility.getFacilityStation())) {
				vapCookie = new Cookie("facility", newDefaultFacility.getFacilityStation());
				vapCookie.setMaxAge(5 * 365 * 24 * 60 * 60);
			}
		}
		return vapCookie;
	}

    private boolean isAlphaNumeric(String s){
        String pattern= "^[a-zA-Z0-9]*$";

        return s.matches(pattern);
    }

    public FacilityHelper getFacilityHelper() {
		final FacilityHelper facilityHelper = this.getBean("facilityHelper",
				FacilityHelper.class);
		return facilityHelper;
	}

    private AllowedRoleDAO getAllowedRoleDAO() {
        return this.getBean("AllowedRoleDAO", AllowedRoleDAO.class);
    }

    private UserRoleDAO getUserRoleDAO() {
        return this.getBean("UserRoleDAO", UserRoleDAO.class);
    }

    private DescriptorDAO getDescriptorDAO() {
        return this.getBean("DescriptorDAO", DescriptorDAO.class);
    }

    private void addDescriptors(HttpSession session) {
        List<Descriptor> dees = this.getDescriptorDAO().getAll();
        HashMap hm = new HashMap();

        for(Descriptor d : dees) {
            hm.put(d.getName(), d.getText());
        }

        session.setAttribute("descriptors", hm);
    }

    public <E> E getBean(final String beanID, final Class<E> clazz) {
		return clazz.cast(this.webApplicationContext.getBean(beanID));
	}
}
