/**
 *
 */
package gov.va.nvap.web.announce;

import gov.va.nvap.common.date.DateUtil;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.OrganizationType;
import gov.va.nvap.service.pdq.PatientMatchQuery;
import gov.va.nvap.service.pdq.PatientMatchResponse;
import gov.va.nvap.service.pdq.PatientProfile;
import gov.va.nvap.service.pdq.PdqService;
import gov.va.nvap.svc.consentmgmt.PIPInterface;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.Announcement;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.AnnouncementBatchSummary;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.AnnouncementOrg;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.AnnouncerInterface;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.PatientConsentBatchAnnounce;
import gov.va.nvap.svc.consentmgmt.stub.data.Organization;
import gov.va.nvap.web.app.ResponseDispatcherHttpServlet;
import gov.va.nvap.web.helper.facility.FacilityHelper;
import gov.va.nvap.web.helper.privacy.ConsentManagementHelper;
import gov.va.nvap.web.user.UserHelper;
import gov.va.nvap.web.util.file.FileUploadUtil;
import gov.va.nvap.web.util.file.FileUploader;

import java.io.IOException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;

import javax.ejb.EJB;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;

import org.apache.commons.fileupload.FileItem;
import org.apache.commons.io.FilenameUtils;

/**
 * @author DNS   
 *
 */
public class BatchAnnounce extends ResponseDispatcherHttpServlet {

    private static final String FLOW_QUERY = "query";
    private static final String FLOW_RESULT = "result";
    private static final String FLOW_SUCCEED = "succeed";
    /**
     *
     */
    private static final long serialVersionUID = -3136937675789090144L;

    // Toggles sequential vs. fan-out implementation for batch announce
    private static final boolean SEQUENTIAL_ANNOUNCE = false;

    private static final int MAX_ANNOUNCEMENT_RETRIEVAL = 1000000;

    @EJB(beanInterface = AnnouncerInterface.class, mappedName = "AnnouncerService")
    private AnnouncerInterface announcer;

    @EJB(beanInterface = PdqService.class, mappedName = "PdqService")
    private PdqService pdqService;

    @EJB(beanInterface = PIPInterface.class, mappedName = "PipService")
    private PIPInterface pip;

    public void accept(final HttpServletRequest request,
            final HttpServletResponse response) throws ServletException,
            IOException {
        final HttpSession session = request.getSession(false);

        final PatientConsentBatchAnnounce batchAnnouncer = this.getBean(
                "batchAnnouncer", PatientConsentBatchAnnounce.class);
        batchAnnouncer.batchAnnounce(
                (String[]) session.getAttribute("batchIds"),
                UserHelper.getUserName(request),
                this.getFacilityHelper().getFaciltyStationIdByUserId(
                        UserHelper.getUserName(request)));
        this.initAttributes(session, null, null, false,
                BatchAnnounce.FLOW_SUCCEED, "");
        session.removeAttribute("batchIds");
        session.removeAttribute("batchSummary");
        this.forward(request, response, BatchAnnounce.FLOW_SUCCEED);
    }

    public void cancel(final HttpServletRequest request,
            final HttpServletResponse response) throws ServletException,
            IOException {
        final HttpSession session = request.getSession(false);
        final String[] batchIdsArray = (String[]) session
                .getAttribute("batchIds");
        if (!NullChecker.isNullOrEmpty(batchIdsArray)) {
            Collection<Announcement> announcements = new ArrayList<Announcement>();
            for (String batchId : batchIdsArray) {
                announcements.addAll(this.announcer.getUnscheduledAnnouncementsByBatchId(batchId, MAX_ANNOUNCEMENT_RETRIEVAL, 0));
            }
            for (Announcement announcement : announcements) {
                this.announcer.removeByAnnouncement(announcement);
            }
            this.announcer.removeAnnouncementBatches(Arrays.asList(batchIdsArray));
        }
        try {
            String startDate = session.getAttribute("startDate") == null ? "" : session.getAttribute("startDate").toString();
            String endDate = session.getAttribute("endDate") == null ? "" : session.getAttribute("endDate").toString();
            boolean reannounce = session.getAttribute("reannounce") == null ? false : (Boolean) session.getAttribute("reannounce");
            this.initAttributes(session, startDate, endDate, reannounce, BatchAnnounce.FLOW_QUERY, "Batch Announcement has been cancelled.");
        } catch (Exception ex) {
            String msg = ex.getMessage();
        }
        this.forward(request, response, BatchAnnounce.FLOW_QUERY);
    }

    public ConsentManagementHelper getCmsHelper() {
        final ConsentManagementHelper cmsHelper = this.getBean("cmsHelper",
                ConsentManagementHelper.class);
        return cmsHelper;
    }

    /**
     * Get the facility helper from Spring.
     */
    public FacilityHelper getFacilityHelper() {
        final FacilityHelper facilityHelper = this.getBean("facilityHelper",
                FacilityHelper.class);
        return facilityHelper;
    }

    private HashSet<String> getUploadedIens(List<Map<String, String>> listOfPatients) {
        HashSet<String> uploadedIens = new HashSet<String>();

        for (Map<String, String> patient : listOfPatients) {
            final PatientMatchQuery patientSearchRequest = new PatientMatchQuery();
            final PatientProfile patientProfile = new PatientProfile();
            patientProfile.setFirstName(patient.get("firstname"));
            patientProfile.setLastName(patient.get("lastname"));
            patientProfile.setSsn(patient.get("ssn"));
            patientSearchRequest.setPatientProfile(patientProfile);

            try {
                final PatientMatchResponse patientSearchResponse = this.pdqService.searchPatient(patientSearchRequest);
                final List<PatientProfile> patients = patientSearchResponse.getPatients().getPatientProfile();

                if (patients.size() == 1) {
                    uploadedIens.add(patients.get(0).getIcn());
                }
            } catch (Exception e) {
                // TODO log?
            }
        }

        return uploadedIens;
    }

    private void initAttributes(final HttpSession session,
            final String startDate, final String endDate,
            final boolean reannounce, final String flow, final String outcome) {
            final List<OrganizationType> orgs = (List) this.getCmsHelper().getAllowedNonConsumerOnlyOrganizations();
        /*
         * Collections.sort((List) orgs, new Comparator<OrganizationType>() {
         *
         * @Override public int compare(OrganizationType o1, OrganizationType
         * o2) { return o1.getOrgName().compareTo(o2.getOrgName()); } });
         */
        session.setAttribute("organizations", orgs);
        session.setAttribute("startDate", startDate);
        session.setAttribute("endDate", endDate);
        session.setAttribute("reannounce", reannounce);
        session.setAttribute("flow", flow);
        session.setAttribute("outcome", outcome);
    }

    private Date parseDate(final String strDate, final boolean endOfDay) {
        final Date date = null;
        if ((null != strDate) && (0 < strDate.length())) {
            final String format = "%s "
                    + (endOfDay ? "11:59:59 PM" : "00:00:00 AM");
			// matches the date input format
            // if (strDate.matches("\\d{1,2}/\\d{1,2}/\\d{4}")) {
            try {
                return DateUtil.parseDateFromString(String.format(format,
                        strDate));
            } catch (final ParseException ex) {
                Logger.getLogger(BatchAnnounce.class.getName()).log(
                        Level.SEVERE, null, ex);
                throw new IllegalArgumentException(ex);
            }
			// }
            // throw new IllegalArgumentException("Date must be MM/dd/yyyy");
        }
        return date;
    }

    public void query(final List<FileItem> fileItems, final HttpServletRequest request,
            final HttpServletResponse response) throws ServletException,
            IOException {

        final HttpSession session = request.getSession(false);
     	final Map<String, String> formFields = FileUploadUtil.getFormFields(fileItems);
     	final String orgIdsList = formFields.get("remoteOrganization");
     	final String startDateStr = formFields.get("startDate");
     	final String endDateStr = formFields.get("endDate");
        final boolean reannounce = "reannounce".equals(formFields.get("reannounce"));
     	final String batchAnnounceType = formFields.get("batchAnnounceType");
        String errorMessage = "";
        String[] orgIds;
        List<String> batchIds = null;
        List<AnnouncementBatchSummary> summary = null;

        if (orgIdsList == null) {
            errorMessage = "No patients found for criteria.";
        } else {
            orgIds = orgIdsList.split(",");

            if (batchAnnounceType.equals("1")) {
                if (this.validate(formFields, request, response, "query")) {
                    Date endDate;
                    if (NullChecker.isEmpty(endDateStr)) {
                        endDate = gov.va.nvap.web.util.date.DateUtil.addTime(new Date(), Calendar.HOUR, 24);
                    } else {
                        endDate = this.parseDate(endDateStr, true);
                    }

                    if (SEQUENTIAL_ANNOUNCE) {
                        batchIds = this.createBatchSequential(orgIds, UserHelper.getUserName(request),
                                this.parseDate(startDateStr, false), endDate, reannounce);
                    } else {
                        batchIds = this.createBatchConcurrent(orgIds, UserHelper.getUserName(request),
                                this.parseDate(startDateStr, false), endDate, reannounce);
                    }

                    summary = this.announcer.getBatchSummary(batchIds);
                    if (summary.isEmpty()) {
                        errorMessage = "No patients found for criteria.";
                    }
                } else {
                    this.forward(request, response, "validate");
                    return;
                }
            } else {
                final List<FileItem> filesUploaded = FileUploadUtil.getFilesUploaded(fileItems);
                List<Map<String, String>> listOfPatients = null;
                HashSet<String> uploadedIens;
                FileItem fileItem = filesUploaded.get(0);
                String fileName = FilenameUtils.getName(fileItem.getName());

                if (!NullChecker.isEmpty(fileName)) {
                    try {
                        if (fileName.toLowerCase(Locale.ENGLISH).endsWith(".csv")) {
                            listOfPatients = FileUploader.readCSV(fileItem);
                        } else if (fileName.toLowerCase(Locale.ENGLISH).endsWith(".xls") || fileName.toLowerCase(Locale.ENGLISH).endsWith(".xlsx")) {
                            listOfPatients = FileUploader.readExcel(fileItem);
                        }
                        if (listOfPatients == null || listOfPatients.isEmpty()) {
                            errorMessage = "Uploaded file doesn't contain any patients";
                        } else {
                            uploadedIens = getUploadedIens(listOfPatients);

                            if (SEQUENTIAL_ANNOUNCE) {
                                batchIds = this.createBatchSequential(orgIds, UserHelper.getUserName(request), uploadedIens);
                            } else {
                                batchIds = this.createBatchConcurrent(orgIds, UserHelper.getUserName(request), uploadedIens);
                            }

                            summary = this.announcer.getBatchSummary(batchIds);
                            if (summary.isEmpty()) {
                                errorMessage = "Uploaded file doesn't contain any known patients.";
                            }
                        }
                    } catch (Exception e) {
                        errorMessage = e.getMessage();
                    }
                }
            }
        }

        session.setAttribute("batchIds", batchIds == null ? null : batchIds.toArray(new String[batchIds.size()]));
        session.setAttribute("batchSummary", summary);
        session.setAttribute("errorMessage", errorMessage);

        this.initAttributes(session, startDateStr, endDateStr, reannounce, BatchAnnounce.FLOW_RESULT, "");
        this.forward(request, response, BatchAnnounce.FLOW_RESULT);
    }

    ArrayList<String> createBatchSequential(String[] orgIds, String userId, Date startDate, Date endDate, boolean reannounce) {
        ArrayList<String> batchIds = new ArrayList<String>(orgIds.length);
        for (final String orgOid : orgIds) {
            final Organization organization = this.pip.getOrganizationByOid(orgOid);
            final String batchId = this.announcer.createAnnouncementsBatch(organization, userId, startDate, endDate, reannounce);
            batchIds.add(batchId);
        }

        return batchIds;
    }

    ArrayList<String> createBatchSequential(String[] orgIds, String userId, HashSet<String> uploadedIens) {
        ArrayList<String> batchIds = new ArrayList<String>(orgIds.length);
        for (final String orgOid : orgIds) {
            final Organization organization = this.pip.getOrganizationByOid(orgOid);
            final String batchId = this.announcer.createAnnouncementsBatch(organization, userId, uploadedIens);
            batchIds.add(batchId);
        }

        return batchIds;
    }

    ArrayList<String> createBatchConcurrent(String[] orgIds, String userId, Date startDate, Date endDate, boolean reannounce) {
        ArrayList<String> batchIds = new ArrayList<String>();
        Collection<Organization> organizations = new ArrayList<Organization>();
        for (final String orgOid : orgIds) {
            final Organization organization = this.pip.getOrganizationByOid(orgOid);
            organizations.add(organization);
        }

        String batchId = this.announcer.createAnnouncementsBatch(organizations, userId, startDate, endDate, reannounce);
        Collection<Announcement> announcements = this.announcer.getUnscheduledAnnouncementsByBatchId(batchId, MAX_ANNOUNCEMENT_RETRIEVAL, 0);
        for (Announcement announce : announcements) {
            for (Organization org : organizations) {
                AnnouncementOrg announceOrg = new AnnouncementOrg();
                announceOrg.setAnnouncement(announce);
                announceOrg.setTargetOrganization(org);
                try {
                    this.announcer.createAnnouncementOrg(announceOrg);
                } catch (Exception ex) {
                    // AnnounceOrg storage fail
                }
            }
        }
        batchIds.add(batchId);
        return batchIds;
    }

    ArrayList<String> createBatchConcurrent(String[] orgIds, String userId, HashSet<String> uploadedIens) {
        ArrayList<String> batchIds = new ArrayList<String>(orgIds.length);
        Collection<Organization> organizations = new ArrayList<Organization>();
        for (final String orgOid : orgIds) {
            final Organization organization = this.pip.getOrganizationByOid(orgOid);
            organizations.add(organization);
        }

        String batchId = this.announcer.createAnnouncementsBatch(organizations, userId, uploadedIens);
        Collection<Announcement> announcements = this.announcer.getUnscheduledAnnouncementsByBatchId(batchId, MAX_ANNOUNCEMENT_RETRIEVAL, 0);
        for (Announcement announce : announcements) {
            for (Organization org : organizations) {
                AnnouncementOrg announceOrg = new AnnouncementOrg();
                announceOrg.setAnnouncement(announce);
                announceOrg.setTargetOrganization(org);
                try {
                    this.announcer.createAnnouncementOrg(announceOrg);
                } catch (Exception ex) {
                    // AnnounceOrg storage fail
                }
            }
        }
        batchIds.add(batchId);
        return batchIds;
    }

    /*
     * (non-Javadoc)
     *
     * @see gov.va.nvap.web.app.ResponseDispatcherHttpServlet#unspecified
     * (javax.servlet.http.HttpServletRequest,
     * javax.servlet.http.HttpServletResponse)
     */
    @Override
    protected void unspecified(final HttpServletRequest request,
            final HttpServletResponse response) throws ServletException,
            IOException {
        final HttpSession session = request.getSession(false);
        session.removeAttribute("errorMessage");
        this.initAttributes(session, null, null, false,
                BatchAnnounce.FLOW_QUERY, "");

        this.forward(request, response, BatchAnnounce.FLOW_QUERY);
    }

}
