package gov.va.nvap.notification.jmsutilities;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.jms.JMSException;
import javax.jms.MapMessage;
import javax.jms.Message;
import javax.jms.Session;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.jms.core.MessageCreator;

class VapJMSMessageCreator implements MessageCreator {

	private static final long serialVersionUID = 1L;
	private Serializable message;
	private Set<String> identifiers;
	private String xml;
	private Map<String, Object> messageSelectors;
	private VapJMSPublisher worker;
	private static final Log LOG = LogFactory.getLog(VapJMSMessageCreator.class);

	private VapJMSMessageCreator(VapJMSPublisher worker, Set<String> identifiers) {
		this.identifiers = identifiers;
		this.worker = worker;
	}

	VapJMSMessageCreator(VapJMSPublisher worker, Serializable message, Set<String> identifiers) {
		// this.message = new VapJMSPublisherMessage(message, identifiers);
		this(worker,identifiers);
		this.message = message;
	}

	VapJMSMessageCreator(VapJMSPublisher worker, String xml, Set<String> identifiers) {
		// this.message = new VapJMSPublisherMessage(message, identifiers);
		this(worker,identifiers);
		this.xml = xml;
	}

	VapJMSMessageCreator(VapJMSPublisher worker, String xml, Set<String> identifiers,
			Map<String, Object> messageSelectors) {
		this(worker,xml, identifiers);
		this.messageSelectors = messageSelectors;
	}

	VapJMSMessageCreator(VapJMSPublisher worker, Serializable message, Set<String> identifiers,
			Map<String, Object> messageSelectors) {
		this(worker,message, identifiers);
		this.messageSelectors = messageSelectors;
	}

	@Override
	public Message createMessage(Session session) throws JMSException {
		// final Message msg = session.createObjectMessage(message);
		final MapMessage msg = session.createMapMessage();
		addSelectors(msg);
		if (identifiers != null) {
			msg.setString(worker.identifierKey, stringSetToCSV(identifiers));
		}
		if (xml != null) {
			// xml message
			msg.setString(worker.xmlKey, xml);
		} else {
			// an object message
			ByteArrayOutputStream bos = new ByteArrayOutputStream();
			ObjectOutputStream out = null;
			try {
				out = new ObjectOutputStream(bos);
				out.writeObject(message);
			} catch (IOException e) {
				LOG.error("An un-serializable message was sent into the JMSMesssageCreator.", e);
				e.printStackTrace();
			}

			byte[] objBytes = bos.toByteArray();
			msg.setBytes(worker.objectKey, objBytes);
			try {
				if (out != null)
					out.close();
				bos.close();
			} catch (IOException e) {// unclosed streams on stack can be gc'ed
			}
		}

		return msg;
	}

	private void addSelectors(MapMessage msg) {
		if (messageSelectors != null) {
			Set<Entry<String, Object>> entries = messageSelectors.entrySet();
			for (Entry<String, Object> entry : entries) {
				String key = entry.getKey();
				Object value = entry.getValue();
				try {
					if (value instanceof Boolean) {
						msg.setBooleanProperty(key, (Boolean) value);
					} else if (value instanceof Byte) {
						msg.setByteProperty(key, (Byte)value);
					} else if (value instanceof Double) {
						msg.setDoubleProperty(key, (Double)value);
					} else if (value instanceof Float) {
						msg.setFloatProperty(key, (Float) value);
					} else if (value instanceof Integer) {
						msg.setIntProperty(key, (Integer) value);
					} else if (value instanceof Long) {
						msg.setLongProperty(key, (Long) value);
					} else if (value instanceof Short) {
						msg.setShortProperty(key, (Short) value);
					} else if (value instanceof String) {
						msg.setStringProperty(key, (String) value);
					} else if (value instanceof Object) {
						msg.setObjectProperty(key, value);
					}
				} catch (JMSException e) {
					LOG.warn("A JMS Exception occured while trying to set some selectors. Some selectors may be missing.  ErrorCode= " + e.getErrorCode(),e );
					e.printStackTrace();
				}
			}
		}
	}
	
	static String stringSetToCSV(Set<String> identifiers) {
		StringBuilder b = new StringBuilder();
		for (String s : identifiers) {
			b.append(s);
			b.append(',');
		}
		b.deleteCharAt(b.length() - 1);
		return b.toString();
	}
}
