package gov.va.nvap.notification.jmsutilities;

import gov.va.nvap.notification.interfaces.VapListener;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.Serializable;
import java.util.HashSet;
import java.util.Observable;
import java.util.Observer;
import java.util.Set;
import java.util.StringTokenizer;

import javax.jms.JMSException;
import javax.jms.MapMessage;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.jms.core.JmsTemplate;

public abstract class VapJMSListener implements Observer, VapListener {

	protected JmsTemplate jmsTemplate;
	protected Set<String> subIdentifiers = new HashSet<String>();
	String identifierKey = "identifiers"; // inject with spring if
	String objectKey = "object";// inject with spring if desired
	String xmlKey = "xml";// inject with spring if desired
	String trigger = "trigger";// inject with spring if desired
	private VapTriggerEvent triggerEvent;

	private static final Log LOG = LogFactory.getLog(VapJMSListener.class);

	@Override
	public void update(Observable mdb, Object jmsMessage) {
		if (jmsMessage instanceof MapMessage) {
			MapMessage m = (MapMessage) jmsMessage;
			Set<Serializable> clone = new HashSet<Serializable>(subIdentifiers);
			try {
				clone.retainAll(CSVToSet(m.getString(identifierKey)));
			} catch (JMSException e) {
				throw new RuntimeException(e);// motivate a retry
			}
			if (!clone.isEmpty()) {
				try {
					if(m.propertyExists(trigger)) {
						try {
							triggerEvent = VapTriggerEvent.valueOf(m.getStringProperty(trigger));
						} catch (IllegalArgumentException e) {
							LOG.error("A vap developer put a string that does not map to " + VapTriggerEvent.class +" -- the property key is " + trigger, e);
						}
					}
				} catch (JMSException e) {
					LOG.warn("The motivating VapTriggerEvent could not be pulled.  Reason = " + e.getErrorCode());
				}
				this.onMessage(m);
			}
		}
	}

	public void setIdentifierKey(String identifierKey) {
		this.identifierKey = identifierKey;
	}

	public void setObjectKey(String objectKey) {
		this.objectKey = objectKey;
	}

	public void setTrigger(String trigger) {
		this.trigger = trigger;
	}

	public void setXmlKey(String xmlKey) {
		this.xmlKey = xmlKey;
	}

	@Required
	public void setJmsTemplate(final JmsTemplate jmsTemplate) {
		this.jmsTemplate = jmsTemplate;
	}

	protected VapTriggerEvent getTrigger() {
		return triggerEvent;
	}

	protected Serializable getObjectMessage(MapMessage m) {
		Serializable rVal = null;
		try {
			if (m.itemExists(objectKey)) {
				byte[] b = m.getBytes(objectKey);
				ByteArrayInputStream bis = new ByteArrayInputStream(b);
				ObjectInputStream in = new ObjectInputStream(bis);
				rVal = (Serializable) in.readObject();
				bis.close();
				in.close();
			}
		} catch (IOException e) {
			LOG.warn(
					"An error occurred trying to deserialize an object.  Message might be replayed.",
					e);
			throw new RuntimeException(e);// motivate a retry
		} catch (JMSException e) {
			LOG.warn(
					"A JMS exception occured while trying to pull data from the object message.  Error code = "
							+ e.getErrorCode() + ". Message might be replayed.", e);
			throw new RuntimeException(e);// motivate a retry
		} catch (ClassNotFoundException e) {
			LOG.error(
					"There is a configuration error in VAP. Is this VapJMSListener is listening to the right destination?", e);
			e.printStackTrace();
		}
		return rVal;
	}

	public void setSubMessageIdentifiers(Set<String> identifier) {
		this.subIdentifiers = identifier;
	}

	protected String getXMLMessage(MapMessage m) {
		String rVal = null;
		try {
			if (m.itemExists(xmlKey)) {
				rVal = m.getString(xmlKey);
			}
		} catch (JMSException e) {
			LOG.warn(
					"A JMS exception occured while trying to pull data from the xml/text message.  Error code = "
							+ e.getErrorCode() + ". Message might be replayed.",
					e);
			throw new RuntimeException(e);// motivate a retry
		}
		return rVal;
	}

	static Set<String> CSVToSet(String csv) {
		StringTokenizer t = new StringTokenizer(csv, ",");
		Set<String> s = new HashSet<String>();
		while (t.hasMoreElements()) {
			s.add(t.nextToken());
		}
		return s;
	}
}
