/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package gov.va.nvap.svc.consentmgmt.integration;

import gov.va.nvap.privacy.ConsentDirectiveDetailedExpirationRequest;
import gov.va.nvap.svc.consentmgmt.PIPInterface;
import gov.va.nvap.svc.consentmgmt.PolicyInformationPoint;
import gov.va.nvap.svc.consentmgmt.jpa.AuthorityJpaController;
import gov.va.nvap.svc.consentmgmt.jpa.ConsentDirectiveJpaController;
import gov.va.nvap.svc.consentmgmt.jpa.FacilityJpaController;
import gov.va.nvap.svc.consentmgmt.jpa.FunctionalRoleJpaController;
import gov.va.nvap.svc.consentmgmt.jpa.OrganizationsJpaController;
import gov.va.nvap.svc.consentmgmt.jpa.PatientDocumentJpaController;
import gov.va.nvap.svc.consentmgmt.jpa.PatientDocumentTypeJpaController;
import gov.va.nvap.svc.consentmgmt.jpa.PurposeOfUseJpaController;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.Announcement;
import gov.va.nvap.svc.consentmgmt.stub.data.ConsentDirective;
import gov.va.nvap.svc.consentmgmt.stub.data.Facility;
import gov.va.nvap.svc.consentmgmt.stub.data.FunctionalRole;
import gov.va.nvap.svc.consentmgmt.stub.data.OptoutReason;
import gov.va.nvap.svc.consentmgmt.stub.data.Organization;
import gov.va.nvap.svc.consentmgmt.stub.data.PatientDocument;
import gov.va.nvap.svc.consentmgmt.stub.data.PatientDocumentType;
import gov.va.nvap.svc.consentmgmt.stub.data.PurposeOfUse;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.Persistence;
import javax.persistence.Query;

import junit.framework.Assert;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.Test;
/**
 *
 * @author vhaislegberb
 */
public class PolicyInformationPointTest {

    private static EntityManagerFactory emf = Persistence.createEntityManagerFactory("nwhin_vap_pip");
    private EntityManager em = null;
    private PIPInterface pip;

    @Before
    public void setUp() throws Exception {

        this.em = emf.createEntityManager();

        ConsentDirectiveJpaController consentDirectiveJpaController = new ConsentDirectiveJpaController();
        FunctionalRoleJpaController functionalRoleJpaController = new FunctionalRoleJpaController();
        OrganizationsJpaController organizationsJpaController = new OrganizationsJpaController();
        PurposeOfUseJpaController purposeOfUseJpaController = new PurposeOfUseJpaController();
        PatientDocumentJpaController patientDocumentJpaController = new PatientDocumentJpaController();
        FacilityJpaController facilityJpaController = new FacilityJpaController();
        AuthorityJpaController authorityController = new AuthorityJpaController();

        consentDirectiveJpaController.setEntityManager(em);
        functionalRoleJpaController.setEntityManager(em);
        organizationsJpaController.setEntityManager(em);
        purposeOfUseJpaController.setEntityManager(em);
        patientDocumentJpaController.setEntityManager(em);
        facilityJpaController.setEntityManager(em);
        authorityController.setEntityManager(em);

        PolicyInformationPoint policyInformationPoint = new PolicyInformationPoint();
        policyInformationPoint.setConsentDirectiveController(consentDirectiveJpaController);
        policyInformationPoint.setFunctionalRoleController(functionalRoleJpaController);
        policyInformationPoint.setOrganizationsController(organizationsJpaController);
        policyInformationPoint.setPurposeOfUseController(purposeOfUseJpaController);
        policyInformationPoint.setPatientDocumentController(patientDocumentJpaController);
        policyInformationPoint.setFacilityController(facilityJpaController);
        policyInformationPoint.setAuthorityController(authorityController);

        this.pip = policyInformationPoint;
    }

    @After
    public void tearDown()
    {
        if( this.em != null)
        {
            this.em.close();
            this.em = null;
        }
    }

    @Test
    public void testStoreOrganization()
    {
        // add an organization to batch announce to
        Organization org = new Organization();
        org.setActive('Y');
        org.setOrgCommunityIdPrefix("NEW");
        org.setOrgContact("ORG CONTACT");
        org.setOrgDomain("NEW.ORG.DOMAIN.1");
        org.setOrgName("New Organization 1");
        org.setOrgNumber("200NEW");
        org.setOrgOid("2.16.840.1.113883.3.555");
        org.setOrgPhoneNumber("555-555-5555");
        org.setOrgConsumerOnly('N');

        this.em.getTransaction().begin();
        pip.storeOrganization(org);
        this.em.getTransaction().commit();

        Collection<Organization> allowedOrgs = pip.getAllowedOrganizations();

        for(Organization o : allowedOrgs)
        {
            if(o.getOrgName().equals("New Organization 1"))
            {
                return;
            }
        }

        Assert.fail();
    }

    @Test(expected=IllegalArgumentException.class)
    public void storeNullOrganization()
    {
        pip.storeOrganization(null);
    }

    @Test
    public void testGetAllowedOrganizations() {

        Collection<Organization> organizations = pip.getAllowedOrganizations();
        Assert.assertNotNull(organizations);
        Assert.assertFalse(organizations.isEmpty());
    }

    @Test
    public void testGetOrganizationByOid()
    {
        String oid = "1.3.6.1.4.1.12009.6.1";
        Organization org = pip.getOrganizationByOid(oid);
        Assert.assertNotNull(org);
        Assert.assertTrue( org.getOrgOid().equals(oid));
    }

    @Test
    public void testGetAllowedVistAFacilities() {

        Collection<Facility> facilities = pip.getAllowedVistAFacilities();
        Assert.assertNotNull(facilities);
        Assert.assertFalse(facilities.isEmpty());
    }

    @Test
    public void testGetAuthorityByOIDMatch()
    {
        String authorityName = pip.getAuthorityName("2.16.840.1.113883.3.190.6.100.1");
        Assert.assertNotNull(authorityName);
        Assert.assertEquals("MVA", authorityName);
    }

    @Test
    public void testGetAuthorityByOIDNoMatch()
    {
        String authorityName = pip.getAuthorityName("BOGUS.OID");
        Assert.assertNull(authorityName);
    }

    @Test
    public void testGetAllowedFunctionalRoles() {
        Collection<FunctionalRole> functionalRoles = pip.getAllowedFunctionalRoles();
        Assert.assertNotNull(functionalRoles);
        Assert.assertFalse(functionalRoles.isEmpty());
    }

    @Test
    public void testGetAllowedPurposeOfUse() {
        Collection<PurposeOfUse> purposes = pip.getAllowedPurposeOfUse();
        Assert.assertNotNull(purposes);
        Assert.assertFalse(purposes.isEmpty());
    }

    @Test
    public void testFindOptoutReasons() {
        Collection<OptoutReason> reasons = pip.getOptoutReasons();
        Assert.assertNotNull(reasons);
        Assert.assertTrue(reasons.size() == 5);
    }

    @Test
    public void testGetConsentDirOptIn() {
        List<String> patientIens = new ArrayList();
        patientIens.add("123456");

        ConsentDirective consentDirective = pip.getActiveConsentDirective(patientIens, pip.getConsentTypeByName("NwHIN Authorization"));
        Assert.assertNotNull(consentDirective);
        Assert.assertTrue(consentDirective.getExcludedOrganizations().isEmpty());
    }
    
    @Test
    public void testFindExpiringConsentDirectives() {
    	Collection<String> patientIens = Arrays.asList(new String[] {
    			"8888888", 
    			"123460"
    			});
    	ConsentDirectiveDetailedExpirationRequest cdExpReq = new ConsentDirectiveDetailedExpirationRequest();
    	boolean exceptionThrown = false;
    	
    	try
    	{
    		cdExpReq.setDayRange(30);
	    	Collection<ConsentDirective> expiring = pip.getExpiringPatientConsentDirectives(patientIens, cdExpReq);
	    	Collection<ConsentDirective> expiringAll = pip.getExpiringPatientConsentDirectives(null, cdExpReq);
	    	cdExpReq.setDayRange(-1);
	    	Collection<ConsentDirective> expiringFutureAll = pip.getExpiringPatientConsentDirectives(null, cdExpReq);
	    	Collection<ConsentDirective> expiringFuture = pip.getExpiringPatientConsentDirectives(patientIens, cdExpReq);
    	}
    	catch (Exception e)
    	{
    		exceptionThrown = true;
    	}
    	
    	Assert.assertTrue(! exceptionThrown);
    }

    @Test
    public void testGetPatientConsentDirOptInWithExclusions() {
        List<String> patientIens = new ArrayList();
        patientIens.add("123457");

        ConsentDirective consentDirective = pip.getActiveConsentDirective(patientIens, pip.getConsentTypeByName("NwHIN Authorization"));
        Assert.assertNotNull(consentDirective);
        Assert.assertTrue(consentDirective.getExcludedOrganizations().size() == 2);
    }

    @Test
    public void testGetPatientConsentDirExpired() {
        List<String> patientIens = new ArrayList();
        patientIens.add("123458");

        ConsentDirective consentDirective = pip.getActiveConsentDirective(patientIens, pip.getConsentTypeByName("NwHIN Authorization")); // expired
        Assert.assertNull(consentDirective); // no current consent
    }

    @Test
    public void testGetPatientConsentDirOptedOut() {
        List<String> patientIens = new ArrayList();
        patientIens.add("123459");

        ConsentDirective consentDirective = pip.getActiveConsentDirective(patientIens, pip.getConsentTypeByName("NwHIN Authorization")); // opted out
        Assert.assertNull(consentDirective); // no current consent
    }

    @Test
    public void testGetPatientConsentDirMultipleOptedIn() {
        List<String> patientIens = new ArrayList();
        patientIens.add("123460");

        ConsentDirective consentDirective = pip.getActiveConsentDirective(patientIens, pip.getConsentTypeByName("NwHIN Authorization")); // patient with multiple consent, one active
        Assert.assertNotNull(consentDirective);
    }

    @Test
    public void testGetPatientConsentHistory() {
        List<String> patientIens = new ArrayList();
        patientIens.add("123460");

        Collection<ConsentDirective> patientConsentHistory = pip.getConsentHistory(patientIens);
        Assert.assertNotNull(patientConsentHistory);
        Assert.assertTrue(patientConsentHistory.size() == 2);
    }

    @Test
    public void testCreatePatientConsentDirectiveBasicOptin() throws Exception {
        PatientDocumentTypeJpaController patientDocumentController = new PatientDocumentTypeJpaController();
        PatientDocumentType optinDocumentType = null;

        patientDocumentController.setEntityManager(em);
        optinDocumentType = patientDocumentController.findPatientDocumentTypeByName("Authorize");

        DateFormat df = new SimpleDateFormat("yyyy-MM-DD HH:mm:SS");
        Date optinTS = df.parse("2011-05-24 12:00:00");
        Date optinDate = df.parse("2011-05-01 00:00:00");
        Date expirationDate = df.parse("2016-05-01 00:00:00");

        ConsentDirective consentDirective = new ConsentDirective();
        consentDirective.setOptinTS(optinTS);
        consentDirective.setOptinDate(optinDate);
        consentDirective.setExpirationDate(expirationDate);
        consentDirective.setPatientIen("8888888");
        consentDirective.setOptinConsentType(this.pip.getConsentTypeByName("NwHIN Authorization"));
        consentDirective.setExcludedOrganizations(null);

        PatientDocument optInDocument = new PatientDocument();
        optInDocument.setPatientDocumentType(optinDocumentType);
        optInDocument.setDocument("This is not a PDF".getBytes());
        optInDocument.setConsentDirective(consentDirective);

        this.em.getTransaction().begin();
        pip.storeConsentDirective(consentDirective);
        pip.storePatientDocument(optInDocument);
        this.em.getTransaction().commit();

        List<String> patientIens = new ArrayList();
        patientIens.add("8888888");

        consentDirective = pip.getActiveConsentDirective(patientIens, pip.getConsentTypeByName("NwHIN Authorization"));
        Collection<PatientDocument> documents = pip.getConsentDirectiveDocuments(consentDirective);
        optInDocument = pip.getConsentDirectiveDocumentByType(consentDirective, optinDocumentType);

        // Try update
        this.em.getTransaction().begin();
        String textToStore = "This is another text thats not a PDF";
        optInDocument.setDocument(textToStore.getBytes());
        pip.updatePatientDocument(optInDocument);
        this.em.getTransaction().commit();

        optInDocument = pip.getConsentDirectiveDocumentByType(consentDirective, optinDocumentType);
        String text = new String((byte[])optInDocument.getDocument());
        Assert.assertEquals(textToStore, text);

        Assert.assertNotNull(consentDirective);
        Assert.assertNotNull(consentDirective.getConsentDirId());
        Assert.assertTrue(consentDirective.getPatientIen().equals("8888888"));
        Assert.assertTrue(documents.size() == 1);
        Assert.assertNotNull(optInDocument);
    }

    @Test
    public void testCreatePatientConsentDirectiveOptinWithOrgExclusions() throws Exception {

        PatientDocumentTypeJpaController patientDocumentController = new PatientDocumentTypeJpaController();
        PatientDocumentType optinDocumentType = null;

        OrganizationsJpaController organizationsController = new OrganizationsJpaController();
        Organization exOrg1 = null;
        Organization exOrg2 = null;

        patientDocumentController.setEntityManager(em);
        optinDocumentType = patientDocumentController.findPatientDocumentTypeByName("Revoke");

        organizationsController.setEntityManager(em);
        exOrg1 = organizationsController.findOrganizationByName("Med Virginia");
        exOrg2 = organizationsController.findOrganizationByName("Kaiser Permanente");

        DateFormat df = new SimpleDateFormat("yyyy-MM-DD HH:mm:SS");
        Date optinTS = df.parse("2011-05-24 12:00:00");
        Date optinDate = df.parse("2011-05-01 00:00:00");
        Date expirationDate = df.parse("2016-05-01 00:00:00");

        ConsentDirective consentDirective = new ConsentDirective();
        consentDirective.setOptinTS(optinTS);
        consentDirective.setOptinDate(optinDate);
        consentDirective.setExpirationDate(expirationDate);
        consentDirective.setPatientIen("999999999");
        consentDirective.setOptinConsentType(this.pip.getConsentTypeByName("NwHIN Authorization"));

        Collection<Organization> excludedOrganizations = new ArrayList<Organization>();
        excludedOrganizations.add(exOrg1);
        excludedOrganizations.add(exOrg2);
        consentDirective.setExcludedOrganizations(excludedOrganizations);

        PatientDocument optInDocument = new PatientDocument();
        optInDocument.setPatientDocumentType(optinDocumentType);
        optInDocument.setDocument("This is not a PDF".getBytes());
        optInDocument.setConsentDirective(consentDirective);

        this.em.getTransaction().begin();
        pip.storeConsentDirective(consentDirective);
        pip.storePatientDocument(optInDocument);
        this.em.getTransaction().commit();

        List<String> patientIens = new ArrayList();
        patientIens.add("999999999");

        consentDirective = pip.getActiveConsentDirective(patientIens, pip.getConsentTypeByName("NwHIN Authorization"));
        Collection<PatientDocument> documents = pip.getConsentDirectiveDocuments(consentDirective);
        optInDocument = pip.getConsentDirectiveDocumentByType(consentDirective, optinDocumentType);

        Assert.assertNotNull(consentDirective);
        Assert.assertNotNull(consentDirective.getConsentDirId());
        Assert.assertTrue(consentDirective.getPatientIen().equals("999999999"));
        Assert.assertTrue(documents.size() == 1);
        Assert.assertNotNull(optInDocument);
        Assert.assertTrue(consentDirective.getExcludedOrganizations().size() == 2);
    }

    @Test
    public void testOptOut() throws Exception
    {
        ConsentDirectiveJpaController consentDirectiveController = new ConsentDirectiveJpaController();
        PatientDocumentTypeJpaController patientDocumentTypeController = new PatientDocumentTypeJpaController();

        PatientDocumentType optoutDocumentType = null;
        OptoutReason optoutReason = null;

        List<String> patientIens = new ArrayList();
        patientIens.add("999999999");

        // look up the record that needs to be opted out.
        ConsentDirective cd = pip.getActiveConsentDirective(patientIens, pip.getConsentTypeByName("NwHIN Authorization"));

        // set the opt-out date of the record
        DateFormat df = new SimpleDateFormat("yyyy-MM-DD HH:mm:SS");
        Date optoutTS = df.parse("2011-12-01 00:00:00");
        cd.setOptoutTS(optoutTS);
        Date optoutDate = df.parse("2011-12-01 00:00:00");
        cd.setOptoutDate(optoutDate);

        patientDocumentTypeController.setEntityManager(em);
        optoutDocumentType = patientDocumentTypeController.findPatientDocumentTypeByName("Revoke");

        consentDirectiveController.setEntityManager(em);
        optoutReason = consentDirectiveController.getOptoutReasonByText("Revoked");

        cd.setOptoutReason(optoutReason);

        PatientDocument optOutDocument = new PatientDocument();
        optOutDocument.setPatientDocumentType(optoutDocumentType);
        optOutDocument.setDocument("This is not a PDF".getBytes());
        optOutDocument.setConsentDirective(cd);

        // store the record.
        this.em.getTransaction().begin();
        pip.storeConsentDirective(cd);
        pip.storePatientDocument(optOutDocument);
        this.em.getTransaction().commit();

        // try to read the active consent directive for the patient to make sure
        // that non are active.
        cd = pip.getActiveConsentDirective(patientIens, pip.getConsentTypeByName("NwHIN Authorization"));
        Assert.assertTrue(cd == null);

        Collection<ConsentDirective> cds = pip.getConsentHistory(patientIens);
        Assert.assertTrue(cds.size() == 1);

        cd = (ConsentDirective)cds.toArray()[0];
        Assert.assertNotNull(cd.getOptoutReason());
        Assert.assertEquals("Revoked", cd.getOptoutReason().getText());

    }

    @AfterClass
    public static void afterClass() throws Exception
    {
        EntityManager em = null;

        try {
            em = emf.createEntityManager();
            em.getTransaction().begin();

            // delete all of the announcements that were added during the testing
            Query q = em.createQuery("from Announcement a");
            List<Announcement> announcements = q.getResultList();
            for(Announcement a : announcements)
            {
            	em.remove(a);
            }

            // delete the new organization that was added during testing.
            q = em.createNamedQuery("Organization.findByOrgName");
            q.setParameter("orgName", "New Organization 1");
            Organization org = (Organization)q.getSingleResult();
            if(org != null)
            {
                em.remove(org);
            }

            ConsentDirectiveJpaController consentDirectiveController = new ConsentDirectiveJpaController();
            consentDirectiveController.setEntityManager(em);

            PatientDocumentJpaController patientDocumentController = new PatientDocumentJpaController();
            patientDocumentController.setEntityManager(em);

            List<String> cleanupIds = new ArrayList<String>();
            cleanupIds.add("8888888");
            cleanupIds.add("999999999");

            for(ConsentDirective cd : consentDirectiveController.findPatientConsentDirectives(cleanupIds))
            {
                for(PatientDocument pd : patientDocumentController.findByConsentDirective(cd))
                {
                    patientDocumentController.destroy(pd.getDocumentId());
                }

                consentDirectiveController.destroy(cd.getConsentDirId());
            }
        } finally {
            if(em != null)
            {
                em.getTransaction().commit();
                em.close();
            }
        }
    }
}
