package gov.va.nvap.svc.consentmgmt.stub.dao;

import static gov.va.nvap.common.date.DateUtil.voidTime;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.svc.consentmgmt.stub.data.MailNotification;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import org.springframework.transaction.annotation.Transactional;

/**
 *
 * @author Johann Sonnenberg
 * @since 04/05/02016
 */
public class MailNotificationDAO {

    @PersistenceContext
    private EntityManager em;

    public void setEntityManager(EntityManager entityManager) {
        this.em = entityManager;
    }

    @Transactional
    public Long create(MailNotification m) throws Exception {
        try {
            boolean exists = false;

            // Strip time from the sentDate property.
            m.setSentDate(voidTime(m.getSentDate()));

            //check to see if we already have a mail notification for the date supplied for either a delayed consent or a regular consent 
            if (NullChecker.isNullOrEmpty(m.getConsentDirId())) {
                exists = this.existByDelayedConsentId(m.getDelayedConsentId().getDelayedConsentId(), m.getSentDate());
            } else {
                exists = this.existByConsentDirId(m.getConsentDirId().getConsentDirId(), m.getSentDate());
            }

            //only add one if one does not already exist
            if (!exists) {
                this.em.persist(m);
            }
        } catch (final Exception ex) {
            throw ex;
        }

        return m.getMailNotificationId();
    }

    @Transactional
    public MailNotification update(MailNotification m) throws Exception {
        try {
            // Strip time from the sentDate property.
            m.setSentDate(voidTime(m.getSentDate()));

            this.em.merge(m);
        } catch (final Exception ex) {
            throw ex;
        }

        return m;
    }

    @Transactional
    public void delete(MailNotification m) throws Exception {
        try {
            m = this.em.merge(m);
            this.em.remove(m);
        } catch (final Exception ex) {
            throw ex;
        }
    }

    public MailNotification find(final Long id) {
        return this.em.find(MailNotification.class, id);
    }

    @Transactional
    public List<MailNotification> findByConsentDirId(String consentDirId) {
        try {
            final Query q = this.em.createQuery("SELECT m FROM MailNotification m WHERE m.consentDirId.consentDirId = :consentDirId ORDER BY m.sentDate ASC");
            q.setParameter("consentDirId", Long.parseLong(consentDirId));

            return (List<MailNotification>) q.getResultList();
        } catch (final NoResultException nre) {
            return new ArrayList<MailNotification>();
        }
    }

    @Transactional
    public List<MailNotification> findByDelayedConsentId(String delayedConsentId) {
        try {
            final Query q = this.em.createQuery("SELECT m FROM MailNotification m WHERE m.delayedConsentId.delayedConsentId = :delayedConsentId ORDER BY m.sentDate ASC");
            q.setParameter("delayedConsentId", Long.parseLong(delayedConsentId));

            return (List<MailNotification>) q.getResultList();
        } catch (final NoResultException nre) {
            return new ArrayList<MailNotification>();
        }
    }

    @Transactional
    public List<MailNotification> findByConsentDirIds(List<Long> consentIds) {
        try {
            if (NullChecker.isNullOrEmpty(consentIds)) {
                return new ArrayList<MailNotification>();
            }
            final Query q = this.em.createQuery("SELECT m FROM MailNotification m WHERE m.consentDirId.consentDirId IN (:consentIds) ORDER BY m.sentDate ASC");
            q.setParameter("consentIds", consentIds);

            return (List<MailNotification>) q.getResultList();
        } catch (final NoResultException nre) {
            return new ArrayList<MailNotification>();
        }
    }

    @Transactional
    public List<MailNotification> findByDelayedConsentIds(List<Long> delayedConsentIds) {
        try {
            if (NullChecker.isNullOrEmpty(delayedConsentIds)) {
                return new ArrayList<MailNotification>();
            }
            final Query q = this.em.createQuery("SELECT m FROM MailNotification m WHERE m.delayedConsentId.delayedConsentId IN (:delayedConsentIds) ORDER BY m.sentDate ASC");
            q.setParameter("delayedConsentIds", delayedConsentIds);

            return (List<MailNotification>) q.getResultList();
        } catch (final NoResultException nre) {
            return new ArrayList<MailNotification>();
        }
    }

    @Transactional
    public boolean existByConsentDirId(Long consentDirId, Date sentDate) {
        final Query q = this.em.createQuery("SELECT COUNT(mn) FROM MailNotification mn WHERE mn.consentDirId.consentDirId = :consentDirId AND mn.sentDate = :sentDate");
        q.setParameter("consentDirId", consentDirId);
        q.setParameter("sentDate", sentDate);

        return (Long) q.getSingleResult() > 0;
    }

    @Transactional
    public boolean existByDelayedConsentId(Long delayedConsentId, Date sentDate) {
        final Query q = this.em.createQuery("SELECT COUNT(mn) FROM MailNotification mn WHERE mn.delayedConsentId.delayedConsentId = :delayedConsentId AND mn.sentDate = :sentDate");
        q.setParameter("delayedConsentId", delayedConsentId);
        q.setParameter("sentDate", sentDate);

        return (Long) q.getSingleResult() > 0;
    }
    
}
