package gov.va.nvap.svc.consentmgmt.stub.dao;

import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.svc.consentmgmt.stub.data.DelayedConsentRpt;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import javax.persistence.EntityManager;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceContext;
import javax.persistence.Query;
import javax.persistence.TemporalType;
import org.springframework.transaction.annotation.Transactional;

/**
 * This DAO handles queries for Delayed Consent detailed and summary reports.
 *
 * @since 04/05/2016
 * @author Johann Sonnenberg
 */
public class DelayedConsentReportDAO {

    @PersistenceContext
    private EntityManager em;

    // Public methods

    public void setEntityManager(EntityManager entityManager) {
        this.em = entityManager;
    }

    public DelayedConsentRpt find(final Long id) {
        return this.em.find(DelayedConsentRpt.class, id);
    }

    public SearchAllResponse find(final SearchAllRequest searchRequest) {
        SearchAllResponse searchResponse = new SearchAllResponse();

        final Query q = this.em.createNamedQuery("DelayedConsentRpt.findAll");
        searchResponse.delayedConsents = (List<DelayedConsentRpt>) q.getResultList();
        searchResponse.count = searchResponse.delayedConsents.size();

        return searchResponse;
    }

    public SearchAllResponse searchAll(final SearchAllRequest searchRequest) {
        SearchAllResponse searchResponse = new SearchAllResponse();
        Query query;

        // Execute list query
        query = constructDetailedQuery(searchRequest, false);
        searchResponse.delayedConsents = (List<DelayedConsentRpt>) query.getResultList();

        // Execute count query
        query = constructDetailedQuery(searchRequest, true);
        searchResponse.count = query.getResultList().size();

        return searchResponse;
    }

    public SearchAllResponse searchAllForSummary(final SearchAllRequest searchRequest) {
        SearchAllResponse searchResponse = new SearchAllResponse();
        final StringBuffer queryString = new StringBuffer();
        final StringBuffer whereClause = new StringBuffer();
        final HashMap<String, Object> queryParams = new HashMap<String, Object>();
        final String facilityNameColumn = searchRequest.aggregateAtFacilityLevel ? "parentFacilityName" : "facilityName";

        queryString.append("SELECT COUNT(DISTINCT d.delayedConsentId), d.");
        queryString.append(facilityNameColumn);
        queryString.append(", d.consentTypeId.name FROM DelayedConsentRpt d");

        // Construct WHERE clause
        constructWhereClause(searchRequest, queryString, whereClause, queryParams);
        queryString.append(whereClause.toString());

        // Construct GROUP BY clause
        queryString.append(" GROUP BY d.");
        queryString.append(facilityNameColumn);
        queryString.append(", d.consentTypeId.name");

        // Construct ORDER BY clause
        queryString.append(constructOrderByClause(searchRequest, true));

        // Create query
        Query query = em.createQuery(queryString.toString());

        // Set start and length
        query.setFirstResult(searchRequest.start);
        if (searchRequest.length != -1) {
            query.setMaxResults(searchRequest.length);
        }

        // Set query parameters
        DAOUtil.setQueryParams(query, queryParams);

        // Execute query
        List<Object[]> results = query.getResultList();
        searchResponse.delayedConsents = new ArrayList<DelayedConsentRpt>();
        for (Object[] r : results) {
            DelayedConsentRpt rpt = new DelayedConsentRpt();
            rpt.setTotal((Long) r[0]);
            rpt.setFacilityName((String) r[1]);
            rpt.setConsentTypeName((String) r[2]);
            searchResponse.delayedConsents.add(rpt);
        }

        searchResponse.count = searchResponse.delayedConsents.size();

        return searchResponse;
    }

    public List<DelayedConsentRpt> findByPatientIen(String patientIen) {
        try {
            final Query q = this.em.createNamedQuery("DelayedConsentRpt.findByPatientIen");
            q.setParameter("patientIen", patientIen);

            return (List<DelayedConsentRpt>) q.getResultList();
        } catch (final NoResultException nre) {
            return new ArrayList<DelayedConsentRpt>();
        }
    }

    // Private methods

    private void appendToWhereClause(final StringBuffer buffer, final String str) {
        buffer.append(buffer.length() == 0 ? " WHERE " : " AND ");
        buffer.append(str);
    }

    private Query constructDetailedQuery(SearchAllRequest searchRequest, boolean isCountQuery) {
        final StringBuffer queryString = new StringBuffer();
        final StringBuffer whereClause = new StringBuffer();
        final HashMap<String, Object> queryParams = new HashMap<String, Object>();

        // Construct SELECT clause
        queryString.append("SELECT DISTINCT d FROM DelayedConsentRpt d");

        // Construct WHERE clause
        constructWhereClause(searchRequest, queryString, whereClause, queryParams);
        queryString.append(whereClause.toString());

        // Construct ORDER BY clause
        queryString.append(constructOrderByClause(searchRequest, false));

        // Create queries
        Query query = em.createQuery(queryString.toString());

        // Set start and length for the list query
        if (!isCountQuery) {
            query.setFirstResult(searchRequest.start);
            if (searchRequest.length != -1) {
                query.setMaxResults(searchRequest.length);
            }
        }

        // Set query parameters
        DAOUtil.setQueryParams(query, queryParams);

        return query;
    }

    private String constructOrderByClause(SearchAllRequest searchRequest, boolean forSummary) {
        String orderByClause = " ORDER BY ";
        Integer sortColumn = NullChecker.isNullOrEmpty(searchRequest.sortBy) ? 0 : Integer.parseInt(searchRequest.sortBy);
        String sortDirection = "ASC".equalsIgnoreCase(searchRequest.sortOrder) ? "ASC" : "DESC";

        if (forSummary) {
            final String facilityNameColumn = searchRequest.aggregateAtFacilityLevel ? "parentFacilityName" : "facilityName";

            switch (sortColumn) {
                case 0:
                    orderByClause += "LOWER(d." + facilityNameColumn + ") {direction}, d.consentTypeId.name ASC";
                    break;
                case 1:
                    orderByClause += "d.consentTypeId.name {direction}, LOWER(d." + facilityNameColumn + ") ASC";
                    break;
                case 2:
                    orderByClause += "COUNT(DISTINCT d.delayedConsentId) {direction}, LOWER(d." + facilityNameColumn + ") ASC";
                    break;
                default:
                    orderByClause += "LOWER(d." + facilityNameColumn + ") ASC, d.consentTypeId.name ASC";
                    break;
            }
        } else {
            switch (sortColumn) {
                case 0:
                    orderByClause += "d.dateAdded {direction}";
                    break;
                case 1:
                    orderByClause += "d.patientIen {direction}";
                    break;
                case 2:
                    orderByClause += "d.patientSsn {direction}";
                    break;
                case 3:
                    orderByClause += "LOWER(d.patientLastName) {direction}, LOWER(d.patientFirstName) ASC";
                    break;
                case 4:
                    orderByClause += "LOWER(d.patientFirstName) {direction}, LOWER(d.patientLastName) ASC";
                    break;
                case 5:
                    orderByClause += "LOWER(d.patientMiddleName) {direction}, LOWER(d.patientLastName) ASC";
                    break;
                case 6:
                    orderByClause += "d.consentTypeId {direction}, d.dateAdded DESC";
                    break;
                case 8:
                    orderByClause += "LOWER(d.userId) {direction}, d.dateAdded DESC";
                    break;
                case 9:
                    orderByClause += "LOWER(d.facilityName) {direction}, d.dateAdded DESC";
                    break;
                default:
                    orderByClause += "d.dateAdded DESC";
                    break;
            }
        }

        orderByClause = orderByClause.replace("{direction}", sortDirection);

        return orderByClause;
    }

    private void constructWhereClause(SearchAllRequest searchRequest, StringBuffer queryString, StringBuffer whereClause,
        HashMap<String, Object> queryParams) {
        final String stationNumberColumn = searchRequest.aggregateAtFacilityLevel ? "parentStationNumber" : "stationNumber" ;

        // Consent type
        if (!NullChecker.isNullOrEmpty(searchRequest.consentType) && !"ALL".equals(searchRequest.consentType)) {
            appendToWhereClause(whereClause, "d.consentTypeId.name = :consentType");
            queryParams.put("consentType", searchRequest.consentType);
        }

        // Days since delayed
        if (!NullChecker.isNullOrEmpty(searchRequest.daysSinceDelayed) && !"ALL".equals(searchRequest.daysSinceDelayed)) {
            Date dateBeforeXDays;
            Calendar cal = Calendar.getInstance();
            //If selection is consents in delayed status for 0-5 days, check for dates AFTER 6 days ago
            if (searchRequest.daysSinceDelayed.equals("0")) {
                appendToWhereClause(whereClause, "d.dateAdded > :daysSinceDelayed");
                cal.add(Calendar.DATE, -6);
                dateBeforeXDays = cal.getTime();
            }
            else {
                appendToWhereClause(whereClause, "d.dateAdded < :daysSinceDelayed");
                cal.add(Calendar.DATE, -Integer.parseInt(searchRequest.daysSinceDelayed));
                dateBeforeXDays = cal.getTime();
            }
            queryParams.put("daysSinceDelayed", dateBeforeXDays);
        }

        // Entered by
        if (!NullChecker.isNullOrEmpty(searchRequest.enteredBy)) {
            if (searchRequest.enteredBy.toUpperCase(Locale.ENGLISH).equals("EMPLOYEE")) {
                appendToWhereClause(whereClause, "LOWER(d.userId) NOT LIKE 'ebenefits'");
                appendToWhereClause(whereClause, "LOWER(d.userId) NOT LIKE 'automatic service'");
                appendToWhereClause(whereClause, "LOWER(d.userId) NOT LIKE 'system'");
            }
            else {
                appendToWhereClause(whereClause, "LOWER(d.userId) LIKE LOWER(:enteredBy)");
                queryParams.put("enteredBy", "%" + searchRequest.enteredBy + "%");
            }
        }

        // First name
        if (!NullChecker.isNullOrEmpty(searchRequest.patientFirstName)) {
            appendToWhereClause(whereClause, "LOWER(d.patientFirstName) = LOWER(:patientFirstName)");
            queryParams.put("patientFirstName", searchRequest.patientFirstName);
        }

        // Last name
        if (!NullChecker.isNullOrEmpty(searchRequest.patientLastName)) {
            appendToWhereClause(whereClause, "LOWER(d.patientLastName) = LOWER(:patientLastName)");
            queryParams.put("patientLastName", searchRequest.patientLastName);
        }

        // Patient types
        if (searchRequest.patientTypes == 1) {
            appendToWhereClause(whereClause, "d.isTestPatient = 0");
        } else if (searchRequest.patientTypes == 2) {
            appendToWhereClause(whereClause, "d.isTestPatient = 1");
        }

        // Reason(s) for delay
        if (!NullChecker.isNullOrEmpty(searchRequest.reasonsForDelay) && !"ALL".equals(searchRequest.reasonsForDelay)) {
            queryString.append(" JOIN d.delayReasonCollection dr");
            appendToWhereClause(whereClause, "dr.delayReasonId IN (:delayReasons)");
            List<String> stringReasons = Arrays.asList(searchRequest.reasonsForDelay.split(","));
            // Convert list of Strings to list of Longs
            List<Long> longReasons = new ArrayList<Long>();
            for (String s : stringReasons) longReasons.add(Long.valueOf(s));
            queryParams.put("delayReasons", longReasons);
        }

        // SSN(s)
        if (!NullChecker.isNullOrEmpty(searchRequest.patientSsn)) {
            appendToWhereClause(whereClause, "d.patientSsn IN (:patientSsn)");
            List<String> ssns = Arrays.asList(searchRequest.patientSsn.split(","));
            queryParams.put("patientSsn", ssns);
        }

        // Station number(s) (authenticating facility)
        if ("ALL".equals(searchRequest.stationNumbers)) {
            if (searchRequest.includeUnknownVisn) {
                // User chose to include everything, so there's no need for the WHERE clause.
            } else {
                // User chose to see everything BUT unknown VISN.
                appendToWhereClause(whereClause, "d.visnNumber IS NOT NULL");
            }
        } else if (NullChecker.isNullOrEmpty(searchRequest.stationNumbers)) {
            if (searchRequest.includeUnknownVisn) {
                // User chose to see unknown VISN only.
                appendToWhereClause(whereClause, "d.visnNumber IS NULL");
            } else {
                // This should never be the case. TODO log an error
            }
        } else {
            if (searchRequest.includeUnknownVisn) {
                // User chose to see selected station numbers AND unknown VISNs.
                appendToWhereClause(whereClause, "d." + stationNumberColumn + " IN (:stationNumbers) OR d.visnNumber IS NULL");
            } else {
                // User chose to see selected stations only.
                appendToWhereClause(whereClause, "d." + stationNumberColumn + " IN (:stationNumbers)");
            }
            // Client will send station numbers enclosed in single quotes. For Hibernate queries we must get rid of them.
            List<String> items = Arrays.asList(searchRequest.stationNumbers.split(","));
            queryParams.put("stationNumbers", items);
        }

        // For Delayed Consent reports, we're only interested in PENDING delayed authorizations.
        appendToWhereClause(whereClause, "d.status = 'PENDING'");
    }

    // Inner classes

    public class SearchAllRequest {

        public boolean aggregateAtFacilityLevel;
        public String authenticatingFacility;
        public String consentType;
        public String daysSinceDelayed;
        public String daysSinceLastNotification;
        public String enteredBy;
        public boolean includeUnknownVisn;
        public int length;
        public String patientFirstName;
        public String patientLastName;
        public String patientSsn;
        public int patientTypes;
        public String reasonsForDelay;
        public String sortBy;
        public String sortOrder;
        public int start;
        public String stationNumbers;

    }

    public class SearchAllResponse {

        public List<DelayedConsentRpt> delayedConsents = null;
        public long count = 0;

    }

}
