package gov.va.nvap.svc.consentmgmt.schedule;

import gov.va.nvap.common.date.hl7.HL7DateUtil;
import gov.va.nvap.common.jaxb.JaxbUtil;
import gov.va.nvap.common.transformer.Transformer;
import gov.va.nvap.common.transformer.TransformerException;
import gov.va.nvap.common.transformer.xml.XMLToString;
import gov.va.nvap.common.uuid.UUIDUtil;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.ConsentDirectiveOptOutReasonType;
import gov.va.nvap.privacy.ConsentDirectiveRevocationRequest;
import gov.va.nvap.privacy.ConsentType;
import gov.va.nvap.privacy.ServiceConsumer;
import gov.va.nvap.privacy.ServiceConsumerContextType;
import gov.va.nvap.privacy.data.ConsentDirectiveData;
import gov.va.nvap.service.audit.AuditService;
import gov.va.nvap.service.audit.data.ConsentAudit;
import gov.va.nvap.service.pdq.PatientDemographics;
import gov.va.nvap.service.pdq.PatientDemographicsQuery;
import gov.va.nvap.service.pdq.PatientDemographicsResponse;
import gov.va.nvap.service.pdq.PdqException;
import gov.va.nvap.service.pdq.PdqService;
import gov.va.nvap.service.privacy.ConsentManagementService;
import gov.va.nvap.service.privacy.ConsentManagementServiceException;
import gov.va.nvap.svc.consentmgmt.PIPInterface;
import gov.va.nvap.svc.consentmgmt.stub.data.ConsentDirective;
import java.text.ParseException;
import java.util.Collection;
import java.util.Date;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.bind.JAXBException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.context.ApplicationContext;
import org.w3c.dom.Document;

/**
 *
 * @author Elan
 * @since 07/05/2016
 */
public class RecentlyExpiredConsentController {

    private PIPInterface pip;
    private PdqService pdqService;
    private Transformer<Document, Document> dataToConsentDirectiveDocument;
    private JaxbUtil consentDirectiveDataJaxbHelper;
    private XMLToString xml2String;
    private ConsentManagementService cms;
    private AuditService auditService;
    @Required
    public void setPip(final PIPInterface pip) {
        this.pip = pip;
    }

    @Required
    public void setPdqService(PdqService pdqService) {
        this.pdqService = pdqService;
    }

    @Required
    public void setDataToConsentDirectiveDocument(Transformer<Document, Document> dataToConsentDirectiveDocument) {
        this.dataToConsentDirectiveDocument = dataToConsentDirectiveDocument;
    }

    @Required
    public void setConsentDirectiveDataJaxbHelper(JaxbUtil consentDirectiveDataJaxbHelper) {
        this.consentDirectiveDataJaxbHelper = consentDirectiveDataJaxbHelper;
    }

    @Required
    public void setXml2String(final XMLToString xml2String) {
        this.xml2String = xml2String;
    }

    @Required
    public void setCms(ConsentManagementService cms) {
        this.cms = cms;
    }
    
    @Required
	public void setAuditService(final AuditService auditService) {
		this.auditService = auditService;
	}
    
    @Autowired
    private ApplicationContext applicationContext;

    public void onCall() {
        Logger.getLogger(RecentlyExpiredConsentController.class.getName()).log(Level.INFO, "RecentlyExpiredConsent has started at " + new Date().toString());
        Collection<ConsentDirective> expired = pip.getRecentlyExpiredConsent();
        ScheduleDAO schedule = applicationContext.getBean("scheduleDAO", ScheduleDAO.class);
        if (schedule.checkDaily("ExpiredConsent")) {
            return;
        }
        if (!schedule.create("ExpiredConsent")) {
            return;
        }
        if (expired != null && !expired.isEmpty()) {
            Logger.getLogger(RecentlyExpiredConsentController.class.getName()).log(Level.INFO, "RecentlyExpiredConsent found " + expired.size() + " entries");
            for (ConsentDirective consentDirective : expired) {
                if (NullChecker.isEmpty(consentDirective)) {
                    continue;
                }
                PatientDemographicsQuery demographicsQuery = new PatientDemographicsQuery();
                demographicsQuery.setPatientId(consentDirective.getPatientIen());
                PatientDemographicsResponse demographicsResponse;
                try {
                    demographicsResponse = this.pdqService.getPatientDemographics(demographicsQuery);

                } catch (PdqException e) {
                    Logger.getLogger(RecentlyExpiredConsentController.class.getName()).log(Level.SEVERE, null, e);
                    continue;
                }
                // Create a consent directive data object
                ConsentDirectiveData data = new ConsentDirectiveData();
                data.setIcn(consentDirective.getPatientIen());
                PatientDemographics demographics = demographicsResponse.getPatientDemographics();
                if (NullChecker.isNotEmpty(demographics)) {
                    data.setPatientRoleCity(demographics.getResidenceCity());
                    data.setPatientRoleState(demographics.getResidenceState());
                    data.setPatientRoleGivenName(demographics.getFirstName());
                    data.setPatientRoleFamilyName(demographics.getLastName());
                    data.setPatientRoleFamilyNameAlias(demographics.getAlias1());
                    data.setPatientRoleMiddleName(demographics.getMiddleName());
                    data.setPatientRoleMiddleNameAlias(demographics.getAlias2());
                    data.setPatientRoleEthnicGroupCodeDisplayText(demographics
                        .getEthnicityDescription());
                    data.setPatientRoleSsn(demographics.getSsn());
                    data.setPatientRoleGenderCode(demographics.getGender());
                    data.setPatientRoleGenderDisplayText(demographics
                        .getGenderDescription());
                    data.setPatientRoleMaritalStatusCode(demographics
                        .getMaritalStatus());
                    data.setPatientRoleMaritalStatusDisplayText(demographics
                        .getMaritalStatusDescription());
                    data.setPatientRolePostalCode(demographics.getResidenceZip4());
                    data.setPatientRolePrefix(demographics.getPrefix());
                    data.setPatientRoleStreetAddressLine(demographics
                        .getStreetAddressLine1());
                    data.setPatientRoleSuffix(demographics.getSuffix());
                    data.setPatientRoleTelecom(demographics.getResidencePhoneNumber());
                    data.setPatientRoleProviderOrganizationName(demographics
                        .getFacilityName());
                    data.setPatientRoleProviderOrganizationNumber(demographics
                        .getFacilityNumber());
                    if (NullChecker.isNotEmpty(demographics.getDob())) {
                        try {
                            // Need to use the HL7 date, because the CDA R2 needs it
                            data.setPatientRoleDob(HL7DateUtil
                                .yyyyMMddhhmmssZ(demographics.getDob()));
                        } catch (final ParseException ex) {
                            Logger.getLogger(RecentlyExpiredConsentController.class.getName()).log(Level.SEVERE, null, ex);
                            continue;
                        }
                    }
                }
                // Set a unique id per document
                data.setDocId(UUIDUtil.generateUUID());

                // Set the status code to aborted
                data.setComponentStatusCode("aborted");

                // Set document dates and times to now
                try {
                    final String signatureDateString = HL7DateUtil
                        .yyyyMMddhhmmssZ(consentDirective.getExpirationDate());
                    // Set the effective date
                    data.setEffectiveDateTime(signatureDateString);
                    // Create the begin and end date
                    data.setDocumentationBeginTime(signatureDateString);
                    // Start and end time are required
                    data.setDocumentationEndTime(signatureDateString);
                } catch (final ParseException ex) {
                    Logger.getLogger(RecentlyExpiredConsentController.class.getName()).log(Level.SEVERE, null, ex);
                }

                // Set the consent directive
                data.setPreviousConsentDirectiveId(consentDirective.getConsentDirId().toString());
                data.setOptoutReason(ConsentDirectiveOptOutReasonType.AUTHORIZATION_EXPIRED.getValue());

                // For now sets user to an automatic service
                data.setAuthorPersonOid("Automatic Service");
                data.setAuthorPersonOrgOid("Automatic Service");

                // Make the request to revoke
                final ServiceConsumerContextType sc = new ServiceConsumerContextType();
                sc.setUser("Automatic Service");
                ConsentAudit ca = auditService.getLatestAudit(consentDirective.getPatientIen(), consentDirective.getOptinConsentType().getName());
                if(ca != null){
                    sc.setFacility(ca.getFacility());
                }
                sc.setServiceConsumerType(ServiceConsumer.ADMINISTRATOR_PORTAL);

                if (consentDirective.getPurposeOfUse() != null) {
                    data.setComponentPurposeOfUseDisplayName(consentDirective.getPurposeOfUse().getPouValue());
                }

                if (consentDirective.getOptinConsentType().getName().equals(ConsentType.NW_HIN_AUTHORIZATION.getValue())) {
                    sc.setConsentType(ConsentType.NW_HIN_REVOCATION);
                } else if (consentDirective.getOptinConsentType().getName().equals(ConsentType.SSA_AUTHORIZATION.getValue())) {
                    sc.setConsentType(ConsentType.SSA_REVOCATION);
                } else {
                    throw new RuntimeException();
                }

                try {
                        // Convert the PDF into byte string
                    // Convert the ConsentDirectiveData to XML document
                    final Document consentDirectiveDataDoc = this.consentDirectiveDataJaxbHelper
                        .marshal(data);
                    // Convert ConsentDirectiveData XML to CDA R2 XML
                    final Document consentDirectiveDocument = this.dataToConsentDirectiveDocument
                        .transform(consentDirectiveDataDoc);
                    // Convert CDA R2 XML to string
                    final String consentDirectiveDocumentString
                        = this.xml2String.transform(consentDirectiveDocument);
                    final byte[] consentDirectiveDocumentBytes = consentDirectiveDocumentString.getBytes();

                    final ConsentDirectiveRevocationRequest request = new ConsentDirectiveRevocationRequest();
                    request.setServiceConsumerContext(sc);
                    request.setOptoutReason(ConsentDirectiveOptOutReasonType
                        .fromValue(ConsentDirectiveOptOutReasonType.AUTHORIZATION_EXPIRED.getValue()));
                    request.setDocument(consentDirectiveDocumentBytes);
                    // Process revoke
                    this.cms.processOldConsentDirectiveRevocation(request, consentDirective);
                } catch (final ConsentManagementServiceException ex) {
                    Logger.getLogger(RecentlyExpiredConsentController.class.getName()).log(Level.SEVERE, null, ex);
                } catch (JAXBException ex) {
                    Logger.getLogger(RecentlyExpiredConsentController.class.getName()).log(Level.SEVERE, null, ex);
                } catch (TransformerException ex) {
                    Logger.getLogger(RecentlyExpiredConsentController.class.getName()).log(Level.SEVERE, null, ex);
                }
            }
        }
    }

}
