/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package gov.va.nvap.svc.consentmgmt;

import gov.va.med.nhin.adapter.audit.SortFieldType;
import gov.va.nvap.privacy.ConsentDirectiveDetailedExpirationRequest;
import gov.va.nvap.svc.consentmgmt.jpa.AuthorityJpaController;
import gov.va.nvap.svc.consentmgmt.jpa.FacilityJpaController;
import gov.va.nvap.svc.consentmgmt.jpa.FunctionalRoleJpaController;
import gov.va.nvap.svc.consentmgmt.jpa.LocationJpaController;
import gov.va.nvap.svc.consentmgmt.jpa.OrganizationsJpaController;
import gov.va.nvap.svc.consentmgmt.jpa.PurposeOfUseJpaController;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.Announcement;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.AnnouncementOrg;
import gov.va.nvap.svc.consentmgmt.stub.data.Authority;
import gov.va.nvap.svc.consentmgmt.stub.data.ConsentDirective;
import gov.va.nvap.svc.consentmgmt.stub.data.ConsentType;
import gov.va.nvap.svc.consentmgmt.stub.data.DetailedConsentDirective;
import gov.va.nvap.svc.consentmgmt.stub.data.ExpiringConsentConfiguration;
import gov.va.nvap.svc.consentmgmt.stub.data.Facility;
import gov.va.nvap.svc.consentmgmt.stub.data.FacilityOptInConsent;
import gov.va.nvap.svc.consentmgmt.stub.data.FunctionalRole;
import gov.va.nvap.svc.consentmgmt.stub.data.Location;
import gov.va.nvap.svc.consentmgmt.stub.data.OptoutReason;
import gov.va.nvap.svc.consentmgmt.stub.data.Organization;
import gov.va.nvap.svc.consentmgmt.stub.data.PatientDocument;
import gov.va.nvap.svc.consentmgmt.stub.data.PatientDocumentType;
import gov.va.nvap.svc.consentmgmt.stub.data.PurposeOfUse;
import java.util.Collection;
import java.util.Date;
import java.util.logging.Level;
import java.util.logging.Logger;

/**
 *
 * @author vhaislegberb
 * @author Zack Peterson
 */
public class PolicyInformationPoint implements PIPInterface {

    private static final Logger logger = Logger
        .getLogger(PolicyInformationPoint.class.getName());

    private AuthorityControllerInterface authorityController;
    private ConsentDirectiveControllerInterface consentDirectiveController;
    private ExpiringConsentControllerInterface expiringConsentController;
    private FacilityControllerInterface facilityController;
    private FunctionalRoleControllerInterface functionalRoleController;
    private LocationControllerInterface locationController;
    private OrganizationsControllerInterface organizationsController;
    private PatientDocumentControllerInterface patientDocumentController;
    private PurposeOfUseControllerInterface purposeOfUseController;
    private AnnouncementOrgControllerInterface announcementOrgController;

    @Override
    public ConsentDirective getActiveConsentDirective(
        final Collection<String> patientIens, final ConsentType consentType) {
        return this.consentDirectiveController
            .findPatientActiveConsentDirective(patientIens, consentType);
    }

    @Override
    public Collection<ConsentDirective> getActiveConsentDirectives(
        final Collection<String> patientIens) {
        return this.consentDirectiveController
            .findPatientActiveConsentDirectives(patientIens);
    }

    @Override
    public Collection<FunctionalRole> getAllowedFunctionalRoles() {
        return this.functionalRoleController.findFunctionalRoleEntities();
    }

    @Override
    public Collection<Organization> getAllowedOrganizations() {
        return this.organizationsController.getActiveOrganizations();
    }

    @Override
    public Boolean getIsOrganizationTrustedSource(String organizationId) {

        return this.organizationsController.getIsOrganizationTrustedSource(organizationId);
    }

    @Override
    public Collection<PurposeOfUse> getAllowedPurposeOfUse() {
        return this.purposeOfUseController.findPurposeOfUseEntities();
    }

    @Override
    public Collection<Facility> getAllowedVistAFacilities() {
        return this.facilityController.findAllowedFacilities();
    }

    @Override
    public String getAuthorityName(final String oid) {
        final Authority authority = this.authorityController
            .findAuthortiyByOid(oid);
        if (authority != null) {
            return authority.getAuthorityName();
        }
        return null;
    }

    @Override
    public PatientDocument getConsentDirectiveDocumentByType(
        final ConsentDirective consentDirective,
        final PatientDocumentType patientDocumentType) {
        return this.patientDocumentController.findByConsentDirectiveAndType(
            consentDirective, patientDocumentType);
    }

    @Override
    public Collection<PatientDocument> getConsentDirectiveDocuments(
        final ConsentDirective consentDirective) {
        return this.patientDocumentController
            .findByConsentDirective(consentDirective);
    }

    @Override
    public Collection<ConsentDirective> getConsentHistory(
        final Collection<String> patientIens) {
        return this.consentDirectiveController
            .findPatientConsentDirectives(patientIens);
    }

    @Override
    public ConsentType getConsentTypeByName(final String name) {
        return this.consentDirectiveController.findConsentTypeByName(name);
    }

    @Override
    public PurposeOfUse getPurposeOfUseByValue(String pouValue) {
        return this.consentDirectiveController.findPurposeOfUseByValue(pouValue);
    }

    @Override
    public Facility getFacilityByStationId(final String id) {
        return this.facilityController.findFacilityByStation(id);
    }

    @Override
    public Location getLocationByCode(final String locationCode) {
        return this.locationController.findByLocationCode(locationCode);
    }

    @Override
    public Collection<OptoutReason> getOptoutReasons() {
        return this.consentDirectiveController.findOptoutReasons();
    }

    @Override
    public Organization getOrganizationById(final Long id) {
        return this.organizationsController.findOrganization(id);
    }

    @Override
    public Organization getOrganizationByOid(final String oid) {
        return this.organizationsController.findOrganizationByOid(oid);
    }

    @Override
    public Collection<PatientDocumentType> getPatientDocumentTypes() {
        return this.patientDocumentController.findPatientDocumentTypes();
    }

    @Override
    public PolicyConstraints getPolicyConstraints(
        final Collection<String> patientIens, final ConsentType consentType) {

        //***
        // Before this check, an NPE would just be thrown, and all callers
        // would handle that by using null in place of the return from this
        // method. But that caused disturbing stack traces in the log, especially
        // from the EJB Facade.
        //***
        ConsentDirective patientDirective =
            getActiveConsentDirective(patientIens, consentType);
        if(patientDirective == null)
        {
            return null;
        }

        final PolicyConstraints policyConstraints = new PolicyConstraints();
        policyConstraints.setAllowedFunctionalRoles(this
            .getAllowedFunctionalRoles());
        policyConstraints.setAllowedOrganizations(this
            .getAllowedOrganizations());
        policyConstraints.setAllowedPurposeOfUse(this.getAllowedPurposeOfUse());
        policyConstraints.setPatientExcludedOrganizations(
                patientDirective.getExcludedOrganizations());
        policyConstraints.setAllowedVistaFacilities(this
            .getAllowedVistAFacilities());

        return policyConstraints;
    }

    public void setAuthorityController(
        final AuthorityJpaController authorityController) {
        this.authorityController = authorityController;
    }

    public void setConsentDirectiveController(
        final ConsentDirectiveControllerInterface consentDirectiveController) {
        this.consentDirectiveController = consentDirectiveController;
    }

    public void setExpiringConsentController(
        final ExpiringConsentControllerInterface expiringConsentController) {
        this.expiringConsentController = expiringConsentController;
    }

    public void setFacilityController(
        final FacilityJpaController facilityController) {
        this.facilityController = facilityController;
    }

    public void setFunctionalRoleController(
        final FunctionalRoleJpaController functionalRoleController) {
        this.functionalRoleController = functionalRoleController;
    }

    public void setLocationController(
        final LocationJpaController locationController) {
        this.locationController = locationController;
    }

    public void setOrganizationsController(
        final OrganizationsJpaController organizationsController) {
        this.organizationsController = organizationsController;
    }

    public void setPatientDocumentController(
        final PatientDocumentControllerInterface patientDocumentController) {
        this.patientDocumentController = patientDocumentController;
    }

    public void setPurposeOfUseController(
        final PurposeOfUseJpaController purposeOfUseController) {
        this.purposeOfUseController = purposeOfUseController;
    }

    public AnnouncementOrgControllerInterface getAnnouncementOrgController() {
        return announcementOrgController;
    }

    public void setAnnouncementOrgController(AnnouncementOrgControllerInterface announcementOrgController) {
        this.announcementOrgController = announcementOrgController;
    }

    @Override
    public void storeConsentDirective(final ConsentDirective consentDirective) {
        if (consentDirective.getConsentDirId() == null) {
            try {
                this.consentDirectiveController.create(consentDirective);
            } catch (final Exception ex) {
                throw new PolicyInformationPointException(
                    "An error occurred while creating the new patient consent directive in the persistent store.",
                    ex);
            }
        } else {
            try {
                this.consentDirectiveController.edit(consentDirective);
            } catch (final Exception ex) {
                throw new PolicyInformationPointException(
                    "An error occured while updating the patient's consent directive.",
                    ex);
            }
        }
    }

    @Override
    public void storeOrganization(final Organization organization) {

        if (organization == null) {
            throw new IllegalArgumentException(
                "Cannot store null organization.");
        }

        try {
            if (organization.getOrgId() == null) {
                this.organizationsController.create(organization);
            } else {
                this.organizationsController.edit(organization);
            }
        } catch (final Exception ex) {
            throw new PolicyInformationPointException(
                "Error storing organization.", ex);
        }
    }

    @Override
    public void storePatientDocument(final PatientDocument patientDocument) {
        try {
            this.patientDocumentController.create(patientDocument);
        } catch (final Exception ex) {
            throw new PolicyInformationPointException(
                "Error creating patient document.", ex);
        }
    }

    @Override
    public void updatePatientDocument(final PatientDocument patientDocument) {
        if (patientDocument == null) {
            throw new IllegalArgumentException(
                "Cannot update a null patient document!");
        }
        try {
            this.patientDocumentController.edit(patientDocument);
        } catch (final Exception ex) {
            throw new PolicyInformationPointException(
                "Could not update patient document!");
        }

    }

    @Override
    public Collection<ConsentDirective> getExpiringPatientConsentDirectives(
        Collection<String> patientIens, ConsentDirectiveDetailedExpirationRequest consentDirectiveExpirationRequest) {

        int daysTilExpiration = consentDirectiveExpirationRequest.getDayRange();
        Date startDate = consentDirectiveExpirationRequest.getStartDate();
        Date endDate = consentDirectiveExpirationRequest.getEndDate();

        int fromPage = 0;
        int toPage = 0;

        if (consentDirectiveExpirationRequest.getPageInfo() != null) {
            int pageNumber = consentDirectiveExpirationRequest.getPageInfo().getPageNumber();
            int pageSize = consentDirectiveExpirationRequest.getPageInfo().getPageSize();
            fromPage = pageNumber * pageSize + 1;

            if (pageNumber <= 0) {
                toPage = pageSize;
            } else {
                toPage = (pageNumber + 1) * pageSize;
            }
        } else {
            // Expired consent email notification case
            toPage = -1;
        }

        if (patientIens == null) {
            if (startDate != null || endDate != null) {
                return this.consentDirectiveController.findAllExpiringPatientConsentDirectives(startDate, endDate, fromPage, toPage);
            }
            if (daysTilExpiration == -1) {
                return this.consentDirectiveController.findAllFutureExpiringPatientConsentDirectives(fromPage, toPage);
            }
            return this.consentDirectiveController.findAllExpiringPatientConsentDirectives(daysTilExpiration, fromPage, toPage);
        }
        return this.consentDirectiveController.findExpiringPatientConsentDirectives(daysTilExpiration, fromPage, toPage, patientIens);
    }

    @Override
    public Collection<DetailedConsentDirective> getExpiringPatientDetailedConsentDirectives(
        ConsentDirectiveDetailedExpirationRequest consentDirectiveExpirationRequest) {

        Date startDate = consentDirectiveExpirationRequest.getStartDate();
        Date endDate = consentDirectiveExpirationRequest.getEndDate();
        String facilityStation = consentDirectiveExpirationRequest.getFacilityStation();

        String consentTypeName = consentDirectiveExpirationRequest.getConsentTypeName();
        String userId = consentDirectiveExpirationRequest.getUserId();
        int fromPage = 0;
        int toPage = 0;

        if (consentDirectiveExpirationRequest.getPageInfo() != null) {
            int pageNumber = consentDirectiveExpirationRequest.getPageInfo().getPageNumber();
            int pageSize = consentDirectiveExpirationRequest.getPageInfo().getPageSize();
            fromPage = pageNumber * pageSize + 1;

            if (pageNumber <= 0) {
                toPage = pageSize;
            } else {
                toPage = (pageNumber + 1) * pageSize;
            }
        }

        // Default sort values
        String sortColumn = "OPTIN_DATE";
        String sortDirection = "DESC";
        if (consentDirectiveExpirationRequest.getSortFields() != null) {
            SortFieldType sortField = consentDirectiveExpirationRequest.getSortFields().getSortField().get(0);
            sortColumn = sortField.getField().name();
            sortDirection = sortField.getDirection().value();
        }

        int patientTypes = consentDirectiveExpirationRequest.getPatientTypes();

        return this.consentDirectiveController.findAllExpiringDetailedConsentDirectives(
            startDate, endDate, facilityStation, consentTypeName, userId, fromPage, toPage, sortColumn, sortDirection, patientTypes);
    }

    @Override
    public Collection<FacilityOptInConsent> getAuthorizedConsentForSummary() {
        return this.consentDirectiveController.findAuthorizedConsentForOptInSummary();
    }

    @Override
    public Collection<Organization> getAllowedNonConsumerOnlyOrganizations() {
        return this.organizationsController.getActiveNonConsumerOnlyOrganizations();
    }

    @Override
    public Collection<Organization> getAllowedConsumerOnlyOrganizations() {
        return this.organizationsController.getActiveConsumerOnlyOrganizations();
    }

    @Override
    public ExpiringConsentConfiguration getExpiringConsentConfiguration() {
        ExpiringConsentConfiguration configuration
            = this.expiringConsentController.getConfiguration();

        return configuration;
    }

    @Override
    public void updateExpiringConsentConfiguration(ExpiringConsentConfiguration config) {
        try {
            this.expiringConsentController.edit(config);
        } catch (Exception e) {
            logger.log(Level.INFO, "Could not update Expiring Consent Configuration");
        }
    }

    @Override
    public Collection<AnnouncementOrg> getAllAnnouncementOrgs() {
        return this.announcementOrgController.findAllAnnouncementOrgs();
    }

    @Override
    public Collection<AnnouncementOrg> getAnnouncementOrgsByAnnouncement(Announcement announcement) {
        return this.announcementOrgController.findAnnouncementOrgsByAnnouncement(announcement);
    }

    @Override
    public void storeAnnouncementOrg(AnnouncementOrg announceOrg) {
        try {
            this.announcementOrgController.create(announceOrg);
        } catch (final Exception ex) {
            throw new PolicyInformationPointException(
                "Error creating announcement org.", ex);
        }
    }

    @Override
    public void updateAnnouncementOrg(AnnouncementOrg announceOrg) {
        try {
            this.announcementOrgController.edit(announceOrg);
        } catch (final Exception ex) {
            throw new PolicyInformationPointException(
                "Error updating announcement org.", ex);
        }
    }

    @Override
    public Collection<ConsentDirective> getRecentlyExpiredConsent() {
        return this.consentDirectiveController.findRecentlyExpiredConsent();
    }

    @Override
    public ConsentDirective getConsentDirective(Long id) {
        return this.consentDirectiveController.findConsentDirective(id);
    }

    @Override
    public Facility getFacilityByName(String facilityName) {
        return this.facilityController.findFacilityByName(facilityName);
    }
}
