/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package gov.va.nvap.svc.consenteval.impl;

import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.extsvc.psim.intf.PersonServiceInterface;
import gov.va.nvap.svc.consenteval.common.PatientIdUtil;
import gov.va.nvap.svc.consenteval.common.XACMLContextConstants;
import gov.va.nvap.svc.consenteval.common.XACMLContextUtil;
import gov.va.nvap.svc.consenteval.common.XACMLMessageMarshaller;
import gov.va.nvap.svc.consenteval.common.XACMLRequestParser;
import gov.va.nvap.server.endpoint.adapter.doc.AdapterDocQueryEndpoint;
import static gov.va.nvap.server.endpoint.adapter.doc.AdapterDocQueryEndpoint.C_CDA_CLASS_CODE;
import static gov.va.nvap.server.endpoint.adapter.doc.AdapterDocQueryRetrieveFacade.DQ_MAP_DOCUMENT_REPOSITORY_ID;
import static gov.va.nvap.server.endpoint.adapter.doc.AdapterDocQueryRetrieveFacade.DQ_MAP_DOCUMENT_UNIQUE_ID;
import static gov.va.nvap.server.endpoint.adapter.doc.AdapterDocQueryRetrieveFacade.DQ_MAP_HOME_COMMUNITY_ID;
import gov.va.nvap.server.endpoint.adapter.doc.AdapterDocRetrieveEndpoint;
import gov.va.nvap.server.endpoint.permission.sls.SLSClient;
import gov.va.nvap.service.pdq.PatientCorrelationsQuery;
import gov.va.nvap.service.pdq.PatientCorrelationsResponse;
import gov.va.nvap.service.pdq.PdqException;
import gov.va.nvap.service.pdq.PdqService;
import gov.va.nvap.service.permission.pdp.PDPInterface;
import gov.va.nvap.svc.consenteval.intf.PolicyEvaluationService;
import gov.va.nvap.svc.consentmgmt.PIPInterface;
import gov.va.nvap.svc.consentmgmt.PolicyConstraints;
import gov.va.nvap.svc.consentmgmt.stub.dao.PatientConfidentialityDAO;
import gov.va.nvap.svc.consentmgmt.stub.data.ConfidentialityCodeType;
import gov.va.nvap.svc.consentmgmt.stub.data.ConsentDirective;
import gov.va.nvap.svc.consentmgmt.stub.data.Organization;
import gov.va.nvap.svc.consentmgmt.stub.data.PatientConfidentiality;
import gov.va.nvap.svc.facility.data.Facility;
import gov.va.nvap.svc.facility.intf.FacilityService;

import java.io.StringWriter;
import java.nio.charset.StandardCharsets;
import java.text.DateFormat;
import java.text.MessageFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;


import oasis.names.tc.xacml._2_0.context.schema.os.DecisionType;
import oasis.names.tc.xacml._2_0.context.schema.os.RequestType;
import oasis.names.tc.xacml._2_0.context.schema.os.ResponseType;
import oasis.names.tc.xacml._2_0.context.schema.os.ResultType;
import org.apache.commons.lang.StringUtils;
import org.slf4j.LoggerFactory;

import org.springframework.beans.factory.annotation.Required;

/**
 * 
 * @author Anand Sastry
 */
public class PolicyEvaluationServiceImpl implements  PolicyEvaluationService {

	private static final Logger logger = Logger
			.getLogger(PolicyEvaluationServiceImpl.class.getName());
    private static final org.slf4j.Logger LOGGER =
        LoggerFactory.getLogger(PolicyEvaluationServiceImpl.class);

    private static final String USER_FOR_ADAPTER_RTRV = "VAP_SYSTEM";
    private static final Date JAN_1900;
    static
    {
        DateFormat df = new SimpleDateFormat("MM-dd-yyyy");
        try
        {
            JAN_1900 = df.parse("01-01-1900");
        }
        catch(ParseException ex)
        {
            throw new RuntimeException("Fixed date parse failed!");
        }
    }
    
    // Spring Injected
    		
	private String localHomeCommunityId;// = "urn:oid:2.16.840.1.113883.4.349";
	private String dodHomeCommunityId;// = "urn:oid:2.16.840.1.113883.3.42.10001.100001.12";
	private PDPInterface pdpProxy;
	private PdqService pdqService;
	private PersonServiceInterface personService;
	private String nwhinConsentType;
	private String nwhinRestrictionType;
	private String ssaConsentType;
	private AdapterDocQueryEndpoint adapterDocQueryEndpoint;
	private AdapterDocRetrieveEndpoint adapterDocRetrieveEndpoint;
    private SLSClient slsEndpoint;
    private PatientConfidentialityDAO patientConfidentialityDAO;
    private boolean enableSLSIntegration = false;
	
	
//	@EJB(name = "PipService", beanInterface = PIPInterface.class)
	private PIPInterface pipInterface;
	
//	@EJB(name = "PipService", beanInterface = PIPInterface.class)
	private FacilityService facilityService;



	private String addOidPrefix(final String value) {
		if (value.startsWith("urn:oid:")) {
			return value;
		}
		return "urn:oid:" + value;
	}

	private boolean checkOptedInIfPertinent(final String optInStatus,
			final String hcid) {
		return (XACMLContextConstants.RESOURCE_OPTIN_YES_VALUE
				.equals(optInStatus) || this.dodHomeCommunityId.equals(hcid));
	}

	/**
	 * @param request
	 * @return
	 */
    @Override
	public ResponseType checkPolicy(final RequestType request) {
		ResponseType response = null;
		PolicyEvaluationServiceImpl.logger.entering(this.getClass().getName(),
				"checkPolicy");
		this.dumpRequestContents("In CheckPolicy - Received from Adapter",
				request);
		this.info("checkPolicy", "PolicyEvaluationServiceImpl Processing Begins ...");
		if (XACMLRequestParser.isNHINOut(request)) {
			response = this.evaluateForNHINOut(request);
		} else {
			response = this.evaluateNwHINTransactions(request);
		}
		this.info("checkPolicy", "PolicyEvaluationServiceImpl Processing Ended.");
		PolicyEvaluationServiceImpl.logger.exiting(this.getClass().getName(),
				"checkPolicy", response.getResult().get(0).getDecision());

		return response;
	}

	private void dumpRequestContents(final String context, final RequestType req) {
		if (LOGGER.isDebugEnabled())
        {
			final StringWriter sw = new StringWriter();
			XACMLMessageMarshaller.marshalXACMLRequestToWriter(context, req, sw);
            String msg = "PolicyEvaluationServiceImpl " + context + System.lineSeparator();
            LOGGER.debug(msg + sw.getBuffer());
		}
	}

	private void dumpResponseContents(final String context,
			final ResponseType res) {
		if (LOGGER.isDebugEnabled())
        {
			final StringWriter sw = new StringWriter();
			XACMLMessageMarshaller.marshalXACMLResponseToWriter(context, res, sw);
            String msg = "PolicyEvaluationServiceImpl " + context + System.lineSeparator();
            LOGGER.debug(msg + sw.getBuffer());
		}
	}

	private ResponseType evaluateForNHINOut(final RequestType requestType) {
		final String userId = this.getSubjectId(requestType);
		final boolean authorized = !NullChecker.isNullOrEmpty(userId)
				&& this.valueStartsWithInArray(userId,
						this.getVistAFacilities());
		final ResponseType resType = this.generateResponse(authorized);
		this.dumpResponseContents("Response from Within PolicyEvaluationServiceImpl ",
				resType);
		return resType;
	}

	private ResponseType evaluateNwHINTransactions(final RequestType request) {
		// try {
		try {
			this.loadConstraintsFromPIP(request);
		} catch (final Exception e) {
			LOGGER.warn(
                "An exception occured while attempting to load contraints " +
                "from PIP. Not delegating to PDP.", e);
			final ResponseType resType = this.generateResponse(false);
			this.dumpResponseContents(
					"Response from Within PolicyEvaluationServiceImpl ", resType);
			return resType;
		}
		this.transformAttributes(request);
		// } catch (Throwable e) {
		// System.out
		// .println("XACMLContextHandlerService:evaluateNwHINTransactions - Exception - "
		// + e.getLocalizedMessage());
		// Should we let the evaluation to PDP play it out ????
		// Or should we force a DENY ??
		// Play it out for now ...
		// }

		this.dumpRequestContents(
				"In evaluateNwHINTransactions - After data has been loaded from the PIP. Payload to PDP ...",
				request);
		return this.evaluateRequestAgainstPDPPolicy(request);
	}

	private ResponseType evaluateRequestAgainstPDPPolicy(
			final RequestType request) {
		PolicyEvaluationServiceImpl.logger.entering(this.getClass().getName(),
				"evaluateRequestAgainstPDPPolicy");
		final ResponseType resType = this.pdpProxy.checkPolicy(request);
		this.dumpResponseContents("Response from PDP ", resType);
		PolicyEvaluationServiceImpl.logger.exiting(this.getClass().getName(),
				"evaluateRequestAgainstPDPPolicy");
		return resType;
	}

	private ResponseType generateResponse(final boolean authorized) {
		final ResponseType response = new ResponseType();
		final List<ResultType> results = response.getResult();
		final ResultType result = new ResultType();
		result.setDecision(authorized ? DecisionType.PERMIT : DecisionType.DENY);
		results.add(result);
		return response;
	}

	// The only reason NHINOut processing is being done here is that Allowed
	// VistA Facilities are being
	// maintained by VAP. It should not be. It should be directly available to
	// PDP. The only way it makes
	// sense for VAP to maintain it, is if an exclusion is applied for a
	// patient.
	// Everything below this be eliminated. All evaluation, including for
	// NHINOut should happen in the PDP.

	private String getOptInStatus(final ConsentDirective consentDir) {
		if (consentDir == null) {
			return XACMLContextConstants.RESOURCE_OPTIN_VALUE_NO;
		}
		return XACMLContextConstants.RESOURCE_OPTIN_YES_VALUE;
	}

	private Map<String, String> getPatientAllowedOrganizationOidsAndNumbers() {

		/*
		 * Map<String, String> patientAllowedOrganizationOidsAndNumbers = new
		 * HashMap<String, String>(); if (pc == null) { //DOD is always allowed
		 * patientAllowedOrganizationOidsAndNumbers.put(dodHomeCommunityId,
		 * dodHomeCommunityNumber); return
		 * patientAllowedOrganizationOidsAndNumbers; }
		 * 
		 * Iterator<Organization> allowedOrganizationsIt = pip
		 * .getAllowedOrganizations().iterator(); Collection<Organization>
		 * patientExcludedOrgs = pc .getPatientExcludedOrganizations();
		 * 
		 * while (allowedOrganizationsIt.hasNext()) { Organization allowedOrg =
		 * allowedOrganizationsIt.next(); if
		 * (!patientExcludedOrgs.contains(allowedOrg)) {
		 * patientAllowedOrganizationOidsAndNumbers
		 * .put(addOidPrefix(allowedOrg.getOrgOid()),
		 * allowedOrg.getOrgNumber()); }
		 * 
		 * }
		 * 
		 * return patientAllowedOrganizationOidsAndNumbers;
		 */

		/* Reevaluate needing this method in I2 */

		final Map<String, String> patientAllowedOrganizationOidsAndNumbers = new HashMap<String, String>();
		final Iterator<Organization> allowedOrganizationsIt = this.getAllowedOrganizations().iterator();
		while (allowedOrganizationsIt.hasNext()) {
			final Organization allowedOrg = allowedOrganizationsIt.next();
			patientAllowedOrganizationOidsAndNumbers.put(
					this.addOidPrefix(allowedOrg.getOrgOid()),
					allowedOrg.getOrgNumber());
		}
		return patientAllowedOrganizationOidsAndNumbers;

	}
	
	private boolean isRequestorSSA(RequestType request) {
		return false;
	}
	
	private boolean isRequestorAllowed(RequestType request) {
		return true;
	}
	
	private Collection<Organization> getAllowedOrganizations() {
		return this.pipInterface.getAllowedOrganizations();
	}

	private List<String> getPatientAllowedOrganizations(
        final String optInStatus, final String authorizationType,
        final Collection<String> patientIens, RequestType request,
        String patientId /* for logging */)
    {
		final List<String> patientAllowedOrganizations = new ArrayList<String>();
		// if patient opted in to share with NwHIN orgs
		if (XACMLContextConstants.RESOURCE_OPTIN_VALUE_NO.equals(optInStatus) &&
				nwhinConsentType.equals(authorizationType)) {
			// Patient opted out for data sharing with NwHIN, add DOD
			patientAllowedOrganizations.add(this.dodHomeCommunityId);	
		}
		else if (XACMLContextConstants.RESOURCE_OPTIN_YES_VALUE.equals(optInStatus) &&
				nwhinConsentType.equals(authorizationType)) {
			// patient opted in....See if restrictions exist...
			//
			// Contact PIP and obtain Constraints
			PolicyConstraints pc = null;
			try {
				pc = this.pipInterface.getPolicyConstraints(
						patientIens,
						this.pipInterface.getConsentTypeByName(nwhinRestrictionType));
			} catch (final Exception e) {
                String msg = MessageFormat.format(
                    "getPatientAllowedOrganizations() failed while attempting to " +
                    "find the patient policy constraints for Patient ICN {0}.",
                    patientId);
                LOGGER.warn(msg, e);
			}
			return getPatientAllowedOrganizations(pc);
		}
		else if (XACMLContextConstants.RESOURCE_OPTIN_YES_VALUE.equals(optInStatus) &&
				ssaConsentType.equals(authorizationType)) {
			patientAllowedOrganizations.add(this.addOidPrefix(XACMLRequestParser
					.getRequestorHomeCommunityId(request)));
		}
		return patientAllowedOrganizations; // DOD only

	}

	private List<String> getPatientAllowedOrganizations(
			final PolicyConstraints restrictions) {
		final List<String> patientAllowedOrganizations = new ArrayList<String>();

		final Iterator<Organization> allowedOrganizationsIt = this
				.getAllowedOrganizations().iterator();
		Collection<Organization> patientExcludedOrgs = new ArrayList<Organization>();
		if (restrictions != null ) {
			patientExcludedOrgs = restrictions.getPatientExcludedOrganizations();
		}				

		while (allowedOrganizationsIt.hasNext()) {
			final Organization allowedOrg = allowedOrganizationsIt.next();
			if (!patientExcludedOrgs.contains(allowedOrg)) {
				patientAllowedOrganizations.add(this.addOidPrefix(allowedOrg
						.getOrgOid()));
			}

		}

		return patientAllowedOrganizations;

	}

	private String getRequestedPatientId(final RequestType request) {
		return XACMLRequestParser.getRequestedResourceId(request);
	}

	// Used only for NHINOut
	private String getSubjectId(final RequestType requestType) {
		return XACMLRequestParser.getRequestorSubjectId(requestType);
	}

	private List<String> getVistAFacilities() {
		final List<String> VistAFacilities = new ArrayList<String>();

		final Collection<Facility> allowedVistAFacilities = this.facilityService
				.getAllowedVistAFacilities();

		for (final Facility facility : allowedVistAFacilities) {
			VistAFacilities.add(facility.getFacilityStation());
		}

		return VistAFacilities;

	}

	private void info(final String methodName, final String message) {
		if (PolicyEvaluationServiceImpl.logger.isLoggable(Level.INFO)) {
			//PolicyEvaluationServiceImpl.logger.logp(Level.INFO, this.getClass()
			//		.getName(), methodName, message);
		}
	}

	private boolean isLocalHomeCommunity(final String requestorHCID) {
		return requestorHCID.contains(this.localHomeCommunityId);
	}

	private boolean isPatientCorrelatedToRequestorsHomeCommunity(
			final String patientId, final String requestorHCID) {
		PolicyEvaluationServiceImpl.logger.entering(this.getClass().getName(),
				"isPatientCorrelatedToRequestorsHomeCommunity");

		final String facilityFromRequest = this
				.getPatientAllowedOrganizationOidsAndNumbers().get(
						requestorHCID);
		String correlatedPatientId = null;

		try {
			final PatientCorrelationsQuery correlationsQuery = new PatientCorrelationsQuery();
			correlationsQuery.setPatientId(patientId);
			final PatientCorrelationsResponse correlationsResponse = this.pdqService
					.getCorrelatedFacilities(correlationsQuery);
			final List<gov.va.nvap.service.pdq.Facility> facilities = correlationsResponse
					.getFacilities();
			for (final gov.va.nvap.service.pdq.Facility facility : facilities) {
				final String facilityNumber = facility.getFacilityNumber();
				if (facilityNumber.equals(facilityFromRequest)) {
					correlatedPatientId = facility.getCorrelatedPatientId();
					break;
				}
			}
		} catch (final PdqException e) {
			if (PolicyEvaluationServiceImpl.logger.isLoggable(Level.FINE)) {
				e.printStackTrace();
			}
			PolicyEvaluationServiceImpl.logger.logp(Level.SEVERE, this
					.getClass().getName(),
					"isPatientCorrelatedToRequestorsHomeCommunity", e
							.getMessage());
		}

		this.info("isPatientCorrelatedToRequestorsHomeCommunity",
				"Patient correlation id is  [" + correlatedPatientId + "]");
		PolicyEvaluationServiceImpl.logger.exiting(this.getClass().getName(),
				"isPatientCorrelatedToRequestorsHomeCommunity");
		return ((correlatedPatientId != null) && (correlatedPatientId.trim()
				.length() > 0));
	}

	private void loadConstraintsFromPIP(final RequestType request)
			throws Exception {
		PolicyEvaluationServiceImpl.logger.entering(this.getClass().getName(),
				"loadConstraintsFromPIP");
		// Obtain Patient IEN from request
		String patientId = this.getRequestedPatientId(request);
		this.info("loadConstraintsFromPIP", "Get IEN from Request ["
				+ patientId + "]");


		if (!NullChecker.isNullOrEmpty(patientId)) {
			// Patient ID might be in the HL7V2 CX Format.
			patientId = PatientIdUtil.parsePatientIdFromHL7CXType(patientId);
			
            determinePatConfCode(patientId, request);
			//
			// Contact PIP and obtain Constraints
			ConsentDirective consentDir = null;
		    Collection<String> patientIens = null;
		    String authorizationType = null;
			try {
				patientIens = this.personService.getCorrelatedIds(patientId);
				authorizationType = getApplicableConsentType(request);
                // Was calling getPolicyContraints, but usage of return was
                // simply to see if it threw an exception on a missing consent
                // directive. Just go for the consent directive and save DB time
                // getting the other constraint data that is ignored.
				consentDir = pipInterface.getActiveConsentDirective(
						patientIens,
						pipInterface.getConsentTypeByName(authorizationType));
				// List<String> vpids = new ArrayList<String>();
				// vpids.add(patientId);
				// pc =
				// pip.getPolicyConstraints(vpids,pip.getConsentTypeByName(consentType));
			} catch (final Exception e) {
                String msg = MessageFormat.format(
                    "loadConstraintsFromPIP() failed while attempting to " +
                    "find the patient consent directive for Patient ICN {0}.",
                    patientId);
                LOGGER.warn(msg, e);
			}            
            
			// Evaluate Constraints and set Opt-In information on request
			final String status = this.getOptInStatus(consentDir);
			XACMLContextUtil.addResourceAttributeValue(
					XACMLContextConstants.RESOURCE_OPTIN_ATTR_ID,
					XACMLContextConstants.STRING_ATTR_DATA_TYPE, status,
					request);

			this.info("loadConstraintsFromPIP",
					"Setting OPT-IN attribute on request - "
							+ XACMLContextConstants.RESOURCE_OPTIN_ATTR_ID
							+ " " + status);

			// Evaluate Constraints and set patient-allowed list (TODO: For now,
			// could change to patient disallowed list)
			final List<String> patientAllowedOrgs =
                getPatientAllowedOrganizations(status, authorizationType,
                patientIens, request, patientId);
			XACMLContextUtil
					.addResourceAttributeValues(
							XACMLContextConstants.RESOURCE_PATIENT_ALLOWED_ORGS_ATTR_ID,
							XACMLContextConstants.STRING_ATTR_DATA_TYPE,
							patientAllowedOrgs, request);

			this.info(
					"loadConstraintsFromPIP",
					"Setting Patient allowed organizations attribute(s) on request - "
							+ XACMLContextConstants.RESOURCE_PATIENT_ALLOWED_ORGS_ATTR_ID
							+ " " + patientAllowedOrgs);

			// Hack - Should be done by PDP.
			// Being done here as it requires some info from the PIP.
			// This method throws an exception
			this.performPatientCorrelationCheckIfRequired(request, patientId,
					status);
		}
		PolicyEvaluationServiceImpl.logger.exiting(this.getClass().getName(),
				"loadConstraintsFromPIP");
	}

    private void determinePatConfCode(String patientId, final RequestType request) throws Exception
    {
        if(!enableSLSIntegration)
        {
            return;
        }
        
        PatientConfidentiality cachedConf =
            patientConfidentialityDAO.getConfidentialityCode(patientId);
        
        String confidentialityCode;
        if(cachedConf != null)
        {
            ConfidentialityCodeType confCodeEnum = cachedConf.getConfidentialityCode();
            confidentialityCode =
                (confCodeEnum == ConfidentialityCodeType.NORMAL) ?
                XACMLContextConstants.RESOURCE_CONF_CODE_VAL_NORMAL :
                XACMLContextConstants.RESOURCE_CONF_CODE_VAL_RESTRICTED;
            if(LOGGER.isDebugEnabled())
            {
                String cachedDebugMsg = MessageFormat.format(
                    "For Patient ICN {0}, using cached confidentiality code: {1}",
                    patientId, confidentialityCode);
                LOGGER.debug(cachedDebugMsg);
            }
        }
        else
        {
            String patientSummaryDoc = retrievePatientSummary(patientId);
            if(LOGGER.isDebugEnabled())
            {
                String docDebugMsg = MessageFormat.format(
                        "For Patient ICN {0}, received the following patient " +
                                "summary from Adapter for SLS: {1}", patientId, patientSummaryDoc);
                LOGGER.debug(docDebugMsg);
            }
            // Adding Confidentiality Code
            confidentialityCode = slsEndpoint.evalConfidentialityCode(patientSummaryDoc.getBytes(StandardCharsets.UTF_8));
            if(LOGGER.isDebugEnabled())
            {
                String slsDebugMsg = MessageFormat.format(
                        "For Patient ICN {0}, SLS returned confidentiality code: {1}",
                        patientId, confidentialityCode);
                LOGGER.debug(slsDebugMsg);
            }
            
            // update the DB cache
            ConfidentialityCodeType confCodeEnum =
                XACMLContextConstants.RESOURCE_CONF_CODE_VAL_NORMAL.equals(confidentialityCode) ?
                ConfidentialityCodeType.NORMAL :
                ConfidentialityCodeType.RESTRICTED;
            if(LOGGER.isDebugEnabled())
            {
                String cachedDebugMsg = MessageFormat.format(
                    "For Patient ICN {0}, adding / updating cached confidentiality code: {1} "+
                    "from SLS result: {2}",
                    patientId, confCodeEnum, confidentialityCode);
                LOGGER.debug(cachedDebugMsg);
            }
            patientConfidentialityDAO.storeConfidentialityCode(patientId, confCodeEnum);
        }
        XACMLContextUtil.addResourceAttributeValue(
                XACMLContextConstants.RESOURCE_CONF_CODE_ATTR_ID,
                XACMLContextConstants.ANYURI_ATTR_DATA_TYPE,
                confidentialityCode,
                request);
    }

    private String retrievePatientSummary(String patientId) throws Exception
    {
        Date toDate = Calendar.getInstance().getTime();
        
        // Call DocQuery
        List<Map<String, String>> documentList = 
            adapterDocQueryEndpoint.getDocumentsByClassCode(
            patientId, JAN_1900, toDate, C_CDA_CLASS_CODE);

        if (NullChecker.isNotEmpty(documentList))
        {
            Map<String, String> documentMap = documentList.get(0);
            String homeCommunityId = documentMap.get(DQ_MAP_HOME_COMMUNITY_ID);
            String repositoryId = documentMap.get(DQ_MAP_DOCUMENT_REPOSITORY_ID);
            String documentUniqueId = documentMap.get(DQ_MAP_DOCUMENT_UNIQUE_ID);
            
            boolean completeId =
                StringUtils.isNotEmpty(homeCommunityId) &&
                StringUtils.isNotEmpty(repositoryId) &&
                StringUtils.isNotEmpty(documentUniqueId);

            if(completeId)
            {
                // Call DocRetrieve
                return adapterDocRetrieveEndpoint.getDocument(
                    patientId, documentUniqueId,
                    repositoryId, homeCommunityId,
                    USER_FOR_ADAPTER_RTRV);
            }
            
        }
        
        // would return above if document found
        String msg = MessageFormat.format(
            "For Patient ICN {0}, Adapter Document Query did not " +
            "produce a patient summary document.", patientId);
        throw new Exception(msg);
    }

	private String getApplicableConsentType(RequestType request) throws Exception {
		// TODO Ugly, what else can I say, maybe, will be replaced.
		// 
		final String requestorHCID = this.addOidPrefix(XACMLRequestParser
				.getRequestorHomeCommunityId(request));
		this.info("getApplicableConsentType",
				"Get Requestors HCID from Request [" + requestorHCID + "]");
		
		if (isRequestorConsumerOnly(requestorHCID)) {
			return this.ssaConsentType;
		}
		return this.nwhinConsentType;
	}

	private void performPatientCorrelationCheckIfRequired(
			final RequestType request, final String patientId,
			final String status) throws Exception {
		final String requestorHCID = this.addOidPrefix(XACMLRequestParser
				.getRequestorHomeCommunityId(request));
		this.info("performPatientCorrelationCheckIfRequired",
				"Get Requestors HCID from Request [" + requestorHCID + "]");

		if (!this.isLocalHomeCommunity(requestorHCID)
				&& XACMLRequestParser.isInbound(request)
				&& (XACMLRequestParser.isDocQuery(request) || XACMLRequestParser
						.isDocRetrieve(request))
				&& this.checkOptedInIfPertinent(status, requestorHCID)
				&& !isRequestorConsumerOnly(requestorHCID)) {
			final boolean correlationsExist = this
					.isPatientCorrelatedToRequestorsHomeCommunity(patientId,
							requestorHCID);
			if (!correlationsExist) {
				throw new Exception(
						"performPatientCorrelationCheckIfRequired - No Correlations Exist for patient, "
								+ patientId);
			}
		} else {
			this.info("performPatientCorrelationCheckIfRequired",
					"Patient Correlation Check not required.");
		}
	}
	
	private boolean isRequestorConsumerOnly (
			final String requestorHCID) throws Exception {

		for (Organization allowedOrg : this.getAllowedOrganizations()) {
			if (requestorHCID.endsWith(allowedOrg.getOrgOid()) && allowedOrg.isOrgConsumerOnly()) {
				this.info("isRequestorConsumerOnly", requestorHCID + " is a consumer only Org's OID");
				return true;
		    }
		}
		this.info("isRequestorConsumerOnly", requestorHCID + " is not a consumer only Org's OID");
		return false;
	}

	private void transformAttributes(final RequestType request) {
		PolicyEvaluationServiceImpl.logger.entering(this.getClass().getName(),
				"transformAttributes");
		// 1)Set XSPA Attribute, urn:oasis:names:tc:xacml:2.0:subject:locality"
		// with requestor's Home Community
		final String subjectLocality = this.addOidPrefix(XACMLRequestParser
				.getRequestorHomeCommunityId(request));

		this.info("transformAttributes", "Setting "
				+ XACMLContextConstants.SUBJECT_LOCALITY_ATTR_ID + " = "
				+ subjectLocality);
		XACMLContextUtil.addSubjectAttributeValue(
				XACMLContextConstants.SUBJECT_LOCALITY_ATTR_ID,
				XACMLContextConstants.STRING_ATTR_DATA_TYPE, subjectLocality,
				request);

		// 2)Set XSPA Attribute,
		// urn:oasis:names:tc:xacml:2.0:environment:locality"
		// with resource Home Community
		final String environmentLocality = this.addOidPrefix(XACMLRequestParser
				.getResourceHomeCommunityId(request));

		this.info("transformAttributes", "Setting "
				+ XACMLContextConstants.ENV_LOCALITY_ATTR_ID + " = "
				+ environmentLocality);
		XACMLContextUtil.addResourceAttributeValue(
				XACMLContextConstants.ENV_LOCALITY_ATTR_ID,
				XACMLContextConstants.STRING_ATTR_DATA_TYPE,
				environmentLocality, request);

		// 3)Based on action set HL7 Resource type to
		// urn:oasis:names:tc:xspa:1.0:resource:hl7:type:patient-search
		// and
		// urn:oasis:names:tc:xspa:1.0:resource:hl7:type:medical-record
		// Note: This step might not be necessary. Evaluate and remove in VAP
		// 2.0
		if (XACMLRequestParser.isDocQuery(request)
				|| XACMLRequestParser.isDocRetrieve(request)) {

			this.info("transformAttributes", "Setting "
					+ XACMLContextConstants.RESOURCE_TYPE_ATTR_ID + " = "
					+ XACMLContextConstants.RESOURCE_TYPE_MEDICAL_RECORD_VALUE);
			XACMLContextUtil.addResourceAttributeValue(
					XACMLContextConstants.RESOURCE_TYPE_ATTR_ID,
					XACMLContextConstants.STRING_ATTR_DATA_TYPE,
					XACMLContextConstants.RESOURCE_TYPE_MEDICAL_RECORD_VALUE,
					request);
		} else if (XACMLRequestParser.isPatientDiscovery(request)) {

			this.info("transformAttributes", "Setting "
					+ XACMLContextConstants.RESOURCE_TYPE_ATTR_ID + " = "
					+ XACMLContextConstants.RESOURCE_TYPE_PATIENT_SEARCH_VALUE);
			XACMLContextUtil.addResourceAttributeValue(
					XACMLContextConstants.RESOURCE_TYPE_ATTR_ID,
					XACMLContextConstants.STRING_ATTR_DATA_TYPE,
					XACMLContextConstants.RESOURCE_TYPE_PATIENT_SEARCH_VALUE,
					request);
		}

		// 4 Create ServiceType Attributes with same values as ActionId
		// attributes
		// The new XACML Policies evaluate based on ServiceType Attributes as
		// opposed to the old version which used ActionId attributes
		// We are nor removing ActionId attributes yet, even though they will
		// not be evaluated. We can chose to remove this in the future.

		// Action value might not be needed. Evaluate and delete in VAP 2.0
		final String actionValue = XACMLRequestParser
				.getRequestedAction(request);
		this.info("transformAttributes", "Setting "
				+ XACMLContextConstants.RESOURCE_SERVICE_TYPE_ATTR_ID + " = "
				+ actionValue);
		XACMLContextUtil.addResourceAttributeValue(
				XACMLContextConstants.RESOURCE_SERVICE_TYPE_ATTR_ID,
				XACMLContextConstants.STRING_ATTR_DATA_TYPE, actionValue,
				request);

		// 5) Ensure that HomeCommunityId attributes have the "urn:oid" prefix
		// Might not have to do this if only subjectLocality is evaluated by PDP
		// REFACTOR

		String requestorHCID = XACMLRequestParser
				.getRequestorHomeCommunityId(request);
		if (!NullChecker.isNullOrEmpty(requestorHCID)
				&& !requestorHCID.startsWith("urn:oid:")) {
			this.info("transformAttributes",
					"Prefixing requestor HCID with urn:oid:");
			requestorHCID = this.addOidPrefix(requestorHCID);
			// remove existing attribute
			XACMLRequestParser.extractRequestorHomeCommunityId(request);
			// add the new attribute
			XACMLContextUtil
					.addSubjectAttributeValue(
							XACMLContextConstants.XACML_REQUEST_SUBJECT_HOME_COMMUNITY_ID_ATTR_ID,
							XACMLContextConstants.STRING_ATTR_DATA_TYPE,
							requestorHCID, request);
		}

		String resourceHCID = XACMLRequestParser
				.getResourceHomeCommunityId(request);
		if (!NullChecker.isNullOrEmpty(resourceHCID)
				&& !resourceHCID.startsWith("urn:oid:")) {
			this.info("transformAttributes",
					"Prefixing resource HCID with urn:oid:");
			resourceHCID = this.addOidPrefix(resourceHCID);
			// remove existing attribute
			XACMLRequestParser.extractResourceHomeCommunityId(request);
			// add the new attribute
			XACMLContextUtil
					.addResourceAttributeValue(
							XACMLContextConstants.XACML_REQUEST_RESOURCE_HOME_COMMUNITY_ID_ATTR_ID,
							XACMLContextConstants.STRING_ATTR_DATA_TYPE,
							resourceHCID, request);
		}
                //fix SAML Auth
                String resourceSaml = XACMLRequestParser
				.getSamlAuth(request);
		if (!NullChecker.isNullOrEmpty(resourceSaml)) {
			this.info("transformAttributes",
					"Fixing resource Saml to be string");
			
			// remove existing attribute
			XACMLContextUtil.extractResourceAttributeValues(request, XACMLContextConstants.RESOURCE_SAML_AUTH);

			// add the new attribute
			XACMLContextUtil.addResourceAttributeValue(
							XACMLContextConstants.RESOURCE_SAML_AUTH,
							XACMLContextConstants.STRING_ATTR_DATA_TYPE,
							resourceSaml, request);
		}
                
                
                
		// 6 Add empty environment element
		this.info("transformAttributes", "Adding empty environment element");
		XACMLContextUtil.addEmptyEnvironmentElement(request);

		// 7 Iterate through Subject, Resource and Action attributes. If they
		// are missing
		// data types create data types. If they are missing data type values,
		// default to String
		// Might not be the best solution. Re-evaluate in VAP 2.0
		//
		this.info("transformAttributes",
				"Performing message validation and structural integrity");
		XACMLRequestParser.validateAndFixMessage(request);
		// 8 Replace POU attribute's, data type with String
		// Might not be the best solution. Re-evaluate in VAP 2.0
		//
		this.info("transformAttributes",
				"Performing message validation and structural integrity");
		// remove existing attribute
		final String pou = XACMLRequestParser.extractSubjectPOU(request);
		// add the new attribute
		XACMLContextUtil
				.addSubjectAttributeValue(
						XACMLContextConstants.XACML_REQUEST_SUBJECT_PURPOSE_FOR_USE_ATTR_ID,
						XACMLContextConstants.STRING_ATTR_DATA_TYPE, pou,
						request);

                
                
		PolicyEvaluationServiceImpl.logger.exiting(this.getClass().getName(),
				"transformAttributes");
                
	}

	private boolean valueStartsWithInArray(final String value,
			final List<String> array) {
		boolean ret = false;
		final String toUpperValue = value.toUpperCase(Locale.ENGLISH);

		for (final String s : array) {
			ret = (s.toUpperCase(Locale.ENGLISH).equals("*") || toUpperValue.startsWith(s.toUpperCase(Locale.ENGLISH)));
			if (ret) {
				break;
			}
		}

		return ret;
	}

	@Required
	public void setPersonService(final PersonServiceInterface personService) {
		this.personService = personService;
	}
	
	@Required
	public void setPdqService(final PdqService pdqService) {
		this.pdqService = pdqService;
	}
	
	@Required
	public void setPdpProxy(final PDPInterface pdpProxy) {
		this.pdpProxy = pdpProxy;
	}
	
	@Required
	public void setLocalHomeCommunityId(final String localHomeCommunityId) {
		this.localHomeCommunityId = localHomeCommunityId;
	}
	
	@Required
	public void setDodHomeCommunityId(final String dodHomeCommunityId) {
		this.dodHomeCommunityId = dodHomeCommunityId;
	}
	
	@Required
	public void setPipInterface(final PIPInterface pipInterface) {
		this.pipInterface = pipInterface;
	}
	
	@Required
	public void setFacilityService(final FacilityService facilityService) {
		this.facilityService = facilityService;
	}
	
	@Required
	public void setNwhinConsentType(final String nwhinConsentType) {
		this.nwhinConsentType = nwhinConsentType;
	}
	
	@Required
	public void setSsaConsentType(final String ssaConsentType) {
		this.ssaConsentType = ssaConsentType;
	}
	@Required
	public void setNwhinRestrictionType(final String nwhinRestrictionType) {
		this.nwhinRestrictionType = nwhinRestrictionType;
	}
    
    @Required
    public void setAdapterDocQueryEndpoint(AdapterDocQueryEndpoint adapterDocQueryEndpoint)
    {
        this.adapterDocQueryEndpoint = adapterDocQueryEndpoint;
    }
    
    @Required
    public void setAdapterDocRetrieveEndpoint(AdapterDocRetrieveEndpoint adapterDocRetrieveEndpoint)
    {
        this.adapterDocRetrieveEndpoint = adapterDocRetrieveEndpoint;
    }

    @Required
    public void setSlsEndpoint(SLSClient slsEndpoint)
    {
        this.slsEndpoint = slsEndpoint;
    }

    @Required
    public void setPatientConfidentialityDAO(PatientConfidentialityDAO patientConfidentialityDAO)
    {
        this.patientConfidentialityDAO = patientConfidentialityDAO;
    }

    // Not Required, defaults to false. Will remove this setting altogether
    // once this feature is ready for release.
    public void setEnableSLSIntegration(boolean enableSLSIntegration)
    {
        this.enableSLSIntegration = enableSLSIntegration;
    }
}
