/*
 * To change this template, choose Tools | Templates
 * and open the template in the editor.
 */
package gov.va.nvap.server.announce;

import gov.va.nvap.server.announce.AnnouncerImpl;
import gov.va.nvap.server.announce.jpa.AnnouncementJpaController;
import gov.va.nvap.svc.consentmgmt.PIPInterface;
import gov.va.nvap.svc.consentmgmt.PolicyInformationPoint;
import gov.va.nvap.svc.consentmgmt.jpa.ConsentDirectiveJpaController;
import gov.va.nvap.svc.consentmgmt.jpa.OrganizationsJpaController;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.Announcement;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.AnnouncementBatchSummary;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.AnnouncerInterface;
import gov.va.nvap.svc.consentmgmt.stub.data.Organization;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import javax.persistence.NoResultException;
import javax.persistence.Persistence;
import javax.persistence.Query;

import junit.framework.Assert;

import org.junit.After;
import org.junit.AfterClass;
import org.junit.Before;
import org.junit.Test;

/**
 *
 * @author vhaislegberb
 */
public class BatchAnnouncerTest {

    private static EntityManagerFactory emf = Persistence.createEntityManagerFactory("nwhin_vap_pip");
    private EntityManager em = null;
    private AnnouncerInterface patientAnnouncer;
    private PIPInterface pip;
    private static String batchId;
    private static Organization testOrg;

    @Before
    public void setUp() throws Exception {

        this.em = emf.createEntityManager();
        pip = new PolicyInformationPoint();

        AnnouncementJpaController announcementController = new AnnouncementJpaController();
        announcementController.setEntityManager(em);

        announcementController.setPip(pip);

        OrganizationsJpaController organizationController = new OrganizationsJpaController();
        organizationController.setEntityManager(em);

        ConsentDirectiveJpaController consentDirectiveController = new ConsentDirectiveJpaController();
        consentDirectiveController.setEntityManager(em);

        patientAnnouncer = new AnnouncerImpl();
        ((AnnouncerImpl)patientAnnouncer).setPatientAnnounceController(announcementController);

        ((PolicyInformationPoint)pip).setOrganizationsController(organizationController);
        ((PolicyInformationPoint)pip).setConsentDirectiveController(consentDirectiveController);

        // add an organization to batch announce to
        testOrg = getOrganization("New Organization 1");
        if (null == testOrg)
        {

            Organization org = new Organization();
            org.setActive('Y');
            org.setOrgCommunityIdPrefix("NEW");
            org.setOrgContact("ORG CONTACT");
            org.setOrgDomain("NEW.ORG.DOMAIN.1");
            org.setOrgName("New Organization 1");
            org.setOrgNumber("200NEW");
            org.setOrgOid("2.16.840.1.113883.3.555");
            org.setOrgPhoneNumber("555-555-5555");

            this.em.getTransaction().begin();
            pip.storeOrganization(org);
            this.em.getTransaction().commit();
        }

    }

    @After
    public void tearDown()
    {
        if( this.em != null)
        {
            this.em.close();
            this.em = null;
        }
    }

    private Organization getOrganization(String orgName)
    {
        if (null == testOrg) {
            Query q = em.createNamedQuery("Organization.findByOrgName");
            q.setParameter("orgName", orgName);
            try
            {
                Long id = ((Organization) q.getSingleResult()).getOrgId();
                testOrg = pip.getOrganizationById(id);
            } catch (NoResultException e)
            {
                // the testOrg hasn't been created yet.
            }
        }
        return testOrg;
    }

    @Test
    public void testCreateAnnouncementsBatch() {
        // This has to be done to start off the test case. It is anticipated that
        // the GUIs that are triggering the capability could provide the Long
        // identifier of the organzation that is being targeted, so, this lookup
        // step could be replaced.
        Organization targetedOrg = getOrganization("New Organization 1");
        if (null == targetedOrg) {
            Assert.fail("Expected organization with ORG_ID: 10 does not exist in persitent store.");
        }

        em.getTransaction().begin();
        batchId = patientAnnouncer.createAnnouncementsBatch(targetedOrg, "testUser", null, null, false);
        Collection<Announcement> announcements = patientAnnouncer.getUnscheduledAnnouncementsByBatchId(batchId, 2, 0);
        for (Announcement announcement : announcements) {
            announcement.setScheduledTs(new Date());
            em.persist(announcement);
        }
        em.getTransaction().commit();

        Assert.assertNotNull(announcements);
        Assert.assertTrue(announcements.size() == 2);
    }

    @Test
    public void testGetBatchSummary()
    {
        List<String> batchIds = new ArrayList<String>();
        batchIds.add(batchId);
        List<AnnouncementBatchSummary> summaries = patientAnnouncer.getBatchSummary(batchIds);
        Assert.assertNotNull(summaries);
        Assert.assertTrue(summaries.size() == 1);
        Assert.assertTrue("New Organization 1".equals(summaries.get(0).getTargetOrganizationName()));
        Assert.assertTrue(3 == summaries.get(0).getBatchSize());
    }

    @Test
    public void testGetBatchSummaryByDates()
    {
        Organization targetedOrg = getOrganization("New Organization 1");
        Announcement announcement = new Announcement();
        announcement.setTargetOrganization(targetedOrg);
        announcement.setCreatedTs(new Date());
        announcement.setUserId("SYSTEM");
        announcement.setPatientConsentDir(pip.getActiveConsentDirective(Collections.singletonList("123457"), pip.getConsentTypeByName("NwHIN Authorization")));

        this.em.getTransaction().begin();
        patientAnnouncer.storeAnnouncement(announcement);
        this.em.getTransaction().commit();

        List<AnnouncementBatchSummary> summaries = patientAnnouncer.getBatchSummaryByDateRange(null, null);
        Assert.assertNotNull(summaries);
        Assert.assertTrue(summaries.size() == 1);
        Assert.assertTrue("New Organization 1".equals(summaries.get(0).getTargetOrganizationName()));
        Assert.assertTrue(3 == summaries.get(0).getBatchSize());

        Date createTs = summaries.get(0).getCreatedDate();
        // startDate == tomorrow; it shouldn't return any summaries.
        Date startDate = new Date(createTs.getTime() + 86400000L);
        summaries = patientAnnouncer.getBatchSummaryByDateRange(startDate, null);
        Assert.assertNotNull(summaries);
        Assert.assertTrue(summaries.isEmpty());

        // should find the same as when param == null
        summaries = patientAnnouncer.getBatchSummaryByDateRange(null, createTs);
        Assert.assertNotNull(summaries);
        Assert.assertTrue(summaries.size() == 1);
        Assert.assertTrue("New Organization 1".equals(summaries.get(0).getTargetOrganizationName()));
        Assert.assertTrue(3 == summaries.get(0).getBatchSize());

        // should find the same as when param == null
        // endDate is tomorrow
        Date endDate = startDate;
        // startDate is yesterday
        startDate = new Date(createTs.getTime() - 86400000L);
        summaries = patientAnnouncer.getBatchSummaryByDateRange(startDate, endDate);
        Assert.assertNotNull(summaries);
        Assert.assertTrue(summaries.size() == 1);
        Assert.assertTrue("New Organization 1".equals(summaries.get(0).getTargetOrganizationName()));
        Assert.assertTrue(3 == summaries.get(0).getBatchSize());
    }

    @Test
    public void testRemoveBatchByBatchId()
    {
        List<String> batchIds = new ArrayList<String>();
        batchIds.add(batchId);

        em.getTransaction().begin();
        patientAnnouncer.removeAnnouncementBatches(batchIds);
        em.getTransaction().commit();

        Collection<Announcement> announcements = patientAnnouncer.getUnscheduledAnnouncementsByBatchId(this.batchId, 1000, 0);
        Assert.assertTrue(announcements == null || announcements.isEmpty());
    }

    @AfterClass
    public static void afterClass() throws Exception
    {
        EntityManager em = null;

        try {
            em = emf.createEntityManager();
            em.getTransaction().begin();

            // delete all of the announcements that were added during the testing
            Query q = em.createQuery("from Announcement a");
            List<Announcement> announcements = q.getResultList();
            for(Announcement a : announcements) {
            	em.remove(a);
            }

            // delete the new organization that was added during testing.
            q = em.createNamedQuery("Organization.findByOrgName");
            q.setParameter("orgName", "New Organization 1");
            List<Organization> orgs = q.getResultList();

            for(Organization org : orgs)
            {
                em.remove(org);
            }

        } finally {
            if(em != null)
            {
                em.getTransaction().commit();
                em.close();
            }
        }
    }
}
