package gov.va.nvap.server.service.privacy.transformer;

import gov.va.nvap.common.date.hl7.HL7DateUtil;
import gov.va.nvap.common.jaxb.JaxbUtil;
import gov.va.nvap.common.transformer.Transformer;
import gov.va.nvap.common.transformer.TransformerException;
import gov.va.nvap.common.transformer.xml.StringToXML;
import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.data.ConsentDirectiveData;
import gov.va.nvap.service.audit.ConsentAuditType;
import gov.va.nvap.service.audit.data.ConsentAudit;
import gov.va.nvap.svc.facility.data.Facility;
import gov.va.nvap.svc.facility.intf.FacilityService;

import java.text.ParseException;
import java.util.Date;

import javax.xml.bind.JAXBException;

import org.springframework.beans.factory.annotation.Required;
import org.w3c.dom.Document;

/**
 * Convert from CDA R2 XML document to Audit. It converts the document to an
 * intermediate ConsentDirectiveData and then populates the values in the
 * ConsentAudit.
 *
 * @author Asha Amritraj
 *
 */
public class ConsentDirectiveDocumentToConsentAudit implements
		Transformer<String, ConsentAudit> {

	/**
	 * Convert to the intermediate format.
	 */
	private Transformer<Document, Document> consentDirectiveDocumentToData;
	/**
	 * Facility Svc to resolve from facility station to facility name.
	 */
	private FacilityService facilityService;
	/**
	 * The marshaller to convert from ConsentDirectiveData XML to Objects.
	 */
	private JaxbUtil dataJaxb2Marshaller;
	/**
	 * For storing the string in the detail section of the audit.
	 */
	private StringToXML stringToXml;

	@Required
	public void setConsentDirectiveDocumentToData(
			final Transformer<Document, Document> consentDirectiveDocumentToData) {
		this.consentDirectiveDocumentToData = consentDirectiveDocumentToData;
	}

	@Required
	public void setDataJaxb2Marshaller(final JaxbUtil dataJaxb2Marshaller) {
		this.dataJaxb2Marshaller = dataJaxb2Marshaller;
	}

	@Required
	public void setStringToXml(final StringToXML stringToXml) {
		this.stringToXml = stringToXml;
	}
	
	@Required
	public void setFacilityService(final FacilityService facilityService) {
		this.facilityService = facilityService;
	}

	@Override
	public ConsentAudit transform(final String str) throws TransformerException {
		Assert.assertNotEmpty(str, "ConsentDirectiveDocument cannot be null!");
		// Convert the CDA R2 XML to Document object
		final Document doc = this.stringToXml.transform(str);
		// Convert from the CDA R2 XML to a value object ConsentDirectiveData
		final Document consentDirectiveDataDoc = this.consentDirectiveDocumentToData
				.transform(doc);
		try {
			// Unmarshall to JAXB ConsentDirectiveData object for easier parsing
			// of data
			final ConsentDirectiveData consentDirectiveData = (ConsentDirectiveData) this.dataJaxb2Marshaller
					.unmarshal(consentDirectiveDataDoc);
			Assert.assertNotEmpty(consentDirectiveData,
					"ConsentDirectiveData cannot be null!");
			// Construct the audit object
			final ConsentAudit consentAudit = new ConsentAudit();
			if ("active".equals(consentDirectiveData.getComponentStatusCode())) {
				consentAudit.setActionType(ConsentAuditType.OPTIN_AUDIT_TYPE
						.value());
			} else if ("aborted".equals(consentDirectiveData
					.getComponentStatusCode())
					|| "expired".equals(consentDirectiveData
							.getComponentStatusCode())) {
				consentAudit.setActionType(ConsentAuditType.OPTOUT_AUDIT_TYPE
						.value());
				consentAudit.setOptoutReason(consentDirectiveData
						.getOptoutReason());
			}

			String pouValue = consentDirectiveData
					.getComponentPurposeOfUseDisplayName();
			consentAudit.setPouValue(pouValue);
			consentAudit.setCreatedDate(HL7DateUtil
					.dateFromString(consentDirectiveData
							.getDocumentationBeginTime()));
			consentAudit.setTimeOfEvent(new Date());
			consentAudit.setPatientId(consentDirectiveData.getIcn());
			consentAudit.setFacility(consentDirectiveData
					.getAuthorPersonOrgOid());
			Facility fac = null;
			try {
				fac = facilityService.getFacilityByStationNumber(consentAudit.getFacility());
			} catch (Exception e) {
				//swallow
			}
			if (fac != null && NullChecker.isNotEmpty(fac)) {
				consentAudit.setFacilityName(fac.getFacilityName());
			} else {
				consentAudit.setFacilityName("Unknown");
			}
			consentAudit.setDetail(str);
			// Store demographics
			consentAudit.setPatientGivenName(consentDirectiveData
					.getPatientRoleGivenName());
			consentAudit.setPatientMiddleName(consentDirectiveData.getPatientRoleMiddleName());
			consentAudit.setPatientLastName(consentDirectiveData
					.getPatientRoleFamilyName());
			consentAudit
					.setPatientSsn(consentDirectiveData.getPatientRoleSsn());
			consentAudit.setUserId(consentDirectiveData.getAuthorPersonOid());
			// consentAudit.setPatientFacility(consentDirectiveData.getPatientRoleProviderOrganizationNumber());
			return consentAudit;
		} catch (final JAXBException ex) {
			throw new TransformerException(
					"Cannot marshall ConsentDirectiveData");
		} catch (final ParseException ex) {
			throw new TransformerException("Cannot parse effectiveDate");
		}
	}

}
