package gov.va.nvap.server.service.privacy.pdf;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.pdfbox.exceptions.COSVisitorException;
import org.apache.pdfbox.pdmodel.PDDocument;
import org.apache.pdfbox.pdmodel.PDPage;
import org.apache.pdfbox.pdmodel.common.PDRectangle;
import org.apache.pdfbox.pdmodel.edit.PDPageContentStream;
import org.apache.pdfbox.pdmodel.font.PDFont;
import org.apache.pdfbox.pdmodel.font.PDType1Font;
import org.apache.pdfbox.pdmodel.graphics.xobject.PDJpeg;

/**
 * @author Sridhar Varma Alluri
 * @author Zack Peterson This class is a helper class to generate a pdf by replacing a template string with actual values
 */
public class PdfGenerator {

    private PDDocument document = null;
    private PDFont font = null;
    private PDFont bold_font = null;
    private int begin_x = 0;
    private PDRectangle rect = null;
    private int line = 0;
    private int lineSpacing = 15;
    private float begin_y = 0;

    private final int PAGE_WIDTH = 120;
    private final int INDENT_WIDTH = 65;
    private final int MAX_LINES = 48;

    private boolean firstPage = true;

    /**
     *
     * Default constructor initializing member variables
     */
    public PdfGenerator() {
        document = new PDDocument();
        font = null;
        bold_font = null;
        begin_x = 0;
        rect = null;
        line = 0;
        lineSpacing = 13;
        begin_y = 0;
    }

    /**
     * Creates a pdf document from template and patientsDetails. A page is created for each patient in the collection.
     *
     * @param template The template that is used to create the pdf document
     * @param patientsDetails a collection of patient details that are replaced in template
     * @param type is a string denoting type of letter being created
     * @return a stream that has the pdf document
     */
    public ByteArrayOutputStream create(String template, ArrayList<HashMap<String, String>> patientsDetails, String type) {
        ByteArrayOutputStream ostream = new ByteArrayOutputStream();

        try {
            for (HashMap<String, String> patientsDetail : patientsDetails) {
                // Replace the address placeholder before splitting the template since it's a multi-line field
                String templateForPage = template.replace("[patientAddress]", patientsDetail.get("[patientAddress]"));
                
                // Split up the template into individual lines
                String[] lines = templateForPage.split("\n");
                // Split lines at space nearest page width limit
                ArrayList<String> adjustedLines = new ArrayList<String>();
                for (String str : lines) {
                    while (str.length() > PAGE_WIDTH) {
                        int spaceDelim = str.lastIndexOf(32, PAGE_WIDTH);
                        adjustedLines.add(str.substring(0, spaceDelim));
                        str = str.substring(spaceDelim + 1);
                    }
                    adjustedLines.add(str);
                }
                
                PDPage page = addPage(adjustedLines, patientsDetail);
                document.addPage(page);
            }
            document.save(ostream);
            document.close();
        } catch (COSVisitorException e) {
            ostream = null;
        } catch (IOException e) {
            ostream = null;
        }
        return ostream;
    }

    /**
     * Creates a pdf document from multiple templates and patientsDetails collections. A page is created for each patient in the
     * collections.
     *
     * @param templateArray The templates that are used to create the pdf document
     * @param patientsDetailsList A List of patient details that are replaced in the templates
     * @return a stream that has the pdf document
     */
    public ByteArrayOutputStream create(String[] templateArray,
        ArrayList<ArrayList<HashMap<String, String>>> patientsDetailsList) {
        ByteArrayOutputStream ostream = new ByteArrayOutputStream();

        // there must be exactly one template for each patient detail collection
        if (templateArray.length != patientsDetailsList.size()) {
            return null;
        }

        for (int i = 0; i < templateArray.length; i++) {
            String template = templateArray[i];
            ArrayList<HashMap<String, String>> patientsDetails = patientsDetailsList.get(i);

            // Split up the template into individual lines
            String[] lines = template.split("\n");
            // Split lines at space nearest page width limit
            ArrayList<String> adjustedLines = new ArrayList<String>();
            for (String str : lines) {
                while (str.length() > PAGE_WIDTH) {
                    int spaceDelim = str.lastIndexOf(32, PAGE_WIDTH);
                    adjustedLines.add(str.substring(0, spaceDelim));
                    str = str.substring(spaceDelim + 1);
                }
                adjustedLines.add(str);
            }
            try {
                for (HashMap<String, String> hashMap : patientsDetails) {
                    PDPage page = addPage(adjustedLines, hashMap);
                    document.addPage(page);
                }
            } catch (IOException e) {
                ostream = null;
            }
        }

        try {
            document.save(ostream);
            document.close();
        } catch (COSVisitorException e) {
            ostream = null;
        } catch (IOException e) {
            ostream = null;
        }

        return ostream;
    }

    /**
     * A pdf page per patient is created from template. A page is created for each patient in the collection.
     *
     * @param lines An array of each individual line
     * @param patientDetail hashmap with patient information
     * @return a stream that has the pdf document
     */
    private PDPage addPage(ArrayList<String> lines, HashMap<String, String> patientDetail)
        throws IOException {
        PDPage page = new PDPage();
        font = PDType1Font.TIMES_ROMAN;
        bold_font = PDType1Font.TIMES_BOLD;

        line = 0;
        begin_x = (int) (INDENT_WIDTH * .75);
        rect = page.getMediaBox();
        begin_y = (float) (rect.getHeight() - (INDENT_WIDTH * 0.65));

        PDPageContentStream contentStream = new PDPageContentStream(document, page);
        InputStream imgIn = null;
        
        //Add the VA Seal image to the document.
        if (firstPage) {
            try {
                imgIn = getClass().getClassLoader().getResourceAsStream("Official_VA_Seal_hres.jpg");
                PDJpeg img = new PDJpeg(document, imgIn);
                contentStream.drawImage(img, begin_x, begin_y - img.getHeight());
            } catch (IOException ex) {
                Logger.getLogger(PdfGenerator.class.getName()).log(Level.SEVERE, null, ex);
            }
        }
        
        if(imgIn != null) {
            imgIn.close();
        }

        int i = 0;
        for (String str : lines) {
            i++;
            if (i > MAX_LINES) {
                break;
            }
            contentStream.beginText();
            contentStream.setFont(font, 11);
            if (line < 6) { //bold and center the facility name and address
                contentStream.setFont(bold_font, 11);
                float lineWidth = bold_font.getStringWidth(getUpdatedString(str, patientDetail)) / 1000 * 10;
                contentStream.moveTextPositionByAmount((page.getMediaBox().getWidth() - lineWidth) / 2, begin_y - lineSpacing * (++line));
            } else if (line >= 8 && line <= 11) {
                contentStream.setFont(bold_font, 11);
                contentStream.moveTextPositionByAmount(begin_x, begin_y - lineSpacing * (++line));
            } else if (str.contains("Expiration Date:")) {
                contentStream.setFont(bold_font, 11);
                contentStream.moveTextPositionByAmount(begin_x, begin_y - lineSpacing * (++line));
            } else {
                contentStream.moveTextPositionByAmount(begin_x, begin_y - lineSpacing * (++line));
            }
            contentStream.drawString(getUpdatedString(str, patientDetail));
            contentStream.endText();
        }
        contentStream.close();
        return page;
    }

    /**
     * This method replaces string that has the keys with patient information.
     *
     * @param templateString Template string that has keys
     * @param patientDetail a collection of patient details that are replaced in template
     * @return String that has keys replaced with values
     */
    private String getUpdatedString(String templateString,
        HashMap<String, String> patientDetail) {
        String updatedString = templateString;
        for (Map.Entry<String, String> entry : patientDetail.entrySet()) {
            String key = entry.getKey();
            String value = entry.getValue();
            updatedString = updatedString.replace(key, value);
        }
        updatedString = updatedString.replaceAll("\r", "\n");
        return updatedString;
    }
}
