package gov.va.nvap.server.service.privacy.interceptor;

import gov.va.nvap.common.interceptor.Interceptor;
import gov.va.nvap.common.interceptor.InterceptorException;
import gov.va.nvap.common.transformer.TransformerException;
import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.ConsentDirectiveAuthorizationRequest;
import gov.va.nvap.privacy.ConsentDirectiveRevocationRequest;
import gov.va.nvap.privacy.ConsentDirectiveUpdateRequest;
import gov.va.nvap.privacy.ServiceConsumer;
import gov.va.nvap.privacy.ServiceConsumerContextType;
import gov.va.nvap.server.service.privacy.transformer.ConsentDirectiveDocumentToConsentAudit;
import gov.va.nvap.service.audit.AuditException;
import gov.va.nvap.service.audit.AuditService;
import gov.va.nvap.service.audit.ConsentAuditType;
import gov.va.nvap.service.audit.data.ConsentAudit;

import org.springframework.beans.factory.annotation.Required;

/**
 * Convert from the CDA R2 XML to the ConsentAudit.
 * 
 * @author Asha Amritraj
 * 
 */
public class ConsentDirectiveDocumentAuditInterceptor implements
		Interceptor<Object, Object> {

	/**
	 * Reference to the audit service.
	 */
	private AuditService auditService;

	/**
	 * Convert from CDA R2 XML to ConsentAudit.
	 */
	private ConsentDirectiveDocumentToConsentAudit consentDirectiveDocumentToConsentAudit;

	private ServiceConsumerContextType getServiceConsumerContextType(
			final Object obj) {
		ServiceConsumerContextType scct = null;
		if (ConsentDirectiveAuthorizationRequest.class.isInstance(obj)) {
			final ConsentDirectiveAuthorizationRequest request = (ConsentDirectiveAuthorizationRequest) obj;
			scct = request.getServiceConsumerContext();
			if (scct == null) {
				scct = new ServiceConsumerContextType();
				request.setServiceConsumerContext(scct);
			}
		} else if (ConsentDirectiveRevocationRequest.class.isInstance(obj)) {
			final ConsentDirectiveRevocationRequest request = (ConsentDirectiveRevocationRequest) obj;
			scct = request.getServiceConsumerContext();
			if (scct == null) {
				scct = new ServiceConsumerContextType();
				request.setServiceConsumerContext(scct);
			}
		} else if (ConsentDirectiveUpdateRequest.class.isInstance(obj)) {
			final ConsentDirectiveUpdateRequest request = (ConsentDirectiveUpdateRequest) obj;
			scct = request.getServiceConsumerContext();
			if (scct == null) {
				scct = new ServiceConsumerContextType();
				request.setServiceConsumerContext(scct);
			}
		}

		return scct;
	}

	@Override
	public Object intercept(final Object obj) throws InterceptorException {

		if (NullChecker.isEmpty(obj)) {
			return obj;
		}
		// Get the consent directive CDA R2 XML string
		String consentDirectiveDocumentString = null;
		String optoutReason = null;
		ServiceConsumerContextType scct = null;
		// It could be authorization or revocation request
		if (ConsentDirectiveAuthorizationRequest.class.isInstance(obj)) {
			final ConsentDirectiveAuthorizationRequest request = (ConsentDirectiveAuthorizationRequest) obj;
			Assert.assertNotEmpty(request.getDocument(),
					"Document cannot be null!");
			scct = this.getServiceConsumerContextType(request);
			consentDirectiveDocumentString = new String(request.getDocument());

		} else if (ConsentDirectiveRevocationRequest.class.isInstance(obj)) {
			final ConsentDirectiveRevocationRequest request = (ConsentDirectiveRevocationRequest) obj;
			Assert.assertNotEmpty(request.getDocument(),
					"Document cannot be null!");
			scct = this.getServiceConsumerContextType(request);
			consentDirectiveDocumentString = new String(request.getDocument());
			if (NullChecker.isNotEmpty(request.getOptoutReason())) {
				optoutReason = request.getOptoutReason().value();
			}

		} else if (ConsentDirectiveUpdateRequest.class.isInstance(obj)) {
			final ConsentDirectiveUpdateRequest request = (ConsentDirectiveUpdateRequest) obj;
			scct = this.getServiceConsumerContextType(request);
			Assert.assertNotEmpty(request.getDocument(),
					"Document cannot be null!");
			consentDirectiveDocumentString = new String(request.getDocument());
		} else {
			throw new InterceptorException("Unknown type in interceptor!");
		}

		try {
			// Convert to ConsentAudit object
			final ConsentAudit consentAudit = this.consentDirectiveDocumentToConsentAudit
					.transform(consentDirectiveDocumentString);
			// Set the special optout reason, since its not part of the CDA R2
			// XML document
			// Get the information from the request
			consentAudit.setOptoutReason(optoutReason);
			if (NullChecker.isNotEmpty(scct)) {
				final ServiceConsumer sc = scct.getServiceConsumerType();
				if (NullChecker.isNotEmpty(sc)) {
					// override consent Audit as well.
					consentAudit.setUserId(scct.getUser());
					consentAudit.setFacility(scct.getFacility());
				}
				if (NullChecker.isNotEmpty(scct.getConsentType())) {
					consentAudit.setConsentType(scct.getConsentType().value());
                    if (scct.getConsentType().value().contains("Authorization")) {
                        consentAudit.setActionType(ConsentAuditType.OPTIN_AUDIT_TYPE.value());
                    } else if (scct.getConsentType().value().contains("Revocation")) {
                        consentAudit.setActionType(ConsentAuditType.OPTOUT_AUDIT_TYPE.value());
                    }
				}
			}
			this.auditService.auditConsent(consentAudit);
		} catch (final TransformerException ex) {
			throw new InterceptorException(ex);
		} catch (final AuditException ex) {
			throw new InterceptorException(ex);
		}
		// return the original document back
		return obj;
	}

	@Required
	public void setAuditService(final AuditService auditService) {
		this.auditService = auditService;
	}

	@Required
	public void setConsentDirectiveDocumentToConsentAudit(
			final ConsentDirectiveDocumentToConsentAudit consentDirectiveDocumentToConsentAudit) {
		this.consentDirectiveDocumentToConsentAudit = consentDirectiveDocumentToConsentAudit;
	}
}
