package gov.va.nvap.server.service.privacy;

import gov.va.med.nhin.adapter.audit.PageInfoType;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.ConsentDirectiveDetailedExpirationRequest;
import gov.va.nvap.privacy.ConsentType;
import gov.va.nvap.privacy.ExpiringConsentConfigurationReferenceType;
import gov.va.nvap.privacy.ExpiringConsentConfigurationRequest;
import gov.va.nvap.privacy.ExpiringConsentConfigurationResponse;
import gov.va.nvap.privacy.ExpiringConsentLetterResponse;
import gov.va.nvap.server.endpoint.psim.PersonServiceInterface;
import gov.va.nvap.server.service.privacy.emailer.Emailer;
import gov.va.nvap.server.service.privacy.pdf.PdfGenerator;
import gov.va.nvap.service.pdq.PatientDemographics;
import gov.va.nvap.service.pdq.PatientDemographicsQuery;
import gov.va.nvap.service.pdq.PatientDemographicsResponse;
import gov.va.nvap.service.pdq.PdqException;
import gov.va.nvap.service.pdq.PdqService;
import gov.va.nvap.service.privacy.ExpiringConsentService;
import gov.va.nvap.svc.consentmgmt.PIPInterface;
import gov.va.nvap.svc.consentmgmt.stub.data.ConsentDirective;
import gov.va.nvap.svc.consentmgmt.stub.data.DetailedConsentDirective;
import gov.va.nvap.svc.consentmgmt.stub.data.ExpiringConsentConfiguration;
import gov.va.nvap.svc.consentmgmt.stub.data.Facility;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Scanner;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.apache.commons.io.IOUtils;
import org.dozer.Mapper;
import org.quartz.CronTrigger;
import org.quartz.Scheduler;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.context.ApplicationContext;

/**
 *
 * @author Zack Peterson
 */
public class ExpiringConsentServiceImpl implements ExpiringConsentService {

    private static final Logger logger = Logger
        .getLogger(ExpiringConsentServiceImpl.class.getName());

    private PIPInterface pip;
    private Mapper mapper;
    private PersonServiceInterface personService;
    private PdqService pdqService;

    @Autowired
    private ApplicationContext appContext;

    @Override
    public ExpiringConsentConfigurationResponse
        getConfiguration(ExpiringConsentConfigurationRequest configRequest) {

        // Get the active consent configuration
        final ExpiringConsentConfiguration configuration = pip.getExpiringConsentConfiguration();

        // Convert to reference type
        final ExpiringConsentConfigurationReferenceType configReference = this.mapper
            .map(configuration, ExpiringConsentConfigurationReferenceType.class);

        // Make the response
        ExpiringConsentConfigurationResponse response = new ExpiringConsentConfigurationResponse();
        if (configReference != null) {
            response.setConfigurationReference(configReference);
        }
        return response;
    }

    @Override
    public void updateConfiguration(ExpiringConsentConfigurationRequest configRequest) {

        final ExpiringConsentConfiguration configuration
            = new ExpiringConsentConfiguration(configRequest.getFrequency(), configRequest.getWindow(),
                configRequest.getEmailAddresses());

        Scheduler scheduler = (Scheduler) appContext.getBean("scheduleFactory");

        // Periods assume first instance of time period
        final String HOURLY_PREFIX = "0 0 * ";
        final String MIDNIGHT_PREFIX = "0 0 0 ";
        final String DAILY = "* * ?";
        final String WEEKLY = "? * 1";
        final String MONTHLY = "1 * ?";
        final String YEARLY = "1 1 ? *";

        // Set job trigger to new time period
        try {
            CronTrigger trigger = (CronTrigger) scheduler.getTrigger("expiringConsentCronTrigger", "DEFAULT");

            String cronExpression;
            if (configRequest.getFrequency().equals("HOURLY")) {
                cronExpression = HOURLY_PREFIX + DAILY;
            } else if (configRequest.getFrequency().equals("DAILY")) {
                cronExpression = MIDNIGHT_PREFIX + DAILY;
            } else if (configRequest.getFrequency().equals("WEEKLY")) {
                cronExpression = MIDNIGHT_PREFIX + WEEKLY;
            } else if (configRequest.getFrequency().equals("MONTHLY")) {
                cronExpression = MIDNIGHT_PREFIX + MONTHLY;
            } else if (configRequest.getFrequency().equals("YEARLY")) {
                cronExpression = MIDNIGHT_PREFIX + YEARLY;
            } else {
                // Monthly
                cronExpression = MIDNIGHT_PREFIX + MONTHLY;
                // Every 20 seconds
                //cronExpression = "0/20 * * * * ?";
            }

            trigger.setCronExpression(cronExpression);
            scheduler.rescheduleJob("expiringConsentCronTrigger", "DEFAULT", trigger);
        } catch (Exception ex) {
            //TODO: Handle exception
        }

        this.pip.updateExpiringConsentConfiguration(configuration);
    }

    @Override
    public ExpiringConsentLetterResponse getReportLetters(ConsentDirectiveDetailedExpirationRequest configRequest) throws IOException {
        // Set page info to include all results

        // Get expiring active consent directives
        PageInfoType pageInfo = new PageInfoType();
        pageInfo.setPageNumber(0);
        pageInfo.setPageSize(-1);
        configRequest.setPageInfo(pageInfo);

        final SimpleDateFormat letterFormat = new SimpleDateFormat("MM/dd/yyyy");
        final Collection<DetailedConsentDirective> consentDirectives = this.pip.getExpiringPatientDetailedConsentDirectives(configRequest);

        final ArrayList<HashMap<String, String>> results = new ArrayList<HashMap<String, String>>();
        String template = configRequest.getTemplate(); //we need to mod this for the patient address so that's why we are grabbing it here
        Facility fac = this.pip.getFacilityByStationId(configRequest.getFacilityStation());
        for (final DetailedConsentDirective cd : consentDirectives) {
            if (NullChecker.isNotEmpty(cd)) {
                // Temporary fix to print out NwHIN Authorizations only
                if (NullChecker.isEmpty(cd.getOptinConsentType())) {
                    continue;
                }
                if (!cd.getOptinConsentType().getName().equals("NwHIN Authorization")) {
                    continue;
                }
                final HashMap<String, String> resultMap = new HashMap<String, String>();

                final String ien = cd.getPatientIen();
                // Assumes that there is only one icn for every ien
                final String icn = this.personService.getCorrelatedIds(ien).get(0);

                PatientDemographicsQuery demographicsQuery = new PatientDemographicsQuery();
                demographicsQuery.setPatientId(icn);

                PatientDemographicsResponse demographicsResponse;

                try {
                    demographicsResponse = this.pdqService.getPatientDemographics(demographicsQuery);
                    final PatientDemographics demographics = demographicsResponse.getPatientDemographics();
                    
                    if (demographics != null) {
                        try {
                            resultMap.put("[lastName]", demographics.getLastName());
                            resultMap.put("[firstName]", demographics.getFirstName());
                            resultMap.put("[middleName]", demographics.getMiddleName());

                            final String streetAddressLine1 = demographics.getStreetAddressLine1();
                            final String streetAddressLine2 = demographics.getStreetAddressLine2();
                            final String streetAddressLine3 = demographics.getStreetAddressLine3();
                            final String city = demographics.getResidenceCity();
                            final String state = demographics.getResidenceState();
                            final String zip = demographics.getResidenceZip4();
                            String patientAddress = streetAddressLine1 + "\n";
                            if (streetAddressLine2 != null && !streetAddressLine2.equals("")) {
                                patientAddress += " " + streetAddressLine2 + "\n";
                            }
                            if (streetAddressLine3 != null && !streetAddressLine3.equals("")) {
                                patientAddress += " " + streetAddressLine3 + "\n";
                            }
                            patientAddress += city + ", " + state + " " + zip;

                            //need to actually replace the string here so the line breaks work...
                            resultMap.put("[patientAddress]", patientAddress);

                            resultMap.put("[facilityName]", fac.getFacilityName());
                            String address1 = (fac.getAddress() != null) ? fac.getAddress() : "";
                            resultMap.put("[facilityAddress1]", address1);
                            String address2 = (fac.getCity() != null) ? fac.getCity() + ", " : "";
                            address2 += (fac.getState() != null) ? fac.getState() + " " : "";
                            address2 += (fac.getPostalCode() != null) ? fac.getPostalCode() : "";
                            resultMap.put("[facilityAddress2]", address2);
                            String phone = (fac.getPhone() != null) ? fac.getPhone() : "";
                            if (!NullChecker.isNullOrEmpty(phone)) {
                                phone = "(" + phone.substring(0, 3) + ") " + phone.substring(3, 6) + "-" + phone.substring(6);
                            }
                            resultMap.put("[facilityPhone]", phone);
                        } catch (Exception e) {
                            resultMap.clear();
                            ExpiringConsentServiceImpl.logger.log(Level.INFO, e.getMessage());
                        }
                    }
                    
                    if (demographics == null || resultMap.size() == 0) {
                        resultMap.put("[lastName]", "[lastName]");
                        resultMap.put("[firstName]", "[firstName]");
                        resultMap.put("[middleName]", "[middleName]");
                        resultMap.put("[patientAddress]", "[patientAddress]");

                        // This will be changed to an actual address when available
                        resultMap.put("[facilityName]", "VA Office");
                        resultMap.put("[facilityAddress1]", "");
                        resultMap.put("[facilityAddress2]", "");
                        resultMap.put("[facilityPhone]", "");
                    }

                    //add all the consent attributes to the letter
                    resultMap.put("[date]", new SimpleDateFormat("MMMM dd, yyyy").format(new Date()));
                    resultMap.put("[expirationDate]", letterFormat.format(cd.getExpirationDate()));

                    String signature = "The VLER Health National Program Office";
                    resultMap.put("[signature]", signature);
                    
                    results.add(resultMap);
                } catch (PdqException e) {
                    ExpiringConsentServiceImpl.logger.log(Level.INFO,
                        "Error getting patient demographics: %s", e.getMessage());
                } catch (Exception e) {
                    ExpiringConsentServiceImpl.logger.log(Level.INFO,
                        "Error processing patient: %s", e.getMessage());
                }
            }
        }

        InputStream inputStream = null;
        String templateString = "";
        try {
            inputStream = IOUtils.toInputStream(template, "UTF-8");
            templateString = new Scanner(inputStream, "UTF-8").useDelimiter("\\A").next();
        }
        finally {
            try {
                if (inputStream != null) {
                   inputStream.close();
                }
            }
            catch (Exception e) {

            }
        }

        PdfGenerator pdfGenerator = new PdfGenerator();
        ByteArrayOutputStream pdfStream = pdfGenerator.create(templateString, results, "expiring");

        byte[] byteData = pdfStream.toByteArray();

        ExpiringConsentLetterResponse response = new ExpiringConsentLetterResponse();
        response.setDownloadByteArray(byteData);

        return response;
    }

    @Override
    public void sendReport(ExpiringConsentConfigurationRequest configRequest) {

        // Get current day window
        final int window = Integer.valueOf(pip.getExpiringConsentConfiguration().getWindow().toString());

        ConsentDirectiveDetailedExpirationRequest cdExpReq = new ConsentDirectiveDetailedExpirationRequest();
        cdExpReq.setDayRange(window);
        PageInfoType pageInfo = new PageInfoType();
        pageInfo.setPageNumber(0);
        pageInfo.setPageSize(-1);
        cdExpReq.setPageInfo(pageInfo);

        // Get expiring active consent directives
        final Collection<ConsentDirective> consentDirectives = this.pip.getExpiringPatientConsentDirectives(null, cdExpReq);

        // Expiring Consent Counters
        int totalExpirations = 0;
        int eheExpirations = 0;
        int ssaExpirations = 0;
        int eagExpirations = 0;
        int dasExpirations = 0;
        int mhvExpirations = 0;
        int unknownExpirations = 0;

        final List<Map<String, Object>> results = new ArrayList<Map<String, Object>>();
        for (final ConsentDirective cd : consentDirectives) {
            if (NullChecker.isNotEmpty(cd)) {

                totalExpirations++;

                final Map<String, Object> resultMap = new HashMap<String, Object>();

                final String ien = cd.getPatientIen();
                // Assumes that there is only one icn for every ien
                final String icn = this.personService.getCorrelatedIds(ien).get(0);

                PatientDemographicsQuery demographicsQuery = new PatientDemographicsQuery();
                demographicsQuery.setPatientId(icn);

                PatientDemographicsResponse demographicsResponse = new PatientDemographicsResponse();
                try {
                    demographicsResponse = this.pdqService.getPatientDemographics(demographicsQuery);
                    final PatientDemographics demographics = demographicsResponse.getPatientDemographics();

                    String ssn = demographics.getSsn();
                    if (!ssn.equalsIgnoreCase("Unknown") && !ssn.isEmpty()) {
                        ssn = ssn.replaceAll("\\p{Z}", "");
                        ssn = ssn.substring(0, 3) + "-"
                            + ssn.substring(3, 5) + "-"
                            + ssn.substring(5, 9);
                    }
                    final String maskedPatientSSN = "***-**" + ssn.substring(6);
                    resultMap.put("ssn", maskedPatientSSN);
                    resultMap.put("lastName", demographics.getLastName());
                    resultMap.put("firstName", demographics.getFirstName());
                    resultMap.put("middleName", demographics.getMiddleName());
                } catch (PdqException e) {
                    // TODO: Handle Exception
                }

                String consentType = cd.getOptinConsentType().getName();

                // Count for consent type
                if (consentType.equals(ConsentType.NW_HIN_AUTHORIZATION.value())) {
                    eheExpirations++;
                } else if (consentType.equals(ConsentType.SSA_AUTHORIZATION.value())) {
                    ssaExpirations++;
                } else if (consentType.equals(ConsentType.EAG_AUTHORIZATION.value())) {
                    eagExpirations++;
                } else if (consentType.equals(ConsentType.DAS_AUTHORIZATION.value())) {
                    dasExpirations++;
                } else if (consentType.equals(ConsentType.MHV_AUTHORIZATION.value())) {
                    mhvExpirations++;
                } else {
                    unknownExpirations++;
                }

                resultMap.put("consentType", consentType);
                resultMap.put("optInDate", cd.getOptinDate());
                resultMap.put("expirationDate", cd.getExpirationDate());

                results.add(resultMap);
            }
        }

        String bodyString = "";
        bodyString = "Consent expiration for the next " + window + " days:\n\n";
        if (eheExpirations > 0) {
            bodyString += ConsentType.NW_HIN_AUTHORIZATION.value() + " expirations: " + eheExpirations + "\n";
        }
        if (ssaExpirations > 0) {
            bodyString += ConsentType.SSA_AUTHORIZATION.value() + " expirations: " + ssaExpirations + "\n";
        }
        if (eagExpirations > 0) {
            bodyString += ConsentType.EAG_AUTHORIZATION.value() + " expirations: " + eagExpirations + "\n";
        }
        if (dasExpirations > 0) {
            bodyString += ConsentType.DAS_AUTHORIZATION.value() + " expirations: " + dasExpirations + "\n";
        }
        if (mhvExpirations > 0) {
            bodyString += ConsentType.MHV_AUTHORIZATION.value() + " expirations: " + mhvExpirations + "\n";
        }
        if (unknownExpirations > 0) {
            bodyString += "Unknown Authorization" + ": " + unknownExpirations + "\n";
        }
        bodyString += "Total expirations: " + totalExpirations + "\n";

        bodyString += "\nTo see the current full expiring consent report, log on to VAP "
            + "and follow the following link:\n";

        Map<String, String> messageAttributes = new HashMap<String, String>();
        messageAttributes.put("subject", "VAP Expiring Consent Report");
        messageAttributes.put("body", bodyString);

        Emailer emailer = new Emailer();
        emailer.emailReport(pip.getExpiringConsentConfiguration().getEmailAddresses(),
            messageAttributes, results);
    }

    public PIPInterface getPip() {
        return this.pip;
    }

    @Required
    public void setPersonService(PersonServiceInterface personService) {
        this.personService = personService;
    }

    @Required
    public void setPdqService(PdqService pdqService) {
        this.pdqService = pdqService;
    }

    @Required
    public void setPip(PIPInterface pip) {
        this.pip = pip;
    }

    @Required
    public void setMapper(final Mapper mapper) {
        this.mapper = mapper;
    }

}
