package gov.va.nvap.server.service.privacy;

import gov.va.nvap.common.date.hl7.HL7DateUtil;
import gov.va.nvap.common.endpoint.Endpoint;
import gov.va.nvap.common.endpoint.EndpointException;
import gov.va.nvap.common.interceptor.Interceptor;
import gov.va.nvap.common.interceptor.InterceptorException;
import gov.va.nvap.common.jaxb.JaxbUtil;
import gov.va.nvap.common.transformer.Transformer;
import gov.va.nvap.common.transformer.TransformerException;
import gov.va.nvap.common.transformer.xml.StringToXML;
import gov.va.nvap.common.transformer.xml.XMLToString;
import gov.va.nvap.common.util.DocumentBuilderFactoryUtil;
import gov.va.nvap.common.uuid.UUIDUtil;
import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.*;
import gov.va.nvap.privacy.data.ConsentDirectiveData;
import gov.va.nvap.server.endpoint.psim.PersonServiceInterface;
import gov.va.nvap.service.audit.AuditService;
import gov.va.nvap.service.audit.data.ConsentAudit;
import gov.va.nvap.service.pdq.PatientDemographics;
import gov.va.nvap.service.pdq.PatientDemographicsQuery;
import gov.va.nvap.service.pdq.PatientDemographicsResponse;
import gov.va.nvap.service.pdq.PdqException;
import gov.va.nvap.service.pdq.PdqService;
import gov.va.nvap.service.privacy.ConsentManagementService;
import gov.va.nvap.service.privacy.ConsentManagementServiceException;
import gov.va.nvap.svc.consentmgmt.PIPInterface;
import gov.va.nvap.svc.consentmgmt.PolicyConstraints;
import gov.va.nvap.svc.consentmgmt.stub.FacilityResolver;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.Announcement;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.AnnouncementOrg;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.AnnouncerInterface;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.PatientAnnouncer;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.data.PatientAnnouncerException;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.AnnouncePatientRequest;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.AuthorizedOrganizationType;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.PatientType;
import gov.va.nvap.svc.consentmgmt.stub.adapter.announce.jaxb.UserType;
import gov.va.nvap.svc.consentmgmt.stub.dao.DelayedConsentDAO;
import gov.va.nvap.svc.consentmgmt.stub.data.ConsentDirective;
import gov.va.nvap.svc.consentmgmt.stub.data.DelayedConsent;
import gov.va.nvap.svc.consentmgmt.stub.data.DetailedConsentDirective;
import gov.va.nvap.svc.consentmgmt.stub.data.FacilityOptInConsent;
import gov.va.nvap.svc.consentmgmt.stub.data.OptoutReason;
import gov.va.nvap.svc.consentmgmt.stub.data.Organization;
import gov.va.nvap.svc.consentmgmt.stub.data.PatientDocument;
import gov.va.nvap.svc.consentmgmt.stub.data.PatientDocumentType;
import gov.va.nvap.svc.consentmgmt.stub.data.PurposeOfUse;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.text.ParseException;
import java.util.*;

import static java.util.Calendar.DAY_OF_MONTH;

import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.bind.JAXBContext;
import javax.xml.bind.JAXBException;
import javax.xml.bind.Unmarshaller;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.dom.DOMSource;
import org.apache.commons.lang.time.DateUtils;
import org.dozer.Mapper;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.oxm.jaxb.Jaxb2Marshaller;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * The implementation of the consent management service.
 *
 * @author Asha Amritraj
 * @author Zack Peterson
 *
 */
public class ConsentManagementServiceImpl implements ConsentManagementService,
        InitializingBean
{

    private static final Logger logger
            = Logger.getLogger(ConsentManagementServiceImpl.class.getName());

    private static final boolean SEQUENTIAL_ANNOUNCE = false;

    private Transformer<Announcement, AnnouncePatientRequest> announcementToAnnouncePatientRequest;

    /**
     * The patient announcer object.
     */
    private AnnouncerInterface announcer;
    /**
     * Marshall/Unmarshall the ConsentDirective types.
     */
    private Jaxb2Marshaller cmsJaxb2Marshaller;
    /**
     * Audit the CDA R2 XML Privacy Consent Directive forms.
     */
    private Interceptor<Object, Object> consentDirectiveDocumentAuditInterceptor;
    /**
     * Convert the consent directive CDA R2 Document to
     * ConsentDirectiveAuthorizationResponse.
     */
    private Transformer<Document, Document> consentDirectiveDocumentToAuthorizationResponse;
    /**
     * Convert to the intermediate format.
     */
    private Transformer<Document, Document> consentDirectiveDocumentToData;
    /**
     * Transformer to transform from the value object to CDA R2 XML document.
     */
    private Transformer<Document, Document> dataToConsentDirectiveDocument;
    /**
     * Convert the consent directive CDA R2 Document to
     * ConsentDirectiveRevocationResponse.
     */
    private Transformer<Document, Document> consentDirectiveDocumentToRevocationResponse;
    /**
     * The marshaller to convert from ConsentDirectiveData XML to Objects.
     */
    private JaxbUtil dataJaxb2Marshaller;
    /*
     * Facility Resolver
     */
    private FacilityResolver facilityResolver;

    /**
     * Dozer mapper to convert from web services types to PIP types.
     */
    private Mapper mapper;
    /**
     * Keep a copy of the OptoutReason type collections. So we dont have to go
     * to the PIP everytime.
     */
    private Collection<OptoutReason> optOutReasons;
    /**
     * Announce patients over the NwHIN. Called during the Authorization.
     */
    private PatientAnnouncer patientAnnouncer;
    // Store for performance
    /**
     * Keep a copy of the PatientDocumenType collections. So we dont have to go
     * to the PIP everytime.
     */
    private Collection<PatientDocumentType> patientDocumenTypes;
    /**
     * Person Service
     */
    private PersonServiceInterface personService;
    /**
     * Person Service
     */
    private PdqService pdqService;
    /**
     * The policy information point.
     */
    private PIPInterface pip;
    /**
     * Transforming from String CDA R2 XML to Document object.
     */
    private StringToXML stringToXML;
    /**
     * XML to String Transformer.
     */
    private XMLToString xmlToString;

    public void setXmlToString(XMLToString xmlToString)
    {
        this.xmlToString = xmlToString;
    }

    private DelayedConsentDAO delayedConsentDAO;

    /**
     * Reference to the audit service.
     */
    private AuditService auditService;

    @Override
    public void afterPropertiesSet() throws Exception
    {
        // Update information into the cache
        this.patientDocumenTypes = this.pip.getPatientDocumentTypes();
        this.optOutReasons = this.pip.getOptoutReasons();
    }

    /*
     * @deprecated as of VAP 2.1. Please do not use - AMS
     */
    private Collection<Organization> getAuthorizedOrganizations(
            final ConsentDirective consentDirective)
    {

        final List<Organization> optedInOrganizations = new ArrayList<Organization>();
        // select only to authorized organizations

        // Get Restriction Consent Type
        final gov.va.nvap.svc.consentmgmt.stub.data.ConsentType type = this.pip
                .getConsentTypeByName(ConsentType.NW_HIN_ORGANIZATION_RESTRICTION_AUTHORIZATION.value());
        // Get Patient IENs
        final List<String> patientIens = new ArrayList<String>();
        patientIens.add(consentDirective.getPatientIen());

        final ConsentDirective restriction = this.pip
                .getActiveConsentDirective(patientIens, type);
        // If no restrictions or no orgs, return 
        if (NullChecker.isEmpty(restriction)
                || NullChecker.isEmpty(restriction.getExcludedOrganizations())) {
            return this.pip
                    .getAllowedOrganizations();
        }

        for (final Organization allowedOrganization : this.pip
                .getAllowedOrganizations()) {
            boolean toExclude = false;
            if (NullChecker.isNotEmpty(restriction
                    .getExcludedOrganizations())) {
                for (final Organization excludedOrganization : restriction
                        .getExcludedOrganizations()) {
                    if (allowedOrganization.getOrgNumber().equals(
                            excludedOrganization.getOrgNumber())) {
                        toExclude = true;
                        break;
                    }
                }
                if (!toExclude) {
                    optedInOrganizations.add(allowedOrganization);
                }
            }
            else {
                optedInOrganizations.add(allowedOrganization);
            }
        }
        return optedInOrganizations;
    }

    @Override
    public ConsentDirectiveDocumentRetrieveResponse getConsentDirectiveDocuments(
            final ConsentDirectiveDocumentRetrieveRequest consentDirectiveDocumentRetrieveRequest)
            throws ConsentManagementServiceException
    {

        if (NullChecker.isEmpty(consentDirectiveDocumentRetrieveRequest)) {
            throw new ConsentManagementServiceException("Error: ConsentDirectiveDocumentRetrieveRequest cannot be null");
        }
        if (NullChecker.isEmpty(consentDirectiveDocumentRetrieveRequest.getConsentDirectiveReference())) {
            throw new ConsentManagementServiceException("Error: ConsentDirectiveDocumentReference cannot be null");
        }

        // Get the ConsentDirective from the ConsentDirectiveReference
        final ConsentDirectiveReferenceType reference = consentDirectiveDocumentRetrieveRequest.getConsentDirectiveReference();
        ConsentDirectiveDocumentType type = consentDirectiveDocumentRetrieveRequest.getDocumentType();

        if (NullChecker.isEmpty(reference.getConsentDirId())) {
            throw new ConsentManagementServiceException("Error: Consent Dir Id cannot be null. Please ensure it is non-null and namespaced");
        }
        // Temporary Solution - till the interfaces for CMS are cleaned up, there is no sense creating more dozer mappings
        // Unnecessary elements to query the consent directive documents will just be set to null
        reference.setPurposeOfUse(null);
        reference.setOptinConsentType(null);
        reference.setOptoutConsentType(null);
        reference.setOptoutReason(null);
        //done 
        final ConsentDirective consentDirective = this.mapper.map(reference, ConsentDirective.class);
        // Assume ALL if type is not specified
        if (NullChecker.isEmpty(type)) {
            type = ConsentDirectiveDocumentType.ALL;
        }
        // Lookup the PatientDocumentType
        final PatientDocumentType currentType = this.getPatientDocumentType(type.value());

        final List<byte[]> documents = new ArrayList<byte[]>();
        // Get all the documents if type is not present
        if (NullChecker.isEmpty(currentType)) {
            final Collection<PatientDocument> patientDocuments = this.pip.getConsentDirectiveDocuments(consentDirective);
            if (NullChecker.isNotEmpty(patientDocuments)) {
                for (final PatientDocument patientDocument : patientDocuments) {
                    if (NullChecker.isNotEmpty(patientDocument)) {
                        if (!NullChecker.isNullOrEmpty(patientDocument.getExternalDocumentId())) {
                            documents.add(patientDocument.getExternalDocumentId().getBytes());
                        }
                        else if (!NullChecker.isNullOrEmpty(patientDocument.getDocument())) {
                            documents.add(patientDocument.getDocument().toString().getBytes());
                        }
                    }
                }
            }
        }
        else {
            // Get only the type of document specified
            final PatientDocument patientDocument = this.pip.getConsentDirectiveDocumentByType(consentDirective, currentType);
            if (NullChecker.isNotEmpty(patientDocument)) {
                if (!NullChecker.isNullOrEmpty(patientDocument.getExternalDocumentId())) {
                    documents.add(patientDocument.getExternalDocumentId().getBytes());
                }
                else if (!NullChecker.isNullOrEmpty(patientDocument.getDocument())) {
                    documents.add(patientDocument.getDocument().toString().getBytes());
                }
            }
        }
        // Make the response and return
        final ConsentDirectiveDocumentRetrieveResponse response = new ConsentDirectiveDocumentRetrieveResponse();
        response.getDocuments().addAll(documents);
        return response;
    }

    @Override
    public ConsentDirectiveDetailedExpirationResponse getAllExpiringConsentDirectives(
            final ConsentDirectiveDetailedExpirationRequest consentDirectiveExpirationRequest)
            throws ConsentManagementServiceException
    {

        if (consentDirectiveExpirationRequest.getDayRange() < -1) {
            throw new ConsentManagementServiceException("Error: Day range cannot be negative");
        }

        // Get expiring active consent directives
        final Collection<ConsentDirective> consentDirectives = this.pip.getExpiringPatientConsentDirectives(null, consentDirectiveExpirationRequest);

        // Convert to the PIP consent directive
        final Collection<ConsentDirectiveReferenceType> consentDirectiveReferences = new ArrayList<ConsentDirectiveReferenceType>();
        final Collection<PatientDemographics> patientDemographicsReferences = new ArrayList<PatientDemographics>();
        final ConsentDirectiveDetailedExpirationResponse response = new ConsentDirectiveDetailedExpirationResponse();
        Long totalRows = new Long(0);

        for (final ConsentDirective cd : consentDirectives) {
            if (NullChecker.isNotEmpty(cd)) {
                //For pagination
                totalRows = cd.getTotalRows();

                final ConsentDirectiveReferenceType reference = this.mapper.map(cd, ConsentDirectiveReferenceType.class);
                consentDirectiveReferences.add(reference);

                final String ien = reference.getPatientIen();
                // Assumes that there is only one icn for every ien
                final String icn = this.personService.getCorrelatedIds(ien).get(0);

                PatientDemographicsQuery demographicsQuery = new PatientDemographicsQuery();
                demographicsQuery.setPatientId(icn);

                PatientDemographicsResponse demographicsResponse = new PatientDemographicsResponse();
                try {
                    demographicsResponse = this.pdqService.getPatientDemographics(demographicsQuery);
                    patientDemographicsReferences.add(demographicsResponse.getPatientDemographics());
                }
                catch (PdqException e) {
                    // TODO: Handle Exception
                }
                catch (RuntimeException e) { //Catches if icn does not exist
                    patientDemographicsReferences.add(new PatientDemographics());
                }
            }
        }

        if (NullChecker.isNotEmpty(consentDirectiveReferences)) {
            response.getConsentDirectiveReference().addAll(consentDirectiveReferences);
        }
        if (NullChecker.isNotEmpty(patientDemographicsReferences)) {
            response.getPatientDemographicsReference().addAll(patientDemographicsReferences);
        }

        response.setTotalRows(totalRows);

        return response;
    }

    @Override
    public ConsentDirectiveDetailedExpirationResponse getAllExpiringDetailedConsentDirectives(
            final ConsentDirectiveDetailedExpirationRequest consentDirectiveExpirationRequest)
            throws ConsentManagementServiceException
    {

        // Get expiring active consent directives
        final Collection<DetailedConsentDirective> consentDirectives = this.pip.getExpiringPatientDetailedConsentDirectives(consentDirectiveExpirationRequest);

        // Convert to the PIP consent directive
        final Collection<DetailedConsentDirectiveReferenceType> consentDirectiveReferences = new ArrayList<DetailedConsentDirectiveReferenceType>();
        final ConsentDirectiveDetailedExpirationResponse response = new ConsentDirectiveDetailedExpirationResponse();
        Long totalRows = (long) 0;

        for (final DetailedConsentDirective cd : consentDirectives) {
            if (NullChecker.isNotEmpty(cd)) {
                //For pagination
                totalRows = cd.getTotalRows();

                final DetailedConsentDirectiveReferenceType reference = this.mapper.map(cd, DetailedConsentDirectiveReferenceType.class);
                consentDirectiveReferences.add(reference);
            }
        }

        if (NullChecker.isNotEmpty(consentDirectiveReferences)) {
            response.getDetailedConsentDirectiveReference().addAll(consentDirectiveReferences);
        }

        response.setTotalRows(totalRows);

        return response;
    }

    @Override
    public ConsentDirectiveDetailedExpirationResponse getAuthorizedConsentForSummary(
            final ConsentDirectiveDetailedExpirationRequest consentDirectiveExpirationRequest)
            throws ConsentManagementServiceException
    {

        // Get expiring active consent directives
        final Collection<FacilityOptInConsent> consentDirectives = this.pip.getAuthorizedConsentForSummary();

        // Convert to the PIP consent directive
        final Collection<FacilityOptInConsentType> consentDirectiveReferences = new ArrayList<FacilityOptInConsentType>();
        final ConsentDirectiveDetailedExpirationResponse response = new ConsentDirectiveDetailedExpirationResponse();

        for (final FacilityOptInConsent cd : consentDirectives) {
            if (NullChecker.isNotEmpty(cd)) {
                final FacilityOptInConsentType reference = this.mapper.map(cd, FacilityOptInConsentType.class);
                consentDirectiveReferences.add(reference);
            }
        }

        if (NullChecker.isNotEmpty(consentDirectiveReferences)) {
            response.getFacilityOptInConsentReference().addAll(consentDirectiveReferences);
        }

        return response;
    }

    @Override
    public ConsentDirectiveQueryResponse getConsentDirectives(
            final ConsentDirectiveQueryRequest consentDirectiveQueryRequest)
            throws ConsentManagementServiceException
    {
        if (NullChecker.isEmpty(consentDirectiveQueryRequest)) {
            throw new ConsentManagementServiceException("Error: ConsentDirectiveQueryRequest cannot be null");
        }
        if (NullChecker.isEmpty(consentDirectiveQueryRequest.getPatientId())) {
            throw new ConsentManagementServiceException("Error: Patient ID cannot be null");
        }

        String consentType = "";
        if (NullChecker.isNotEmpty(consentDirectiveQueryRequest.getServiceConsumerContext())
                && NullChecker.isNotEmpty(consentDirectiveQueryRequest.getServiceConsumerContext().getConsentType())) {
            consentType = consentDirectiveQueryRequest.getServiceConsumerContext().getConsentType().value();
        }

        // Get the active consent directive
        final String patientId = consentDirectiveQueryRequest.getPatientId();

        // Skip correlation if SSA
        List<String> patientIens = new ArrayList<String>();
        if (consentType.equals(ConsentType.SSA_AUTHORIZATION.value())) {
            patientIens.add(patientId);
        }
        else {
            patientIens = this.personService.getCorrelatedIds(patientId);
        }

        final Collection<ConsentDirective> consentDirectives = new ArrayList<ConsentDirective>();
        if (NullChecker.isNotEmpty(consentDirectiveQueryRequest.getQueryParam())
                && consentDirectiveQueryRequest.getQueryParam().equals(ConsentDirectiveQueryParamType.ACTIVE)) {
            // TODO: Optimize
            if (NullChecker.isEmpty(consentType)) {
                for (final ConsentType type : ConsentType.values()) {
                    final ConsentDirective consentDirective = this.pip.getActiveConsentDirective(patientIens, this.pip.getConsentTypeByName(type.value()));
                    if (NullChecker.isNotEmpty(consentDirective)) {
                        consentDirectives.add(consentDirective);
                    }
                }
            }
            else {
                final ConsentDirective consentDirective = this.pip.getActiveConsentDirective(patientIens, this.pip.getConsentTypeByName(consentType));
                if (NullChecker.isNotEmpty(consentDirective)) {
                    consentDirectives.add(consentDirective);
                }
            }
            if (NullChecker.isEmpty(consentDirectives)) {
                return new ConsentDirectiveQueryResponse();
            }
        } // Get all consent directives
        else {
            Collection<ConsentDirective> allConsentDirectives = new ArrayList<ConsentDirective>();
            allConsentDirectives = this.pip.getConsentHistory(patientIens);
            // Filter only the consent directives requested by consent type
            if (NullChecker.isNotEmpty(consentType)) {
                for (final ConsentDirective directive : allConsentDirectives) {
                    if (NullChecker.isNotEmpty(directive)) {
                        if (directive.getOptinConsentType().getName().equals(consentType)) {
                            consentDirectives.add(directive);
                        }
                    }
                }
            }
            else {
                consentDirectives.addAll(allConsentDirectives);
            }
        }

        // Convert to the PIP consent directive
        final Collection<ConsentDirectiveReferenceType> consentDirectiveReferences = new ArrayList<ConsentDirectiveReferenceType>();
        for (final ConsentDirective cd : consentDirectives) {
            if (NullChecker.isNotEmpty(cd)) {
                final ConsentDirectiveReferenceType reference = this.mapper.map(cd, ConsentDirectiveReferenceType.class);
                consentDirectiveReferences.add(reference);
            }
        }
        // Make the response
        final ConsentDirectiveQueryResponse response = new ConsentDirectiveQueryResponse();
        if (NullChecker.isNotEmpty(consentDirectiveReferences)) {
            response.getConsentDirectiveReference().addAll(consentDirectiveReferences);
        }
        return response;
    }

    /**
     * Convenience method to get the PIP Organization from an oid.
     */
    private Organization getOrganizationByOid(final String oid)
    {
        for (final Organization org : this.pip.getAllowedOrganizations()) {
            if (org.getOrgOid().equals(oid)) {
                return org;
            }
        }
        return null;
    }

    @Override
    public OrganizationsQueryResponse getOrganizations(
            final OrganizationsQueryRequest organizationsQueryRequest)
            throws ConsentManagementServiceException
    {
        // Get the list of organizations from the pip.
        final Collection<Organization> organizations = this.pip.getAllowedOrganizations();
        if (NullChecker.isEmpty(organizations)) {
            throw new ConsentManagementServiceException("Error: ConsentDirectiveOrganizations cannot be null");
        }
        // sort organizations collection
        Collections.sort((List) organizations, new Comparator<Organization>()
        {
            @Override
            public int compare(final Organization o1, final Organization o2)
            {
                return o1.getOrgName().compareToIgnoreCase(o2.getOrgName());
            }
        });
        // Construct response
        final OrganizationsQueryResponse response = new OrganizationsQueryResponse();
        // Convert responses using Dozer to Webservice types and add to the
        // response
        for (final Organization organization : organizations) {
            final OrganizationType org = this.mapper.map(organization, OrganizationType.class);
            response.getOrganization().add(org);
        }
        // Return response
        return response;
    }
    
    @Override
    public OrganizationTrustedSourceResponse getIsOrganizationTrustedSource(final OrganizationTrustedSourceRequest organizationTrustedSourceRequest) throws ConsentManagementServiceException {
        OrganizationTrustedSourceResponse resp = new OrganizationTrustedSourceResponse();
        
        String orgId = organizationTrustedSourceRequest.getOrganizationId();
        
        // Exchange will be passing OIDs that start with community id prefix "urn:oid:", which we must strip.
        if (orgId.toLowerCase().startsWith("urn:oid:")) {
            orgId = orgId.substring(8);
        }
        
        Boolean isTrusted = this.pip.getIsOrganizationTrustedSource(orgId);
        
        resp.setIsTrusted(isTrusted);
        
        return resp;
    }
    
    /**
     * Convenience method to get the PatientDocumentType from a string.
     */
    private PatientDocumentType getPatientDocumentType(final String type)
    {
        for (final PatientDocumentType pType : this.patientDocumenTypes) {
            if (pType.getDocumentName().equals(type)) {
                return pType;
            }
        }
        return null;
    }

    private ConsentDirectiveData unmarshalConsentDirectiveData(String consentDirectiveString)
    {
        ConsentDirectiveData consentDirectiveData = null;

        try {
            JAXBContext jaxbContext = JAXBContext.newInstance(ConsentDirectiveData.class);
            Unmarshaller jaxbUnmarshaller = jaxbContext.createUnmarshaller();
            InputStream consentDirectiveStream = new ByteArrayInputStream(consentDirectiveString.getBytes("UTF-8"));
            DocumentBuilderFactory dbf = DocumentBuilderFactoryUtil.getDocumentBuilderFactory(null, true, false, false);
            DocumentBuilder db = dbf.newDocumentBuilder();
            Document d = db.parse(consentDirectiveStream);
            consentDirectiveData = (ConsentDirectiveData) jaxbUnmarshaller.unmarshal(d);
        }
        catch (JAXBException ex) {
            Logger.getLogger(ConsentManagementServiceImpl.class.getName()).log(Level.SEVERE, null, ex);
        }
        catch (UnsupportedEncodingException ex) {
            Logger.getLogger(ConsentManagementServiceImpl.class.getName()).log(Level.SEVERE, null, ex);
        }
        catch (ParserConfigurationException ex) {
            Logger.getLogger(ConsentManagementServiceImpl.class.getName()).log(Level.SEVERE, null, ex);
        }
        catch (SAXException ex) {
            Logger.getLogger(ConsentManagementServiceImpl.class.getName()).log(Level.SEVERE, null, ex);
        }
        catch (IOException ex) {
            Logger.getLogger(ConsentManagementServiceImpl.class.getName()).log(Level.SEVERE, null, ex);
        }

        return consentDirectiveData;
    }

    /**
     * Convert from the ConsentDirectiveData to the CDA R2 XML Privacy consent
     * directive document and then convert that to string.
     */
    private String makeConsentDirectiveDocumentString(
            final ConsentDirectiveData data)
    {
        try {
            // Convert the ConsentDirectiveData to XML document
            final Document consentDirectiveDataDoc = this.dataJaxb2Marshaller.marshal(data);
            // Convert ConsentDirectiveData XML to CDA R2 XML
            final Document consentDirectiveDocument = this.dataToConsentDirectiveDocument.transform(consentDirectiveDataDoc);
            // Convert CDA R2 XML to string
            final String consentDirectiveDocumentString = this.xmlToString.transform(consentDirectiveDocument);
            return consentDirectiveDocumentString;
        }
        catch (final TransformerException ex) {
            throw new RuntimeException(ex);
        }
        catch (final JAXBException ex) {
            throw new RuntimeException(ex);
        }
    }

    @Override
    public ConsentDirectiveAuthorizationResponse processConsentDirectiveAuthorization(
            final ConsentDirectiveAuthorizationRequest consentDirectiveAuthorizationRequest)
            throws ConsentManagementServiceException
    {
        if (NullChecker.isEmpty(consentDirectiveAuthorizationRequest)) {
            throw new ConsentManagementServiceException("Error: ConsentDirectiveAuthorizationRequest cannot be null");
        }
        if (NullChecker.isEmpty(consentDirectiveAuthorizationRequest.getDocument())) {
            throw new ConsentManagementServiceException("Error: ConsentDirectiveDocument cannot be null");
        }

        // Get the CDA R2 XML document
        String consentDirectiveString = new String(consentDirectiveAuthorizationRequest.getDocument());
        consentDirectiveString = consentDirectiveString.trim();

        try {
            ServiceConsumerContextType scct = consentDirectiveAuthorizationRequest.getServiceConsumerContext();
            ConsentDirectiveData consentDirectiveData;

            // Convert to XML
            Document consentDirectiveDoc = this.stringToXML.transform(consentDirectiveString);

            if (scct.getServiceConsumerType().value().equalsIgnoreCase("Exchange")
                    && scct.getConsentType().value().equalsIgnoreCase("SSA Authorization")) {
                consentDirectiveData = unmarshalConsentDirectiveData(consentDirectiveString);
                consentDirectiveString = this.makeConsentDirectiveDocumentString(consentDirectiveData);
                consentDirectiveAuthorizationRequest.setDocument(consentDirectiveString.getBytes());
                consentDirectiveDoc = this.stringToXML.transform(consentDirectiveString);
            }
            else {
                // Need to get the User information from the CDA R2
                final Document consentDirectiveDataDoc = this.consentDirectiveDocumentToData.transform(consentDirectiveDoc);
                consentDirectiveData = (ConsentDirectiveData) this.dataJaxb2Marshaller.unmarshal(consentDirectiveDataDoc);
            }

            if (NullChecker.isEmpty(consentDirectiveData)) {
                throw new ConsentManagementServiceException("Error: ConsentDirectiveData cannot be null");
            }
            if (NullChecker.isEmpty(consentDirectiveData.getIcn())) {
                throw new ConsentManagementServiceException("Error: ConsentDirectiveData ICN cannot be null");
            }

            // TODO: Temporary Fix for eBenefits not to populate
            // ServiceContextConsumerType
            // The above approach involves changing CDAR2, which we should not
            // be doing if submitted by "eBenefits". Remove comment
            scct = this.validateAndFixServiceConsumerContext(scct, consentDirectiveData.getIcn());
            consentDirectiveAuthorizationRequest.setServiceConsumerContext(scct);
            if (NullChecker.isEmpty(scct)) {
                throw new ConsentManagementServiceException("Error: ServiceConsumerContextType is required");
            }
            if (NullChecker.isEmpty(scct.getConsentType())) {
                throw new ConsentManagementServiceException("Error: ServiceConsumerContextType ConsentType is required");
            }
            ConsentDirective consentDirective;

            // Make the response
            final Document consentDirectiveAuthorizationResponseDoc = this.consentDirectiveDocumentToAuthorizationResponse.transform(consentDirectiveDoc);
            // Unmarshal the response from XML to objects
            final ConsentDirectiveAuthorizationResponse response = (ConsentDirectiveAuthorizationResponse) this.cmsJaxb2Marshaller.unmarshal(new DOMSource(consentDirectiveAuthorizationResponseDoc));
            // Convert reference from the request to the PIP consent directive
            // Mostly the opt-in date, created date, expiration date, patient id
            // are the only fields populated
            consentDirective = this.mapper.map(response.getConsentDirectiveReference(), ConsentDirective.class);

            // Add the excluded organizations
            final Collection<Organization> excludedOrganizations = consentDirective.getExcludedOrganizations();
            // Add exclusions
            if (NullChecker.isNotEmpty(excludedOrganizations)) {
                final Collection<Organization> resetExcludedOrg = new ArrayList<Organization>();
                for (final Organization org : excludedOrganizations) {
                    final Organization storedOrg = this.getOrganizationByOid(org.getOrgOid());
                    if (NullChecker.isNotEmpty(storedOrg)) {
                        resetExcludedOrg.add(storedOrg);
                    }
                }
                consentDirective.setExcludedOrganizations(resetExcludedOrg);
            }
            // Consent Type is required!
            final gov.va.nvap.svc.consentmgmt.stub.data.ConsentType type = this.pip.getConsentTypeByName(scct.getConsentType().value());
            final PurposeOfUse pou = this.pip.getPurposeOfUseByValue(consentDirectiveData.getComponentPurposeOfUseDisplayName());

            if (NullChecker.isEmpty(type)) {
                throw new ConsentManagementServiceException("Error: Not a valid consent type " + scct.getConsentType().value());
            }
            if (NullChecker.isEmpty(consentDirective.getOptinDate())) {
                throw new ConsentManagementServiceException("Error: OptinDate cannot be null");
            }
            if (NullChecker.isEmpty(consentDirective.getOptinTS())) {
                throw new ConsentManagementServiceException("Error: OptinTS cannot be null");
            }
            consentDirective.setOptinConsentType(type);
            consentDirective.setPurposeOfUse(pou);

            final List<String> patientIens = new ArrayList<String>();
            patientIens.add(consentDirective.getPatientIen());

            // Check if there's a delayed consent of the same type (ehx vs ssa) in PENDING status, we need to first cancel that consent by 
            // setting status to CANCELED and resolution date to now.
            DelayedConsent dc = this.delayedConsentDAO.findByTypeAndStatus(scct.getConsentType().getValue(), "PENDING", consentDirective.getPatientIen());

            if (!NullChecker.isNullOrEmpty(dc)) {
                // There is a delayed consent in pending status, so we need to set it to CANCELLED and the resolution date to now.
                try {
                    dc.setStatus("CANCELED");
                    dc.setResolutionDate(new Date());
                    this.delayedConsentDAO.update(dc);
                } catch (Exception ex) {
                    throw new ConsentManagementServiceException("There was an error cancelling the delayed consent.");
                }
            }

            final ConsentDirectiveReferenceType consentDirectiveReference = this.mapper.map(consentDirective, ConsentDirectiveReferenceType.class);
            final ConsentDirective existingDirective = this.pip.getActiveConsentDirective(patientIens, type);
            ConsentType authConsentType = scct.getConsentType();

            Date docStartDate = consentDirective.getOptinDate();
            if (existingDirective != null && existingDirective.getOptinDate() != null && docStartDate.before(existingDirective.getOptinDate()))  {
                throw new ConsentManagementServiceException(
                        "Error: An active consent directive " + existingDirective.getConsentDirId()
                        + " created on " + existingDirective.getOptinDate()
                        + " already exists for the patient for " + existingDirective.getOptinConsentType().getName());
            } else if(existingDirective != null && docStartDate.after(existingDirective.getOptinDate())) {    // Revoke old and commit new
                // Try Revoke
                try {
                    revokeOldConsent(scct, consentDirectiveReference, existingDirective);
                } catch (Exception ex) {
                    ex.printStackTrace();
                    throw new ConsentManagementServiceException("There was an error while revoking the old consent. ", ex);
                }
            }
            scct.setConsentType(authConsentType);
            consentDirective.setOptinTS(consentDirective.getOptinDate());

            Calendar c = Calendar.getInstance();
            c.setTime(consentDirective.getOptinDate());
            c.add(Calendar.YEAR, 10);                           // add 10 years to optin to get the expiration date
            consentDirective.setExpirationDate(c.getTime());

            this.pip.storeConsentDirective(consentDirective);

            // Get the consent directive Id
            final PatientDocument patientDocument = new PatientDocument();
            patientDocument.setConsentDirective(consentDirective);
            patientDocument.setDocument(consentDirectiveString);
            patientDocument.setPatientDocumentType(this.getPatientDocumentType("Authorize"));
            patientDocument.setExternalDocumentId(consentDirectiveAuthorizationRequest.getDasDocumentId());
            // Save Document
            this.pip.storePatientDocument(patientDocument);
            // Send Response
            final ConsentDirectiveAuthorizationResponse responseType = new ConsentDirectiveAuthorizationResponse();
            // Convert from PIP consent directive to webservice types using
            // dozer
            // TODO: ConsentDirective Id is not passed back in the response
//            final ConsentDirectiveReferenceType consentDirectiveReference = this.mapper.map(consentDirective, ConsentDirectiveReferenceType.class);
            responseType.setConsentDirectiveReference(consentDirectiveReference);
            if(consentDirective.getConsentDirId() != null)
                {
                responseType.getConsentDirectiveReference().setConsentDirId(consentDirective.getConsentDirId().toString());
                }
            // Announce only for NwHIN ConsentType
            if (ConsentType.NW_HIN_AUTHORIZATION.equals(scct.getConsentType())) {
                // Check if announcement is in exclusion timeframe
                int announceExclusionTimeframeHours = this.announcer.getAnnounceExclusionTimeframeHours();
                announceExclusionTimeframeHours = 0 - announceExclusionTimeframeHours; //Make negative
                Calendar cal = Calendar.getInstance();
                cal.add(Calendar.HOUR, announceExclusionTimeframeHours);
                Date timeframeBeginDate = cal.getTime();
                List<Announcement> timeframeAnnouncements
                        = this.announcer.getNonBatchAnnouncementsByCreatedTs(timeframeBeginDate, consentDirective.getPatientIen());
                if (timeframeAnnouncements == null || timeframeAnnouncements.isEmpty()) {
                    // Get the opted-in organizations
                    final Collection<Organization> optedInOrganizations = this.pip.getAllowedNonConsumerOnlyOrganizations();
                    //.getAuthorizedOrganizations(consentDirective);

                    if (NullChecker.isNotEmpty(optedInOrganizations)) {
                        try {
                            // For each authorized organizations
                            final Announcement announcement = new Announcement();
                            final List<AuthorizedOrganizationType> authorizedOrgs = new ArrayList<AuthorizedOrganizationType>();
                            if (!SEQUENTIAL_ANNOUNCE) {
                                // Store announcement
                                announcement.setCreatedTs(new Date());
                                announcement.setPatientConsentDir(consentDirective);
                                announcement.setScheduledTs(new Date());
                                announcement.setTargetOrganization(null);
                                announcement.setUserId(consentDirectiveData
                                        .getAuthorPersonOid());
                                this.announcer.storeAnnouncement(announcement);
                            }
                            for (final Organization organization : optedInOrganizations) {
                                if (SEQUENTIAL_ANNOUNCE) {
                                    // Store announcement
                                    announcement.setCreatedTs(new Date());
                                    announcement.setPatientConsentDir(consentDirective);
                                    announcement.setScheduledTs(new Date());
                                    announcement.setTargetOrganization(organization);
                                    announcement.setUserId(consentDirectiveData
                                            .getAuthorPersonOid());
                                    this.announcer.storeAnnouncement(announcement);
                                }
                                else {
                                    AnnouncementOrg announceOrg = new AnnouncementOrg();
                                    announceOrg.setAnnouncement(announcement);
                                    announceOrg.setTargetOrganization(organization);
                                    try {
                                        this.announcer.createAnnouncementOrg(announceOrg);
                                    }
                                    catch (Exception ex) {
                                        // AnnounceOrg storage fail
                                    }
                                }

                                // Convert orgs for request insertion
                                final AuthorizedOrganizationType authorizedOrganization = new AuthorizedOrganizationType();
                                authorizedOrganization.setOrgNumber(organization
                                        .getOrgNumber());
                                authorizedOrganization.setOrgName(organization
                                        .getOrgName());
                                authorizedOrganization.setOrgOid(organization
                                        .getOrgOid());
                                authorizedOrgs.add(authorizedOrganization);
                            }
                            // Convert into AnnouncePatientRequest
                            final AnnouncePatientRequest announcePatientRequest = this.announcementToAnnouncePatientRequest
                                    .transform(announcement);
                            // Strip the last announcement target organization so it is not counted twice
                            announcePatientRequest.getAuthorizedOrganizations().clear();
                            // Add all organizations
                            announcePatientRequest.getAuthorizedOrganizations().addAll(authorizedOrgs);
                            // Get the user information from the consent
                            // directive
                            // data object
                            final UserType userType = new UserType();
                            // Anand - 08/27/2011
                            // If Service Consumer is "Veterans_Portal" aka
                            // eBenefits, and transaction invoked is
                            // opt-in/opt-out,
                            // the
                            // business requirement is to
                            // 1) Set User Id to "eBenefits"
                            // 2) Set Facility to an ESR(Patient
                            // Preferred)/MVI(Last
                            // Visited), in that order
                            //
                            // Todo: Since Announce is implicit, how do we
                            // handle it
                            // ?
                            // For the moment, only overriding userid
                            final ServiceConsumer sc = scct.getServiceConsumerType();

                            if (NullChecker.isNotEmpty(sc)) {
                                userType.setUserId(scct.getUser());
                                userType.setFacilityNumber(scct.getFacility());
                            }
                            else {
                                userType.setUserId(consentDirectiveData.getAuthorPersonOid());
                                userType.setFacilityNumber(consentDirectiveData.getAuthorPersonFacilityNumber());
                            }
                            userType.setFullName(consentDirectiveData.getAuthorPersonFamilyName());
                            final PatientType patientType = new PatientType();
                            patientType.setGivenName(consentDirectiveData.getPatientRoleGivenName());
                            patientType.setLastName(consentDirectiveData.getPatientRoleFamilyName());
                            patientType.setMiddleName(consentDirectiveData.getPatientRoleMiddleName());
                            patientType.setSsn(consentDirectiveData.getPatientRoleSsn());
                            announcePatientRequest.setUser(userType);
                            // Announce patients synchronously over the NwHIN
                            // Create the request for async announce
                            // Call the announce bean
                            this.patientAnnouncer.announceAsync(announcePatientRequest);
                        }
                        catch (final PatientAnnouncerException ex) {
                            throw new ConsentManagementServiceException(ex);
                        }
                    }
                }
            }

            try {
                this.consentDirectiveDocumentAuditInterceptor.intercept(consentDirectiveAuthorizationRequest);
            }
            catch (final InterceptorException ex) {
                throw new ConsentManagementServiceException(ex);
            }

            // Return response
            return responseType;
        }
        catch (final JAXBException ex) {
            throw new ConsentManagementServiceException(ex);
        }
        catch (final TransformerException ex) {
            throw new ConsentManagementServiceException("Error: ConsentDirectiveDocument is not a valid XML document");
        }
    }
    private Endpoint<Object, String> vistaPatientPrimaryViewEndpoint;
    private Transformer<String, Map<String, Object>> patientPrimaryViewRpcResponseToPropertyMap;

    private void revokeOldConsent(ServiceConsumerContextType scct, ConsentDirectiveReferenceType reference, Object oldConsent)
            throws ConsentManagementServiceException, JAXBException, TransformerException {
        final String ien = reference.getPatientIen();
        // Assumes that there is only one icn for every ien
        final String icn = this.personService.getCorrelatedIds(ien).get(0);
        PatientDemographics demographics = null;
        PatientDemographicsQuery demographicsQuery = new PatientDemographicsQuery();
        demographicsQuery.setPatientId(icn);

        PatientDemographicsResponse demographicsResponse = new PatientDemographicsResponse();
        final Collection<PatientDemographics> patientDemographicsReferences = new ArrayList<PatientDemographics>();
        try {
            demographicsResponse = this.pdqService.getPatientDemographics(demographicsQuery);
            demographics = demographicsResponse.getPatientDemographics();
        }
        catch (PdqException e) {
            // TODO: Handle Exception
        }
        catch (RuntimeException e) { //Catches if icn does not exist
            patientDemographicsReferences.add(new PatientDemographics());
        }

        ConsentDirectiveData data = new ConsentDirectiveData();
        data.setIcn(icn);

        if (NullChecker.isNotEmpty(demographics))
            {
            data.setPatientRoleCity(demographics.getResidenceCity());
            data.setPatientRoleState(demographics.getResidenceState());
            data.setPatientRoleGivenName(demographics.getFirstName());
            data.setPatientRoleFamilyName(demographics.getLastName());
            data.setPatientRoleFamilyNameAlias(demographics.getAlias1());
            data.setPatientRoleMiddleName(demographics.getMiddleName());
            data.setPatientRoleMiddleNameAlias(demographics.getAlias2());
            data.setPatientRoleEthnicGroupCodeDisplayText(demographics
                    .getEthnicityDescription());
            data.setPatientRoleSsn(demographics.getSsn());
            data.setPatientRoleGenderCode(demographics.getGender());
            data.setPatientRoleGenderDisplayText(demographics
                    .getGenderDescription());
            data.setPatientRoleMaritalStatusCode(demographics
                    .getMaritalStatus());
            data.setPatientRoleMaritalStatusDisplayText(demographics
                    .getMaritalStatusDescription());
            data.setPatientRolePostalCode(demographics.getResidenceZip4());
            data.setPatientRolePrefix(demographics.getPrefix());
            data.setPatientRoleStreetAddressLine(demographics
                    .getStreetAddressLine1());
            data.setPatientRoleSuffix(demographics.getSuffix());
            data.setPatientRoleTelecom(demographics.getResidencePhoneNumber());
            data.setPatientRoleProviderOrganizationName(demographics
                    .getFacilityName());
            data.setPatientRoleProviderOrganizationNumber(demographics
                    .getFacilityNumber());
            if (NullChecker.isNotEmpty(demographics.getDob())) {
                try {
                    // Need to use the HL7 date, because the CDA R2 needs it
                    data.setPatientRoleDob(HL7DateUtil
                            .yyyyMMddhhmmssZ(demographics.getDob()));
                } catch (final ParseException ex) {
                    throw new RuntimeException(ex);
                }
            }
        }
        // Set a unique id per document
        data.setDocId(UUIDUtil.generateUUID());

        // Set the status code to aborted
        data.setComponentStatusCode("aborted");

        // Set document dates and times to now
        try {
            final String signatureDateString = HL7DateUtil
                    .yyyyMMddhhmmssZ(new Date());
            // Set the effective date
            data.setEffectiveDateTime(HL7DateUtil.yyyyMMddhhmmssZ(new Date()));
            // Create the begin and end date
            data.setDocumentationBeginTime(signatureDateString);
            // Start and end time are required
            data.setDocumentationEndTime(signatureDateString);
        } catch (final ParseException ex) {
            throw new RuntimeException(ex);
        }

        // Set the consent directive
        ConsentDirective cd = ((ConsentDirective) oldConsent);
        data.setPreviousConsentDirectiveId(cd.getConsentDirId().toString());
        data.setOptoutReason("New Authorization");
        // For now sets user to an automatic service
        data.setAuthorPersonOid("Automatic Service");
        data.setAuthorPersonOrgOid("Automatic Service");

        // Make the request to revoke
        final ServiceConsumerContextType sc = new ServiceConsumerContextType();
        sc.setUser("Automatic Service");

        ConsentAudit ca = auditService.getLatestAudit(cd.getPatientIen(), cd.getOptinConsentType().getName());
        if(ca != null){
            sc.setFacility(ca.getFacility());
        }
        sc.setServiceConsumerType(ServiceConsumer.ADMINISTRATOR_PORTAL);

        if (cd.getPurposeOfUse() != null) {
            data.setComponentPurposeOfUseDisplayName(cd.getPurposeOfUse().getPouValue());
        }
        ;
        String auth = gov.va.nvap.privacy.ConsentType.fromValue(cd.getOptinConsentType().getName()).name();
        String deAuthName = auth.substring(0, auth.lastIndexOf("_")) + "_REVOCATION";
        sc.setConsentType(ConsentType.valueOf(deAuthName));

        // Convert the PDF into byte string
        // Convert the ConsentDirectiveData to XML document
        final Document consentDirectiveDataDoc = this.dataJaxb2Marshaller.marshal(data);
            // Convert ConsentDirectiveData XML to CDA R2 XML
        final Document consentDirectiveDocument = this.dataToConsentDirectiveDocument.transform(consentDirectiveDataDoc);
        // Convert CDA R2 XML to string
        final String consentDirectiveDocumentString = this.xmlToString.transform(consentDirectiveDocument);
        final byte[] consentDirectiveDocumentBytes = consentDirectiveDocumentString.getBytes();

        final ConsentDirectiveRevocationRequest request = new ConsentDirectiveRevocationRequest();
        request.setServiceConsumerContext(sc);
        request.setOptoutReason(ConsentDirectiveOptOutReasonType.fromValue("New Authorization"));
        request.setDocument(consentDirectiveDocumentBytes);

        processConsentDirectiveRevocation(request);
        }

    /**
     * Processes the revocation of a consent for a particular consent directive.
     * This method also inserts a patient document into the database if the
     * revocation is not because of a deceased patient.
     *
     * @param consentDirectiveRevocationRequest Request object for consent
     * revocation
     *
     * @return the response object for the revocation
     *
     * @throws gov.va.nvap.service.privacy.ConsentManagementServiceException
     */
    @Override
    public ConsentDirectiveRevocationResponse processConsentDirectiveRevocation(
        final ConsentDirectiveRevocationRequest consentDirectiveRevocationRequest) throws ConsentManagementServiceException {
        return processOldConsentDirectiveRevocation(consentDirectiveRevocationRequest, null);
    }
    
    @Override
    public  ConsentDirectiveRevocationResponse processOldConsentDirectiveRevocation(
        final ConsentDirectiveRevocationRequest consentDirectiveRevocationRequest, Object oldConsent)
        throws ConsentManagementServiceException {

        if (NullChecker.isEmpty(consentDirectiveRevocationRequest)) {
            throw new ConsentManagementServiceException("Error: ConsentDirectiveRevocationRequest element cannot be null");
        }
        if (NullChecker.isEmpty(consentDirectiveRevocationRequest.getDocument())) {
            throw new ConsentManagementServiceException("Error: document element is null or invalid");
        }
        if (NullChecker.isEmpty(consentDirectiveRevocationRequest.getOptoutReason())) {
            throw new ConsentManagementServiceException("Error: optoutReason element is null or invalid");
        }

        ServiceConsumerContextType scct = consentDirectiveRevocationRequest.getServiceConsumerContext();

        // Get the CDA R2 XML String
        String consentDirectiveString = new String(consentDirectiveRevocationRequest.getDocument());
        consentDirectiveString = consentDirectiveString.trim();

        if (scct.getServiceConsumerType() == null) {
            throw new ConsentManagementServiceException("Error: serviceConsumerContext.serviceConsumerType element is null or invalid");
        }

        if (scct.getServiceConsumerType().value().equalsIgnoreCase("Exchange")
                && scct.getConsentType().value().equalsIgnoreCase("SSA Revocation")) {
            ConsentDirectiveData consentDirectiveData = unmarshalConsentDirectiveData(consentDirectiveString);
            consentDirectiveString = this.makeConsentDirectiveDocumentString(consentDirectiveData);
            consentDirectiveRevocationRequest.setDocument(consentDirectiveString.getBytes());
        }

        // Temp fix: To include opt-out reason in the content.
        try {
            // Convert to XML consent directive document
            final Document doc = this.stringToXML.transform(consentDirectiveString);
            // Make the response by transforming from the CDA R2 Document
            final Document consentDirectiveRevocationResponseDoc = this.consentDirectiveDocumentToRevocationResponse.transform(doc);
            // Convert the XML response to JAXB objects
            final ConsentDirectiveRevocationResponse response = (ConsentDirectiveRevocationResponse) this.cmsJaxb2Marshaller.unmarshal(new DOMSource(consentDirectiveRevocationResponseDoc));
            // Get the ConsentDirectiveReference
            final ConsentDirectiveReferenceType reference = response.getConsentDirectiveReference();

            // TODO: Temporary Fix for eBenefits not to populate
            // ServiceContextConsumerType
            // The above approach involves changing CDAR2, which we should not
            // be doing if submitted by "eBenefits". Remove comment
            scct = this.validateAndFixServiceConsumerContext(scct, reference.getPatientIen());
            consentDirectiveRevocationRequest.setServiceConsumerContext(scct);
            if (scct == null) {
                throw new ConsentManagementServiceException("Error: serviceConsumerContext element is null or invalid");
            }
            if (scct.getConsentType() == null) {
                throw new ConsentManagementServiceException("Error: serviceConsumerContext.consentType element is null or invalid");
            }

            // Call pip to get the active consent directive, and use that to
            // store the optOut information
            // Has to be atleast one active consent directive or throw error
            ConsentType activeConsentType = null;
            if (ConsentType.NW_HIN_REVOCATION.value().equals(scct.getConsentType().value())) {
                activeConsentType = ConsentType.NW_HIN_AUTHORIZATION;
            }
            else if (ConsentType.SSA_REVOCATION.value().equals(scct.getConsentType().value())) {
                activeConsentType = ConsentType.SSA_AUTHORIZATION;
            }
            else if (ConsentType.NW_HIN_ORGANIZATION_RESTRICTION_REVOCATION.value().equals(scct.getConsentType().value())) {
                activeConsentType = ConsentType.NW_HIN_ORGANIZATION_RESTRICTION_AUTHORIZATION;
            }

            if (activeConsentType == null) {
                throw new ConsentManagementServiceException("Error: serviceConsumerContext.consentType element is invalid");
            }

            final gov.va.nvap.svc.consentmgmt.stub.data.ConsentType dbConsentType = this.pip.getConsentTypeByName(scct.getConsentType().value());
            final gov.va.nvap.svc.consentmgmt.stub.data.ConsentType activeDbConsentType = this.pip.getConsentTypeByName(activeConsentType.value());

            //Skip correlation for SSA or if passed old consent for expired patients
            ConsentDirective activeConsentDirective;
            if (oldConsent != null && oldConsent instanceof ConsentDirective) {
                activeConsentDirective = (ConsentDirective)oldConsent;
            }
            else if (scct.getConsentType().value().equalsIgnoreCase("SSA Revocation")) {
                List ienList = new ArrayList();
                ienList.add(reference.getPatientIen());
                activeConsentDirective = this.pip.getActiveConsentDirective(ienList, activeDbConsentType);
            }
            else {
                activeConsentDirective = this.pip.getActiveConsentDirective(this.personService.getCorrelatedIds(reference.getPatientIen()), activeDbConsentType);
            }

            if (NullChecker.isEmpty(activeConsentDirective)) {
                throw new ConsentManagementServiceException("Error: The Patient does not have an active consent directive");
            }
            // Set the optout date

            activeConsentDirective.setOptoutTS(reference.getOptoutTS());
            activeConsentDirective.setOptoutDate(reference.getOptoutDate());
            activeConsentDirective.setOptoutConsentType(dbConsentType);

            if (scct.getServiceConsumerType().value().equalsIgnoreCase("Exchange")
                    && scct.getConsentType().value().equalsIgnoreCase("SSA Revocation")) {
                activeConsentDirective.setOptoutTS(new Date());
                activeConsentDirective.setOptoutDate(new Date());
                scct.setUser("SSA_USER");
            }

            // Set the optout reason
            for (final OptoutReason optOutReason : this.optOutReasons) {
                final String reason = consentDirectiveRevocationRequest.getOptoutReason().value();
                if (optOutReason.getText().equals(reason)) {
                    activeConsentDirective.setOptoutReason(optOutReason);
                    break;
                }
            }
            // Save Directive
            this.pip.storeConsentDirective(activeConsentDirective);
            // Store the CDA R2 XML as the patient document if patient is not deceased
            if (!activeConsentDirective.getOptoutReason().getText().equals("Patient Deceased")) {
                final PatientDocument patientDocument = new PatientDocument();
                patientDocument.setConsentDirective(activeConsentDirective);
                patientDocument.setDocument(consentDirectiveString);
                patientDocument.setPatientDocumentType(this.getPatientDocumentType("Revoke"));
                // Save Document
                this.pip.storePatientDocument(patientDocument);
            }
            // Send Response
            final ConsentDirectiveRevocationResponse responseType = new ConsentDirectiveRevocationResponse();
            // Convert from PIP consent directive to webserive types using dozer
            final ConsentDirectiveReferenceType consentDirectiveReference = this.mapper.map(activeConsentDirective, ConsentDirectiveReferenceType.class);
            responseType.setConsentDirectiveReference(consentDirectiveReference);
            // Return response

            // Intercept & Audit
            try {
                this.consentDirectiveDocumentAuditInterceptor.intercept(consentDirectiveRevocationRequest);
            }
            catch (final InterceptorException ex) {
                throw new ConsentManagementServiceException(ex);
            }

            return responseType;
        }
        catch (final TransformerException ex) {
            throw new ConsentManagementServiceException("Error: document element is not a valid XML document");
        }
    }

    @Override
    public ConsentDirectiveUpdateResponse processConsentDirectiveUpdate(
            final ConsentDirectiveUpdateRequest consentDirectiveUpdateRequest)
            throws ConsentManagementServiceException
    {

        if (NullChecker.isEmpty(consentDirectiveUpdateRequest)) {
            throw new ConsentManagementServiceException("Error: ConsentDirectiveUpdateRequest cannot be null");
        }
        if (NullChecker.isEmpty(consentDirectiveUpdateRequest.getDocument())) {
            throw new ConsentManagementServiceException("Error: Consent Directive Document cannot be null");
        }

        // Get the CDA R2 XML document
        String consentDirectiveString = new String(consentDirectiveUpdateRequest.getDocument());
        consentDirectiveString = consentDirectiveString.trim();
        try {
            // Convert to XML
            final Document consentDirectiveDoc = this.stringToXML.transform(consentDirectiveString);
            // Need to get the information from the CDA R2
            final Document consentDirectiveDataDoc = this.consentDirectiveDocumentToData.transform(consentDirectiveDoc);
            // Convert to helper VO
            final ConsentDirectiveData consentDirectiveData = (ConsentDirectiveData) this.dataJaxb2Marshaller.unmarshal(consentDirectiveDataDoc);

            final String icn = consentDirectiveData.getIcn();
            final String id = consentDirectiveData.getDocId();
            final String status = consentDirectiveData.getComponentStatusCode();
            // Verify the required fields
            if (NullChecker.isEmpty(icn)) {
                throw new ConsentManagementServiceException("Error: ICN cannot be null");
            }
            if (NullChecker.isEmpty(id)) {
                throw new ConsentManagementServiceException("Error: Document Id cannot be null");
            }
            if (NullChecker.isEmpty(status)) {
                throw new ConsentManagementServiceException("Error: Status code cannot be null");
            }

            // TODO: Temporary Fix for eBenefits not to populate
            // ServiceContextConsumerType
            // The above approach involves changing CDAR2, which we should not
            // be doing if submitted by "eBenefits". Remove comment
            ServiceConsumerContextType scct = consentDirectiveUpdateRequest.getServiceConsumerContext();
            scct = this.validateAndFixServiceConsumerContext(scct, icn);
            consentDirectiveUpdateRequest.setServiceConsumerContext(scct);
            if (NullChecker.isEmpty(scct)) {
                throw new ConsentManagementServiceException("Error: ServiceConsumerContextTye is required");
            }
            if (NullChecker.isEmpty(scct.getConsentType())) {
                throw new ConsentManagementServiceException("Error: ServiceConsumerType ConsentType is required");
            }

            // Get the type for query
            PatientDocumentType type = null;
            if ("aborted".equals(status)) {
                type = this.getPatientDocumentType("OPT-OUT");
            }
            else if ("active".equals(status)) {
                type = this.getPatientDocumentType("OPT-IN");
            }
            else {
                throw new ConsentManagementServiceException("Error: Consent directive status code is not supported");
            }
            final List<String> ids = new ArrayList<String>();
            ids.add(icn);
            final Collection<ConsentDirective> consentDirectives = this.pip.getConsentHistory(ids);
            PatientDocument patientDocument = null;
            // Get the document to update
            for (final ConsentDirective consentDir : consentDirectives) {
                patientDocument = this.pip.getConsentDirectiveDocumentByType(consentDir, type);
                if (NullChecker.isNotEmpty(patientDocument)) {
                    final String document = (String) patientDocument.getDocument();
                    if (document.contains(consentDirectiveData.getDocId())) {
                        break;
                    }
                }

            }
            // Set the new content
            if (patientDocument != null) {
                patientDocument.setDocument(consentDirectiveString);
            }

            // Update
            this.pip.updatePatientDocument(patientDocument);
            // Return the consent directive associated with the update
            final ConsentDirectiveUpdateResponse response = new ConsentDirectiveUpdateResponse();

            if (patientDocument != null) {
                final ConsentDirectiveReferenceType consentDirectiveReference = this.mapper.map(patientDocument.getConsentDirective(), ConsentDirectiveReferenceType.class);
                response.setConsentDirectiveReference(consentDirectiveReference);
            }

            // Intercept and audit the CDA R2 XML Request
            try {
                this.consentDirectiveDocumentAuditInterceptor.intercept(consentDirectiveUpdateRequest);
            }
            catch (final InterceptorException ex) {
                throw new ConsentManagementServiceException(ex);
            }

            return response;
        }
        catch (final TransformerException ex) {
            throw new ConsentManagementServiceException(ex);
        }
        catch (final JAXBException ex) {
            throw new ConsentManagementServiceException(ex);
        }
    }

    @Override
    public RestrictionsQueryResponse getRestrictions(final RestrictionsQueryRequest rqr)
            throws ConsentManagementServiceException
    {

        PolicyConstraints pc = null;
        try {
            ArrayList<String> patients = new ArrayList<String>();
            patients.add(rqr.getPatientId());
            //get the constarnts for the patient
            pc = this.pip.getPolicyConstraints(patients, this.pip.getConsentTypeByName(ConsentType.NW_HIN_ORGANIZATION_RESTRICTION_AUTHORIZATION.value()));
        }
        catch (final Exception e) {
            if (ConsentManagementServiceImpl.logger.isLoggable(Level.FINE)) {
                e.printStackTrace();
            }
            ConsentManagementServiceImpl.logger.logp(Level.WARNING, this.getClass().getName(), "getRestrictions", e.getMessage());
        }
        RestrictionsQueryResponse response = new RestrictionsQueryResponse();
        if (pc != null) {
            Collection<Organization> organizations = pc.getPatientExcludedOrganizations();
            for (final Organization organization : organizations) {
                final OrganizationType org = this.mapper.map(organization, OrganizationType.class);
                response.getOrganization().add(org);
            }
        }

        return response;
    }

    @Override
    public StatusQueryResponse getStatus(final StatusQueryRequest sqr)
            throws ConsentManagementServiceException
    {

        ArrayList<String> expiringTypes = new ArrayList<String>();//consents that do expire
        ArrayList<String> nonexpiringTypes = new ArrayList<String>();//things that don't expire
        expiringTypes.add(ConsentType.NW_HIN_AUTHORIZATION.value());
        expiringTypes.add(ConsentType.SSA_AUTHORIZATION.value());
        nonexpiringTypes.add(ConsentType.NW_HIN_ORGANIZATION_RESTRICTION_AUTHORIZATION.value());
        Collection<ConsentDirective> allConsentDirectives = null;
        
        //get the last millisecond from yesterday because we want to get all the active consents including today at midnight (12:00 AM)
        Date startOfToday = DateUtils.truncate(new Date(), Calendar.DAY_OF_MONTH);
        Date lastMillisecondOfYesterday = DateUtils.addMilliseconds(startOfToday, -1);
        
        try {
            ArrayList<String> patients = new ArrayList<String>();
            patients.add(sqr.getPatientId());
            allConsentDirectives = this.pip.getConsentHistory(patients);
        }
        catch (final Exception e) {
            if (ConsentManagementServiceImpl.logger.isLoggable(Level.FINE)) {
                e.printStackTrace();
            }
            ConsentManagementServiceImpl.logger.logp(Level.WARNING, this.getClass().getName(), "getStatus", e.getMessage());
        }
        StatusQueryResponse response = new StatusQueryResponse();
        if (allConsentDirectives != null) {
            for (final ConsentDirective directive : allConsentDirectives) {
                if (NullChecker.isNotEmpty(directive) && NullChecker.isEmpty(directive.getOptoutTS()) && NullChecker.isEmpty(directive.getOptoutDate())) {
                    if (nonexpiringTypes.contains(directive.getOptinConsentType().getName())) {
                        response.getAuthorization().add(directive.getOptinConsentType().getName());
                    }
                    else if (expiringTypes.contains(directive.getOptinConsentType().getName())
                            && directive.getExpirationDate().after(lastMillisecondOfYesterday)) {
                        response.getAuthorization().add(directive.getOptinConsentType().getName());
                    }
                }
            }
        }
        for (int c = 0; c < response.getAuthorization().size(); c++) {
            response.getAuthorization().set(c, response.getAuthorization().get(c).replace("NwHIN", "eHealth"));
        }
        return response;
    }

    @Required
    public void setDataToConsentDirectiveDocument(Transformer<Document, Document> dataToConsentDirectiveDocument)
    {
        this.dataToConsentDirectiveDocument = dataToConsentDirectiveDocument;
    }

    @Required
    public void setAnnouncementToAnnouncePatientRequest(
            final Transformer<Announcement, AnnouncePatientRequest> announcementToAnnouncePatientRequest)
    {
        this.announcementToAnnouncePatientRequest = announcementToAnnouncePatientRequest;
    }

    @Required
    public void setAnnouncer(final AnnouncerInterface announcer)
    {
        this.announcer = announcer;
    }

    @Required
    public void setCmsJaxb2Marshaller(final Jaxb2Marshaller jaxb2Marshaller)
    {
        this.cmsJaxb2Marshaller = jaxb2Marshaller;
    }

    @Required
    public void setConsentDirectiveDocumentAuditInterceptor(
            final Interceptor<Object, Object> consentDirectiveDocumentAuditInterceptor)
    {
        this.consentDirectiveDocumentAuditInterceptor = consentDirectiveDocumentAuditInterceptor;
    }

    @Required
    public void setConsentDirectiveDocumentToAuthorizationResponse(
            final Transformer<Document, Document> consentDirectiveDocumentToAuthorizationResponse)
    {
        this.consentDirectiveDocumentToAuthorizationResponse = consentDirectiveDocumentToAuthorizationResponse;
    }

    @Required
    public void setConsentDirectiveDocumentToData(
            final Transformer<Document, Document> consentDirectiveDocumentToData)
    {
        this.consentDirectiveDocumentToData = consentDirectiveDocumentToData;
    }

    @Required
    public void setConsentDirectiveDocumentToRevocationResponse(
            final Transformer<Document, Document> consentDirectiveDocumentToRevocationResponse)
    {
        this.consentDirectiveDocumentToRevocationResponse = consentDirectiveDocumentToRevocationResponse;
    }

    @Required
    public void setDataJaxb2Marshaller(final JaxbUtil dataJaxb2Marshaller)
    {
        this.dataJaxb2Marshaller = dataJaxb2Marshaller;
    }

    @Required
    public void setFacilityResolver(final FacilityResolver facilityResolver)
    {
        this.facilityResolver = facilityResolver;
    }

    @Required
    public void setMapper(final Mapper mapper)
    {
        this.mapper = mapper;
    }

    @Required
    public void setPatientAnnouncer(final PatientAnnouncer patientAnnouncer)
    {
        this.patientAnnouncer = patientAnnouncer;
    }

    @Required
    public void setPersonService(final PersonServiceInterface personService)
    {
        this.personService = personService;
    }

    @Required
    public void setPdqService(final PdqService pdqService)
    {
        this.pdqService = pdqService;
    }

    @Required
    public void setPip(final PIPInterface pip)
    {
        this.pip = pip;
    }

    @Required
    public void setStringToXML(final StringToXML stringToXML)
    {
        this.stringToXML = stringToXML;
    }
    
    @Required
    public void setDelayedConsentDAO(final DelayedConsentDAO delayedConsentDAO) {
        this.delayedConsentDAO = delayedConsentDAO;
    }

    @Required
    public void setAuditService(final AuditService auditService) {
        this.auditService = auditService;
    }

    // TODO: Fix - eBenefits can send other consent types
    private ServiceConsumerContextType validateAndFixServiceConsumerContext(
            ServiceConsumerContextType scct, final String icn)
    {
        // if (NullChecker.isEmpty(scct)) {
        // scct = new ServiceConsumerContextType();
        // scct.setConsentType(ConsentType.NW_HIN_AUTHORIZATION);
        // scct.setServiceConsumerType(ServiceConsumer.VETERANS_PORTAL);
        // }
        // If Facility is Empty it must be from eBenefits
        if (NullChecker.isEmpty(scct.getServiceConsumerType())) {
            scct.setServiceConsumerType(ServiceConsumer.VETERANS_PORTAL);
        }
        if (scct.getServiceConsumerType().value().equals(ServiceConsumer.VETERANS_PORTAL.value())) {
            scct.setUser("eBenefits");
            // Get PatientPreferred Facility
            String ppf = "";
            try {
                ppf = this.facilityResolver.getPatientPreferredFacility(icn);
            }
            catch (final Exception e) {
                e.printStackTrace();
            }
            scct.setFacility(ppf);
        }
        else if (scct.getServiceConsumerType().value().equals(ServiceConsumer.KIOSK.value())) {
            scct.setUser(ServiceConsumer.KIOSK.value());
            // Get PatientPreferred Facility
            String ppf = "";
            try {
                ppf = this.facilityResolver.getPatientPreferredFacility(icn);
            }
            catch (final Exception e) {
                e.printStackTrace();
            }
            scct.setFacility(ppf);
        }
        // if (NullChecker.isEmpty(scct.getConsentType())) {
        // scct.setConsentType(ConsentType.NW_HIN_AUTHORIZATION);
        // }
        return scct;
    }
}
