package gov.va.nvap.server.service.pdq.transformer;

import gov.va.nvap.common.splitter.Splitter;
import gov.va.nvap.common.splitter.SplitterException;
import gov.va.nvap.common.transformer.Transformer;
import gov.va.nvap.common.transformer.TransformerException;
import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.service.pdq.Facility;
import gov.va.nvap.service.pdq.PatientCorrelationsResponse;

import java.util.HashMap;
import java.util.Map;

import org.springframework.beans.factory.annotation.Required;

/**
 * Convert the MPI responses to the PatientCorrelationsResponse Jaxb object.
 * 
 * @author Asha Amritraj
 * 
 */
public class RpcResponsesToPatientCorrelationsResponse implements
		Transformer<String, PatientCorrelationsResponse> {
	/**
	 * Split the one single RPC into multiple responses.
	 */
	private Splitter<String, Map<String, String>> rpcResponsesSplitter;
	/**
	 * Split the correlations into a map record with responses and properties.
	 */
	private Transformer<Map<String, String>, Map<String, Map<String, String>>> rpcResponsesToSiteMapRecord;

	@Required
	public void setRpcResponsesSplitter(
			final Splitter<String, Map<String, String>> rpcResponsesSplitter) {
		this.rpcResponsesSplitter = rpcResponsesSplitter;
	}

	@Required
	public void setRpcResponsesToSiteMapRecord(
			final Transformer<Map<String, String>, Map<String, Map<String, String>>> rpcResponsesToSiteMapRecord) {
		this.rpcResponsesToSiteMapRecord = rpcResponsesToSiteMapRecord;
	}

	@Override
	public PatientCorrelationsResponse transform(final String rpcResponse)
			throws TransformerException {

		try {
			Assert.assertNotEmpty(rpcResponse, "RPC Response cannot be empty!");
			
			// Split and filter (by latest correlation date) responses
			final Map<String, String> splitAndFilteredResponses = this.rpcResponsesSplitter
					.splitAndFilter(rpcResponse);
			
			// Make a record of split rpc responses and properties
			final Map<String, Map<String, String>> mapRecord = this.rpcResponsesToSiteMapRecord
					.transform(splitAndFilteredResponses);
			
			Assert.assertNotEmpty(mapRecord, "Map record cannot be empty!");

			final PatientCorrelationsResponse response = new PatientCorrelationsResponse();
			// Make the properties
			for (final Map<String, String> values : mapRecord.values()) {
				final String facilityId = values.get("facilityId");
				final String correlatedPatientId = values.get("dfn");
				final String facilityName = values.get("facilityName");
				final String facilityNumber = values.get("facilityNumber");
				final String assigningAuthorityName = values
						.get("assigningAuthorityName");
				final String assigningAuthorityNumber = values
						.get("assigningAuthorityNumber");
				final Facility facility = new Facility();
				facility.setCorrelatedPatientId(correlatedPatientId);
				facility.setFacilityId(facilityId);
				facility.setFacilityName(facilityName);
				facility.setAssigningAuthorityName(assigningAuthorityName);
				facility.setAssigningAuthorityNumber(assigningAuthorityNumber);
				facility.setFacilityNumber(facilityNumber);
				response.getFacilities().add(facility);
			}
			// Send back response
			return response;
		} catch (final SplitterException ex) {
			throw new TransformerException(ex);
		}
	}

}
