package gov.va.nvap.server.service.pdq;

import gov.va.nvap.common.aggregator.Aggregator;
import gov.va.nvap.common.aggregator.AggregatorException;
import gov.va.nvap.common.date.hl7.HL7DateUtil;
import gov.va.nvap.common.endpoint.Endpoint;
import gov.va.nvap.common.endpoint.EndpointException;
import gov.va.nvap.common.filter.Filter;
import gov.va.nvap.common.filter.FilterException;
import gov.va.nvap.common.interceptor.Interceptor;
import gov.va.nvap.common.interceptor.InterceptorException;
import gov.va.nvap.common.jaxb.JaxbUtil;
import gov.va.nvap.common.splitter.Splitter;
import gov.va.nvap.common.splitter.SplitterException;
import gov.va.nvap.common.transformer.Transformer;
import gov.va.nvap.common.transformer.TransformerException;
import gov.va.nvap.common.transformer.xml.XMLToString;
import gov.va.nvap.common.uuid.UUIDUtil;
import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.common.validation.NullChecker;
import gov.va.nvap.privacy.ConsentDirectiveOptOutReasonType;
import gov.va.nvap.privacy.ConsentDirectiveQueryParamType;
import gov.va.nvap.privacy.ConsentDirectiveQueryRequest;
import gov.va.nvap.privacy.ConsentDirectiveQueryResponse;
import gov.va.nvap.privacy.ConsentDirectiveReferenceType;
import gov.va.nvap.privacy.ConsentDirectiveRevocationRequest;
import gov.va.nvap.privacy.ConsentType;
import gov.va.nvap.privacy.ServiceConsumer;
import gov.va.nvap.privacy.ServiceConsumerContextType;
import gov.va.nvap.privacy.data.ConsentDirectiveData;
import gov.va.nvap.service.audit.AuditService;
import gov.va.nvap.service.audit.data.ConsentAudit;
import gov.va.nvap.service.pdq.Facility;
import gov.va.nvap.service.pdq.PatientCorrelationsQuery;
import gov.va.nvap.service.pdq.PatientCorrelationsResponse;
import gov.va.nvap.service.pdq.PatientDemographics;
import gov.va.nvap.service.pdq.PatientDemographicsQuery;
import gov.va.nvap.service.pdq.PatientDemographicsResponse;
import gov.va.nvap.service.pdq.PatientMatchQuery;
import gov.va.nvap.service.pdq.PatientMatchResponse;
import gov.va.nvap.service.pdq.PdqException;
import gov.va.nvap.service.pdq.PdqService;
import gov.va.nvap.service.pdq.RemovePatientCorrelationRequest;
import gov.va.nvap.service.privacy.ConsentManagementService;
import gov.va.nvap.service.privacy.ConsentManagementServiceException;
import java.text.ParseException;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.xml.bind.JAXBException;
import net.sf.ehcache.Cache;
import net.sf.ehcache.CacheManager;
import net.sf.ehcache.Element;
import org.springframework.beans.factory.annotation.Required;
import org.w3c.dom.Document;

/**
 * The pdq preference implementation, which will be the facade for the GUI.
 * 
 * @author Asha Amritraj
 * @author Zack Peterson
 */
public class PdqServiceImpl implements PdqService {

	private static final Logger logger = Logger.getLogger(PdqServiceImpl.class
			.getName());
	private static final String PDQ_CORRELATIONS_CACHE_NAME = "gov.va.nvap.service.pdq.PdqService.Correlations";
	private static final String PDQ_DEMOGPHICS_CACHE_NAME = "gov.va.nvap.service.pdq.PdqService.Demographics";

	private static final String PDQ_PATIENT_MATCH_CACHE_NAME = "gov.va.nvap.service.pdq.PdqService.PatientMatch";
	// Cache Manager
	CacheManager cacheManager;
	// Endpoints
	private Endpoint<Document, Document> mviSocketEndpoint;

	private Transformer<String, Map<String, Object>> patientExtendedViewRpcResponseToPropertyMap;
	private Transformer<Document, Document> patientMatchQueryToQBPQ22Transformer;
	private Transformer<String, Map<String, Object>> patientPrimaryViewRpcResponseToPropertyMap;
    /**
	 * Transformer to transform from the value object to CDA R2 XML document.
	 */
	private Transformer<Document, Document> dataToConsentDirectiveDocument;
	// Interceptor
	private Interceptor<Object, Object> pdqAuditInterceptor;
	// Transformers
	private JaxbUtil pdqJaxbHelper;
    /**
	 * To convert from ConsentDirectiveData to XML for transformation into the
	 * CDA XML.
	 */
	private JaxbUtil consentDirectiveDataJaxbHelper;
	// Aggregators
	private Aggregator<Map<String, Object>, Map<String, ?>> propertyMapAggregator;
	private Transformer<Document, Document> removePatientCorrelationRequestToADTA37;
	// Splitter
	private Splitter<String, Map<String, String>> rpcExtendedResponseSplitter;
	// Filter
	private Filter<Map<String, Map<String, Object>>, String> rpcResponsesRecentDateFilter;
	private Transformer<String, PatientCorrelationsResponse> rpcResponsesToPatientCorrelationResponse;
	private Transformer<Map<String, Object>, Document> rpcResponsesToPatientDemographicsResponse;
	private Transformer<Map<String, String>, Map<String, Map<String, Object>>> rpcResponsesToPropertyMapRecordTwoPass;
	private Transformer<Document, Document> rspK22ToPatientMatchResponseTransformer;
	private Filter<Map<String, String>, Map<String, String>> vaFacilityNumberFilter;
	private Endpoint<Object, String> vistaPatientExtendedViewEndpoint;

	private Endpoint<Object, String> vistaPatientPrimaryViewEndpoint;
	//
	private XMLToString xml2String;
    
	private ConsentManagementService cms;
        
        private AuditService auditService;

	@Override
	public PatientCorrelationsResponse getCorrelatedFacilities(
			final PatientCorrelationsQuery query) throws PdqException {
		Assert.assertNotEmpty(query.getPatientId(),
				"Patient Id cannot be empty!");
		try {
			// Cache lives for 5 mins. Check the ehCache.xml for more
			// information.
			if (CacheManager.getInstance().cacheExists(
					PdqServiceImpl.PDQ_CORRELATIONS_CACHE_NAME)) {
				final Cache cache = CacheManager.getInstance().getCache(
						PdqServiceImpl.PDQ_CORRELATIONS_CACHE_NAME);
				if (cache.isKeyInCache(query.getPatientId())) {
					final Element element = cache.get(query.getPatientId());
					if (NullChecker.isNotEmpty(element)) {
						if (PdqServiceImpl.logger.isLoggable(Level.INFO)) {
							//PdqServiceImpl.logger.log(Level.INFO,
							//		"Obtaining cached correlations for patient, "
							//				+ query.getPatientId());
							PdqServiceImpl.logger.log(Level.INFO,
									"Obtaining cached correlations for patient ");
						}
						return (PatientCorrelationsResponse) element
								.getObjectValue();
					}
				}
			}
			// Execute the Extended View RPC
			final long startTime = System.currentTimeMillis();
			final String extendedRpcResponse = this.vistaPatientExtendedViewEndpoint
					.invoke(query.getPatientId());
			final long endTime = System.currentTimeMillis();
			Assert.assertNotEmpty(
					extendedRpcResponse,
					"Extended RPC Response from MPI is empty for "
							+ query.getPatientId() + "!");
			// Return response
			final PatientCorrelationsResponse response = this.rpcResponsesToPatientCorrelationResponse
					.transform(extendedRpcResponse);
			if (PdqServiceImpl.logger.isLoggable(Level.INFO)) {
				//PdqServiceImpl.logger.log(Level.INFO,
				//		"Obtaining correlations (Extended View RPC) for patient, "
				//				+ query.getPatientId());
				final List<String> facilityCoordinates = new ArrayList<String>();
				final List<Facility> facs = response.getFacilities();
				for (final Facility fac : facs) {
					facilityCoordinates.add(fac.getFacilityNumber() + ":"
							+ fac.getFacilityName());
				}
				PdqServiceImpl.logger.log(Level.INFO, "Facilities in Response["
						+ facilityCoordinates + "]. Time elapsed = "
						+ ((endTime - startTime) / 1000) + "s");
			}
			// Store in cache
			if (CacheManager.getInstance().cacheExists(
					PdqServiceImpl.PDQ_CORRELATIONS_CACHE_NAME)) {
				final Cache cache = CacheManager.getInstance().getCache(
						PdqServiceImpl.PDQ_CORRELATIONS_CACHE_NAME);
				cache.put(new Element(query.getPatientId(), response));
			}
			return response;

		} catch (final TransformerException ex) {
			throw new PdqException(ex);
		} catch (final EndpointException ex) {
			throw new PdqException(ex);
		}
	}

    /**
	 * Retrieves the demographics for the patient contained in the patient demographics
     * request. If the patient is found to be deceased, a consent revocation is processed
     * for all consent directives.
     * 
     * @param query query object containing patient for which to retrieve demographics
     * 
     * @return the response object containing the requested demographics if they exist
     * 
     * @throws gov.va.nvap.service.pdq.PdqException
	 */
	@Override
	public PatientDemographicsResponse getPatientDemographics(
			final PatientDemographicsQuery query) throws PdqException {
		Assert.assertNotEmpty(query.getPatientId(),
				"Patient identiifer cannot be null!");
		try {
			if (CacheManager.getInstance().cacheExists(
					PdqServiceImpl.PDQ_DEMOGPHICS_CACHE_NAME)) {
				final Cache cache = CacheManager.getInstance().getCache(
						PdqServiceImpl.PDQ_DEMOGPHICS_CACHE_NAME);
				if (cache.isKeyInCache(query.getPatientId())) {
					final Element element = cache.get(query.getPatientId());
					if (NullChecker.isNotEmpty(element)) {
						//if (PdqServiceImpl.logger.isLoggable(Level.INFO)) {
							//PdqServiceImpl.logger.log(Level.INFO,
							//		"Obtaining cached demographics for patient, "
							//				+ query.getPatientId());
						//}
						return (PatientDemographicsResponse) element
								.getObjectValue();
					}
				}
			}
			// Execute Primary View RPC
			long startTime = System.currentTimeMillis();
			final String primaryRpcResponse = this.vistaPatientPrimaryViewEndpoint
					.invoke(query.getPatientId());
			long endTime = System.currentTimeMillis();
			if (PdqServiceImpl.logger.isLoggable(Level.INFO)) {
				//PdqServiceImpl.logger.log(Level.INFO,
				//		"Obtaining demographics (Primary View RPC) for patient, "
				//				+ query.getPatientId() + ". Time elapsed = "
				//				+ ((endTime - startTime) / 1000) + "s");
				// logger.log(Level.INFO, "Response[" + primaryRpcResponse +
				// "]");
			}
			Assert.assertNotEmpty(
					primaryRpcResponse,
					"Primary RPC Response from MPI is empty for "
							+ query.getPatientId() + "!");
			// Create a map of properties from the primary rpc response
			final Map<String, Object> patientPrimaryViewPropertyMap = this.patientPrimaryViewRpcResponseToPropertyMap
					.transform(primaryRpcResponse);
			// Execute the Extended View RPC
			startTime = System.currentTimeMillis();
			final String extendedRpcResponse = this.vistaPatientExtendedViewEndpoint
					.invoke(query.getPatientId());
			endTime = System.currentTimeMillis();
			if (PdqServiceImpl.logger.isLoggable(Level.INFO)) {
	//			PdqServiceImpl.logger.log(Level.INFO,
	//					"Obtaining demographics (Extended View RPC) for patient, "
	//							+ query.getPatientId() + ". Time elapsed = "
	//							+ ((endTime - startTime) / 1000) + "s");
				// logger.log(Level.INFO, "Response[" + extendedRpcResponse +
				// "]");
			}
			Assert.assertNotEmpty(
					extendedRpcResponse,
					"Extended RPC Response from MPI is empty for "
							+ query.getPatientId() + "!");
			// Split the responses for each facility
			final Map<String, String> splitRpcResponses = this.rpcExtendedResponseSplitter
					.split(extendedRpcResponse);
			// Filter by known VA facility if present
			final Map<String, String> splitRpcResponsesFiltered = this.vaFacilityNumberFilter
					.filter(splitRpcResponses);

			final Map<String, Map<String, Object>> rpcResponsesMapRecord = this.rpcResponsesToPropertyMapRecordTwoPass
					.transform(splitRpcResponsesFiltered);
			// Get the most recent updated facility to work with from the
			// response
			final String extendedRecentRpcResponse = this.rpcResponsesRecentDateFilter
					.filter(rpcResponsesMapRecord);
			Assert.assertNotEmpty(
					extendedRecentRpcResponse,
					"Extended Filtered Response cannot be null for "
							+ query.getPatientId() + "!");
			// Create a map of properties from the extended rpc respnse
			final Map<String, Object> patientExtendedViewPropertyMap = this.patientExtendedViewRpcResponseToPropertyMap
					.transform(extendedRecentRpcResponse);
			// Aggregate the primary, extended and sites maps
			final Map<String, Object> patientViewPropertyMap = this.propertyMapAggregator
					.aggregate(patientExtendedViewPropertyMap,
							patientPrimaryViewPropertyMap);

			// Construct the patient demographics response return value document
			final Document patientDemographicsResponseDocument = this.rpcResponsesToPatientDemographicsResponse
					.transform(patientViewPropertyMap);
			// Unmarshal document
			final PatientDemographicsResponse patientDemographicsResponse = (PatientDemographicsResponse) this.pdqJaxbHelper
					.unmarshal(patientDemographicsResponseDocument);

            // Store demographics information in cache
			if (CacheManager.getInstance().cacheExists(
					PdqServiceImpl.PDQ_DEMOGPHICS_CACHE_NAME)) {
				final Cache cache = CacheManager.getInstance().getCache(
						PdqServiceImpl.PDQ_DEMOGPHICS_CACHE_NAME);
				cache.put(new Element(query.getPatientId(),
						patientDemographicsResponse));
			}
            
            // Opt out patient if deceased
            Date dateOfDeath = patientDemographicsResponse.getPatientDemographics().getDateOfDeath();
            
            if (dateOfDeath != null) {
                // look up the records that need to be opted out.
                ConsentDirectiveQueryResponse consentDirectiveQueryResponse = null;
                try {
                    // Construct request
                    final ConsentDirectiveQueryRequest consentDirectiveQueryRequest = 
                            new ConsentDirectiveQueryRequest();
                    // Set the patient id
                    consentDirectiveQueryRequest.setPatientId(query.getPatientId());
                    // Query only for active
                    consentDirectiveQueryRequest
                            .setQueryParam(ConsentDirectiveQueryParamType.ACTIVE);
                    consentDirectiveQueryResponse = this.cms
                            .getConsentDirectives(consentDirectiveQueryRequest);
                } catch (final ConsentManagementServiceException ex) {
                    throw new RuntimeException(ex);
                }
                
                for (ConsentDirectiveReferenceType consentDirective : 
                        consentDirectiveQueryResponse.getConsentDirectiveReference()) {
                    if (NullChecker.isEmpty(consentDirective)) {
                        throw new RuntimeException(
                                "There has to be one active consent directive to process revocation!");
                    }
                    // Create a consent directive data object
                    ConsentDirectiveData data = new ConsentDirectiveData();
                    data.setIcn(query.getPatientId());
                    PatientDemographics demographics = patientDemographicsResponse.getPatientDemographics();
                    if (NullChecker.isNotEmpty(demographics)) {
                        data.setPatientRoleCity(demographics.getResidenceCity());
                        data.setPatientRoleState(demographics.getResidenceState());
                        data.setPatientRoleGivenName(demographics.getFirstName());
                        data.setPatientRoleFamilyName(demographics.getLastName());
                        data.setPatientRoleFamilyNameAlias(demographics.getAlias1());
                        data.setPatientRoleMiddleName(demographics.getMiddleName());
                        data.setPatientRoleMiddleNameAlias(demographics.getAlias2());
                        data.setPatientRoleEthnicGroupCodeDisplayText(demographics
                                .getEthnicityDescription());
                        data.setPatientRoleSsn(demographics.getSsn());
                        data.setPatientRoleGenderCode(demographics.getGender());
                        data.setPatientRoleGenderDisplayText(demographics
                                .getGenderDescription());
                        data.setPatientRoleMaritalStatusCode(demographics
                                .getMaritalStatus());
                        data.setPatientRoleMaritalStatusDisplayText(demographics
                                .getMaritalStatusDescription());
                        data.setPatientRolePostalCode(demographics.getResidenceZip4());
                        data.setPatientRolePrefix(demographics.getPrefix());
                        data.setPatientRoleStreetAddressLine(demographics
                                .getStreetAddressLine1());
                        data.setPatientRoleSuffix(demographics.getSuffix());
                        data.setPatientRoleTelecom(demographics.getResidencePhoneNumber());
                        data.setPatientRoleProviderOrganizationName(demographics
                                .getFacilityName());
                        data.setPatientRoleProviderOrganizationNumber(demographics
                                .getFacilityNumber());
                        if (NullChecker.isNotEmpty(demographics.getDob())) {
                            try {
                                // Need to use the HL7 date, because the CDA R2 needs it
                                data.setPatientRoleDob(HL7DateUtil
                                        .yyyyMMddhhmmssZ(demographics.getDob()));
                            } catch (final ParseException ex) {
                                throw new RuntimeException(ex);
                            }
                        }
                    }
                    // Set a unique id per document
                    data.setDocId(UUIDUtil.generateUUID());

                    // Set the status code to aborted
                    data.setComponentStatusCode("aborted");
                    
                    // Set document dates and times to now
                    try {
                        final String signatureDateString = HL7DateUtil
                                .yyyyMMddhhmmssZ(new Date());
                        // Set the effective date
                        data.setEffectiveDateTime(HL7DateUtil.yyyyMMddhhmmssZ(new Date()));
                        // Create the begin and end date
                        data.setDocumentationBeginTime(signatureDateString);
                        // Start and end time are required
                        data.setDocumentationEndTime(signatureDateString);
                    } catch (final ParseException ex) {
                        throw new RuntimeException(ex);
                    }
                    
                    // Set the consent directive
                    data.setPreviousConsentDirectiveId(consentDirective.getConsentDirId());
                    data.setOptoutReason("Patient Deceased");
                    
                    // For now sets user to an automatic service
                    data.setAuthorPersonOid("Automatic Service");
                    data.setAuthorPersonOrgOid("Automatic Service");
                    
                    // Make the request to revoke
                    final ServiceConsumerContextType sc = new ServiceConsumerContextType();
                    sc.setUser("Automatic Service");
                    ConsentAudit ca = auditService.getLatestAudit(consentDirective.getPatientIen(), consentDirective.getOptinConsentType().getValue());
                    if(ca != null){
                        sc.setFacility(ca.getFacility());
                    }
                    sc.setServiceConsumerType(ServiceConsumer.ADMINISTRATOR_PORTAL);

                    if (consentDirective.getPurposeOfUse() != null) {
                        data.setComponentPurposeOfUseDisplayName(consentDirective.getPurposeOfUse().name());
                    }
                    String deAuthName = consentDirective.getOptinConsentType().name().
                            substring(0, consentDirective.getOptinConsentType().name().lastIndexOf("_")) + 
                            "_REVOCATION";
                    sc.setConsentType(ConsentType.valueOf(deAuthName));
                    
                    // Convert the PDF into byte string
                    // Convert the ConsentDirectiveData to XML document
                    final Document consentDirectiveDataDoc = this.consentDirectiveDataJaxbHelper
                            .marshal(data);
                    // Convert ConsentDirectiveData XML to CDA R2 XML
                    final Document consentDirectiveDocument = this.dataToConsentDirectiveDocument
                            .transform(consentDirectiveDataDoc);
                    // Convert CDA R2 XML to string
                    final String consentDirectiveDocumentString = 
                            this.xml2String.transform(consentDirectiveDocument);
                    final byte[] consentDirectiveDocumentBytes = consentDirectiveDocumentString.getBytes();
                    
                    final ConsentDirectiveRevocationRequest request = new ConsentDirectiveRevocationRequest();
                    request.setServiceConsumerContext(sc);
                    request.setOptoutReason(ConsentDirectiveOptOutReasonType
                            .fromValue("Patient Deceased"));
                    request.setDocument(consentDirectiveDocumentBytes);

                    try {
                        // Process revoke
                        this.cms.processConsentDirectiveRevocation(request);
                    } catch (final ConsentManagementServiceException ex) {
                        throw new RuntimeException(ex);
                    }
                }
            }
            
			return patientDemographicsResponse;
		} catch (final SplitterException ex) {
			throw new PdqException(ex);
		} catch (final AggregatorException ex) {
			throw new PdqException(ex);
		} catch (final JAXBException ex) {
			throw new PdqException(ex);
		} catch (final FilterException ex) {
			throw new PdqException(ex);
		} catch (final TransformerException ex) {
			throw new PdqException(ex);
		} catch (final EndpointException ex) {
			throw new PdqException(ex);
		}
	}

	@Override
	public RemovePatientCorrelationRequest removePatientCorrelation(
			final RemovePatientCorrelationRequest request) throws PdqException {
		Assert.assertNotEmpty(request,
				"Remove patient correlation request cannot be null!");
		Assert.assertNotEmpty(request.getCorrelatedAssigningAuthorityName(),
				"Correlated assigning authority name cannot be null!");
		Assert.assertNotEmpty(request.getCorrelatedPatientId(),
				"Correlated patient id cannot be null!");
		Assert.assertNotEmpty(request.getPatientId(), "Patient id is required!");
		try {
			// Construct ADT_A37 for the MPI.
			final Document requestDocument = this.pdqJaxbHelper
					.marshal(request);
			final Document adtA37Doc = this.removePatientCorrelationRequestToADTA37
					.transform(requestDocument);
			// Send to MVI
			this.mviSocketEndpoint.invoke(adtA37Doc);
			// Audit the request sent
			RemovePatientCorrelationRequest r = (RemovePatientCorrelationRequest)this.pdqAuditInterceptor.intercept(request);
                        
                        return r;
		} catch (final InterceptorException ex) {
			throw new PdqException(ex);
		} catch (final JAXBException ex) {
			throw new PdqException(ex);
		} catch (final TransformerException ex) {
			throw new PdqException(ex);
		} catch (final EndpointException ex) {
			throw new PdqException(ex);
		}
	}

	@Override
	public PatientMatchResponse searchPatient(
			final PatientMatchQuery patientMatchQuery) throws PdqException {
		Assert.assertNotEmpty(patientMatchQuery,
				"Patient search request cannot be null!");
		Assert.assertNotEmpty(patientMatchQuery.getPatientProfile(),
				"Patient in the patient search request cannot be null!");
		Assert.assertNotEmpty(patientMatchQuery.getPatientProfile()
				.getFirstName(), "First name cannot be null!");
		Assert.assertNotEmpty(patientMatchQuery.getPatientProfile()
				.getLastName(), "Last name is required!");
		Assert.assertNotEmpty(patientMatchQuery.getPatientProfile().getSsn(),
				"SSN is required!");

		final String firstName = patientMatchQuery.getPatientProfile()
				.getFirstName();
		final String lastName = patientMatchQuery.getPatientProfile()
				.getLastName();
		final String ssn = patientMatchQuery.getPatientProfile().getSsn();
		String cacheKey = "ssn=" + ssn + "&firstname=" + firstName
				+ "&lastname=" + lastName;
		cacheKey = cacheKey.toLowerCase();

		// Go to the MPI and match the patient using a match query and return
		// the results in Patient objects. There could be Single Match, Multiple
		// Match or No Matches with the MPI.
		try {

			// Cache lives for 5 mins. Check the ehCache.xml for more
			// information.
			if (CacheManager.getInstance().cacheExists(
					PdqServiceImpl.PDQ_PATIENT_MATCH_CACHE_NAME)) {
				final Cache cache = CacheManager.getInstance().getCache(
						PdqServiceImpl.PDQ_PATIENT_MATCH_CACHE_NAME);
				if (cache.isKeyInCache(cacheKey)) {
					final Element element = cache.get(cacheKey);
					if (NullChecker.isNotEmpty(element)) {
						//if (PdqServiceImpl.logger.isLoggable(Level.INFO)) {
							//PdqServiceImpl.logger.log(Level.INFO,
							//		"Obtaining cached matches for patient, "
							//				+ cacheKey);
						//}
						return (PatientMatchResponse) element.getObjectValue();
					}
				}
			}

			// 1. Construct a patient search request object
			// 2. Use JAXB to make patient search request object into XML
			final Document patientMatchQueryDocument = this.pdqJaxbHelper
					.marshal(patientMatchQuery);
			// 3. Get the patient search request xml document and convert it to
			// a QBP_Q22 MPI
			// query - Refer to MPI Technical Manual
			Assert.assertNotEmpty(patientMatchQueryDocument,
					"PatientMatchQuery document cannot be null!");
			final Document qbpQ22Document = this.patientMatchQueryToQBPQ22Transformer
					.transform(patientMatchQueryDocument);
			if (PdqServiceImpl.logger.isLoggable(Level.FINE)) {
				PdqServiceImpl.logger.log(
						Level.FINE,
						"Search Patient Request["
								+ (qbpQ22Document != null ? this.xml2String
										.transform(qbpQ22Document) : "null")
								+ "]");
			}
			// 4. Get the RSP_K22 xml document from the MPI
			Assert.assertNotEmpty(qbpQ22Document,
					"QBP_Q22 document cannot be null!");
			final long startTime = System.currentTimeMillis();
			final Document rspK22Document = this.mviSocketEndpoint
					.invoke(qbpQ22Document);
			final long endTime = System.currentTimeMillis();
			if (PdqServiceImpl.logger.isLoggable(Level.FINE)) {
				PdqServiceImpl.logger.log(
						Level.FINE,
						"Search Patient Response["
								+ (rspK22Document != null ? this.xml2String
										.transform(rspK22Document) : "null")
								+ "]" + ". Time elapsed = "
								+ ((endTime - startTime) / 1000) + "s");
			}
			// 5. Convert from RSP_K22 to Patient search response xml document
			Assert.assertNotEmpty(rspK22Document,
					"RSP_K22 document cannot be null!");

			final Document patientMatchResponseDocument = this.rspK22ToPatientMatchResponseTransformer
					.transform(rspK22Document);
			Assert.assertNotEmpty(patientMatchResponseDocument,
					"PatientMatchResponse document cannot be null!");
			// 6. Use JAXB to unmarshall the patient search response xml
			// document into a java object
			final PatientMatchResponse patientMatchResponse = (PatientMatchResponse) this.pdqJaxbHelper
					.unmarshal(patientMatchResponseDocument);
			Assert.assertNotEmpty(patientMatchResponse,
					"PatientMatchResponse object cannot be null!");

			// Store in cache
			if (CacheManager.getInstance().cacheExists(
					PdqServiceImpl.PDQ_PATIENT_MATCH_CACHE_NAME)) {
				final Cache cache = CacheManager.getInstance().getCache(
						PdqServiceImpl.PDQ_PATIENT_MATCH_CACHE_NAME);
				cache.put(new Element(cacheKey, patientMatchResponse));
			}
			// 7. Return the patient search response
			return patientMatchResponse;
		} catch (final EndpointException ex) {
			throw new PdqException(ex);
		} catch (final TransformerException ex) {
			throw new PdqException(ex);
		} catch (final JAXBException ex) {
			throw new PdqException(ex);
		}
	}

	@Required
	public void setCacheManager(final CacheManager cacheManager) {
		this.cacheManager = cacheManager;
	}

	@Required
	public void setMviSocketEndpoint(
			final Endpoint<Document, Document> mviSocketEndpoint) {
		this.mviSocketEndpoint = mviSocketEndpoint;
	}

	@Required
	public void setPatientExtendedViewRpcResponseToPropertyMap(
			final Transformer<String, Map<String, Object>> patientExtendedViewRpcResponseToPropertyMap) {
		this.patientExtendedViewRpcResponseToPropertyMap = patientExtendedViewRpcResponseToPropertyMap;
	}

	@Required
	public void setPatientMatchQueryToQBPQ22Transformer(
			final Transformer<Document, Document> patientMatchQueryToQBPQ22Transformer) {
		this.patientMatchQueryToQBPQ22Transformer = patientMatchQueryToQBPQ22Transformer;
	}

	@Required
	public void setPatientPrimaryViewRpcResponseToPropertyMap(
			final Transformer<String, Map<String, Object>> patientPrimaryViewRpcResponseToPropertyMap) {
		this.patientPrimaryViewRpcResponseToPropertyMap = patientPrimaryViewRpcResponseToPropertyMap;
	}

	@Required
	public void setPdqAuditInterceptor(
			final Interceptor<Object, Object> pdqAuditInterceptor) {
		this.pdqAuditInterceptor = pdqAuditInterceptor;
	}

	@Required
	public void setPdqJaxbHelper(final JaxbUtil pdqJaxbHelper) {
		this.pdqJaxbHelper = pdqJaxbHelper;
	}

	@Required
	public void setPropertyMapAggregator(
			final Aggregator<Map<String, Object>, Map<String, ?>> propertyMapAggregator) {
		this.propertyMapAggregator = propertyMapAggregator;
	}

	@Required
	public void setRemovePatientCorrelationRequestToADTA37(
			final Transformer<Document, Document> removePatientCorrelationRequestToADTA37) {
		this.removePatientCorrelationRequestToADTA37 = removePatientCorrelationRequestToADTA37;
	}

	@Required
	public void setRpcExtendedResponseSplitter(
			final Splitter<String, Map<String, String>> rpcExtendedResponseSplitter) {
		this.rpcExtendedResponseSplitter = rpcExtendedResponseSplitter;
	}

	@Required
	public void setRpcResponsesRecentDateFilter(
			final Filter<Map<String, Map<String, Object>>, String> rpcResponsesRecentDateFilter) {
		this.rpcResponsesRecentDateFilter = rpcResponsesRecentDateFilter;
	}

	@Required
	public void setRpcResponsesToPatientCorrelationResponse(
			final Transformer<String, PatientCorrelationsResponse> rpcResponsesToPatientCorrelationResponse) {
		this.rpcResponsesToPatientCorrelationResponse = rpcResponsesToPatientCorrelationResponse;
	}

	@Required
	public void setRpcResponsesToPatientDemographicsResponse(
			final Transformer<Map<String, Object>, Document> rpcResponsesToPatientDemographicsResponse) {
		this.rpcResponsesToPatientDemographicsResponse = rpcResponsesToPatientDemographicsResponse;
	}

	@Required
	public void setRpcResponsesToPropertyMapRecordTwoPass(
			final Transformer<Map<String, String>, Map<String, Map<String, Object>>> rpcResponsesToPropertyMapRecordTwoPass) {
		this.rpcResponsesToPropertyMapRecordTwoPass = rpcResponsesToPropertyMapRecordTwoPass;
	}

	@Required
	public void setRspK22ToPatientMatchResponseTransformer(
			final Transformer<Document, Document> rspK22ToPatientMatchResponseTransformer) {
		this.rspK22ToPatientMatchResponseTransformer = rspK22ToPatientMatchResponseTransformer;
	}

	@Required
	public void setVaFacilityNumberFilter(
			final Filter<Map<String, String>, Map<String, String>> vaFacilityNumberFilter) {
		this.vaFacilityNumberFilter = vaFacilityNumberFilter;
	}

	@Required
	public void setVistaPatientExtendedViewEndpoint(
			final Endpoint<Object, String> vistaPatientExtendedViewEndpoint) {
		this.vistaPatientExtendedViewEndpoint = vistaPatientExtendedViewEndpoint;
	}

	@Required
	public void setVistaPatientPrimaryViewEndpoint(
			final Endpoint<Object, String> vistaPatientPrimaryViewEndpoint) {
		this.vistaPatientPrimaryViewEndpoint = vistaPatientPrimaryViewEndpoint;
	}

	@Required
	public void setXml2String(final XMLToString xml2String) {
		this.xml2String = xml2String;
	}
    
    @Required
    public void setDataToConsentDirectiveDocument(Transformer<Document, Document> dataToConsentDirectiveDocument) {
        this.dataToConsentDirectiveDocument = dataToConsentDirectiveDocument;
    }

    @Required
    public void setConsentDirectiveDataJaxbHelper(JaxbUtil consentDirectiveDataJaxbHelper) {
        this.consentDirectiveDataJaxbHelper = consentDirectiveDataJaxbHelper;
    }

    @Required
    public void setCms(ConsentManagementService cms) {
        this.cms = cms;
    }
    @Required
	public void setAuditService(final AuditService auditService) {
		this.auditService = auditService;
	}
}
