package gov.va.nvap.server.pdq.interceptor;

import gov.va.nvap.common.interceptor.Interceptor;
import gov.va.nvap.common.interceptor.InterceptorException;
import gov.va.nvap.common.transformer.Transformer;
import gov.va.nvap.common.transformer.TransformerException;
import gov.va.nvap.common.validation.Assert;
import gov.va.nvap.service.audit.AuditException;
import gov.va.nvap.service.audit.AuditService;
import gov.va.nvap.service.audit.data.ConsentAudit;
import gov.va.nvap.service.pdq.RemovePatientCorrelationRequest;

import org.springframework.beans.factory.annotation.Required;

/**
 * Audit interceptor to audit the information from the patient demographics
 * service.
 * 
 * @author Asha Amritraj
 * 
 */
public class PdqAuditInterceptor implements Interceptor<Object, Object> {

	/**
	 * The reference to the audit server.
	 */
	AuditService auditService;
	/**
	 * Transforms from the RemovePatientsCorrelationRequest to ConsentAudit.
	 */
	Transformer<RemovePatientCorrelationRequest, ConsentAudit> transformer;

	@Override
	public Object intercept(final Object object) throws InterceptorException {
		Assert.assertNotEmpty(object, "Request cannot be empty!");

		if (RemovePatientCorrelationRequest.class.isInstance(object)) {
			final RemovePatientCorrelationRequest request = (RemovePatientCorrelationRequest) object;
			try {
				// Transform & Audit
				final ConsentAudit audit = this.transformer.transform(request);
				this.auditService.auditConsent(audit);

			} catch (final TransformerException ex) {
				throw new InterceptorException(ex);
			} catch (final AuditException ex) {
				throw new InterceptorException(ex);
			}
		} else {
			throw new RuntimeException(
					"Only RemovePatientCorrelationRequest is supported!");
		}
		return object;
	}

	@Required
	public void setAuditService(final AuditService auditService) {
		this.auditService = auditService;
	}

	@Required
	public void setTransformer(
			final Transformer<RemovePatientCorrelationRequest, ConsentAudit> transformer) {
		this.transformer = transformer;
	}
}
